"use strict";
this.name = "Smugglers_Equipment";
this.author = "phkb";
this.description = "Looks after the smuggling compartment functions.";
this.licence = "CC BY-NC-SA 3.0";

/* TODO:
- Add very occasional messages from traders entering system ("Let hope that setting of 234 is worth the money we paid.")
- When docking with a standard contract that will be completed, but the cargo is now illegal, still give the player the opportunity to bribe
- allow non-core commodities to be put in the compartment

ISSUE:
- bribe opportunity not appearing in scenario where player docks at non-main station galcop station, with a scooped escape pod and illegal cargo (not slaves)

*/

this._disabled = false;
this._debug = false;

// array of all possible installed compartment equipment keys
this._equipmentList = ["EQ_SMUGGLING_COMPARTMENT_1", "EQ_SMUGGLING_COMPARTMENT_2", "EQ_SMUGGLING_COMPARTMENT_3", "EQ_SMUGGLING_COMPARTMENT_4",
	"EQ_SMUGGLING_COMPARTMENT_5", "EQ_SMUGGLING_COMPARTMENT_6", "EQ_SMUGGLING_COMPARTMENT_7", "EQ_SMUGGLING_COMPARTMENT_8",
	"EQ_SMUGGLING_COMPARTMENT_9", "EQ_SMUGGLING_COMPARTMENT_10", "EQ_SMUGGLING_COMPARTMENT_11", "EQ_SMUGGLING_COMPARTMENT_12",
	"EQ_SMUGGLING_COMPARTMENT_13", "EQ_SMUGGLING_COMPARTMENT_14", "EQ_SMUGGLING_COMPARTMENT_15", "EQ_SMUGGLING_COMPARTMENT_16",
	"EQ_SMUGGLING_COMPARTMENT_17", "EQ_SMUGGLING_COMPARTMENT_18", "EQ_SMUGGLING_COMPARTMENT_19", "EQ_SMUGGLING_COMPARTMENT_20"
];

this._commodityIDList = [];

// station types where the smuggling compartment can be purchased.
this._smugglingAllegiance = ["pirate", "chaotic", "neutral"];
this._maxTonnage = 20; // maximum possible size of the hold
this._doCompartmentCheck = false; // flag to indicate that a compartment check will be performed after docking
this._backdoor = false;

this._sc_Phase = 1; // the current phase setting (value between 1 and 999)
this._sc_TechVersion = 1; // the current tech level of the compartment
this._sc_VisibleAs = "food"; // string (commodity name) that the hold will display as for scanners
this._sc_Cargo = []; // array of commodities in hold (commodity:string, quantity:number, unit:t/kg/g);
this._sc_Days = 0; // number of days since the last tech upgrade
this._phaseUpdateDays = 0;
this._phaseScanDiscovery = []; // array of objects of discovered phase scan settings (from purchase or from phase scanner)

this._smugglingScreen = false; // flag to indicate that a mission screen is open
this._originalCredits = 0; // value to hold the players credits (used to refund the cost of the initial item)
this._display = 0;
this._oldday = 0;

this._phaseScannerAvailable = false; // indicates that this system has the phase scanner available
this._phaseScannerDetected = false; // indicates that the player was caught using the phase scanner
this._systemPhase = []; // array of objects (gov, tl, phase)
this._phaseUpdates = []; // stores list of governments/techLevels that have been updated
this._toRemove = []; // list of source and commodities to be removed
this._removeCompartment = false; // flag to indicate that the smuggling compartment should be removed after docking
this._playerFine = 0; // amount to fine the player
this._menuColor = "orangeColor";
this._itemColor = "yellowColor";
this._disabledColor = "darkGrayColor";
this._galJump = false;
this._policeModel = "";
this._NCInstalled = false;
this._IGTInstalled = false;
this._doCargoCheck = false;
this._dockingWithSlaves = false;
this._dockingWithSlavesCount = 0;

//-------------------------------------------------------------------------------------------------------------
this.startUp = function () {
	if (this._disabled) {
		delete this.playerWillSaveGame;
		delete this.playerBoughtEquipment;
		delete this.shipWillDockWithStation;
		delete this.shipDockedWithStation;
		delete this.missionScreenOpportunity;
		delete this.shipExitedWitchspace;
		delete this.shipLaunchedEscapePod;
		delete this.guiScreenChanged;
		delete this.guiScreenWillChange;
		delete this.playerEnteredNewGalaxy;
		delete this.playerRescuedEscapePod;
		delete this.dayChanged;
		delete this.equipmentDamaged;
		delete this.equipmentRemoved;
		delete this.startUpComplete;
		delete this.startUp;
		return;
	}
	this.$createSystemDataArray();
	if (worldScripts.CargoTypeExtension) this._NCInstalled = true;
}

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {
	function compare(a, b) {
		if (a.order < b.order) return -1;
		if (a.order > b.order) return 1;
		return 0;
	}

	// check for illegal goods tweak
	if (worldScripts["illegal_goods_tweak"]) this._IGTInstalled = true;

	// build a list of available commodities
	var m = null;
	// find a station to get market data from
	if (system.mainStation) {
		m = system.mainStation.market;
	} else if (player.ship.dockedStation) {
		m = playership.dockedStation.market;
	} else if (system.stations.length > 0) {
		m = system.stations[0].market;
	}
	if (m) {
		var sbd = worldScripts.Smugglers_DockMaster;
		var c = Object.keys(m);
		var list = [];
		for (var i = 0; i < c.length; i++) {
			list.push({
				commodity: c[i],
				order: (m[c[i]].sort_order ? m[c[i]].sort_order : 0)
			});
			// add any non-t commodities (kg or g, ie gold, platinum, gemstones) to the dockmaster noswitch list
			if (m[c[i]].quantity_unit != 0) sbd._noSwitch.push(c[i]);
		}
		list.sort(compare);
		for (var i = 0; i < list.length; i++) {
			this._commodityIDList.push(list[i].commodity);
		}
	}

	var core = worldScripts.Smugglers_CoreFunctions;
	this.$padTextRight = core.$padTextRight;
	this.$padTextLeft = core.$padTextLeft;
	this.$isBigGuiActive = core.$isBigGuiActive;
	this.$rand = core.$rand;

	if (missionVariables.SmugglingCompartment_Phase) this._sc_Phase = missionVariables.SmugglingCompartment_Phase;
	if (missionVariables.SmugglingCompartment_TechVersion) this._sc_TechVersion = missionVariables.SmugglingCompartment_TechVersion;
	if (missionVariables.SmugglingCompartment_VisibleAs) this._sc_VisibleAs = missionVariables.SmugglingCompartment_VisibleAs;
	if (missionVariables.SmugglingCompartment_Cargo) this._sc_Cargo = JSON.parse(missionVariables.SmugglingCompartment_Cargo);
	this.$addExtraField(); // make sure the "extra" property is added if it's not there yet
	if (missionVariables.SmugglingCompartment_Days) this._sc_Days = missionVariables.SmugglingCompartment_Days;
	if (missionVariables.Smuggling_SystemPhase) this._systemPhase = JSON.parse(missionVariables.Smuggling_SystemPhase);
	if (missionVariables.Smuggling_PhaseScannerAvailable) this._phaseScannerAvailable = (missionVariables.Smuggling_PhaseScannerAvailable === 1 ? true : false);
	if (missionVariables.Smuggling_PhaseDiscovery) this._phaseScanDiscovery = JSON.parse(missionVariables.Smuggling_PhaseDiscovery);
	if (missionVariables.Smuggling_PhaseUpdates) this._phaseUpdates = JSON.parse(missionVariables.Smuggling_PhaseUpdates);
	if (missionVariables.Smuggling_PhaseUpdateDays) this._phaseUpdateDays = missionVariables.Smuggling_PhaseUpdateDays;

	delete missionVariables.SmugglingCompartment_Phase;
	delete missionVariables.SmugglingCompartment_TechVersion;
	delete missionVariables.SmugglingCompartment_VisibleAs;
	delete missionVariables.SmugglingCompartment_Cargo;

	if (this._systemPhase.length === 0) this.$createSystemPhase();

	this._policeModel = this.$getPoliceModel();

	// set up the interface screen, if required
	this.$initInterface(player.ship.dockedStation);

	// check that the inflight eq is given to the player if they have a smuggling compartment
	if (this.$hasSmugglingCompartment() === true) {
		player.ship.awardEquipment("EQ_SMUGGLING_INFLIGHT_STORAGE");
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function () {
	if (this.$hasSmugglingCompartment() === true) {
		missionVariables.SmugglingCompartment_Phase = this._sc_Phase;
		missionVariables.SmugglingCompartment_TechVersion = this._sc_TechVersion;
		missionVariables.SmugglingCompartment_VisibleAs = this._sc_VisibleAs;
		missionVariables.SmugglingCompartment_Cargo = JSON.stringify(this._sc_Cargo);
		missionVariables.SmugglingCompartment_Days = this._sc_Days;
	} else {
		delete missionVariables.SmugglingCompartment_Phase;
		delete missionVariables.SmugglingCompartment_TechVersion;
		delete missionVariables.SmugglingCompartment_VisibleAs;
		delete missionVariables.SmugglingCompartment_Cargo;
		delete missionVariables.SmugglingCompartment_Days;
	}
	missionVariables.Smuggling_SystemPhase = JSON.stringify(this._systemPhase);
	missionVariables.Smuggling_PhaseScannerAvailable = (this._phaseScannerAvailable ? 1 : 0);
	missionVariables.Smuggling_PhaseDiscovery = JSON.stringify(this._phaseScanDiscovery);
	if (this._phaseUpdates.length > 0) missionVariables.Smuggling_PhaseUpdates = JSON.stringify(this._phaseUpdates);
	missionVariables.Smuggling_PhaseUpdateDays = this._phaseUpdateDays;
}

//-------------------------------------------------------------------------------------------------------------
this.playerBoughtEquipment = function (equipmentKey) {
	var p = player.ship;

	// repairing the phase scanner
	if (equipmentKey === "EQ_PHASE_SCANNER_REPAIR") {
		p.removeEquipment(equipmentKey);
		p.setEquipmentStatus("EQ_PHASE_SCANNER", "EQUIPMENT_OK");
		return;
	}

	// repairing the smuggling compartment
	if (equipmentKey.indexOf("EQ_SMUGGLING_COMPARTMENT_REPAIR") >= 0) {
		p.removeEquipment(equipmentKey);
		var eq = this.$getSmugglingCompartmentEquipment();
		p.setEquipmentStatus(eq, "EQUIPMENT_OK");
		return;
	}

	var stn = p.dockedStation;

	// removing the phase scanner
	if (equipmentKey === "EQ_PHASE_SCANNER_REMOVAL") {
		p.removeEquipment(equipmentKey);
		if (stn.allegiance != "galcop") {
			// give the player a refund if we're not at a galcop station
			var tradeinfactor = 1.0;
			// less refund for damaged scanner
			if (p.equipmentStatus("EQ_PHASE_SCANNER") != "EQUIPMENT_OK") tradinfactor = 0.3;

			var oldEqInfo = EquipmentInfo.infoForKey("EQ_PHASE_SCANNER");
			player.credits += ((oldEqInfo.price / 10) * stn.equipmentPriceFactor) * tradeinfactor;
		}
		p.removeEquipment("EQ_PHASE_SCANNER");
	}

	// removing the smuggling compartment
	if (equipmentKey === "EQ_SMUGGLING_COMPARTMENT_REMOVAL") {
		p.removeEquipment(equipmentKey);
		var eqKey = this.$getSmugglingCompartmentEquipment();
		// do a refund, but only if we're not at a galcop station, and only if the existing equipment is undamaged
		if (stn.allegiance != "galcop" && eqKey != "") {
			var tradeinfactor = 1.0;
			// less refund for damaged compartment
			if (p.equipmentStatus(eqKey) != "EQUIPMENT_OK") tradinfactor = 0.3;

			var oldEqInfo = EquipmentInfo.infoForKey(eqKey);
			player.credits += ((oldEqInfo.price / 10) * stn.equipmentPriceFactor) * tradeinfactor;
		}
		if (p.equipmentStatus("EQ_SMUGGLING_PHASE_ADJUSTMENT") === "EQUIPMENT_OK") {
			p.removeEquipment("EQ_SMUGGLING_PHASE_ADJUSTMENT");
		}
		player.ship.removeEquipment("EQ_SMUGGLING_INFLIGHT_STORAGE");
		this._backdoor = true;
		if (eqKey != "") p.removeEquipment(eqKey);
		this._backdoor = false;

		this.$resetCompartmentInfo(stn);
	}

	// buying a new (or expanding/contracting an existing) smuggling compartment
	if (equipmentKey === "EQ_SMUGGLING_COMPARTMENT") {
		p.removeEquipment(equipmentKey);
		// give the player back the original amount, in case they decide not to purchase
		if (this._originalCredits != 0) {
			player.credits = this._originalCredits;
			this._originalCredits = player.credits;
		}
		// mission screen to ask how many TC to allocate
		this._hudHidden = player.ship.hudHidden;
		this.$showPurchasePage();
	}

	if (equipmentKey === "EQ_SMUGGLING_VERSION_UPGRADE") {
		p.removeEquipment(equipmentKey);
		this._sc_TechVersion += 1;
		this._sc_Days = 0;
	}

	if (stn.allegiance === "galcop" && equipmentKey === "EQ_RENOVATION") {
		// is the phase scanner on board
		var msgtype = 0;
		if (p.equipmentStatus("EQ_PHASE_SCANNER") === "EQUIPMENT_OK" || p.equipmentStatus("EQ_PHASE_SCANNER") === "EQUIPMENT_DAMAGED") {
			// oh dear - caught!
			p.setBounty(player.bounty + (this.$rand(10) + 10), "illegal equipment");
			msgtype += 1;
		}
		// is the smuggling compartment installed?
		if (this.$hasSmugglingCompartment() === true) {
			msgtype += 2;
			// increase bounty
			p.setBounty(player.bounty + (this.$rand(10) + 10), "illegal equipment");
		}
		var msgBody = "";
		switch (msgtype) {
			case 0:
				break;
			case 1:
				msgBody = expandDescription("[illegal-equipment-1]");
				break;
			case 2:
				msgBody = expandDescription("[illegal-equipment-2]", {
					size: this.$smugglingCompartmentSize()
				});
				break;
			case 3:
				msgBody = expandDescription("[illegal-equipment-3]", {
					size: this.$smugglingCompartmentSize()
				});
				break;
		}

		if (msgBody != "") {
			var curChoices = {};
			if (player.credits > 0) {
				curChoices["01_BRIBE"] = {
					text: "[illegal-attempt-bribe]",
					color: this._menuColor
				};
			}
			curChoices["98_ACCEPT"] = {
				text: "[illegal-accept-penalty]",
				color: this._menuColor
			};

			this._bribeType = 2;
			var opts = {
				titleKey: "smuggling_illegal_equipment",
				message: msgBody,
				model: "[" + this._policeModel + "]",
				allowInterrupt: false,
				exitScreen: "GUI_SCREEN_EQUIP_SHIP",
				choices: curChoices,
				initialChoicesKey: "98_ACCEPT"
			};

			mission.runScreen(opts, this.$bribeScreenHandler, this);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
/*this.equipmentRemoved = function(equipmentKey) {
	if (this._backdoor === true) return;
	if (this._equipmentList.indexOf(equipmentKey) >= 0) {
		this.$resetCompartmentInfo(player.ship.dockedStation);
		return;
	}
}*/

//-------------------------------------------------------------------------------------------------------------
this.shipWillDockWithStation = function (station) {
	this._policeModel = this.$getPoliceModel();

	this._savedBounty = player.bounty;

	// grab a snapshot of the players illegal cargo
	this._dockIllegal = [];
	this._dockingWithSlaves = false;
	// if the player has slaves on board, and IGT is installed, then turn on the flag to indicate we're docking with slaves
	var escapePods = 0;
	if (worldScripts.EscapePodTweaks) {
		// if the _escapePods array have data, what means the escape pod tweaks "shipWillDOckWithStation" function hasn't been executed yet
		// in that case we need to know how many slaves to ignore
		escapePods = worldScripts.EscapePodTweaks._escapePods.length;
		// if its been run already, the value will be 0 anyway
	}
	if (player.ship.manifest["slaves"] && (player.ship.manifest["slaves"] - escapePods) > 0 && this._IGTInstalled === true) {
		this._dockingWithSlaves = true;
		// make a note of the slaves count, because if we dock with rescued escape pods, the number will have to be reduced during "playerRescuedEscapePod";
		this._dockingWithSlavesCount = (player.ship.manifest["slaves"] - escapePods);
	}

	this._playerHadIllegals = false;
	if (this.$playerHasIllegalCargo(station) === true) {
		var p = player.ship;
		var m = p.manifest;
		this._dockIllegal = [];
		for (var i = 0; i < m.list.length; i++) {
			if (m.list[i].quantity > 0 && station.market[m.list[i].commodity].legality_import > 0) this._dockIllegal.push({
				commodity: m.list[i].commodity,
				quantity: m.list[i].quantity
			});
		}
		this._playerHadIllegals = true;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipDockedWithStation = function (station) {
	if (station.allegiance === "galcop") {
		if (this._dockingWithSlaves === false) this._doCargoCheck = true;
	}
	// set up smuggling interface screen
	this.$initInterface(station);
}

//-------------------------------------------------------------------------------------------------------------
this.shipLaunchedEscapePod = function (escapepod) {
	// clear out smugglers compartment when escape pod is used
	this._sc_Cargo = [];
}

//-------------------------------------------------------------------------------------------------------------
this.missionScreenOpportunity = function () {
	// if IGT is installed, let it run first. it will call the performCargoCheck when it's finished.
	if (this._doCargoCheck === true) {
		this.$performCargoCheck();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerRescuedEscapePod = function (fee, reason, occupant) {
	if (this._dockingWithSlavesCount > 0) {
		this._dockingWithSlavesCount -= 1;
		if (this._dockingWithSlavesCount === 0) this._dockingWithSlaves = false;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipExitedWitchspace = function () {
	if (this._galJump === true) {
		this._galJump = false;
		this.$createSystemPhase();
		this.$createSystemDataArray();
	}
	this._phaseScannerAvailable = false;
	this._phaseScannerDetected = false;
	if (system.government === 0 && system.info.techlevel >= 8) {
		// will this system provide a phase scanner?
		if (system.scrambledPseudoRandomNumber(system.ID) > 0.6) this._phaseScannerAvailable = true;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenWillChange = function (to, from) {
	if (to === "GUI_SCREEN_MANIFEST") this.$updateManifest();
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged = function (to, from) {
	if (guiScreen === "GUI_SCREEN_EQUIP_SHIP") {
		this._originalCredits = player.credits;
		this._savedBounty = player.bounty;
	}
	if (from === "GUI_SCREEN_MISSION" && this._smugglingScreen) {
		this._smugglingScreen = false;
		if (this._hudHidden === false && player.ship.hudHidden === true) player.ship.hudHidden = this._hudHidden;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerEnteredNewGalaxy = function (galaxyNumber) {
	this._systemPhase = [];
	this._phaseUpdates = [];
	this._phaseScanDiscovery = [];
	this._galJump = true;
}

//-------------------------------------------------------------------------------------------------------------
this.dayChanged = function (newday) {
	if (newday != this._oldday) {
		this._sc_Days += 1;
		this._oldday = newday;
		this._phaseUpdateDays += 1;
		if (this._phaseUpdateDays >= 10) {
			this._phaseUpdateDays = 0;
			var item = this._systemData[this.$rand(this._systemData.length - 1)];
			var gov = parseInt(item / 100);
			var tl = item - gov;
			var newPhase = this.$rand(999);
			// remove from the update list
			for (var i = this._phaseUpdates.length - 1; i >= 0; i--) {
				if (this._phaseUpdates[i].gov === gov && this._phaseUpdates[i].tl === tl) {
					this._phaseUpdates.splice(i, 1);
				}
			}
			this.$updatePhase(gov, tl, newPhase);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.equipmentDamaged = function (equipment) {
	if (this._equipmentList.indexOf(equipment) >= 0) {
		// work out how many items will be damaged
		var remove = this.$rand(3);
		var report = [];
		var count = 0;
		do {
			for (var i = 0; i < this._sc_Cargo.length; i++) {
				if (count < remove && this._sc_Cargo[i].quantity > 0) {
					// nc checks
					if (this._sc_Cargo[i].extra != "") {
						var dta = [];
						log(this.name, "pre: " + this._sc_Cargo[i].extra);
						dta = this._sc_Cargo[i].extra.split("/");
						if ((this._sc_Cargo[i].quantity - dta.length) == 0) {
							dta.shift();
							this._sc_Cargo[i].extra = dta.join("/");
							log(this.name, "post: " + this._sc_Cargo[i].extra);
						}
					}
					this._sc_Cargo[i].quantity -= 1;
					report.push(this._sc_Cargo[i].displayName + expandMissionText("smuggling_cargo_destroyed"));
					count += 1;
				}
			}
		} while (count < remove && this._sc_Cargo.length > 0);
		this.$cargoCleanUp();
		// give the player the bad news
		if (report.length > 0) {
			for (var i = 0; i < report.length; i++) {
				player.consoleMessage(report[i]);
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$addExtraField = function () {
	if (this._sc_Cargo.length == 0) return;
	for (var i = 0; i < this._sc_Cargo.length; i++) {
		if (!this._sc_Cargo[i].hasOwnProperty("extra")) this._sc_Cargo[i].extra = "";
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$performCargoCheck = function $performCargoCheck() {
	var msgtype = 0;
	var p = player.ship;

	this._toRemove = [];
	this._removeCompartment = false;
	this._playerFine = 0;

	if (this._doCargoCheck === true) {
		this._doCargoCheck = false;

		if (this._phaseScannerDetected === true) {
			if (player.credits > 1000) {
				this._playerFine += 1000;
			} else {
				this._playerFine = player.credits;
			}
			// player has already had a legal penalty when the scan was detected, so we don't need to do it again here
		}

		// is our compartment damaged (and therefore visible?
		if (this.$hasSmugglingCompartment() === true && this.$isCompartmentDamaged() === true) {
			msgtype = 1;
			this._removeCompartment = true;
			// increase bounty
			p.setBounty(player.bounty + (this.$rand(10) + 10), "illegal equipment");
			if (player.credits > 1000) {
				this._playerFine += 1000;
			} else {
				this._playerFine = player.credits;
			}
		}

		// get the list of illegal goods for this system
		var si = worldScripts.Smugglers_Illegal;
		var goods = si.$illegalGoodsList(system.ID, true);
		// if IGT is installed, add slaves onto the list of illegal goods
		if (this._IGTInstalled) {
			goods.push({
				systemID: system.ID,
				commodity: "slaves",
				started: 0,
				end: 0,
				description: "",
				permit: 0,
				scale: 1
			});
		}

		if (msgtype === 0) {
			for (var i = 0; i < goods.length; i++) {
				// if the visibleAs property equal to an illegal good on this planet? oh dear...it's not going to matter what the chance is...
				if (this.$hasSmugglingCompartment() === true && goods[i].commodity.indexOf(this._sc_VisibleAs) >= 0 && si.$playerHasPermit(system.ID, goods[i].commodity, true) === false) {
					msgtype = 2;
					this._removeCompartment = true;
					// increase bounty
					p.setBounty(player.bounty + (this.$rand(10) + 10), "illegal equipment");
					// fine player
					if (player.credits > 1000) {
						this._playerFine += 1000;
					} else {
						this._playerFine = player.credits;
					}
				}
			}
		}

		// now, goods in the hold that are illegal to import should now automatically fine the player with a bounty increase.
		// so we shouldn't have to worry about bounty from this point on.
		// what we will do here is remove cargo and possibly fine the player a monetary amount

		// base chance is 26% (for tech level 1), but increasing with a systems tech level, to a max of 40% for a tl 15 system
		var base = 25 + system.info.techlevel;
		var phase_diff = base;
		// work out the chance of detection
		// check for phase adjuster
		if (p.equipmentStatus("EQ_SMUGGLING_PHASE_ADJUSTMENT") === "EQUIPMENT_OK") {
			// if we've got the phase adjuster, there's a small boost the the base value, from 22% to 37%
			base = 22 + system.info.techlevel;
			phase_diff = Math.abs(this.$getSystemPhase(system.ID) - this._sc_Phase) - 3;
			// if we're within +/- 3 then it's good enough
			if (phase_diff < 0) phase_diff = 0;
			// anything over base is considered inconsequential
			if (phase_diff > base) phase_diff = base;
		}
		var chance = phase_diff / 100;

		if (this._debug) log(this.name, "Smuggling compartment init chance: " + chance);

		// now apply tech level offset. could be 0 through to an extra 12.5% (30/240 = 0.125)
		chance += (this._sc_Days > 30 ? 30 : this._sc_Days) / 240;

		if (this._debug) log(this.name, "Smuggling compartment final chance: " + chance);

		var s_cargo = this.$getSmugglingCargo();
		var msg = "";
		var permit_failed = false;
		var permit_list = "";
		var permit_multi = 0;
		var permit_ok = "";
		var permit_okcount = 0;
		var displaycheck = false;
		var removecount = 0;
		var successcount = 0;
		var successmsg = "";

		var smuggling_found = false;
		// do this here so we only do it once
		if (Math.random() < chance) smuggling_found = true;

		for (var i = 0; i < goods.length; i++) {
			// are there any illegal goods in our standard hold

			// get a list of commodities (handles case where there are multiple commdoties in the illegal goods entry
			var cmdtylist = goods[i].commodity.split(",");
			var quantity_to_check = 0;

			for (var j = 0; j < cmdtylist.length; j++) {
				// only check if the good is actually marked as illegal
				if (this.$commodityIllegalScale(cmdtylist[j]) > 0) {
					if (p.manifest[cmdtylist[j]] > 0) {
						quantity_to_check += p.manifest[cmdtylist[j]];
					}
					var hyper = this.$getHyperCargoQuantity(cmdtylist[j]);
					if (hyper > 0) {
						quantity_to_check += hyper;
					}
					for (var k = 0; k < s_cargo.length; k++) {
						if (cmdtylist[j] === s_cargo[k].commodity && smuggling_found === true) {
							quantity_to_check += s_cargo[k].quantity;
						}
					}
				}
			}

			if (quantity_to_check > 0) {
				if (si.$playerHasPermit(system.ID, goods[i].commodity, true) === false) {
					// will get here if the player had a permit but it didn't work, or if they didn't have a permit at all.

					// check if a permit permit failed to work
					if (si.$playerHasPermit(system.ID, goods[i].commodity, false) === true) {
						permit_failed = true;
						permit_list += (permit_list === "" ? "" : ", ") + si.$translateCommodityList(goods[i].commodity);
						permit_multi += 1;

						if (goods[i].commodity.indexOf(",") >= 0) permit_multi += 1;

						// remove any permit whether it worked or not
						si.$removePermit(system.ID, goods[i].commodity);
					}

					for (var j = 0; j < cmdtylist.length; j++) {
						// check the visible carge
						if (p.manifest[cmdtylist[j]] > 0) {
							// found!
							msg += " - " + p.manifest[cmdtylist[j]] + " " + this.$getCommodityType(cmdtylist[j]) + " × " + displayNameForCommodity(cmdtylist[j]) + "\n";
							this._toRemove.push({
								source: "hold",
								commodity: cmdtylist[j]
							});
							removecount += 1;
							// stanndard slave bounty gets added here, because it's not set to be flagged as illegal to import by default
							if (this._IGTInstalled === true && cmdtylist[j] === "slaves") {
								p.setBounty(player.bounty + (p.manifest[cmdtylist[j]] * goods[i].scale), "illegal imports");
							}
						}
						// check hypercargo
						var hyper = this.$checkHyperCargo(cmdtylist[j]);
						if (hyper != "") {
							msg += hyper + "\n";
							p.setBounty(player.bounty + (this.$getHyperCargoQuantity(cmdtylist[j]) * goods[i].scale), "illegal imports");
							this._toRemove.push({
								source: "hyper",
								commodity: cmdtylist[j]
							});
							removecount += 1;
						}
						// try to check the smuggling compartment (but only if it hasn't already been discovered)
						if (this._removeCompartment === false) {
							for (var k = 0; k < s_cargo.length; k++) {
								if (cmdtylist[j] === s_cargo[k].commodity) {
									if (smuggling_found === true) {
										// found!
										msg += " - " + s_cargo[k].quantity + this.$getCommodityType(s_cargo[k].commodity) + " × " + displayNameForCommodity(s_cargo[k].commodity) + "\n";
										// apply the bounty here, otherwise it won't get applied
										p.setBounty(player.bounty + (s_cargo[k].quantity * goods[i].scale), "illegal imports");
										//s_cargo[k].quantity = 0;
										this._removeCompartment = true;
										removecount += 1;
									} else {
										successcount += 1;
										successmsg += (successmsg === "" ? "" : " and ") + displayNameForCommodity(s_cargo[k].commodity);
									}
								}
							}
						}
					}
				} else {
					var found = false;
					// for any visible cargo (ie not in the cargo hold) for which the player has a permit, we need to refund the bounty
					for (var j = 0; j < cmdtylist.length; j++) {
						// do we have any of this commodity in our hold?
						if (p.manifest[cmdtylist[j]] > 0) {
							p.setBounty(player.bounty - (p.manifest[cmdtylist[j]] * goods[i].scale), "import permit");
							found = true;
						} else {
							// check that this cargo wasn't removed by the contracts system
							for (var k = 0; k < this._dockIllegal.length; k++) {
								if (this._dockIllegal[k].commodity === cmdtylist[j]) found = true;
							}
						}
						// check hypercargo
						var hyper = this.$checkHyperCargo(cmdtylist[j]);
						if (hyper != "") {
							found = true;
						}

						if (this._removeCompartment === false) {
							for (var k = 0; k < s_cargo.length; k++) {
								if (cmdtylist[j] === s_cargo[k].commodity && smuggling_found === true) {
									// found
									found = true;
								}
							}
						}
					}

					// if we had permitted cargo,
					if (found === true) {
						permit_ok += (permit_ok === "" ? "" : ", ") + si.$translateCommodityList(goods[i].commodity);
						permit_okcount += 1;
						if (goods[i].commodity.indexOf(",") > 0) permit_okcount += 1;
						// turn on flat to make sure a mission screen appears if we've adjusted the player's bounty
						// otherwise the status screen will show an incorrect value in "Legal Status"
						if (msgtype === 0) msgtype = 4;
						// remove any permit whether it worked or not
						si.$removePermit(system.ID, goods[i].commodity);
					}
				}
			}
		}

		// special case: where we have illegal cargo just before docking, but don't now, and no permit.
		// this would mean the contract system removed the cargo after docking
		// we still need to give the player an opportunity to bribe their way to a clean slate.
		if (msgtype === 0 && this._playerHadIllegals === true && this.$playerHasIllegalCargo(p.dockedStation) === false) {
			msgtype = 6;
		}

		// found illegal cargo
		if (msg != "") msgtype = 3;

		// switch to the phase scanner message if that was detected but without any cargo
		if (this._phaseScannerDetected === true) {
			if (msgtype === 0 || msgtype === 4) msgtype = 5;
		}

		if (permit_list != "") permit_list = permit_list.replace(", ", expandMissionText("smuggling_and"));

		var msgTitle = "";
		var msgBody = "";
		var bigScreen = false;

		switch (msgtype) {
			case 0:
				if (successcount > 0) {
					msgTitle = expandMissionText("smuggling_contraband");
					msgBody = expandDescription("[smuggling-success]", {
						cargo: successmsg
					});
				}
				break;
			case 1: //
				msgTitle = expandMissionText("smuggling_galcop_alert");
				msgBody = expandDescription("[smuggling-fail-1]", {
					size: this.$smugglingCompartmentSize()
				}) +
					(this._phaseScannerDetected === false ? "" : expandDescription("[phase-scanner-removal]")) +
					expandDescription("[end-warning]");
				break;
			case 2:
				msgTitle = expandMissionText("smuggling_galcop_alert");
				msgBody = expandDescription("[smuggling-fail-2]", {
					size: this.$smugglingCompartmentSize(),
					commodity: displayNameForCommodity(this._sc_VisibleAs)
				}) +
					(this._phaseScannerDetected === false ? "" : expandDescription("[phase-scanner-removal]")) +
					expandDescription("[end-warning]");
				break;

			case 3:
				var sections = 0;
				if (permit_failed) sections += 1;
				if (this._removeCompartment) sections += 1;
				if (this._phaseScannerDetected) sections += 1;
				if (sections === 3 || (sections === 2 && removecount > 4)) bigScreen = true;

				msgTitle = expandMissionText("smuggling_galcop_alert");
				msgBody = expandDescription("[smuggling-fail-3]", { insert: msg }) +
					(permit_failed === false ? "" :
						"\n\n" + expandMissionText("smuggling_invalid_permit", {
							permit: (permit_multi > 1 ? expandMissionText("smuggling_permits") : expandMissionText("smuggling_permit")),
							list: permit_list,
							waswere: (permit_multi === 1 ? expandMissionText("smuggling_was") : expandMissionText("smuggling_were")),
							fabrication: (permit_multi > 1 ? expandMissionText("smuggling_fabrications") : expandMissionText("smuggling_fabrication"))
						})) +
					(this._removeCompartment === false ? "" : expandDescription("[smuggling-compartment-removal]", { size: this.$smugglingCompartmentSize() })) +
					(this._phaseScannerDetected === false ? "" : expandDescription("[phase-scanner-removal]")) +
					expandDescription("[end-warning]");
				break;
			case 4:
				msgTitle = expandMissionText("smuggling_customs");
				msgBody = expandDescription("[smuggling_permit]", {
					cargo: permit_ok,
					permits: (permit_okcount === 1 ? expandMissionText("smuggling_permit") : expandMissionText("smuggling_permits"))
				});
				break;
			case 5:
				msgTitle = expandMissionText("smuggling_galcop_alert");
				msgBody = expandDescription("[phase-scanner-removal-main]");
				break;
			case 6:
				msgTitle = expandMissionText("smuggling_galcop_alert");
				msgBody = expandDescription("[smuggling-fail-4]");
				break;
		}

		if (msgtype >= 1 && msgtype <= 6 && msgtype != 4) {
			this._hudHidden = p.hudHidden;
			if (bigScreen === true && this.$isBigGuiActive() === false) {
				p.hudHidden = true;
				this._smugglingScreen = true;
			}

			var curChoices = {};
			curChoices["01_BRIBE"] = {
				text: "[illegal-attempt-bribe]",
				color: this._menuColor
			};
			curChoices["99_ACCEPT"] = {
				text: "[illegal-accept-penalty]",
				color: this._menuColor
			};

			this._bribeType = 1;
			if (msgtype === 6) this._bribeType = 3;

			var opts = {
				screenID: "docking_with_illegals",
				title: msgTitle,
				message: msgBody,
				allowInterrupt: false,
				model: "[" + this._policeModel + "]",
				exitScreen: "GUI_SCREEN_STATUS",
				choices: curChoices,
				initialChoicesKey: "99_ACCEPT"
			};

			mission.runScreen(opts, this.$bribeScreenHandler, this);
		}

		if (msgtype === 4 || (msgtype === 0 && successcount > 0)) {
			mission.runScreen({
				screenID: "dock_successful",
				title: msgTitle,
				message: msgBody,
				model: "[" + this._policeModel + "]",
				exitScreen: "GUI_SCREEN_STATUS"
			});
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$bribeScreenHandler = function $bribeScreenHandler(choice) {
	if (!choice) return;
	if (choice === "99_ACCEPT") {
		this.$processRemovals();
	}
	if (choice === "98_ACCEPT") {
		this.$processEquipmentRemovals();
	}
	if (choice === "01_BRIBE") {
		this.$getBribeAmount()
	}
}

//-------------------------------------------------------------------------------------------------------------
// initialise the F4 screen entries
this.$initInterface = function $initInterface(station) {
	if (this.$hasSmugglingCompartment() === true) {
		station.setInterface(this.name, {
			title: expandMissionText("smuggling_compartment_title"),
			category: expandDescription("[interfaces-category-ship-systems]"),
			summary: expandMissionText("smuggling_compartment_summary"),
			callback: this.$showInitialSmugglingConfig.bind(this)
		});
	} else {
		station.setInterface(this.name, null);
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the smuggling compartment is damaged
this.$isCompartmentDamaged = function $isCompartmentDamaged() {
	var eq = this.$getSmugglingCompartmentEquipment();
	if (player.ship.equipmentStatus(eq) === "EQUIPMENT_DAMAGED") return true;
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the smuggling compartment is installed
this.$hasSmugglingCompartment = function $hasSmugglingCompartment() {
	for (var i = 1; i <= this._maxTonnage; i++) {
		var check = player.ship.equipmentStatus("EQ_SMUGGLING_COMPARTMENT_" + i.toString());
		if (check === "EQUIPMENT_OK" || check === "EQUIPMENT_DAMAGED") return true;
	}
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// returns the equipment key of the installed compartment
this.$getSmugglingCompartmentEquipment = function $getSmugglingCompartmentEquipment() {
	for (var i = 1; i <= this._maxTonnage; i++) {
		var check = player.ship.equipmentStatus("EQ_SMUGGLING_COMPARTMENT_" + i.toString());
		if (check === "EQUIPMENT_OK" || check === "EQUIPMENT_DAMAGED") return "EQ_SMUGGLING_COMPARTMENT_" + i.toString();
	}
	return "";
}

//-------------------------------------------------------------------------------------------------------------
// returns the max size of the current compartment
this.$smugglingCompartmentSize = function $smugglingCompartmentSize() {
	for (var i = 1; i <= this._maxTonnage; i++) {
		var check = player.ship.equipmentStatus("EQ_SMUGGLING_COMPARTMENT_" + i.toString());
		if (check === "EQUIPMENT_OK" || check === "EQUIPMENT_DAMAGED") return i;
	}
	return 0;
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the phase scanner is damaged, otherwise false
this.$isPhaseScannerDamaged = function $isPhaseScannerDamaged() {
	if (player.ship.equipmentStatus("EQ_PHASE_SCANNER") === "EQUIPMENT_DAMAGED") return true;
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the phase scanner is installed (even if damaged), otherwise false
this.$isPhaseScannerInstalled = function $isPhaseScannerInstalled() {
	var check = player.ship.equipmentStatus("EQ_PHASE_SCANNER");
	if (check === "EQUIPMENT_OK" || check === "EQUIPMENT_DAMAGED") return true;
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// resets smuggling compartment internal data after a compartment is removed
this.$resetCompartmentInfo = function $resetCompartmentInfo(station) {
	var p = player.ship;
	// move any cargo into the main hold
	if (this._sc_Cargo.length > 0) {
		for (var i = 0; i < this._sc_Cargo.length; i++) {
			p.manifest[this._sc_Cargo[i].commodity] += this._sc_Cargo[i].quantity;
		}
		this._sc_Cargo = [];
	}
	this._sc_VisibleAs = "Food";
	this._sc_Phase = 1;
	this._sc_TechVersion = 1;
	this.$initInterface(station);
}

//-------------------------------------------------------------------------------------------------------------
// returns the amount of used space in the compartment
this.$usedSpace = function $usedSpace() {
	var t = 0.0;
	for (var i = 0; i < this._sc_Cargo.length; i++) {
		switch (this._sc_Cargo[i].unit) {
			case "t":
				t += this._sc_Cargo[i].quantity;
				break;
			case "kg":
				t += (this._sc_Cargo[i].quantity / 1000);
				break;
			case "g":
				t += (this._sc_Cargo[i].quantity / 1000000);
				break;
		}
	}
	return t;
}

//-------------------------------------------------------------------------------------------------------------
// returns the amount of available space in the compartment
this.$availableSpace = function $availableSpace() {
	var total = this.$smugglingCompartmentSize();
	return (total - this.$usedSpace());
}

//-------------------------------------------------------------------------------------------------------------
// clean up any zero amounts
this.$cargoCleanUp = function $cargoCleanUp() {
	for (var i = this._sc_Cargo.length - 1; i >= 0; i--) {
		if (this._sc_Cargo[i].quantity === 0) {
			this._sc_Cargo.splice(i, 1);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// shows the installation page to select a compartment size
this.$showPurchasePage = function $showPurchasePage() {

	var curChoices = {};
	var p = player.ship;
	var stn = p.dockedStation;
	var rowAdjust = 0;
	var pagesize = 0;

	//this._hudHidden = p.hudHidden;
	if (this.$isBigGuiActive() === false) {
		p.hudHidden = true;
	}
	this._smugglingScreen = true;

	var text = expandDescription("[purchase-page-header]", {
		cash: formatCredits(player.credits, true, true)
	});
	var curr_size = this.$smugglingCompartmentSize();

	// what is the maximum allowed for this vessel (35% of available space, maxing out at 20t)
	var shipmax = parseInt((p.cargoSpaceCapacity + curr_size) * 0.35);
	// make sure we can get at least 1tc
	if ((p.cargoSpaceCapacity + curr_size) > 0 && shipmax === 0) shipmax = 1;
	// make sure we don't go over the 20 tc limit
	if (shipmax > this._maxTonnage) shipmax = this._maxTonnage;

	var max;
	var check = (p.cargoSpaceAvailable + curr_size);
	if (check < 0) {
		max += check;
	} else {
		max = check;
	}
	if (max > this._maxTonnage) max = this._maxTonnage;

	if (curr_size > 0) {
		text += expandDescription("[downgrade-warning]");
		rowAdjust += 2;
	}

	pagesize = 23 - rowAdjust;

	var count = 0;

	for (var i = 1; i <= this._maxTonnage; i++) {
		var eq = EquipmentInfo.infoForKey("EQ_SMUGGLING_COMPARTMENT_" + i.toString());
		var inst_time = ((600 + (eq.price * stn.equipmentPriceFactor)) / 60) / 60;
		var price = (eq.price / 10) * stn.equipmentPriceFactor;
		var menuitem = this.$padTextLeft(expandMissionText("smuggling_compartment_item", { size: i }), 6) +
			this.$padTextLeft(formatCredits(price, true, true), 7) +
			this.$padTextLeft(expandMissionText("smuggling_compartment_time", { time: inst_time.toFixed(2) }), 15);
		if (i <= shipmax) {
			count += 1;
			if (i != curr_size && i <= max && player.credits >= price) {
				curChoices["01_HOLD_" + (i < 10 ? "0" : "") + i + "~" + i] = {
					text: menuitem,
					alignment: "LEFT",
					color: this._menuColor
				};
			} else {
				curChoices["01_HOLD_" + (i < 10 ? "0" : "") + i + "~" + i] = {
					text: menuitem,
					alignment: "LEFT",
					color: this._disabledColor,
					unselectable: true
				};
			}
		}
	}

	curChoices["98_SPACER"] = "";
	curChoices["99_EXIT"] = {
		text: "[exit-no-change]",
		color: this._itemColor
	};
	var def = "99_EXIT";

	if ((count + 1) < pagesize) {
		for (var i = (pagesize - (count + 1)); i > 0; i--) {
			curChoices["99_SPACER_" + (i < 10 ? "0" : "") + i] = "";
		}
	}

	var opts = {
		screenID: "oolite-smuggling-purchase-map",
		titleKey: "smuggling_compartment_title",
		allowInterrupt: true,
		overlay: {
			name: "stgu-cupboard.png",
			height: 546
		},
		exitScreen: "GUI_SCREEN_EQUIP_SHIP",
		choices: curChoices,
		initialChoicesKey: def,
		message: text
	};

	mission.runScreen(opts, this.$purchaseScreenHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$purchaseScreenHandler = function $purchaseScreenHandler(choice) {

	if (!choice) return;

	if (choice.indexOf("01_HOLD") >= 0) {
		var p = player.ship;
		var stn = p.dockedStation;

		var newsize = parseInt(choice.substring(choice.indexOf("~") + 1));

		var newEq = this._equipmentList[newsize - 1];
		var oldEq = this.$getSmugglingCompartmentEquipment();

		// truncate any existing cargo
		if (newsize < this.$usedSpace()) {
			var diff = this.$usedSpace - newsize;
			do {
				for (var i = this._sc_Cargo.length; i >= 0; i--) {
					if (diff > 0) {
						switch (this._sc_Cargo[i].unit) {
							case "t":
								this._sc_Cargo[i].quantity -= 1;
								diff -= 1;
								break;
							case "kg":
								if (this._sc_Cargo[i].quantity > 1000) {
									this._sc_Cargo[i].quantity -= 1000;
									diff -= 1;
								}
								break;
							case "g":
								if (this._sc_Cargo[i].quantity > 1000000) {
									this._sc_Cargo[i].quantity -= 1000000;
									diff -= 1;
								}
						}
					}
				}
			} while (diff > 0);

			this.$cargoCleanUp();
		}

		// remove the old one (if present)
		if (oldEq != "") {
			this._backdoor = true;
			p.removeEquipment(oldEq);
			this._backdoor = false;
			// refund the price of the old version
			var oldEqInfo = EquipmentInfo.infoForKey(oldEq);
			player.credits += ((oldEqInfo.price / 10) * stn.equipmentPriceFactor);
		}
		// install the new one
		p.awardEquipment(newEq);
		p.awardEquipment("EQ_SMUGGLING_INFLIGHT_STORAGE");

		var eq = EquipmentInfo.infoForKey(newEq);
		var inst_time = 600 + (eq.price * stn.equipmentPriceFactor);
		var price = (eq.price / 10) * stn.equipmentPriceFactor;

		// charge the player
		player.credits -= price;
		// add the smuggler role to the player
		player.setPlayerRole("trader-smuggler");
		// apply the install time
		clock.addSeconds(inst_time);

		// if StationDockControl is installed, tell it to check for launched ships and repopulate
		var sdc = worldScripts.StationDockControl;
		if (sdc && sdc._disable === false) sdc.$checkForLaunchedShips();

		this.$initInterface(stn);
		this._sc_Days = 0;

		// make sure the right email is sent to the player
		var email = worldScripts.GalCopAdminServices;
		if (email) {
			email._boughtKey = newEq;
			email.$playerBoughtEquip();
		}
	}

}

//-------------------------------------------------------------------------------------------------------------
// initial entry point for the smuggling compartment config screen
this.$showInitialSmugglingConfig = function $showInitialSmugglingConfig() {
	this._display = 0;
	this.$showSmugglingConfig();
}

//-------------------------------------------------------------------------------------------------------------
// main config screen display
this.$showSmugglingConfig = function $showSmugglingConfig() {
	function compare(a, b) {
		if ((a.gov * 100) + a.tl < (b.gov * 100) + b.tl) return -1;
		if ((a.gov * 100) + a.tl > (b.gov * 100) + b.tl) return 1;
		return 0;
	}

	var text = "";
	var curChoices = {};
	var p = player.ship;

	var pagesize = 26;
	var phaseSetting = false;
	if (p.equipmentStatus("EQ_SMUGGLING_PHASE_ADJUSTMENT") === "EQUIPMENT_OK") phaseSetting = true;

	this._hudHidden = p.hudHidden;
	if (this.$isBigGuiActive() === false) p.hudHidden = true;
	this._smugglingScreen = true;

	if (this.$isCompartmentDamaged() === true) this._display = 10;

	// main display
	if (this._display === 0) {
		var lines = 5;
		text = expandDescription("[total-capacity-header]") + this.$smugglingCompartmentSize() + "t    " + Math.floor(this.$usedSpace()) + "t used.\n";
		text += expandDescription("[visible-as-header]") + displayNameForCommodity(this._sc_VisibleAs) + "     ";
		if (phaseSetting) text += expandMissionText("smuggling_phase", { phase: this._sc_Phase }) + "     ";
		text += expandDescription("[tech-version-header]") + this._sc_TechVersion + ".0\n";
		if (this._sc_Days >= 20) {
			text += expandDescription("[tech-version-warning]");
			lines += 1;
		}
		text += "\n";
		lines += 6;
		if (phaseSetting === false) lines -= 1;

		var cargo = this.$getSmugglingCargo();
		if (cargo.length > 0) {
			text += expandDescription("[smuggling-current-contents]");
			lines += 1;
			var additem = "";
			for (var i = 0; i < this._sc_Cargo.length; i++) {
				if (additem === "") {
					additem += "  " + this.$padTextRight(this._sc_Cargo[i].quantity + this._sc_Cargo[i].unit + " × " + displayNameForCommodity(this._sc_Cargo[i].commodity) + (this._sc_Cargo[i].extra != "" ? "*" : ""), 17);
				} else {
					additem += this._sc_Cargo[i].quantity + this._sc_Cargo[i].unit + " × " + displayNameForCommodity(this._sc_Cargo[i].commodity) + (this._sc_Cargo[i].extra != "" ? "*" : "");
					text += additem + "\n";
					lines += 1;
					additem = "";
				}
			}
			if (additem != "") {
				text += additem + "\n";
				lines += 1;
			}
		} else {
			text += expandDescription("[smuggling-no-contents]");
			lines += 2;
		}

		if (phaseSetting) {
			curChoices["01_SET_PHASE"] = {
				text: "[smuggling-set-phase]",
				color: this._menuColor
			};
		}
		curChoices["02_VIEW_DISCOVERED"] = {
			text: "[smuggling-view-phase]",
			color: this._menuColor
		};
		curChoices["03_SET_VISIBLE"] = {
			text: "[smuggling-set-visible-as]",
			color: this._menuColor
		};

		var avail = this.$availableSpace();
		var used = this.$usedSpace();

		if (this.$playerHasCargo() === true && avail > 0) {
			curChoices["04_MOVE_CARGO_IN"] = {
				text: "[smuggling-move-in]",
				color: this._menuColor
			};
		} else {
			curChoices["04_MOVE_CARGO_IN"] = {
				text: "[smuggling-move-in]",
				color: this._disabledColor,
				unselectable: true
			};
		}
		if (used > 0) {
			curChoices["05_MOVE_CARGO_OUT"] = {
				text: "[smuggling-move-out]",
				color: this._menuColor
			};
		} else {
			curChoices["05_MOVE_CARGO_OUT"] = {
				text: "[smuggling-move-out]",
				color: this._disabledColor,
				unselectable: true
			};
		}
		if (used > 0) {
			curChoices["06_SELL_CARGO"] = {
				text: "[smuggling-sell-from]",
				color: this._menuColor
			};
		} else {
			curChoices["06_SELL_CARGO"] = {
				text: "[smuggling-sell-from]",
				color: this._disabledColor,
				unselectable: true
			};
		}
		if (avail > 0) {
			curChoices["07_BUY_CARGO"] = {
				text: "[smuggling-buy-in]",
				color: this._menuColor
			};
		} else {
			curChoices["07_BUY_CARGO"] = {
				text: "[smuggling-buy-in]",
				color: this._disabledColor,
				unselectable: true
			};
		}

		//curChoices["70_SPACER"] = "";
		curChoices["99_EXIT"] = {
			text: "[smuggling-exit-config]",
			color: this._itemColor
		};

		pagesize = 26;
		if (lines < pagesize) {
			for (var i = (pagesize - lines); i > 0; i--) {
				curChoices["99_SPACER_" + (i < 10 ? "0" : "") + i] = "";
			}
		}

		var def = "99_EXIT";

		var opts = {
			screenID: "oolite-smuggling-config-summary",
			titleKey: "smuggling_compartment_config",
			allowInterrupt: true,
			overlay: {
				name: "stgu-cupboard.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	// set visible property
	if (this._display === 1) {
		text = expandDescription("[visible-as-property]", {
			commodity: displayNameForCommodity(this._sc_VisibleAs)
		});

		for (var i = 0; i < this._commodityIDList.length; i++) {
			if (this._sc_VisibleAs === this._commodityIDList[i]) {
				curChoices["40_" + (i < 10 ? "0" : "") + i + "~" + this._commodityIDList[i]] = {
					text: displayNameForCommodity(this._commodityIDList[i]),
					color: this._disabledColor,
					unselectable: true
				};
			} else {
				curChoices["40_" + (i < 10 ? "0" : "") + i + "~" + this._commodityIDList[i]] = {
					text: displayNameForCommodity(this._commodityIDList[i]),
					color: this._menuColor
				};
			}
		}

		curChoices["97_SPACER"] = "";
		curChoices["98_EXIT"] = {
			text: "[exit-no-change]",
			color: this._itemColor
		};

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-config-map",
			titleKey: "smuggling_compartment_config",
			allowInterrupt: true,
			overlay: {
				name: "stgu-cupboard.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	if (this._display === 2) { // move cargo in
		// todo: deal with new cargoes

		var avail = this.$availableSpace();
		text = expandDescription("[move-cargo-in]", {
			smuggling: Math.floor(avail),
			standard: p.cargoSpaceAvailable
		});

		pagesize = 20;

		var cargo = p.manifest.list;
		var sdm = worldScripts.Smugglers_DockMaster;
		var itemcount = 0;

		for (var i = 0; i < cargo.length; i++) {
			// only show standard (non-OXP) commodity items
			if (this._commodityIDList.indexOf(cargo[i].commodity) >= 0) {
				var q = cargo[i].quantity;
				// make sure we don't move relabelled cargo
				if (sdm._cargoLabelled.length > 0) {
					for (var j = 0; j < sdm._cargoLabelled.length; j++) {
						if (sdm._cargoLabelled[j].newCommodity === cargo[i].commodity) q -= sdm._cargoLabelled[j].quantity;
					}
				}
				// don't allow scooped escape pods to be moved
				if (cargo[i].commodity === "slaves" && cargo[i].quantity > 0 && worldScripts.EscapePodTweaks && worldScripts.EscapePodTweaks._heldPods.length > 0) {
					q -= worldScripts.EscapePodTweaks._heldPods;
				}

				if (q > 0) {
					curChoices["50_" + (i < 10 ? "0" : "") + i + "_CARGO~" + cargo[i].commodity] = {
						text: cargo[i].displayName + " (" + q + cargo[i].unit + ")" + (this.$calcCTEcargo(cargo[i].commodity) > 0 ? "*" : ""),
						color: this._menuColor
					};
					itemcount += 1;
				}
			}
		}

		curChoices["97_SPACER"] = "";
		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		for (var i = 1; i <= (pagesize - itemcount); i++) {
			curChoices["99_SPACER_" + (i < 10 ? "0" : "") + i] = "";
		}

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-movecargoin-map",
			titleKey: "smuggling_compartment_cargo",
			allowInterrupt: true,
			overlay: {
				name: "stgu-cupboard.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	if (this._display === 3) { // move cargo out
		// todo: deal with new cargoes

		var avail = this.$availableSpace();
		text = expandDescription("[move-cargo-out]", {
			smuggling: Math.floor(avail),
			standard: p.cargoSpaceAvailable
		});

		pagesize = 20;

		var cargo = this.$getSmugglingCargo();

		for (var i = 0; i < cargo.length; i++) {
			if ((p.cargoSpaceAvailable === 0 && cargo[i].unit != "t") || p.cargoSpaceAvailable > 0) {
				curChoices["51_" + (i < 10 ? "0" : "") + i + "_CARGO~" + cargo[i].commodity] = {
					text: cargo[i].quantity + " " + cargo[i].unit + " × " + cargo[i].displayName + (cargo[i].extra != "" ? "*" : ""),
					color: this._menuColor
				};
			}
		}

		curChoices["97_SPACER"] = "";
		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		for (var i = 1; i <= (pagesize - cargo.length); i++) {
			curChoices["99_SPACER_" + (i < 10 ? "0" : "") + i] = "";
		}
		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-movecargoout-map",
			titleKey: "smuggling_compartment_cagro",
			allowInterrupt: true,
			overlay: {
				name: "stgu-cupboard.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	if (this._display === 4) { // view discovered phase settings

		// look for and remove any TL0 items in the list
		// why are we doing this???
		//for (var i = this._phaseScanDiscovery.length - 1; i >= 0; i--) {
		//	if (this._phaseScanDiscovery[i].tl === 0) {
		//		this._phaseScanDiscovery.splice(i, 1);
		//	}
		//}
		// sort the list by government/techlevel
		this._phaseScanDiscovery.sort(compare);

		var govs = new Array();
		for (var i = 0; i < 8; i++)
			govs.push(String.fromCharCode(i));
		var spc = String.fromCharCode(31);

		text = expandDescription("[discovered-phase-scans]");

		if (this._phaseScanDiscovery.length === 0) {
			text += "  None";
		} else {
			for (var i = 0; i < 24; i++) {
				if (this._phaseScanDiscovery.length - 1 >= i && this._phaseScanDiscovery[i].phase != 0) {
					text += this.$padTextRight(govs[this._phaseScanDiscovery[i].gov] + spc + "TL" + (this._phaseScanDiscovery[i].tl + 1) + ":", 3.7) + this.$padTextRight(this._phaseScanDiscovery[i].phase, 3);
				}
				if (this._phaseScanDiscovery.length - 1 >= (i + 24) && this._phaseScanDiscovery[i + 24].phase != 0) {
					text += this.$padTextRight(govs[this._phaseScanDiscovery[i + 24].gov] + spc + "TL" + (this._phaseScanDiscovery[i + 24].tl + 1) + ":", 3.7) + this.$padTextRight(this._phaseScanDiscovery[i + 24].phase, 3);
				}
				if (this._phaseScanDiscovery.length - 1 >= (i + 48) && this._phaseScanDiscovery[i + 48].phase != 0) {
					text += this.$padTextRight(govs[this._phaseScanDiscovery[i + 48].gov] + spc + "TL" + (this._phaseScanDiscovery[i + 48].tl + 1) + ":", 3.7) + this.$padTextRight(this._phaseScanDiscovery[i + 48].phase, 3);
				}
				if (this._phaseScanDiscovery.length - 1 >= (i + 72) && this._phaseScanDiscovery[i + 72].phase != 0) {
					text += this.$padTextRight(govs[this._phaseScanDiscovery[i + 72].gov] + spc + "TL" + (this._phaseScanDiscovery[i + 72].tl + 1) + ":", 3.7) + this.$padTextRight(this._phaseScanDiscovery[i + 72].phase, 3);
				}
				if (this._phaseScanDiscovery.length - 1 >= (i + 96) && this._phaseScanDiscovery[i + 96].phase != 0) {
					text += this.$padTextRight(govs[this._phaseScanDiscovery[i + 96].gov] + spc + "TL" + (this._phaseScanDiscovery[i + 96].tl + 1) + ":", 3.7) + this._phaseScanDiscovery[i + 96].phase;
				}
				text += "\n";
			}
		}

		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-discovered-map",
			titleKey: "smuggling_compartment_config",
			allowInterrupt: true,
			overlay: {
				name: "stgu-cupboard.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	// sell cargo from compartment
	if (this._display === 5) {
		// deal with new cargoes
		text = expandDescription("[sell-smuggling-cargo]");

		pagesize = 23;

		if (this._extraMessage) {
			text += "\n(Note: " + this._extraMessage + ")";
			pagesize -= 1;
			delete this._extraMessage;
		}

		var cargo = this.$getSmugglingCargo();
		var mkt = p.dockedStation.market;

		for (var i = 0; i < cargo.length; i++) {
			curChoices["52_" + (i < 10 ? "0" : "") + i + "_CARGO~" + cargo[i].commodity] = {
				text: this.$padTextRight(cargo[i].quantity + " " + cargo[i].unit + " × " + cargo[i].displayName, 12) +
					(cargo[i].extra != "" ? "*" : "") +
					this.$padTextLeft(formatCredits((mkt[cargo[i].commodity].price / 10), true, true), 5),
				color: this._menuColor
			};
		}

		curChoices["97_SPACER"] = "";
		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		for (var i = 1; i <= (pagesize - cargo.length); i++) {
			curChoices["99_SPACER_" + (i < 10 ? "0" : "") + i] = "";
		}
		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-sellcargo-map",
			titleKey: "smuggling_compartment_sell",
			allowInterrupt: true,
			overlay: {
				name: "stgu-cupboard.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};
	}

	// buy cargo directly to compartment
	if (this._display === 6) {
		text = expandDescription("[buy-smuggling-cargo]");

		pagesize = 23;

		if (this._insufficientFunds && this._insufficientFunds === true) {
			text += "\n" + expandMissionText("smuggling_no_funds", { commodity: displayNameForCommodity(this._tfrCargo) });
			pagesize = 22;
		}

		var mkt = p.dockedStation.market;

		var itemorder = 0;
		var itemcount = 0;

		for (var i in mkt) {
			if (this._commodityIDList.indexOf(i) >= 0) {
				if (mkt[i].quantity > 0) {
					itemorder = parseInt(mkt[i].sort_order / 100);
					curChoices["53_" + (itemorder < 10 ? "0" : "") + itemorder + "_CARGO~" + i] = {
						text: this.$padTextRight(displayNameForCommodity(i), 8) +
							this.$padTextLeft(formatCredits((mkt[i].price / 10), true, true), 4) +
							this.$padTextLeft(mkt[i].quantity, 4) + " " +
							this.$padTextRight(this.$getCommodityType(i), 2),
						color: this._menuColor
					};
					itemcount += 1;
				}
			}
		}

		curChoices["97_SPACER"] = "";
		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		for (var i = 1; i <= (pagesize - itemcount); i++) {
			curChoices["99_SPACER_" + (i < 10 ? "0" : "") + i] = "";
		}
		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-buycargo-map",
			titleKey: "smuggling_compartment_buy",
			allowInterrupt: true,
			overlay: {
				name: "stgu-cupboard.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	if (this._display === 10) {
		text = expandDescription("[smuggling-compartment-damaged]");

		curChoices["99_EXIT"] = {
			text: "[smuggling-exit-config]",
			color: this._itemColor
		};

		var opts = {
			screenID: "smuggling_compartment_damaged",
			titleKey: "smuggling_compartment_config",
			message: text,
			choices: curChoices,
			overlay: {
				name: "stgu-lock.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES"
		};

	}
	mission.runScreen(opts, this.$configScreenHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$configScreenHandler = function $configScreenHandler(choice) {

	if (!choice) return;

	switch (choice) {
		case "01_SET_PHASE":
			this.$setPhase();
			break;
		case "02_VIEW_DISCOVERED":
			this._display = 4;
			break;
		case "03_SET_VISIBLE":
			this._display = 1;
			break;
		case "04_MOVE_CARGO_IN":
			this._display = 2;
			break;
		case "05_MOVE_CARGO_OUT":
			this._display = 3;
			break;
		case "06_SELL_CARGO":
			this._display = 5;
			break;
		case "07_BUY_CARGO":
			this._display = 6;
			break;
		case "98_EXIT":
			this._display = 0;
			break;
	}

	if (choice.indexOf("40_") >= 0) {
		this._sc_VisibleAs = choice.substring(choice.indexOf("~") + 1);
		this._display = 0;
	}

	var showMain = true;

	if (choice.indexOf("50_") >= 0) {
		showMain = false;
		this._tfrCargo = choice.substring(choice.indexOf("~") + 1);
		var cargo = player.ship.manifest.list;
		var sdm = worldScripts.Smugglers_DockMaster;

		for (var i = 0; i < cargo.length; i++) {
			if (cargo[i].commodity === this._tfrCargo) {
				var q = cargo[i].quantity;

				// make sure we don't move relabelled cargo
				if (sdm._cargoLabelled.length > 0) {
					for (var j = 0; j < sdm._cargoLabelled.length; j++) {
						if (sdm._cargoLabelled[j].newCommodity === cargo[i].commodity) q -= sdm._cargoLabelled[j].quantity;
					}
				}

				// don't allow scooped escape pods to be moved
				if (cargo[i].commodity === "slaves" && cargo[i].quantity > 0 && worldScripts.EscapePodTweaks && worldScripts.EscapePodTweaks._heldPods.length > 0) {
					q -= worldScripts.EscapePodTweaks._heldPods;
				}

				this._tfrCargoName = cargo[i].displayName;
				this._tfrMaxAmount = q;
				this._tfrUnit = cargo[i].unit;
				this._tfrExtra = this.$calcCTEcargo(cargo[i].commodity).toString();
				var avail = this.$smugglingCompartmentSize();
				var used = this.$usedSpace();
				// make sure our max amount set isn't greater than the cargo space limit
				switch (this._tfrUnit) {
					case "t":
						if (this._tfrMaxAmount > (avail - used)) {
							this._tfrMaxAmount = avail - used;
						}
						break;
					case "kg":
						var sub_avail = avail * 1000;
						var sub_used = used * 1000;
						if (this._tfrMaxAmount > (sub_avail - sub_used)) this._tfrMaxAmount = sub_avail - sub_used;
						break;
					case "g":
						var sub_avail = avail * 1000000;
						var sub_used = used * 1000000;
						if (this._tfrMaxAmount > (sub_avail - sub_used)) this._tfrMaxAmount = sub_avail - sub_used;
						break;
				}
				this.$getTransferAmount(1);
			}
		}
	}

	if (choice.indexOf("51_") >= 0) {
		showMain = false;
		this._tfrCargo = choice.substring(choice.indexOf("~") + 1);
		var cargo = this.$getSmugglingCargo();
		for (var i = 0; i < cargo.length; i++) {
			if (cargo[i].commodity === this._tfrCargo) {
				this._tfrCargoName = cargo[i].displayName;
				this._tfrMaxAmount = cargo[i].quantity;
				this._tfrUnit = cargo[i].unit;
				this._tfrExtra = cargo[i].extra.split(";").length.toString();
				var avail = player.ship.cargoSpaceAvailable;
				// make sure our max amount set isn't greater than the cargo space limit
				switch (this._tfrUnit) {
					case "t":
						if (this._tfrMaxAmount > avail) this._tfrMaxAmount = avail;
						break;
					case "kg":
						var sub_avail = avail * 1000;
						if (this._tfrMaxAmount > sub_avail) this._tfrMaxAmount = sub_avail;
						break;
					case "g":
						var sub_avail = avail * 1000000;
						if (this._tfrMaxAmount > sub_avail) this._tfrMaxAmount = sub_avail;
						break;

				}
				this.$getTransferAmount(2);
			}
		}
	}

	if (choice.indexOf("52_") >= 0) {
		showMain = false;
		this._tfrCargo = choice.substring(choice.indexOf("~") + 1);
		var cargo = this.$getSmugglingCargo();
		for (var i = 0; i < cargo.length; i++) {
			if (cargo[i].commodity === this._tfrCargo) {
				this._tfrCargoName = cargo[i].displayName;
				this._tfrMaxAmount = cargo[i].quantity;
				this._tfrExtra = cargo[i].extra.split(";").length.toString();
				var stn = player.ship.dockedStation;
				// limit the max amount by the market capacity
				var check = stn.market[this._tfrCargo].capacity - stn.market[this._tfrCargo].quantity;
				if (this._tfrMaxAmount > check) this._tfrMaxAmount = check;
				// if the max amount we can transfer is zero, return to the previous menu.
				if (this._tfrMaxAmount === 0) {
					this._extraMessage = expandMissionText("smuggling_market_cap", { commodity: cargo[i].displayName });
					this.$showSmugglingConfig();
					return;
				}
				this._tfrUnit = cargo[i].unit;
				this.$getTransferAmount(3);
			}
		}
	}

	if (choice.indexOf("53_") >= 0) {
		showMain = false;
		this._insufficientFunds = false;
		this._tfrCargo = choice.substring(choice.indexOf("~") + 1);
		var cargoItem = player.ship.dockedStation.market[this._tfrCargo];
		this._tfrCargoName = displayNameForCommodity(this._tfrCargo);
		this._tfrUnit = this.$getCommodityType(this._tfrCargo);
		this._tfrMaxAmount = cargoItem.quantity;
		this._tfrExtra = "";
		var avail = this.$availableSpace();
		switch (this._tfrUnit) {
			case "t":
				if (avail < this._tfrMaxAmount) this._tfrMaxAmount = avail;
				break;
			case "kg":
				var sub_avail = avail * 1000;
				if (sub_avail < this._tfrMaxAmount) this._tfrMaxAmount = sub_avail;
				break;
			case "g":
				var sub_avail = avail * 1000000;
				if (sub_avail < this._tfrMaxAmount) this._tfrMaxAmount = sub_avail;
				break;
		}
		if ((this._tfrMaxAmount * (cargoItem.price / 10)) > player.credits) {
			do {
				this._tfrMaxAmount -= 1;
			} while ((this._tfrMaxAmount * (cargoItem.price / 10)) > player.credits)
		}
		if (this._tfrMaxAmount > 0) {
			this.$getTransferAmount(4);
		} else {
			showMain = true;
			this._insufficientFunds = true;
		}
	}

	if (choice != "99_EXIT" && choice != "01_SET_PHASE" && showMain === true) {
		this.$showSmugglingConfig();
	}

}

//-------------------------------------------------------------------------------------------------------------
// returns true if the player has any cargo at all
this.$playerHasCargo = function $playerHasCargo() {
	var m = player.ship.manifest;
	for (var i = 0; i < m.list.length; i++) {
		if (m.list[i].quantity > 0) return true;
	}
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the player has any illegal cargo in the visible cargo hold
this.$playerHasIllegalCargo = function $playerHasIllegalCargo(station) {
	var m = player.ship.manifest;
	for (var i = 0; i < m.list.length; i++) {
		if (m.list[i].quantity > 0 && station.market[m.list[i].commodity].legality_import > 0) {
			return true;
		}
	}
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the player has any illegal cargo in their smuggling compartment
this.$playerHasHiddenIllegalCargo = function $playerHasHiddenIllegalCargo(station) {
	var c = this.$getSmugglingCargo();
	var found = false;
	for (var i = 0; i < c.length; i++) {
		if (station.market[c[i].commodity].legality_import > 0) found = true;
	}
	return found;
}

//-------------------------------------------------------------------------------------------------------------
// prompts the user for a new phase
this.$setPhase = function $setPhase() {

	var text = "";

	text = expandDescription("[set-phase]", {
		current: this._sc_Phase
	});

	var opts = {
		screenID: "oolite-smuggling-setphase-map",
		titleKey: "smuggling_compartment_config",
		allowInterrupt: false,
		exitScreen: "GUI_SCREEN_INTERFACES",
		message: text,
		textEntry: true
	};
	mission.runScreen(opts, this.$getNewPhase, this);
}

//-------------------------------------------------------------------------------------------------------------
// parses the input from the setPhase screen
this.$getNewPhase = function $getNewPhase(param) {
	if (parseInt(param) >= 1 && parseInt(param) <= 999) {
		this._sc_Phase = parseInt(param);
	}
	this.$showSmugglingConfig();
}

//-------------------------------------------------------------------------------------------------------------
// prompts the user for the transfer amount
this.$getTransferAmount = function $getTransferAmount(type) {

	var text = "";

	// don't show a screen when only 1 unit is possible
	if (this._tfrMaxAmount === 1) {
		switch (type) {
			case 1:
				this.$getCargoTransferIn(1);
				return;
			case 2:
				this.$getCargoTransferOut(1);
				return;
			case 3:
				this.$getCargoSell(1);
				return;
			case 4:
				this.$getCargoBuy(1);
				return;
		}
	}

	switch (type) {
		case 1:
			text = expandDescription("[enter-quantity-smuggling]", {
				commodity: this._tfrCargoName,
				quantity: parseInt(this._tfrMaxAmount),
				cte_note: (this._tfrExtra != "" ? "\n\n" + expandMissionText("smuggling_newcargoes_note", {extra:this._tfrExtra}) : "")
			});
			break;
		case 2:
			text = expandDescription("[enter-quantity-standard]", {
				commodity: this._tfrCargoName,
				quantity: parseInt(this._tfrMaxAmount),
				cte_note: (this._tfrExtra != "" ? "\n\n" + expandMissionText("smuggling_newcargoes_note", {extra:this._tfrExtra}) : "")
			});
			break;
		case 3:
			text = expandDescription("[enter-quantity-sell]", {
				commodity: this._tfrCargoName,
				quantity: parseInt(this._tfrMaxAmount),
				cte_note: (this._tfrExtra != "" ? "\n\n" + expandMissionText("smuggling_newcargoes_note", {extra:this._tfrExtra}) : "")
			});
			break;
		case 4:
			text = expandDescription("[enter-quantity-buy]", {
				commodity: this._tfrCargoName,
				quantity: parseInt(this._tfrMaxAmount)
			});
			break;
	}

	var opts = {
		screenID: "oolite-smuggling-transfer-map",
		titleKey: "smuggling_compartment_cargo",
		allowInterrupt: false,
		exitScreen: "GUI_SCREEN_INTERFACES",
		message: text,
		textEntry: true
	};

	switch (type) {
		case 1:
			mission.runScreen(opts, this.$getCargoTransferIn, this);
			break;
		case 2:
			mission.runScreen(opts, this.$getCargoTransferOut, this);
			break;
		case 3:
			mission.runScreen(opts, this.$getCargoSell, this);
			break;
		case 4:
			mission.runScreen(opts, this.$getCargoBuy, this);
			break;
	}
}

//-------------------------------------------------------------------------------------------------------------
// parses the input from the getTransferAmount screen
this.$getCargoTransferIn = function $getCargoTransferIn(param) {
	if (!param) {
		this.$showSmugglingConfig();
		return;
	}
	if (parseInt(param) >= 1 && parseInt(param) <= this._tfrMaxAmount) {
		var p = player.ship;
		var max = p.manifest[this._tfrCargo];
		var amount = parseInt(param);
		var extra = "";
		if (this._NCInstalled) {
			extra = this.$extractCargoFromCTE(this._tfrCargo, amount, max);
		}
		this.$addCargoToCompartment(this._tfrCargo, this._tfrCargoName, amount, this._tfrUnit, extra);
		// take the cargo out of the hold
		var p = player.ship;
		p.manifest[this._tfrCargo] -= amount;
	}
	this._display = 2;
	if (this.$availableSpace() === 0) this._display = 0;
	this.$showSmugglingConfig();
}

//-------------------------------------------------------------------------------------------------------------
this.$calcCTEcargo = function (generic) {
	var cte = worldScripts.CargoTypeExtension;
	// if there isn't enough generic cargo of a particular type
	// throw out special cargo entries of that type until there is
	var totalspecial = 0;
	for (var j = 0; j < cte.specialCargoCarried[generic].length; j++) {
		if (cte.specialCargoRegister[cte.specialCargoCarried[generic][j].type]) {
			totalspecial += cte.specialCargoCarried[generic][j].quantity;
		}
	}
	return totalspecial;
}

//-------------------------------------------------------------------------------------------------------------
// parses the input from the getTransferAmount screen
this.$getCargoTransferOut = function $getCargoTransferOut(param) {
	if (!param) {
		this.$showSmugglingConfig();
		return;
	}
	if (parseInt(param) >= 1 && parseInt(param) <= this._tfrMaxAmount) {
		var p = player.ship;
		var amount = parseInt(param);
		var starting = p.manifest[this._tfrCargo];
		// put the cargo into the main hold
		p.manifest[this._tfrCargo] += amount;
		// check that the hold actually took all the amount
		if (p.manifest[this._tfrCargo] != (starting + amount)) {
			// if not, adjust the amount of transfer
			amount = p.manifest[this._tfrCargo] - starting;
		}
		for (var i = 0; i < this._sc_Cargo.length; i++) {
			if (this._sc_Cargo[i].commodity === this._tfrCargo) {
				if (this._NCInstalled) {
					var ncmanifest = this.$checkForAndRemoveCTECargo(this._tfrCargo, amount);
					if (ncmanifest != "") {
						var cte = worldScripts.CargoTypeExtension;
						cte.mergePlayerManifest(ncmanifest);
					}
				}
				this._sc_Cargo[i].quantity -= amount;
				break;
			}
		}
		this.$cargoCleanUp();
	}
	this._display = 3;
	if (this.$usedSpace() === 0) this._display = 0;
	this.$showSmugglingConfig();
}

//-------------------------------------------------------------------------------------------------------------
// parses the input from the getTransferAmount screen
this.$getCargoSell = function $getCargoSell(param) {
	if (!param) {
		this.$showSmugglingConfig();
		return;
	}
	if (parseInt(param) >= 1 && parseInt(param) <= this._tfrMaxAmount) {
		var amount = parseInt(param);
		var count = 0;
		var check = 0;
		var p = player.ship;
		var stn = p.dockedStation;
		var mkt = stn.market;
		for (var i = this._sc_Cargo.length - 1; i >= 0; i--) {
			if (this._sc_Cargo[i].commodity === this._tfrCargo) {
				check = mkt[this._tfrCargo].quantity;
				var newamount = check + amount;
				var actual = amount;
				// check that we aren't going over the market capacity
				if (newamount > mkt[this._tfrCargo].capacity) {
					newamount = mkt[this._tfrCargo].capacity;
					actual = newamount - check;
				}
				// only do an sell if the actual amount that can be sold is greater than zero
				if (actual > 0) {
					stn.setMarketQuantity(this._tfrCargo, newamount);

					this.$checkForAndRemoveCTECargo(this._tfrCargo, actual);

					this._sc_Cargo[i].quantity -= actual;
					count += actual;
					player.credits += (mkt[this._tfrCargo].price / 10) * actual;
					this.$playSound("sell");
				}
				break;
			}
		}
		this.$cargoCleanUp();
	}
	this._display = 5;
	if (this.$usedSpace() === 0) this._display = 0;
	this.$showSmugglingConfig();
}

//-------------------------------------------------------------------------------------------------------------
// parses the input from the getTransferAmount screen
this.$getCargoBuy = function $getCargoBuy(param) {
	if (!param) {
		this.$showSmugglingConfig();
		return;
	}
	if (parseInt(param) >= 1 && parseInt(param) <= this._tfrMaxAmount) {
		var amount = parseInt(param);
		this.$addCargoToCompartment(this._tfrCargo, this._tfrCargoName, amount, this._tfrUnit, "");
		// take the cargo out of the market
		var stn = player.ship.dockedStation;
		var mkt = stn.market;
		var newamount = mkt[this._tfrCargo].quantity - amount;
		stn.setMarketQuantity(this._tfrCargo, newamount);
		player.credits -= (mkt[this._tfrCargo].price / 10) * amount;
		this.$playSound("buy");
	}
	this._display = 6;
	if (this.$availableSpace() === 0) this._display = 0;
	this.$showSmugglingConfig();
}

//-------------------------------------------------------------------------------------------------------------
// adds cargo to smuggling compartment
this.$addCargoToCompartment = function $addCargoToCompartment(cmdty, name, amount, unit, extra) {
	var found = false;
	for (var i = 0; i < this._sc_Cargo.length; i++) {
		if (this._sc_Cargo[i].commodity === cmdty) {
			// found it
			found = true;
			this._sc_Cargo[i].quantity += amount;
			if (extra != "") {
				if (this._sc_Cargo[i].extra == "") {
					this._sc_Cargo[i].extra = extra;
				} else {
					this._sc_Cargo[i].extra += ";" + extra;
				}
			}
		}
	}
	if (found === false) {
		this._sc_Cargo.push({
			commodity: cmdty,
			displayName: name,
			quantity: amount,
			unit: unit,
			extra: extra
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// create system phases
this.$createSystemPhase = function $createSystemPhase() {
	for (var i = 0; i <= 7; i++) {
		for (var j = 1; j <= 15; j++) {
			this._systemPhase.push({
				gov: i,
				tl: j,
				phase: this.$rand(999)
			});
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// updates the phase setting for a particular government/techlevel
this.$updatePhase = function $updatePhase(govType, techlevel, phase) {
	for (var i = 0; i < this._systemPhase.length; i++) {
		if (this._systemPhase[i].gov === govType && this._systemPhase[i].tl === techlevel) {
			this._systemPhase[i].phase = phase;
			this._phaseUpdates.push({
				gov: govType,
				tl: techlevel
			});
			break;
		}
	}
	for (var i = 0; i < this._phaseScanDiscovery.length; i++) {
		if (this._phaseScanDiscovery[i].gov === govType && this._phaseScanDiscovery[i].tl === techlevel && this._phaseScanDiscovery[i].source < 3) this._phaseScanDiscovery[i].source = 4;
	}

}

//-------------------------------------------------------------------------------------------------------------
// returns the phase for a given system ID
this.$getSystemPhase = function $getSystemPhase(systemID) {
	var sysinfo = System.infoForSystem(galaxyNumber, systemID);
	var result = 0;
	for (var i = 0; i < this._systemPhase.length; i++) {
		if (this._systemPhase[i].gov === sysinfo.government && this._systemPhase[i].tl === sysinfo.techlevel) result = this._systemPhase[i].phase;
	}
	return result;
}

//-------------------------------------------------------------------------------------------------------------
// removes the current smuggling compartment and empties the cargo stored in  it
this.$removeCompartment = function $removeCompartment() {
	player.ship.removeEquipment("EQ_SMUGGLING_PHASE_ADJUSTMENT");
	player.ship.removeEquipment("EQ_SMUGGLING_INFLIGHT_STORAGE");
	var eq = this.$getSmugglingCompartmentEquipment();
	if (eq === "") return;
	this._backdoor = true;
	player.ship.removeEquipment(eq);
	this._backdoor = false;
	this._sc_Cargo = [];
	this._sc_VisibleAs = "Food";
	this._sc_Phase = 1;
	this._sc_TechVersion = 1;
}

//-------------------------------------------------------------------------------------------------------------
// returns a summarised version of the smuggling compartment cargo
this.$getSmugglingCargo = function $getSmugglingCargo() {
	var cargo = [];
	for (var i = 0; i < this._sc_Cargo.length; i++) {
		var found = false;
		for (var j = 0; j < cargo.length; j++) {
			if (cargo[j].commodity === this._sc_Cargo[i].commodity) {
				cargo[j].quantity += this._sc_Cargo[i].quantity;
				found = true;
			}
		}
		if (found === false) {
			cargo.push({
				commodity: this._sc_Cargo[i].commodity,
				displayName: this._sc_Cargo[i].displayName,
				quantity: this._sc_Cargo[i].quantity,
				unit: this._sc_Cargo[i].unit,
				extra: this._sc_Cargo[i].extra
			});
		}
	}
	return cargo;
}

//-------------------------------------------------------------------------------------------------------------
// returns how much of a particular commodity is in the smuggling compartment
this.$getCargoQuantity = function $getCargoQuantity(commodity) {
	var cargo = this.$getSmugglingCargo();
	var result = 0;
	for (var i = 0; i < cargo.length; i++) {
		if (cargo[i].commodity === commodity) result = cargo[i].quantity;
	}
	return result;
}

//-------------------------------------------------------------------------------------------------------------
// removes an given amount of a particular commodity
this.$removeCargo = function $removeCargo(commodity, quantity) {
	var sum = quantity;
	for (var i = this._sc_Cargo.length - 1; i >= 0; i--) {
		if (this._sc_Cargo[i].commodity === commodity && sum > 0) {
			if (this._sc_Cargo[i].quantity >= sum) {
				this.$checkForAndRemoveCTECargo(commodity, sum);
				this._sc_Cargo[i].quantity -= sum;
				sum = 0;
			} else {
				sum -= this._sc_Cargo[i].quantity;
				this._sc_Cargo[i].quantity = 0;
			}
		}
	}
	this.$cargoCleanUp();
	// return any remainder
	return sum;
}

//-------------------------------------------------------------------------------------------------------------
// extracts an certain amount of a particular commodity from new cargoes
this.$extractCargoFromCTE = function (commodity, amount, max) {
	var cte = worldScripts.CargoTypeExtension;
	var special = this.$calcCTEcargo(commodity);
	var normal = max - special;
	if (amount - normal > 0) {
		// we're transferring some special cargo
		var storage = [];
		var ncmanifest = cte.suspendPlayerManifest(commodity);
		var nc = ncmanifest.split("|");
		var lst = nc[1].split("/");
		var counter = 0;

		for (var i = 0; i < lst.length; i++) {
			if (lst[i] != "") {
				var itms = lst[i].split(";");
				for (var j = 0; j < itms.length; j++) {
					if (itms[j] != "") {
						// reduce the original
						var sub = itms[j].split("=");
						var removed = 0;
						while (sub[1] > 0) {
							sub[1]--;
							counter++;
							removed++;
							if ((amount - normal) - counter == 0) break;
						}
						itms[j] = sub.join("=");

						//storage.push(sub[0] + "=" + removed + "=" + sub[2]);
						// make a copy of what we're taking out
						// we'll store them a 1 unit items to make the return process easier
						for (var k = 0; k < removed; k++) {
							var st = "";
							for (var m = 0; m < sub.length; m++) {
								st += (st == "" ? "" : "=") + (m == 1 ? 1 : sub[m]);
							}
							storage.push(st);
						}
					}
				}
				lst[i] = itms.join(";");
			}
			if ((amount - normal) - counter == 0) break;
		}

		nc[1] = lst.join("/");
		ncmanifest = nc.join("|");
		// tell NC about what we've done
		cte.mergePlayerManifest(ncmanifest);
		// pull our storage data together
		extra = storage.join(";");
	}
	return extra;
}

//-------------------------------------------------------------------------------------------------------------
this.$checkForAndRemoveCTECargo = function (commodity, quantity) {
	var extra = "";
	for (var i = 0; i < this._sc_Cargo.length; i++) {
		if (this._sc_Cargo[i].commodity == commodity) {
			if (this._sc_Cargo[i].extra != "") {
				var dta = [];
				dta = this._sc_Cargo[i].extra.split(";");
				var normal = this._sc_Cargo[i].quantity - dta.length;
				if (quantity - normal > 0) {
					var diff = quantity - normal;
					// compile a list of the cargo items to re-merge with nc
					while (diff > 0) {
						extra += (extra == "" ? "" : ";") + dta.shift();
						diff--;
					}
					this._sc_Cargo[i].extra = dta.join(";");
				}
			}
		}
	}
	// because we store everything as 1 unit items, we need to compile it back into totals so NC has good data
	var hold = this.$compileCTEValues(extra);
	var ncmanifest = "2|" + this.$buildMergeManifest(commodity, hold);
	return ncmanifest;
}

//-------------------------------------------------------------------------------------------------------------
// compile multiple 1 unit items down to one entry with a 1+ unit count
this.$compileCTEValues = function (extra) {
	var items = extra.split(";");
	var list = {};
	for (var i = 0; i < items.length; i++) {
		if (!list[items[i]]) {
			list[items[i]] = 1;
		} else {
			list[items[i]] = list[items[i]] + 1;
		}
		log(this.name, "tracking " + list[items[i]]);
	}
	var keys = Object.keys(list);
	var final = "";
	for (var i = 0; i < keys.length; i++) {
		var subs = keys[i].split("=");
		final = (final == "" ? "" : ";") + subs[0] + "=" + list[keys[i]] + "=" + subs[2];
	}
	return final;
}

//-------------------------------------------------------------------------------------------------------------
// put the manifest string together
// blank entries in all but the item we are processing the commodity for
this.$buildMergeManifest = function (commodity, extra) {
	var cte = worldScripts.CargoTypeExtension;
	var goodsList = cte.cargoTypes;
	var serial = "";
	for (var i = 0; i < goodsList.length; i++) {
		if (goodsList[i] == commodity) {
			serial += extra;
		}
		if (i + 1 < goodsList.length) {
			serial += "/";
		}
	}
	return serial;
}

//-------------------------------------------------------------------------------------------------------------
// adds a phase scan setting to our discovered list
// source 1 = from phase scanner, 2 = purchased from black market, 3 = overheard in rumours, 4 = updated after scan
this.$addPhaseScan = function $addPhaseScan(phase, systemID, source) {
	//log(this.name, "checking " + system.ID);
	//log(this.name, "data: " + systemID + ", " + source);
	var sysinfo = System.infoForSystem(galaxyNumber, systemID);
	var found = false;
	for (var i = 0; i < this._phaseScanDiscovery.length; i++) {
		if (this._phaseScanDiscovery[i].gov === sysinfo.government && this._phaseScanDiscovery[i].tl === sysinfo.techlevel) {
			found = true;
			this._phaseScanDiscovery[i].phase = phase;
			// if the previous value was just a remour, or changed, update it now
			if (this._phaseScanDiscovery[i].source >= 3 && source < 3) this._phaseScanDiscovery[i].source = source;
		}
	}
	if (found === false) {
		this._phaseScanDiscovery.push({
			gov: sysinfo.government,
			tl: sysinfo.techlevel,
			phase: phase,
			source: source
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the player has a reliable phase scan for a given system
this.$playerHasPhaseScan = function $playerHasPhaseScan(sysID) {
	var sysinfo = System.infoForSystem(galaxyNumber, sysID);
	var result = false;
	for (var i = 0; i < this._phaseScanDiscovery.length; i++) {
		if (this._phaseScanDiscovery[i].gov === sysinfo.government && this._phaseScanDiscovery[i].tl === sysinfo.techlevel && this._phaseScanDiscovery[i].source < 3) result = true;
	}
	return result;
}

//-------------------------------------------------------------------------------------------------------------
// returns the number of sellable phase scans the player has
this.$sellablePhaseScans = function $sellablePhaseScans() {
	var list = this.$getSellablePhaseScans();
	return list.length;
}

//-------------------------------------------------------------------------------------------------------------
// get a list of sellable phase scan settings for the black market
// note: will only return scans within 7ly of current system
this.$getSellablePhaseScans = function $getSellablePhaseScans() {
	function compare(a, b) {
		if ((a.gov * 100) + a.tl < (b.gov * 100) + b.tl) return -1;
		if ((a.gov * 100) + a.tl > (b.gov * 100) + b.tl) return 1;
		return 0;
	}

	this._phaseScanDiscovery.sort(compare);

	var govs = new Array();
	for (var i = 0; i < 8; i++)
		govs.push(String.fromCharCode(i));
	var spc = String.fromCharCode(31);
	var list = [];
	var added = [];
	// you can only sell phase scans for the local area
	var sys = System.infoForSystem(galaxyNumber, system.ID).systemsInRange(7);

	for (var i = 0; i < this._phaseScanDiscovery.length; i++) {
		if (this._phaseScanDiscovery[i].source === 1) {
			for (var j = 0; j < sys.length; j++) {
				// does this scan match a local system, and have we already added this variation?
				if (sys[j].government === this._phaseScanDiscovery[i].gov && sys[j].techlevel === this._phaseScanDiscovery[i].tl && added.indexOf((sys[j].government * 100) + sys[j].techlevel) === -1) {
					list.push({
						text: this.$padTextRight(govs[this._phaseScanDiscovery[i].gov] + spc + "TL" + (this._phaseScanDiscovery[i].tl + 1) + ":", 3.7) + this.$padTextRight(this._phaseScanDiscovery[i].phase, 3),
						gov: this._phaseScanDiscovery[i].gov,
						tl: this._phaseScanDiscovery[i].tl
					});
					// add the signature to the added list so we can easily check if it's there already
					added.push((sys[j].government * 100) + sys[j].techlevel);
				}
			}
		}
	}
	return list;
}

//-------------------------------------------------------------------------------------------------------------
// switches a source = 1 to source = 0 for a particular phase scan
this.$sellPhaseScan = function $sellPhaseScan(govrn, techlvl) {
	for (var i = 0; i < this._phaseScanDiscovery.length; i++) {
		if (this._phaseScanDiscovery[i].gov === govrn && this._phaseScanDiscovery[i].tl === techlvl && this._phaseScanDiscovery[i].source === 1) {
			this._phaseScanDiscovery[i].source = 0;
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// gets the commodity type for a particular commodity (t , kg or g)
this.$getCommodityType = function $getCommodityType(good) {
	if (system.isInterstellarSpace) {
		var m = player.ship.manifest;
		if (m[good] > 0) {
			for (var i = 0; i < m.list.length; i++) {
				if (m.list[i].commodity == good) return m.list[i].unit;
			}
		} else {
			return "t";
		}
	}
	var types = ["t", "kg", "g"];
	return types[system.mainStation.market[good].quantity_unit];
}

//=============================================================================================================
// hypercargo integration routines

//-------------------------------------------------------------------------------------------------------------
// returns a list of commodity items that are illegal, based on a commodity list
this.$checkHyperCargo = function $checkHyperCargo(commodity) {
	var returnData = "";
	if (missionVariables.hyperCargoMemory === "EMPTY") return "";
	if (!worldScripts["HyperCargo"]) return ""; // just in case hypercargo OXP was removed after being installed for a while
	var hc = worldScripts["HyperCargo"];
	for (var i = 0; i < 14; i++) {
		var cargoItem = hc.cargoNameArray[i];
		if (commodity === cargoItem && hc.storedArray[i] > 0) {
			var cargoCount = hc.storedArray[i];
			returnData = " - " + cargoCount + " " + this.$getCommodityType(cargoItem) + " × " + displayNameForCommodity(cargoItem) + expandMissionText("smuggling_hypercargo");
		}
	}
	return returnData;
}

//-------------------------------------------------------------------------------------------------------------
// returns the quantity of a particular commodity
this.$getHyperCargoQuantity = function $getHyperCargoQuantity(commodity) {
	var returnData = "";
	if (missionVariables.hyperCargoMemory === "EMPTY") return 0;
	if (!worldScripts["HyperCargo"]) return 0; // just in case hypercargo OXP was removed after being installed for a while
	var hc = worldScripts["HyperCargo"];
	for (var i = 0; i < 14; i++) {
		var cargoItem = hc.cargoNameArray[i];
		if (commodity === cargoItem && hc.storedArray[i] > 0) {
			return hc.storedArray[i];
		}
	}
	return -1;
}

//-------------------------------------------------------------------------------------------------------------
// removes illegal cargo from hypercargo and applies a bounty
this.$updateHyperCargo = function $updateHyperCargo(commodity, def) {
	if (missionVariables.hyperCargoMemory === "EMPTY") return;
	if (!worldScripts["HyperCargo"]) return; // just in case hypercargo OXP was removed after being installed for a while
	var hc = worldScripts["HyperCargo"];
	var p = player.ship;
	for (var i = 0; i < 14; i++) {
		var cargoItem = hc.cargoNameArray[i];
		if (commodity === cargoItem && hc.storedArray[i]) {
			p.setBounty(player.bounty + (hc.storedArray[i] * def.scale), "illegal imports");
			hc.storedArray[i] = 0;
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// removes an amount of cargo from hypercargo
this.$removeHyperCargo = function $removeHyperCargo(commodity, amount) {
	if (missionVariables.hyperCargoMemory === "EMPTY") return 0;
	if (!worldScripts["HyperCargo"]) return 0; // just in case hypercargo OXP was removed after being installed for a while
	var hc = worldScripts["HyperCargo"];
	for (var i = 0; i < 14; i++) {
		var cargoItem = hc.cargoNameArray[i];
		if (commodity === cargoItem && hc.storedArray[i] > 0) {
			if (hc.storedArray[i] >= amount) {
				hc.storedArray[i] -= amount;
				return 0;
			} else {
				var remain = amount - hc.storedArray[i];
				hc.storedArray[i] = 0;
				return remain;
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// prompts the user for a new phase
this.$getBribeAmount = function $getBribeAmount() {

	var text = "";
	var sdm = worldScripts.Smugglers_DockMaster;
	var inttype = Math.floor((sdm._bribeChance[system.ID] * 4) + 1);
	var inttypedesc = expandDescription("[interest-type" + inttype + "]");
	var lastBribe = sdm.$previousBribeAmount();
	text = expandDescription("[dockmaster-question]", {
		interesttype: inttypedesc
	}) + "\n\n" +
		"(" + expandDescription("[dockmaster-cost]", {
			credits: formatCredits(player.credits, false, true)
		}) +
		(lastBribe > 0 ? expandDescription("[dockmaster-lastbribe]", {
			bribeamount: formatCredits(lastBribe, false, true)
		}) : "") + ")";

	var opts = {
		screenID: "smuggling_bribe",
		titleKey: "smuggling_bribe_official",
		allowInterrupt: false,
		model: "[" + this._policeModel + "]",
		exitScreen: "GUI_SCREEN_STATUS",
		message: text,
		textEntry: true
	};

	mission.runScreen(opts, this.$getBribeAmountInput, this);
}

//-------------------------------------------------------------------------------------------------------------
// parses the input from the getBribeAmount screen
this.$getBribeAmountInput = function $getBribeAmountInput(param) {

	var sdm = worldScripts.Smugglers_DockMaster;
	var p = player.ship;
	sdm._bribeAttempted = true;

	if (parseInt(param) >= 1 && parseInt(param) <= player.credits) {
		var amount = parseInt(param);
		// will this work
		var chance = sdm._bribeChance[system.ID];
		// higher amounts are more likely to be accepted
		// chance of 0.5 would mean you would need to offer them 256 credits
		//if (this._debug) log(this.name, "min bribe amount = " + (Math.pow(parseInt(chance * 32), 1.8) * (1 + system.productivity / 56300) * sdm._bribeCount[system.ID]));
		if ((Math.pow(parseInt(chance * 32), 1.8) * (1 + system.productivity / 56300) * sdm._bribeCount[system.ID]) <= amount) {
			player.credits -= amount;
			sdm.$addBribe(system.ID);
			sdm._successfulBribes.push({
				system: system.ID,
				bribeAmount: amount
			});
			p.setBounty(this._savedBounty, "bribe");
			mission.runScreen({
				screenID: "smuggling_bribe",
				titleKey: "smuggling_bribe_official",
				model: "[" + this._policeModel + "]",
				message: expandDescription("[dockmaster-complete]"),
				exitScreen: "GUI_SCREEN_STATUS"
			});
		} else {
			if (this._bribeType === 1) this.$processRemovals();
			if (this._bribeType === 2) this.$processEquipmentRemovals();
			if (Math.random() > chance) {
				mission.runScreen({
					screenID: "smuggling_bribe",
					titleKey: "smuggling_bribe_official",
					model: "[" + this._policeModel + "]",
					message: expandDescription("[dockmaster-angry-nopenalty]"),
					exitScreen: "GUI_SCREEN_STATUS"
				});

			} else {
				var penalty = (worldScripts.Smugglers_CoreFunctions.$rand(10) + 3);
				p.setBounty(player.bounty + penalty, "attempted bribe");
				mission.runScreen({
					screenID: "smuggling_bribe",
					titleKey: "smuggling_bribe_official",
					model: "[" + this._policeModel + "]",
					message: expandDescription("[dockmaster-angry-penalty]"),
					exitScreen: "GUI_SCREEN_STATUS"
				});

				// send email (if installed)
				var email = worldScripts.EmailSystem;
				if (email) {
					var ga = worldScripts.GalCopAdminServices;
					email.$createEmail({
						sender: expandMissionText("smuggling_customs"),
						subject: expandMissionText("smuggling_bribe_attempt"),
						date: clock.seconds,
						message: expandDescription("[smugglers-failed-bribe-email]", {
							legal_penalty: penalty,
							stationname: player.ship.dockedStation.displayName,
							systemname: System.systemNameForID(system.ID)
						}),
						expiryDays: ga._defaultExpiryDays
					});
				}
			}
		}
	} else {
		if (this._bribeType === 1) this.$processRemovals();
		if (this._bribeType === 2) this.$processEquipmentRemovals();
		mission.runScreen({
			screenID: "smuggling_bribe",
			title: "smuggling_bribe_official",
			model: "[" + this._policeModel + "]",
			message: expandDescription("[dockmaster-skipbribe]"),
			exitScreen: "GUI_SCREEN_STATUS"
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// processes any cargo and/or smuggling compartment removals after docking
this.$processRemovals = function $processRemovals() {
	var emailText = "";
	var p = player.ship;
	if (this._removeCompartment === true) {
		// remove smuggling compartment and contents
		emailText += "- " + expandMissionText("smuggling_found_compartment") + "\n";
		this.$removeCompartment();
	}
	if (this._phaseScannerDetected === true) {
		emailText += "- " + expandMissionText("smuggling_found_phase_scanner") + ".\n";
		p.removeEquipment("EQ_PHASE_SCANNER");
	}
	for (var i = 0; i < this._toRemove.length; i++) {
		if (this._toRemove[i].source === "hold") {
			emailText += "- " + expandMissionText("smuggling_found_cargo", {commodity:p.manifest[this._toRemove[i].commodity] + this.$getCommodityType(this._toRemove[i].commodity) + " × " + displayNameForCommodity(this._toRemove[i].commodity)}) + "\n";
			p.manifest[this._toRemove[i].commodity] = 0;
		}
		if (this._toRemove[i].source === "hyper") {
			emailText += "- " + expandMissionText("smuggling_found_hypercargo", {commodity:p.manifest[this._toRemove[i].commodity] + this.$getCommodityType(this._toRemove[i].commodity) + " × " + displayNameForCommodity(this._toRemove[i].commodity)}) + "\n";
			this.$removeHyperCargo(this._toRemove[i].commodity, this.$getHyperCargoQuantity(this._toRemove[i].commodity));
		}
	}
	if (this._playerFine > 0) {
		if (player.credits > this._playerFine) {
			emailText += "- " + expandMissionText("smuggling_applied_fine", {fine: formatCredits(this._playerFine, true, true)}) + "\n";
			player.credits -= this._playerFine;
		} else {
			emailText += "- " + expandMissionText("smuggling_applied_fine", {fine:formatCredits(player.credits, true, true)}) + "\n";
			player.credits = 0;
		}
	}
	this._removeCompartment = false;
	this._phaseScannerDetected = false;

	// send email (if installed)
	if (emailText != "") {
		var email = worldScripts.EmailSystem;
		if (email) {
			var ga = worldScripts.GalCopAdminServices;
			email.$createEmail({
				sender: expandMissionText("smuggling_customs"),
				subject: expandMissionText("smuggling_contraband_removal"),
				date: clock.seconds,
				message: expandDescription("[smugglers-removal-email]", {
					contraband: emailText,
					stationname: player.ship.dockedStation.displayName,
					systemname: System.systemNameForID(system.ID)
				}),
				expiryDays: ga._defaultExpiryDays
			});
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// processes any equipment removals after a renovation
this.$processEquipmentRemovals = function $processEquipmentRemovals() {
	var p = player.ship;
	var emailText = "";

	if (this.$isPhaseScannerInstalled() === true) {
		emailText = "- " + expandMissionText("smuggling_found_phase_scanner") + "\n";
		p.removeEquipment("EQ_PHASE_SCANNER");
	}

	if (this.$hasSmugglingCompartment() === true) {
		// remove compartment and contents
		emailText += "- " + expandMissionText("smuggling_found_compartment") + "\n";
		this.$removeCompartment();
	}

	// send email (if installed)
	if (emailText != "") {
		var email = worldScripts.EmailSystem;
		if (email) {
			var ga = worldScripts.GalCopAdminServices;
			email.$createEmail({
				sender: expandMissionText("smuggling_customs"),
				subject: expandMissionText("smuggling_contraband_removal"),
				date: clock.seconds,
				message: expandDescription("[smugglers-removal-email]", {
					contraband: emailText,
					stationname: player.ship.dockedStation.displayName,
					systemname: System.systemNameForID(system.ID)
				}),
				expiryDays: ga._defaultExpiryDays
			});
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// build up an array of values that can be used to determine how many government/techlevel types exist in the system
this.$createSystemDataArray = function $createSystemDataArray() {
	this._systemData = [];
	for (var i = 0; i <= 255; i++) {
		var sys = System.infoForSystem(galaxyNumber, i);
		var item = sys.government * 100 + sys.techlevel;
		if (this._systemData.indexOf(item) === -1) this._systemData.push(item);
	}
}

//-------------------------------------------------------------------------------------------------------------
// updates the manifest screen entry for the smuggling hold contents
this.$updateManifest = function $updateManifest() {
	var list = [];
	var cargo_size = this.$smugglingCompartmentSize();
	if (cargo_size > 0) {
		var damaged = this.$isCompartmentDamaged();
		list.push(expandMissionText("smuggling_compartment_manifest", {used:this.$usedSpace().toFixed(0), size:cargo_size}) + (damaged === true ? " " + expandMissionText("smuggling_damaged") : "") + ":");
		var additem = "";
		for (var i = 0; i < this._sc_Cargo.length; i++) {
			if (additem === "") {
				additem += this.$padTextRight(this._sc_Cargo[i].quantity + " " + this._sc_Cargo[i].unit + " × " + displayNameForCommodity(this._sc_Cargo[i].commodity), 17);
			} else {
				additem += this._sc_Cargo[i].quantity + " " + this._sc_Cargo[i].unit + " × " + displayNameForCommodity(this._sc_Cargo[i].commodity);
				list.push(additem);
				additem = "";
			}
		}
		if (additem != "") list.push(additem);

		if (this._sc_Cargo.length > 0) {
			mission.setInstructions(list, this.name);
		} else {
			list.push(expandMissionText("smuggling_no_cargo"));
			mission.setInstructions(list, this.name);
		}
	} else {
		mission.setInstructions(null, this.name);
	}
}

//-------------------------------------------------------------------------------------------------------------
// play the buy/sell sound effects
this.$playSound = function $playSound(soundtype) {
	var mySound = new SoundSource;

	switch (soundtype) {
		case "buy":
			mySound.sound = "[buy-commodity]";
			break;
		case "sell":
			mySound.sound = "[sell-commodity]";
			break;
		case "mode":
			mySound.sound = "[@click]";
			break;
		case "activate":
			mySound.sound = "[@beep]";
			break;
		case "stop":
			mySound.sound = "[@boop]";
			break;
	}
	mySound.loop = false;
	mySound.play();
}

//-------------------------------------------------------------------------------------------------------------
// returns the illegal scale factor for a particular commodity at the current docked station
this.$commodityIllegalScale = function $commodityIllegalScale(commodity) {
	var p = player.ship;
	var stn = p.dockedStation;
	var scale = parseInt(stn.market[commodity].legality_import);
	if (scale === 0) scale = parseInt(stn.market[commodity].legality_export);
	return scale;
}

//-------------------------------------------------------------------------------------------------------------
// gets a list of all police ship models so we can use them consistently on smuggling screens
this.$getPoliceModel = function $getPoliceModel() {
	var shipkeys = Ship.keysForRole("police");

	if (shipkeys.length === 0) return "viper";
	var key = "";
	var tries = 0;
	do {
		key = shipkeys[Math.floor(Math.random() * shipkeys.length)];
		if (this.$modelIsAllowed(key) === false) key = "";
		tries += 1;
	} while (key === "" && tries < 5)
	if (key === "") key = "viper";
	return key;
}

//-------------------------------------------------------------------------------------------------------------
this.$modelIsAllowed = function $modelIsAllowed(shipkey) {
	var shipdata = Ship.shipDataForKey(shipkey);
	// are we allowed to include this data key in this system? check the conditions if there are some
	var include = true;
	if (shipdata.conditions) {
		var cond = shipdata.conditions.toString().split(",");
		//1,systemGovernment_number equal 4,systemGovernment_number,0,0,4,
		//1,systemEconomy_number notequal 4,systemEconomy_number,1,0,4
		//1,systemEconomy_number lessthan 4,systemEconomy_number,2,0,4
		//1,systemEconomy_number greaterthan 4,systemEconomy_number,3,0,4
		var offset = 0;
		var finish = false;
		var checking = -1;
		do {
			// get the value we're checking
			checking = -1;
			if (cond[offset + 2].substring(0, 6) === "mission") {
				if (missionVariables[cond[offset + 2].replace("mission_", "")]) {
					checking = missionVariables[cond[offset + 2].replace("mission_", "")];
					//log(this.name, "field = " + cond[offset + 2] + ", value = " + checking);
				} else {
					log(this.name, "!!NOTE: Condition value mission variable not set: " + cond[offset + 2]);
				}
			} else {
				switch (cond[offset + 2]) {
					case "systemGovernment_number":
						checking = system.government;
						break;
					case "systemEconomy_number":
						checking = system.economy;
						break;
					case "systemTechLevel_number":
						checking = system.techLevel;
						break;
					case "score_number":
						checking = player.score;
						break;
					case "galaxy_number":
						checking = galaxyNumber;
						break;
					case "planet_number":
						checking = system.ID;
						break;
					default:
						log(this.name, "!!NOTE: Condition value not catered for: " + cond[offset + 2]);
						break;
				}
			}
			// in case a mission variable is a text value of some sort
			if (isNaN(parseInt(checking)) && isNaN(parseFloat(checking))) {
				switch (cond[offset + 3]) {
					case "0": // equals
						if (checking != cond[offset + 5]) include = false;
						break;
					case "1": // not equals
						if (checking == cond[offset + 5]) include = false;
						break;
					default:
						log(this.name, "!!NOTE: Condition comparison not catered for: " + cond[offset + 3]);
						break;
				}
			} else {
				if (checking >= 0) {
					// work out the type of check, but in negative (or opposite)
					switch (cond[offset + 3]) {
						case "0": // equals
							if (checking !== parseInt(cond[offset + 5])) include = false;
							break;
						case "1": // not equals
							if (checking === parseInt(cond[offset + 5])) include = false;
							break;
						case "2": // lessthan
							if (checking >= parseInt(cond[offset + 5])) include = false;
							break;
						case "3": // greaterthan
							if (checking <= parseInt(cond[offset + 5])) include = false;
							break;
						default:
							log(this.name, "!!NOTE: Condition comparison not catered for: " + cond[offset + 3]);
							break;
						// others?
					}
				}
			}
			offset += 6;
			if (offset >= cond.length - 1) finish = true;
		} while (finish === false);
	} else if (shipdata.condition_script) {
		// or the condition script
		// create a dummy object to attach the script to so it can be executed
		if (system.sun && system.mainPlanet) {
			var temppos = system.sun.position.cross(system.mainPlanet.position).direction().multiply(4E9).subtract(system.mainPlanet.position);
			var tempalloy = system.addShips("alloy", 1, temppos, 0);
			tempalloy[0].setScript(shipdata.condition_script);
			include = tempalloy[0].script.allowSpawnShip(shipkey);
			tempalloy[0].remove(true);
		} else {
			include = true;
		}
	} else {
		// otherwise we're free to play
		include = true;
	}
	return include;
}