"use strict";
this.name = "Smugglers_DockMaster";
this.author = "phkb";
this.description = "Looks after the Dock Master interface screen.";
this.licence = "CC BY-NC-SA 3.0";

this._disabled = false;
this._debug = true;

this._bribeChance = []; // holds the chance a each dock master can be bribed
this._cargoLabelled = []; // array of original commodity and quantities, and new labels
// {oldCommodity: name, newCommodity: name, quantity: number of tons relabelled
this._bribeAttempted = false;
this._dockMasterStartTime = 0;
this._dockMasterEndTime = 0;

this._noSwitch = []; // these commodities can't be relabelled - populated through the smugglers_equipment.startUpComplete function
this._display = 0;
this._milkRuns = [];
this._successfulBribes = [];
this._cargoAtLaunch = [];
this._bribeCount = [];
this._menuColor = "orangeColor";
this._itemColor = "yellowColor";
this._disabledColor = "darkGrayColor";
this._galJump = false;
this._destReleaseCount = 0;
this._flagged = [];
this._curPage = 0;
this._marker = {};

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {

	if (this._disabled) {
		delete this.shipDockedWithStation;
		delete this.shipLaunchedFromStation;
		delete this.playerWillSaveGame;
		delete this.shipExitedWitchspace;
		delete this.playerEnteredNewGalaxy;
		delete this.shipSpawned;
		delete this.shipTakingDamage;
		delete this.playerSoldCargo;
		delete this.startUpComplete;
		delete this.guiScreenChanged;
		return;
	}

	// add a mission screen exception to Xenon UI
	if (worldScripts.XenonUI) {
		var wx = worldScripts.XenonUI;
		wx.$addMissionScreenException("oolite-smuggling-dockmaster-milkrun");
	}
	// add a mission screen exception to Xenon Redux UI
	if (worldScripts.XenonReduxUI) {
		var wxr = worldScripts.XenonReduxUI;
		wxr.$addMissionScreenException("oolite-smuggling-dockmaster-milkrun");
	}

	var core = worldScripts.Smugglers_CoreFunctions;
	this.$padTextRight = core.$padTextRight;
	this.$padTextLeft = core.$padTextLeft;
	this.$isBigGuiActive = core.$isBigGuiActive;
	this.$rand = core.$rand;

	this.$getMilkRuns();
	if (missionVariables.Smuggling_BribeChance) {
		this._bribeChance = JSON.parse(missionVariables.Smuggling_BribeChance);
		delete missionVariables.Smuggling_BribeChance;
	} else {
		this.$setupChance();
	}
	if (missionVariables.Smuggling_Relabels) {
		this._cargoLabelled = JSON.parse(missionVariables.Smuggling_Relabels);
		delete missionVariables.Smuggling_Relabels;
	}
	if (missionVariables.Smuggling_SuccessfulBribes) {
		this._successfulBribes = JSON.parse(missionVariables.Smuggling_SuccessfulBribes);
		delete missionVariables.Smuggling_SuccessfulBribes;
	}
	if (missionVariables.Smuggling_BribeAttempted) this._bribeAttemped = (missionVariables.Smuggling_BribeAttemped === 1 ? true : false);

	if (missionVariables.Smuggling_DockMasterStartTime) {
		this._dockMasterStartTime = missionVariables.Smuggling_DockMasterStartTime;
		this._dockMasterEndTime = missionVariables.Smuggling_DockMasterEndTime;
	} else {
		this.$setDockMasterHours();
	}

	if (missionVariables.Smuggling_BribeCount) {
		this._bribeCount = JSON.parse(missionVariables.Smuggling_BribeCount);
		delete missionVariables.Smuggling_BribeCount;
	} else {
		this.$resetBribeCount();
	}

	if (missionVariables.Smuggling_DockMasterFlagged) {
		this._flagged = JSON.parse(missionVariables.Smuggling_DockMasterFlagged);
		delete missionVariables.Smuggling_DockMasterFlagged;
	}
	if (missionVariables.Smuggling_DockMasterUnlockCount) {
		this._destReleaseCount = missionVariables.Smuggling_DockMasterUnlockCount;
		delete missionVariables.Smuggling_DockMasterUnlockCount;
	}

	// set up the interface screen, if required
	this.$initMainInterface(player.ship.dockedStation);
	if (this._cargoLabelled.length > 0) {
		this.$initUnlabelInterface(player.ship.dockedStation);
	}

	// upgraded features in v1.84
	if (0 < oolite.compareVersion("1.84")) {
		delete this.shipDumpedCargo;
	} else {
		delete this.shipSpawned;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipDockedWithStation = function (station) {
	// set up dock master interface screen
	this.$initMainInterface(station);
	if (this._cargoLabelled.length > 0) {
		this.$initUnlabelInterface(station);
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipLaunchedFromStation = function (station) {
	if (missionVariables.Smuggling_BribeChance) delete missionVariables.Smuggling_BribeChance;
	if (missionVariables.Smuggling_Relabels) delete missionVariables.Smuggling_Relabels;
	if (missionVariables.Smuggling_SuccessfulBribes) delete missionVariables.Smuggling_SuccessfulBribes;
	if (missionVariables.Smuggling_BribeCount) delete missionVariables.Smuggling_BribeCount;

	var p = player.ship;
	var m = p.manifest.list;
	this._cargoAtLaunch = [];
	// grab a snapshot of the cargo for comparison later
	for (var i = 0; i < m.length; i++) {
		if (m[i].quantity > 0) this._cargoAtLaunch.push({
			commodity: m[i].commodity,
			quantity: m[i].quantity
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function () {
	missionVariables.Smuggling_BribeChance = JSON.stringify(this._bribeChance);
	missionVariables.Smuggling_Relabels = JSON.stringify(this._cargoLabelled);
	missionVariables.Smuggling_SuccessfulBribes = JSON.stringify(this._successfulBribes);
	missionVariables.Smuggling_BribeCount = JSON.stringify(this._bribeCount);
	missionVariables.Smuggling_BribeAttempted = (this._bribeAttemped ? 1 : 0);
	if (this._flagged.length > 0) {
		missionVariables.Smuggling_DockMasterFlagged = JSON.stringify(this._flagged)
	}
	missionVariables.Smuggling_DockMasterUnlockCount = this._destReleaseCount;
}

//-------------------------------------------------------------------------------------------------------------
this.shipExitedWitchspace = function () {
	this._bribeAttempted = false;
	this._destReleaseCount = 0;
	this._flagged = [];
	this.$setDockMasterHours();
	if (this._galJump === true) {
		this._galJump = false;
		this.$setupChance();
		this.$getMilkRuns();
		this.$resetBribeCount();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerEnteredNewGalaxy = function (galaxyNumber) {
	this._successfulBribes = [];
	this._galJump = true;
}

//-------------------------------------------------------------------------------------------------------------
this.shipSpawned = function (ship) {
	// did the player just dump relabelled cargo?
	// if so, reduce the count in the array
	// and unlabel the dumped cargo
	var p = player.ship;
	if (p && ship && ship.isCargo && ship.position && p.position && p.position.distanceTo(ship) < 300) {
		//gotcha
		var total = this.$getTotalRelabelled(ship.commodity);
		if (p.manifest[ship.commodity] < total) {
			for (var i = 0; i < this._cargoLabelled.length; i++) {
				// find the first matching relabelled commodity and adjust it
				if (this._cargoLabelled[i].newCommodity === ship.commodity && this._cargoLabelled[i].quantity > 0) {
					this._cargoLabelled[i].quantity -= ship.commodityAmount;
					var check = ship.setCargo(this._cargoLabelled[i].oldCommodity);
					break;
				}
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipDumpedCargo = function (cargo) {
	var p = player.ship;
	var total = this.$getTotalRelabelled(cargo.commodity);
	if (p.manifest[cargo.commodity] < total) {
		for (var i = 0; i < this._cargoLabelled.length; i++) {
			// find the first matching relabelled commodity and adjust it
			if (this._cargoLabelled[i].newCommodity === cargo.commodity && this._cargoLabelled[i].quantity > 0) {
				this._cargoLabelled[i].quantity -= cargo.commodityAmount;
				var check = ship.setCargo(this._cargoLabelled[i].oldCommodity);
				break;
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipTakingDamage = function (dmg_amount, whom, type) {
	// work out if the any of the relabelled cargo was destroyed
	// if so, reduce the count in the array
	// basically, relabelled cargo should be destroyed first
	var p = player.ship;
	var se = worldScripts.Smugglers_Equipment;
	for (var i = 0; i < se._commodityIDList.length; i++) {
		var holdamt = p.manifest[se._commodityIDList[i]];
		var origamt = this.$launchAmount(se._commodityIDList[i]);
		var amount = this.$getTotalRelabelled(se._commodityIDList[i]);
		if (holdamt < origamt && amount > 0) {
			// how many units do we have to remove?
			var to_remove = (origamt - holdamt);

			// find the first instance of a relabelled cargo
			for (var j = 0; j < this._cargoLabelled.length; j++) {
				if (this._cargoLabelled[j].quantity > 0 && to_remove > 0) {
					// reduce the cargo by the removed amount
					this._cargoLabelled[j].quantity -= to_remove;
					// if we've just gone negative, we've wiped out all of that cargo batch
					if (this._cargoLabelled[j].quantity < 0) {
						// pick up the remainder
						to_remove = this._cargoLabelled[j].quantity * -1;
						// reset this one back to zero
						this._cargoLabelled[j].quantity = 0;
					}
				}
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// if we scoop cargo make sure it gets added to the at_launch list
this.shipScoopedOther = function (whom) {
	var si = worldScripts.Smugglers_Illegal;
	var check = 0;
	if (si._commodityInfo[whom.commodity]) {
		check = si._commodityInfo[whom.commodity][0];
		if (check > 0) {
			var origamt = this.$launchAmount(whom.commodity);
			if (origamt > 0) {
				for (var i = 0; i < this._cargoAtLaunch.length; i++) {
					if (this._cargoAtLaunch[i].commodity === whom.commodity) this._cargoAtLaunch[i].quantity += whom.commodityAmount;
				}
			} else {
				this._cargoAtLaunch.push({
					commodity: whom.commodity,
					quantity: whom.commodityAmount
				});
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerSoldCargo = function (commodity, units, price) {
	// did the player just sell cargo that has been relabelled?
	// if so, reverse the sale: add the relabelled units back, give the player their money back, and send a console message
	var p = player.ship;
	var relabelled_amount = this.$getTotalRelabelled(commodity);
	if (relabelled_amount > 0 && p.manifest[commodity] < relabelled_amount) {
		var goodunits = units - relabelled_amount;
		if (units === 1) goodunits = 0;
		var reclaimunits = units - goodunits;

		p.manifest[commodity] = relabelled_amount;
		p.dockedStation.setMarketQuantity(commodity, p.dockedStation.market[commodity].quantity - reclaimunits);

		player.credits -= ((reclaimunits * price) / 10);
		player.consoleMessage(expandMissionText("smuggling_sale_reversed"));
	}
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged_hide = function (to, from) {
	if (guiScreen === "GUI_SCREEN_INTERFACES") {
		mission.unmarkSystem({
			system: this._marker.id1,
			name: this.name + "_mark1"
		});
		mission.unmarkSystem({
			system: this._marker.id2,
			name: this.name + "_mark2"
		});
		if (this._marker.id1 < this._marker.id2) {
			SystemInfo.setInterstellarProperty(galaxyNumber, this._marker.id1, this._marker.id2, 2, "link_color", null);
		} else {
			SystemInfo.setInterstellarProperty(galaxyNumber, this._marker.id2, this._marker.id1, 2, "link_color", null);
		}
		this._marker = {};
		delete this.guiScreenChanged;
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns the total number of units that have been relabelled to the passed commodity type
this.$getTotalRelabelled = function $getTotalRelabelled(commodity) {
	var amount = 0;
	for (var i = 0; i < this._cargoLabelled.length; i++) {
		if (this._cargoLabelled[i].newCommodity === commodity) amount += this._cargoLabelled[i].quantity;
	}
	return amount;
}

//-------------------------------------------------------------------------------------------------------------
// returns the total number of units of the original commodity
this.$getTotalUnlabelled = function $getTotalUnlabelled(commodity) {
	var amount = 0;
	for (var i = 0; i < this._cargoLabelled.length; i++) {
		if (this._cargoLabelled[i].oldCommodity === commodity) amount += this._cargoLabelled[i].quantity;
	}
	return amount;
}

//-------------------------------------------------------------------------------------------------------------
// returns the amount of a particular commodity we had at launch
this.$launchAmount = function $launchAmount(commodity) {
	for (var i = 0; i < this._cargoAtLaunch.length; i++) {
		if (this._cargoAtLaunch[i].commodity === commodity) return this._cargoAtLaunch[i].quantity;
	}
	return 0;
}

//-------------------------------------------------------------------------------------------------------------
// initialise the dock master F4 screen entries
this.$initMainInterface = function $initMainInterface(station) {
	// no station? no interface
	if (!station) return;

	// not galcop aligned? no interface
	if (station.allegiance != "galcop") return;

	// did we get to the station when the dock master is on duty?
	if (clock.hoursComponent < this._dockMasterStartTime || clock.hoursComponent > this._dockMasterEndTime) return;

	if (this._bribeAttempted === false) {
		station.setInterface(this.name + "dockmaster", {
			title: expandMissionText("smuggling_dock_master"),
			category: expandMissionText("smuggling_interface_category"),
			summary: expandMissionText("smuggling_interface_summary"),
			callback: this.$initialDockMasterOptions.bind(this)
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// initialises the remove labelling interface screen
this.$initUnlabelInterface = function $initUnlabelInterface(station) {
	station.setInterface(this.name + "unlabel", {
		title: expandMissionText("smuggling_unlabel"),
		category: expandDescription("[interfaces-category-ship-systems]"),
		summary: expandMissionText("smuggling_unlabel_summary"),
		callback: this.$removeLabellingList.bind(this)
	});
}

//-------------------------------------------------------------------------------------------------------------
this.$initialDockMasterOptions = function $initialDockMasterOptions() {
	this._display = 0;
	this.$showDockMasterOptions();
}

//-------------------------------------------------------------------------------------------------------------
this.$showDockMasterOptions = function $showDockMasterOptions() {

	var curChoices = {};
	var text = "";
	var p = player.ship;
	var def = "";

	var pagesize = 20;
	if (this.$isBigGuiActive() === true) pagesize = 26;

	if (this._display === 0) {
		if (clock.hoursComponent < 12) text += expandMissionText("smuggling_good_morning");
		if (clock.hoursComponent >= 12 && clock.hoursComponent < 18) text += expandMissionText("smuggling_good_afternoon");
		if (clock.hoursComponent >= 18) text += expandMissionText("smuggling_good_evening");

		text += " " + expandDescription("[dockmaster-welcome]");

		var itemcount = 1;
		curChoices["01_MILKRUN"] = {
			text: "[dockmaster-milkruns]",
			color: this._menuColor
		};
		if (this._bribeAttempted === false && player.credits > 0 && this.$playerHasBribableCargo() === true) {
			curChoices["02_BRIBE"] = {
				text: "[dockmaster-bribe-cargo]",
				color: this._menuColor
			};
			itemcount += 1;
		}
		if (worldScripts.BountySystem_MostWanted && worldScripts.StationDockControl && p.equipmentStatus("EQ_HUNTER_LICENCE") === "EQUIPMENT_OK" && this._destReleaseCount < 3 && this._flagged.length > 0) {
			curChoices["05_HUNTER_REQ"] = {
				text: "[dockmaster-release-dest]",
				color: this._menuColor
			};
			itemcount += 1;
		}
		curChoices["98_SPACER"] = "";
		itemcount += 1;

		curChoices["99_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		for (var i = 0; i < ((pagesize - 2) - itemcount); i++) {
			curChoices["99_SPACER_" + i] = "";
		}

		def = "99_EXIT";

		var opts = {
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_dock_master"),
			allowInterrupt: true,
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	// milk runs
	if (this._display === 1) {
		text = expandDescription("[milkrun-apology]");
		var found = false;

		for (var i = this._milkRuns.length - 1; i >= 0; i--) {
			var dist1 = System.infoForSystem(galaxyNumber, system.ID).distanceToSystem(System.infoForSystem(galaxyNumber, this._milkRuns[i].id1));
			var dist2 = System.infoForSystem(galaxyNumber, system.ID).distanceToSystem(System.infoForSystem(galaxyNumber, this._milkRuns[i].id2));
			if (dist1 < 15 || dist2 < 15) {
				// found one
				text = "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n" + expandDescription("[milkrun]", {
					system1: System.infoForSystem(galaxyNumber, this._milkRuns[i].id1).name,
					system2: System.infoForSystem(galaxyNumber, this._milkRuns[i].id2).name
				});
				this._marker = this._milkRuns[i];
				this.guiScreenChanged = this.guiScreenChanged_hide;
				mission.markSystem({
					system: this._marker.id1,
					name: this.name + "_mark1",
					markerShape: "MARKER_DIAMOND",
					markerColor: "whiteColor",
					markerScale: 1.1
				});
				mission.markSystem({
					system: this._milkRuns[i].id2,
					name: this.name + "_mark2",
					markerShape: "MARKER_DIAMOND",
					markerColor: "whiteColor",
					markerScale: 1.1
				});
				if (this._marker.id1 < this._marker.id2) {
					SystemInfo.setInterstellarProperty(galaxyNumber, this._marker.id1, this._marker.id2, 2, "link_color", "whiteColor");
				} else {
					SystemInfo.setInterstellarProperty(galaxyNumber, this._marker.id2, this._marker.id1, 2, "link_color", "whiteColor");
				}
				found = true;
				break;
			}
		}
		if (found === false) {
			var opts = {
				screenID: "oolite-smuggling-dockmaster-map",
				title: expandMissionText("smuggling_dock_master"),
				allowInterrupt: true,
				overlay: {
					name: "stgu-chat.png",
					height: 546
				},
				exitScreen: "GUI_SCREEN_INTERFACES",
				message: text
			};
		} else {
			var point1 = System.infoForSystem(galaxyNumber, this._marker.id1).coordinates;
			var point2 = System.infoForSystem(galaxyNumber, this._marker.id2).coordinates;
			var xdiff = (point1.x - point2.x) / 2;
			var ydiff = (point1.y - point2.y) / 2;
			var opts = {
				screenID: "oolite-smuggling-dockmaster-milkrun",
				title: expandMissionText("smuggling_dock_master"),
				allowInterrupt: true,
				backgroundSpecial: "CUSTOM_CHART",
				customChartZoom: 1.4,
				customChartCentreInLY: new Vector3D(point1.x - xdiff, point1.y - ydiff, 0),
				exitScreen: "GUI_SCREEN_INTERFACES",
				message: text
			};
		}
	}

	// display cargo to alter
	if (this._display === 2) {
		var cargo = p.manifest.list;
		var itemcount = 0;
		var list = [];

		for (var i = 0; i < cargo.length; i++) {
			// make sure we don't relabel relabelled cargo
			var q = cargo[i].quantity;
			if (this._cargoLabelled.length > 0) {
				for (var j = 0; j < this._cargoLabelled.length; j++) {
					if (this._cargoLabelled[j].newCommodity === cargo[i].commodity) q -= this._cargoLabelled[j].quantity;
				}
			}
			// make sure we don't relabel New Cargoes
			var cteCargo = worldScripts.Smugglers_Equipment.$calcCTEcargo(cargo[i].commodity);
			if (cteCargo > 0) q -= cteCargo;

			if (q > 0) {
				if (this._noSwitch.indexOf(cargo[i].commodity) === -1) {
					list.push({
						menu: "50_" + (i < 10 ? "0" : "") + i + "_CARGO~" + cargo[i].commodity,
						text: cargo[i].displayName + " (" + q + cargo[i].unit + ")",
						color: this._menuColor
					})
				}
			}
		}

		if (list.length >= pagesize) {
			pagesize -= 4;
		} else {
			pagesize -= 2;
		}

		for (var i = 0; i < pagesize; i++) {
			var ref = i + (this._curPage * pagesize);
			if (ref < list.length) {
				curChoices[list[ref].menu] = {
					text: list[ref].text,
					color: list[ref].color
				};
				itemcount += 1;
			}
		}

		if (list.length > pagesize) {
			if (this._curPage < Math.floor(list.length / pagesize)) {
				curChoices["Z_NEXT"] = {
					text: "[illegal-nextpage]",
					color: this._itemColor
				};
			} else {
				curChoices["Z_NEXT"] = {
					text: "[illegal-nextpage]",
					color: this._disabledColor,
					unselectable: true
				};
			}
			if (this._curPage > 0) {
				curChoices["Z_PREV"] = {
					text: "[illegal-prevpage]",
					color: this._itemColor
				};
			} else {
				curChoices["Z_PREV"] = {
					text: "[illegal-prevpage]",
					color: this._disabledColor,
					unselectable: true
				};
			}
		}

		for (var i = 0; i < ((pagesize + 1) - itemcount); i++) {
			curChoices["99_SPACER_" + i] = "";
		}

		var opts = {
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_dock_master"),
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			allowInterrupt: false,
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			messageKey: "cargo-relabel-accept"
		};
	}

	// display options to switch to
	if (this._display === 3) {

		var se = worldScripts.Smugglers_Equipment;

		text = expandMissionText("cargo-relabel-question", {
			cargo: displayNameForCommodity(this._tfrCargo)
		});

		var itemcount = 0;
		var list = [];

		for (var i = 0; i < se._commodityIDList.length; i++) {
			if (this._noSwitch.indexOf(se._commodityIDList[i]) === -1 && se._commodityIDList != this._tfrCargo) {
				list.push({
					menu: "40_" + (i < 10 ? "0" : "") + i + "~" + se._commodityIDList[i],
					text: displayNameForCommodity(se._commodityIDList[i]),
					color: this._menuColor
				})
			}
		}
		if (list.length >= pagesize) {
			pagesize -= 4;
		} else {
			pagesize -= 2;
		}
		for (var i = 0; i < pagesize; i++) {
			var ref = i + (this._curPage * pagesize);
			if (ref < list.length) {
				curChoices[list[ref].menu] = {
					text: list[ref].text,
					color: list[ref].color
				};
				itemcount += 1;
			}
		}

		if (list.length > pagesize) {
			if (this._curPage < Math.floor(list.length / pagesize)) {
				curChoices["99_NEXT"] = {
					text: "[illegal-nextpage]",
					color: this._itemColor
				};
			} else {
				curChoices["99_NEXT"] = {
					text: "[illegal-nextpage]",
					color: this._disabledColor,
					unselectable: true
				};
			}
			if (this._curPage > 0) {
				curChoices["99_PREV"] = {
					text: "[illegal-prevpage]",
					color: this._itemColor
				};
			} else {
				curChoices["99_PREV"] = {
					text: "[illegal-prevpage]",
					color: this._disabledColor,
					unselectable: true
				};
			}
		}
		for (var i = 0; i < ((pagesize + 1) - itemcount); i++) {
			curChoices["60_SPACER_" + i] = "";
		}

		var opts = {
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_dock_master"),
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			allowInterrupt: false,
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			message: text
		};

	}

	// linking bounty most wanted and SDC
	if (this._display === 4) {
		// display list of flagged ships 
		text = expandMissionText("smuggling_ship_dest");
		var itemcount = 0;

		for (var i = 0; i < this._flagged.length; i++) {
			// if you don't have the dough, it won't let you go
			// saves having to have another response screen saying "You don't have enough cash"
			if (player.credits > parseInt(this._flagged[i].cost)) {
				curChoices["01_FLAGGED~" + i] = {
					text: this._flagged[i].text,
					color: this._menuColor,
					alignment: "LEFT"
				};
			} else {
				curChoices["01_FLAGGED~" + i] = {
					text: this._flagged[i].text,
					color: this._disabledColor,
					alignment: "LEFT",
					unselectable: true
				};
			}
			itemcount += 1;
		}

		curChoices["97_SPACER"] = "";
		itemcount += 1;

		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		for (var i = 0; i < ((pagesize - 2) - itemcount); i++) {
			curChoices["97_SPACER_" + i] = "";
		}

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_dock_master"),
			allowInterrupt: true,
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};
	}

	// ship dest unlock successful
	if (this._display === 5) {
		text = expandMissionText("smuggling_ship_dest_unlocked");

		curChoices["98A_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};
		def = "98A_EXIT";

		var opts = {
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_dock_master"),
			allowInterrupt: true,
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};

	}

	// ship dest unlock unsuccessful
	if (this._display === 6) {
		text = expandMissionText("smuggling_ship_not_found");

		curChoices["98A_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};
		def = "98A_EXIT";

		var opts = {
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_dock_master"),
			allowInterrupt: true,
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};
	}

	mission.runScreen(opts, this.$screenHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$screenHandler = function $screenHandler(choice) {

	if (choice == null) return;

	switch (choice) {
		case "01_MILKRUN":
			this._display = 1;
			break;
		case "02_BRIBE":
			this.$getBribeAmount("manifest");
			return;
		case "05_HUNTER_REQ":
			this._display = 4;
			break;
		case "98_EXIT":
			this._display = 1;
			break;
		case "98A_EXIT":
			this._display = 0;
			break;
		case "99_NEXT":
			this._curPage += 1;
			break;
		case "99_PREV":
			this._curPage -= 1;
			break;
	}

	if (choice.indexOf("01_FLAGGED") >= 0) {
		var idx = parseInt(choice.substring(choice.indexOf("~") + 1));
		var ref = parseInt(this._flagged[idx].index);
		var shpRef = parseInt(this._flagged[idx].shipIndex);
		var cost = parseInt(this._flagged[idx].cost);
		var mw = worldScripts.BountySystem_MostWanted;
		var found = false;
		for (var i = 0; i < mw._wanted.length; i++) {
			if (mw._wanted[i].index === ref) {
				var sdci = worldScripts.StationDockControl_Interface;
				var item = sdci._viewing[shpRef];
				if (item.pilotName === mw._wanted[i].pilotName) {
					player.credits -= cost;
					player.consoleMessage(expandMissionText("smuggling_charged", { amount: formatCredits(cost, false, true) }));
					item.destinationHidden = false;
					this._destReleaseCount += 1;
					this._flagged.splice(idx, 1);
					found = true;
					this._display = 5;
				}
			}
		}
		if (found === false) {
			this._display = 6;
		}
	}

	if (choice.indexOf("50_") >= 0) {
		this._tfrCargo = choice.substring(choice.indexOf("~") + 1);
		this._display = 3;
		this._curPage = 0;
	}

	if (choice.indexOf("40_") >= 0) {
		var newCmdty = choice.substring(choice.indexOf("~") + 1);
		var p = player.ship;
		var amount = p.manifest[this._tfrCargo];

		// if we already have relabelled the selected cargo, get the un-relabelled portion of it
		for (var i = 0; i < this._cargoLabelled.length; i++) {
			if (this._cargoLabelled[i].newCommodity === this._tfrCargo) amount -= this._cargoLabelled[i].quantity;
		}

		// do we already have some relabelled cargo of the same type (matching old and new
		var found = false;
		for (var i = 0; i < this._cargoLabelled.length; i++) {
			if (this._cargoLabelled[i].newCommodity === newCmdty && this._cargoLabelled[i].oldCommodity === this._tfrCargo) {
				this._cargoLabelled[i].quantity += amount
				found = true;
			}
		}
		if (found === false) this._cargoLabelled.push({
			oldCommodity: this._tfrCargo,
			newCommodity: newCmdty,
			quantity: amount
		});

		p.manifest[this._tfrCargo] -= amount;
		p.manifest[newCmdty] += amount;

		this.$initUnlabelInterface(p.dockedStation);

		var extramsg = "";
		var se = worldScripts.Smugglers_Equipment;
		if (se.$hasSmugglingCompartment() === true) extramsg = expandMissionText("cargo-relabel-reminder");

		mission.runScreen({
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_dock_master"),
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			message: expandDescription("[dockmaster-complete]") + extramsg,
			exitScreen: "GUI_SCREEN_INTERFACES"
		});
		p.dockedStation.setInterface(this.name + "dockmaster", null);
	}


	if (choice != "99_EXIT" && choice.indexOf("40_") === -1) {
		this.$showDockMasterOptions();
	}

}

//-------------------------------------------------------------------------------------------------------------
// prompts the user for a new phase
this.$getBribeAmount = function $getBribeAmount(type) {

	var text = "";

	this._bribeAttempted = true;

	var inttype = Math.floor((this._bribeChance[system.ID] * 4) + 1);
	var inttypedesc = expandDescription("[interest-type" + inttype + "]");
	var lastBribe = this.$previousBribeAmount();
	if (type === "manifest") {
		text = expandDescription("[dockmaster-question]", {
			interesttype: inttypedesc
		}) + "\n\n" +
			"(" + expandDescription("[dockmaster-cost]", {
				credits: formatCredits(player.credits, false, true)
			}) +
			(lastBribe > 0 ? expandDescription("[dockmaster-lastbribe]", {
				bribeamount: formatCredits(lastBribe, false, true)
			}) : "") + ")";
	} else {
		text = expandMissionText("smuggling_keys_kingdom") + " " + expandDescription("[dockmaster-question]", {
			interesttype: inttypedesc
		}) + "\n\n" +
			"(" + expandDescription("[dockmaster-cost]", {
				credits: formatCredits(player.credits, false, true)
			})
	}

	var opts = {
		screenID: "oolite-smuggling-dockmaster-map",
		title: expandMissionText("smuggling_dock_master"),
		allowInterrupt: false,
		overlay: {
			name: "stgu-chat.png",
			height: 546
		},
		exitScreen: "GUI_SCREEN_INTERFACES",
		message: text,
		textEntry: true
	};

	mission.runScreen(opts, this.$getBribeAmountInputManifest, this);
}

//-------------------------------------------------------------------------------------------------------------
// parses the input from the getBribeAmount screen
this.$getBribeAmountInputManifest = function $getBribeAmountInputManifest(param) {
	var p = player.ship;
	if (parseInt(param) >= 1 && parseInt(param) <= player.credits) {
		var amount = parseInt(param);
		// will this work
		var chance = this._bribeChance[system.ID];
		// higher amounts are more likely to be accepted
		// chance of 0.5 would mean you would need to offer them 1690 credits
		//if (this._debug) log(this.name, "min bribe amount = " + (Math.pow(parseInt(chance * 32), 1.8) * (1 + system.productivity / 56300) * this._bribeCount[system.ID]));
		if ((Math.pow(parseInt(chance * 32), 1.8) * (1 + system.productivity / 56300) * this._bribeCount[system.ID]) <= amount) {
			player.credits -= amount;
			this._display = 2;
			this._curPage = 0;
			this.$addBribe(system.ID);
			this._successfulBribes.push({
				system: system.ID,
				bribeAmount: amount
			});
			this.$showDockMasterOptions();
		} else {
			this._display = 0;
			if (Math.random() > chance) {
				mission.runScreen({
					screenID: "oolite-smuggling-dockmaster-map",
					title: expandMissionText("smuggling_dock_master"),
					overlay: {
						name: "stgu-chat.png",
						height: 546
					},
					message: expandDescription("[dockmaster-angry-nopenalty]"),
					exitScreen: "GUI_SCREEN_INTERFACES"
				});
			} else {
				var penalty = (this.$rand(10) + 3);
				p.setBounty(player.bounty + penalty, "attempted bribe");
				mission.runScreen({
					screenID: "oolite-smuggling-dockmaster-map",
					title: expandMissionText("smuggling_dock_master"),
					overlay: {
						name: "stgu-chat.png",
						height: 546
					},
					message: expandDescription("[dockmaster-angry-penalty]"),
					exitScreen: "GUI_SCREEN_INTERFACES"
				});
				// send email (if installed)
				var email = worldScripts.EmailSystem;
				if (email) {
					var ga = worldScripts.GalCopAdminServices;
					email.$createEmail({
						sender: "GalCop Customs",
						subject: "Attempt to bribe official",
						date: clock.seconds,
						message: expandDescription("[smugglers-failed-bribe-email]", {
							legal_penalty: penalty,
							stationname: player.ship.dockedStation.displayName,
							systemname: System.systemNameForID(system.ID)
						}),
						expiryDays: ga._defaultExpiryDays
					});
				}
			}
			p.dockedStation.setInterface(this.name + "dockmaster", null);
		}
	} else {
		this._display = 0;
		mission.runScreen({
			screenID: "oolite-smuggling-dockmaster-map",
			title: expandMissionText("smuggling_bribe_official"),
			overlay: {
				name: "stgu-chat.png",
				height: 546
			},
			message: expandDescription("[dockmaster-skipbribe]"),
			exitScreen: "GUI_SCREEN_STATUS"
		});
		p.dockedStation.setInterface(this.name + "dockmaster", null);
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$removeLabellingList = function $removeLabellingList() {

	var curChoices = {};
	var se = worldScripts.Smugglers_Equipment;

	var pagesize = 20;

	if (this.$isBigGuiActive() === true) pagesize = 26;

	for (var i = 0; i < this._cargoLabelled.length; i++) {
		curChoices["01_CARGO~" + (i < 10 ? "0" : "") + i] = {
			text: this._cargoLabelled[i].quantity + se.$getCommodityType(this._cargoLabelled[i].newCommodity) + " " + displayNameForCommodity(this._cargoLabelled[i].newCommodity) +
				", " + expandMissionText("smuggling_back_to") + " " + displayNameForCommodity(this._cargoLabelled[i].oldCommodity),
			color: this._menuColor
		};
	}

	curChoices["98_SPACER_" + i] = "";
	curChoices["99_EXIT"] = {
		text: "[illegal-return]",
		color: this._itemColor
	};

	for (var i = 0; i < ((pagesize - 3) - this._cargoLabelled.length); i++) {
		curChoices["99_SPACER_" + i] = "";
	}

	var def = "99_EXIT";

	var opts = {
		screenID: "oolite-smuggling-dockmaster-map",
		titleKey: "smuggling_unlabel",
		allowInterrupt: true,
		overlay: {
			name: "stgu-id.png",
			height: 546
		},
		exitScreen: "GUI_SCREEN_INTERFACES",
		choices: curChoices,
		initialChoicesKey: def,
		messageKey: "cargo-relabel-heading"
	};

	mission.runScreen(opts, this.$removeLabellingHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$removeLabellingHandler = function $removeLabellingHandler(choice) {
	if (choice == null) return;

	if (choice.indexOf("01_") >= 0) {
		var idx = parseInt(choice.substring(choice.indexOf("~") + 1));
		this.$removeLabelling(idx);
		if (this._cargoLabelled.length > 0) this.$removeLabellingList();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$removeLabelling = function $removeLabelling(idx) {

	var p = player.ship;

	// revert cargo back to original
	// make sure we still have the right amount
	if (p.manifest[this._cargoLabelled[idx].newCommodity] < this._cargoLabelled[idx].quantity)
		this._cargoLabelled[idx].quantity = p.manifest[this._cargoLabelled[idx].newCommodity];
	// deduct from the new commodity name
	p.manifest[this._cargoLabelled[idx].newCommodity] -= this._cargoLabelled[idx].quantity;
	// add to the old commodity name
	p.manifest[this._cargoLabelled[idx].oldCommodity] += this._cargoLabelled[idx].quantity;

	this._cargoLabelled.splice(idx, 1);

	if (this._cargoLabelled.length === 0) {
		p.dockedStation.setInterface(this.name + "unlabel", null);
	}
}

//-------------------------------------------------------------------------------------------------------------
// sets up the chance a dock master can be bribed, for each of the main stations
this.$setupChance = function $setupChance() {
	this._bribeChance = [];
	for (var i = 0; i <= 255; i++) {
		this._bribeChance.push(Math.random());
	}
}

//-------------------------------------------------------------------------------------------------------------
// sets the time frame where the dock master can be called on - he doesn't work 24 hours a day!
this.$setDockMasterHours = function $setDockMasterHours() {
	// set a start time between 7 and 10
	this._dockMasterStartTime = 6 + this.$rand(4);
	// set an end time between 7 and 10 hours later
	this._dockMasterEndTime = this.$dockMasterStartTime + (this.$rand(4) + 6)
}

//-------------------------------------------------------------------------------------------------------------
// removes labelling on a specific amount of cargo
this.$removeRealCargoQuantity = function $removeRealCargoQuantity(commodity, amount) {
	var p = player.ship;
	var remainder = amount;
	for (var i = this._cargoLabelled.length - 1; i >= 0; i--) {
		if (remainder > 0 && this._cargoLabelled[i].oldCommodity === commodity) {
			if (this._cargoLabelled[i].quantity >= remainder) {
				if (p.manifest[this._cargoLabelled[i].newCommodity] >= remainder) {
					p.manifest[this._cargoLabelled[i].newCommodity] -= remainder;
					p.manifest[this._cargoLabelled[i].oldCommodity] += remainder;
					this._cargoLabelled[i].quantity -= remainder;
					remainder = 0;
				} else {
					var subamt = p.manifest[this._cargoLabelled[i].newCommodity];
					p.manifest[this._cargoLabelled[i].newCommodity] = 0;
					p.manifest[this._cargoLabelled[i].oldCommodity] += subamt;
					this._cargoLabelled[i].quantity -= subamt;
					remainder -= subamt;
				}
			} else {
				var correction = remainder - this._cargoLabelled[i].quantity;
				if (p.manifest[this._cargoLabelled[i].newCommodity] >= correction) {
					p.manifest[this._cargoLabelled[i].newCommodity] -= correction;
					p.manifest[this._cargoLabelled[i].oldCommodity] += correction;
					this._cargoLabelled[i].quantity -= correction;
					remainder -= correction;
				} else {
					var subamt = p.manifest[this._cargoLabelled[i].oldCommodity];
					p.manifest[this._cargoLabelled[i].newCommodity] = 0;
					p.manifest[this._cargoLabelled[i].oldCommodity] += subamt;
					this._cargoLabelled[i].quantity = 0;
					remainder -= subamt;
				}
			}
		}
	}
	return remainder;
}

//-------------------------------------------------------------------------------------------------------------
// returns true if the player cargo that the dock master can relabel, otherwise false
this.$playerHasBribableCargo = function $playerHasBribableCargo() {
	var p = player.ship;
	var cargo = p.manifest.list;
	var result = false;
	for (var i = 0; i < cargo.length; i++) {
		if (this._noSwitch.indexOf(cargo[i].commodity) === -1) result = true;
	}
	return result;
}

//-------------------------------------------------------------------------------------------------------------
// increment the number of times the dm has been bribed
this.$addBribe = function $addBribe(sysID) {
	this._bribeCount[sysID] += 0.1 + (system.pseudoRandomNumber > 0.5 ? 0.1 : 0);
}

//-------------------------------------------------------------------------------------------------------------
// reset the bribecount array
this.$resetBribeCount = function $resetBribeCount() {
	this._bribeCount = [];
	for (var i = 0; i <= 255; i++) {
		this._bribeCount.push(1);
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns the last successful bribe amount for the current system
this.$previousBribeAmount = function $previousBribeAmount() {
	var bribeAmt = 0;
	for (var i = 0; i < this._successfulBribes.length; i++) {
		if (this._successfulBribes[i].system === system.ID && this._successfulBribes[i].bribeAmount > bribeAmt) {
			bribeAmt = this._successfulBribes[i].bribeAmount;
		}
	}
	return bribeAmt;
}

//-------------------------------------------------------------------------------------------------------------
// loads the mulk run array with data for the current galaxy
this.$getMilkRuns = function $getMilkRuns() {
	this._milkRuns = [];
	switch (galaxyNumber) {
		case 0:
			this._milkRuns = [
				{ id1: 99, id2: 84, rating: 70 },
				{ id1: 99, id2: 191, rating: 72 },
				{ id1: 12, id2: 116, rating: 73 },
				{ id1: 163, id2: 192, rating: 73 },
				{ id1: 131, id2: 42, rating: 73 },
				{ id1: 254, id2: 187, rating: 74 },
				{ id1: 158, id2: 116, rating: 75 },
				{ id1: 98, id2: 84, rating: 75 },
				{ id1: 98, id2: 191, rating: 75 },
				{ id1: 195, id2: 63, rating: 76 },
				{ id1: 254, id2: 251, rating: 76 },
				{ id1: 79, id2: 174, rating: 76 },
				{ id1: 49, id2: 116, rating: 76 },
				{ id1: 254, id2: 74, rating: 77 },
				{ id1: 100, id2: 42, rating: 77 },
				{ id1: 207, id2: 116, rating: 77 },
				{ id1: 131, id2: 167, rating: 78 },
				{ id1: 240, id2: 187, rating: 78 },
				{ id1: 79, id2: 191, rating: 79 },
				{ id1: 33, id2: 123, rating: 81 },
				{ id1: 99, id2: 185, rating: 81 },
				{ id1: 82, id2: 244, rating: 83 },
				{ id1: 227, id2: 154, rating: 83 },
				{ id1: 35, id2: 186, rating: 84 },
				{ id1: 221, id2: 185, rating: 84 },
				{ id1: 163, id2: 252, rating: 86 },
				{ id1: 141, id2: 101, rating: 89 },
				{ id1: 250, id2: 101, rating: 89 },
				{ id1: 218, id2: 18, rating: 90 },
				{ id1: 82, id2: 154, rating: 90 },
				{ id1: 246, id2: 18, rating: 93 },
			];
			break;
		case 1:
			this._milkRuns = [
				{ id1: 255, id2: 37, rating: 72 },
				{ id1: 191, id2: 37, rating: 74 },
				{ id1: 128, id2: 165, rating: 74 },
				{ id1: 57, id2: 94, rating: 76 },
				{ id1: 24, id2: 20, rating: 78 },
				{ id1: 237, id2: 174, rating: 79 },
				{ id1: 16, id2: 157, rating: 80 },
				{ id1: 81, id2: 20, rating: 80 },
				{ id1: 167, id2: 94, rating: 81 },
				{ id1: 82, id2: 174, rating: 82 },
				{ id1: 40, id2: 68, rating: 82 },
				{ id1: 106, id2: 211, rating: 83 },
				{ id1: 57, id2: 19, rating: 83 },
				{ id1: 57, id2: 115, rating: 84 },
				{ id1: 185, id2: 38, rating: 84 },
				{ id1: 92, id2: 211, rating: 85 },
				{ id1: 191, id2: 83, rating: 85 },
				{ id1: 16, id2: 34, rating: 87 },
				{ id1: 40, id2: 179, rating: 88 },
				{ id1: 209, id2: 115, rating: 88 },
				{ id1: 81, id2: 97, rating: 90 },
				{ id1: 82, id2: 86, rating: 92 },
				{ id1: 167, id2: 115, rating: 92 },
				{ id1: 231, id2: 147, rating: 94 },
			];
			break;
		case 2:
			this._milkRuns = [
				{ id1: 205, id2: 168, rating: 71 },
				{ id1: 144, id2: 147, rating: 73 },
				{ id1: 34, id2: 57, rating: 73 },
				{ id1: 64, id2: 168, rating: 73 },
				{ id1: 76, id2: 152, rating: 76 },
				{ id1: 76, id2: 66, rating: 76 },
				{ id1: 205, id2: 82, rating: 77 },
				{ id1: 155, id2: 57, rating: 77 },
				{ id1: 116, id2: 152, rating: 77 },
				{ id1: 17, id2: 24, rating: 77 },
				{ id1: 205, id2: 114, rating: 79 },
				{ id1: 17, id2: 41, rating: 79 },
				{ id1: 17, id2: 147, rating: 79 },
				{ id1: 22, id2: 140, rating: 79 },
				{ id1: 64, id2: 215, rating: 79 },
				{ id1: 58, id2: 152, rating: 80 },
				{ id1: 50, id2: 65, rating: 82 },
				{ id1: 42, id2: 163, rating: 83 },
				{ id1: 58, id2: 140, rating: 84 },
				{ id1: 205, id2: 241, rating: 84 },
				{ id1: 43, id2: 98, rating: 84 },
				{ id1: 22, id2: 91, rating: 84 },
				{ id1: 42, id2: 135, rating: 85 },
				{ id1: 54, id2: 118, rating: 86 },
				{ id1: 18, id2: 100, rating: 87 },
				{ id1: 5, id2: 66, rating: 87 },
				{ id1: 79, id2: 65, rating: 89 },
				{ id1: 10, id2: 141, rating: 89 },
				{ id1: 6, id2: 180, rating: 89 },
				{ id1: 58, id2: 66, rating: 89 },
				{ id1: 10, id2: 70, rating: 90 },
				{ id1: 5, id2: 140, rating: 90 },
				{ id1: 116, id2: 122, rating: 92 },
				{ id1: 10, id2: 85, rating: 94 },
			];
			break;
		case 3:
			this._milkRuns = [
				{ id1: 38, id2: 160, rating: 70 },
				{ id1: 25, id2: 24, rating: 72 },
				{ id1: 25, id2: 62, rating: 74 },
				{ id1: 220, id2: 171, rating: 75 },
				{ id1: 137, id2: 105, rating: 76 },
				{ id1: 209, id2: 33, rating: 77 },
				{ id1: 29, id2: 168, rating: 78 },
				{ id1: 58, id2: 24, rating: 78 },
				{ id1: 103, id2: 107, rating: 78 },
				{ id1: 65, id2: 239, rating: 78 },
				{ id1: 103, id2: 105, rating: 79 },
				{ id1: 103, id2: 177, rating: 79 },
				{ id1: 38, id2: 53, rating: 82 },
				{ id1: 42, id2: 16, rating: 83 },
				{ id1: 38, id2: 74, rating: 84 },
				{ id1: 103, id2: 108, rating: 84 },
				{ id1: 103, id2: 239, rating: 84 },
				{ id1: 220, id2: 68, rating: 84 },
				{ id1: 58, id2: 109, rating: 89 },
				{ id1: 220, id2: 90, rating: 90 },
			];
			break;
		case 4:
			this._milkRuns = [
				{ id1: 34, id2: 186, rating: 70 },
				{ id1: 34, id2: 250, rating: 70 },
				{ id1: 66, id2: 58, rating: 70 },
				{ id1: 2, id2: 63, rating: 71 },
				{ id1: 226, id2: 31, rating: 71 },
				{ id1: 130, id2: 250, rating: 72 },
				{ id1: 162, id2: 250, rating: 73 },
				{ id1: 34, id2: 183, rating: 74 },
				{ id1: 162, id2: 183, rating: 74 },
				{ id1: 137, id2: 191, rating: 75 },
				{ id1: 98, id2: 123, rating: 75 },
				{ id1: 130, id2: 108, rating: 76 },
				{ id1: 12, id2: 81, rating: 76 },
				{ id1: 213, id2: 81, rating: 77 },
				{ id1: 115, id2: 150, rating: 78 },
				{ id1: 162, id2: 149, rating: 78 },
				{ id1: 254, id2: 183, rating: 79 },
				{ id1: 212, id2: 46, rating: 80 },
				{ id1: 198, id2: 31, rating: 80 },
				{ id1: 254, id2: 250, rating: 82 },
				{ id1: 130, id2: 149, rating: 82 },
				{ id1: 194, id2: 88, rating: 83 },
				{ id1: 198, id2: 63, rating: 84 },
				{ id1: 99, id2: 91, rating: 84 },
				{ id1: 254, id2: 149, rating: 85 },
				{ id1: 99, id2: 200, rating: 85 },
				{ id1: 184, id2: 150, rating: 86 },
				{ id1: 89, id2: 87, rating: 88 },
				{ id1: 249, id2: 16, rating: 90 },
			];
			break;
		case 5:
			this._milkRuns = [
				{ id1: 243, id2: 57, rating: 70 },
				{ id1: 243, id2: 228, rating: 71 },
				{ id1: 182, id2: 160, rating: 71 },
				{ id1: 14, id2: 249, rating: 73 },
				{ id1: 240, id2: 30, rating: 75 },
				{ id1: 20, id2: 57, rating: 75 },
				{ id1: 243, id2: 231, rating: 75 },
				{ id1: 243, id2: 101, rating: 75 },
				{ id1: 243, id2: 98, rating: 76 },
				{ id1: 109, id2: 57, rating: 76 },
				{ id1: 243, id2: 103, rating: 77 },
				{ id1: 34, id2: 24, rating: 78 },
				{ id1: 69, id2: 167, rating: 78 },
				{ id1: 144, id2: 61, rating: 79 },
				{ id1: 109, id2: 228, rating: 79 },
				{ id1: 182, id2: 167, rating: 79 },
				{ id1: 240, id2: 234, rating: 80 },
				{ id1: 144, id2: 253, rating: 81 },
				{ id1: 144, id2: 97, rating: 81 },
				{ id1: 109, id2: 98, rating: 81 },
				{ id1: 20, id2: 244, rating: 81 },
				{ id1: 86, id2: 131, rating: 81 },
				{ id1: 109, id2: 101, rating: 82 },
				{ id1: 28, id2: 108, rating: 82 },
				{ id1: 1, id2: 108, rating: 82 },
				{ id1: 115, id2: 215, rating: 83 },
				{ id1: 109, id2: 231, rating: 83 },
				{ id1: 20, id2: 103, rating: 84 },
				{ id1: 1, id2: 181, rating: 86 },
				{ id1: 240, id2: 140, rating: 88 },
				{ id1: 14, id2: 113, rating: 88 },
				{ id1: 144, id2: 13, rating: 88 },
				{ id1: 102, id2: 98, rating: 88 },
				{ id1: 17, id2: 108, rating: 91 },
				{ id1: 6, id2: 181, rating: 92 },
				{ id1: 40, id2: 215, rating: 94 },
				{ id1: 17, id2: 181, rating: 96 },
				{ id1: 50, id2: 232, rating: 98 },
			];
			break;
		case 6:
			this._milkRuns = [
				{ id1: 170, id2: 105, rating: 75 },
				{ id1: 170, id2: 190, rating: 75 },
				{ id1: 64, id2: 160, rating: 75 },
				{ id1: 13, id2: 160, rating: 76 },
				{ id1: 6, id2: 174, rating: 77 },
				{ id1: 183, id2: 164, rating: 79 },
				{ id1: 28, id2: 223, rating: 79 },
				{ id1: 119, id2: 190, rating: 79 },
				{ id1: 151, id2: 180, rating: 80 },
				{ id1: 64, id2: 81, rating: 80 },
				{ id1: 55, id2: 105, rating: 80 },
				{ id1: 195, id2: 159, rating: 80 },
				{ id1: 55, id2: 190, rating: 80 },
				{ id1: 86, id2: 127, rating: 80 },
				{ id1: 172, id2: 160, rating: 81 },
				{ id1: 199, id2: 233, rating: 82 },
				{ id1: 151, id2: 73, rating: 82 },
				{ id1: 181, id2: 95, rating: 83 },
				{ id1: 103, id2: 58, rating: 83 },
				{ id1: 71, id2: 160, rating: 83 },
				{ id1: 66, id2: 190, rating: 84 },
				{ id1: 135, id2: 80, rating: 84 },
				{ id1: 23, id2: 233, rating: 84 },
				{ id1: 66, id2: 105, rating: 84 },
				{ id1: 215, id2: 95, rating: 86 },
				{ id1: 247, id2: 202, rating: 86 },
				{ id1: 82, id2: 105, rating: 87 },
				{ id1: 82, id2: 190, rating: 87 },
				{ id1: 172, id2: 81, rating: 88 },
				{ id1: 71, id2: 81, rating: 88 },
				{ id1: 67, id2: 209, rating: 88 },
				{ id1: 23, id2: 246, rating: 89 },
				{ id1: 7, id2: 209, rating: 89 },
				{ id1: 215, id2: 133, rating: 91 },
				{ id1: 183, id2: 133, rating: 92 },
				{ id1: 247, id2: 133, rating: 93 },
				{ id1: 151, id2: 5, rating: 95 },
			];
			break;
		case 7:
			this._milkRuns = [
				{ id1: 56, id2: 240, rating: 70 },
				{ id1: 56, id2: 170, rating: 72 },
				{ id1: 82, id2: 100, rating: 75 },
				{ id1: 31, id2: 25, rating: 77 },
				{ id1: 147, id2: 121, rating: 77 },
				{ id1: 41, id2: 35, rating: 78 },
				{ id1: 244, id2: 138, rating: 78 },
				{ id1: 192, id2: 133, rating: 78 },
				{ id1: 151, id2: 230, rating: 78 },
				{ id1: 137, id2: 121, rating: 80 },
				{ id1: 206, id2: 25, rating: 80 },
				{ id1: 232, id2: 86, rating: 81 },
				{ id1: 93, id2: 185, rating: 81 },
				{ id1: 93, id2: 196, rating: 81 },
				{ id1: 192, id2: 35, rating: 81 },
				{ id1: 161, id2: 255, rating: 82 },
				{ id1: 14, id2: 25, rating: 83 },
				{ id1: 110, id2: 172, rating: 84 },
				{ id1: 82, id2: 179, rating: 84 },
				{ id1: 41, id2: 233, rating: 85 },
				{ id1: 251, id2: 30, rating: 85 },
				{ id1: 218, id2: 71, rating: 89 },
				{ id1: 93, id2: 186, rating: 90 },
				{ id1: 206, id2: 71, rating: 90 },
				{ id1: 251, id2: 233, rating: 90 },
				{ id1: 93, id2: 92, rating: 92 },
				{ id1: 232, id2: 81, rating: 92 },
				{ id1: 110, id2: 177, rating: 93 },
				{ id1: 232, id2: 252, rating: 93 },
			];
			break
	}
}