"use strict";
this.name = "Smugglers_Illegal";
this.author = "phkb";
this.description = "Looks after the new illegal goods.";
this.licence = "CC BY-NC-SA 3.0";

/* TODO:
- add more illegal trade items to the possibilities list
*/

this._disabled = false;
this._debug = false;
this._enableMarkedSystems = true;
this._markedSystemRange = 7;
this._IGTInstalled = false;
this._minLength = 100;

/* template: {govTypes:"", commodity:"", properties:"", permit:1, scale:1, period:n, chance:1, description:""},
	govTypes:		Comma-separated list of goverment types (0-7) this definition can apply to
	commodity:		Comma-separated list of commodity names this definition will apply to.
	properties:		Comma-separated list of text search strings to use against the planetary description.
					For instance, "civil war" would only select planets that have "civil war" in their description.
					"drink|brew" would only select planets that have "drink" or "brew" in their description.
	permit:			Boolean value (0 or 1) indicating whether a permit can be purchased for this commodity
	scale:			The legality scale to apply to this commodity. The scale will apply when calculating the bounty to apply to a player caught smuggling illegal goods as a multiplier.
					Most illegal goods have a scale of 1. Slaves have 2.
	period:			The period (in days) this definition will be active for.
	chance:			A chance factor, between 0 (never) and 1 (very often), to apply when selecting this definition
	description:	The description to display for this definition.
*/
this._possibilities = [
	{ govTypes: "0,1,2", commodity: "firearms", properties: "", permit: 0, scale: 2, period: -1, chance: 0, description: "firearms1" },
	{ govTypes: "3,4,5,6,7", commodity: "firearms", properties: "civil war", permit: 0, scale: 1, period: -1, chance: 0, description: "firearms2" },
	{ govTypes: "0", commodity: "computers", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "computers1" },
	{ govTypes: "0,1", commodity: "radioactives", properties: "", permit: 1, scale: 1, period: 200, chance: 1, description: "radioactives1" },
	{ govTypes: "0,1", commodity: "machinery", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "machinery1" },
	{ govTypes: "1", commodity: "food", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "food1" },
	{ govTypes: "1", commodity: "textiles", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "textiles1" },
	{ govTypes: "1", commodity: "liquor_wines", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "liquor_wines1" },
	{ govTypes: "1", commodity: "computers", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "computers2" },
	{ govTypes: "1", commodity: "machinery", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "machinery2" },
	{ govTypes: "1", commodity: "furs", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "furs1" },
	{ govTypes: "1", commodity: "gold,platinum", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gold,platinum1" },
	{ govTypes: "2", commodity: "computers", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "computers3" },
	{ govTypes: "2", commodity: "alien_items", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "alien_items1" },
	{ govTypes: "2", commodity: "food", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "food2" },
	{ govTypes: "2", commodity: "alloys", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "alloys1" },
	{ govTypes: "2", commodity: "liquor_wines", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "liquor_wines2" },
	{ govTypes: "2", commodity: "machinery", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "machinery3" },
	{ govTypes: "2", commodity: "gold,platinum", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gold,platinum2" },
	{ govTypes: "2", commodity: "firearms", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "firearms3" },
	{ govTypes: "3", commodity: "luxuries", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "luxuries1" },
	{ govTypes: "3", commodity: "radioactives", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "radioactives2" },
	{ govTypes: "0,3", commodity: "food", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "food3" },
	{ govTypes: "3", commodity: "machinery", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "machinery4" },
	{ govTypes: "3", commodity: "computers", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "computers4" },
	{ govTypes: "3", commodity: "minerals", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "minerals1" },
	{ govTypes: "3", commodity: "alien_items", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "alien_items2" },
	{ govTypes: "4", commodity: "minerals", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "minerals2" },
	{ govTypes: "4", commodity: "machinery", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "machinery5" },
	{ govTypes: "4", commodity: "computers", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "computers5" },
	{ govTypes: "4", commodity: "luxuries", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "luxuries2" },
	{ govTypes: "4", commodity: "liquor_wines", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "liquor_wines3" },
	{ govTypes: "4", commodity: "furs", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "furs2" },
	{ govTypes: "4", commodity: "food", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "food4" },
	{ govTypes: "4", commodity: "textiles", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "textiles2" },
	{ govTypes: "4", commodity: "gold,platinum,gem_stones", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gold,platinum,gem_stones1" },
	{ govTypes: "5", commodity: "food", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "food5" },
	{ govTypes: "5", commodity: "gold,platinum", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gold,platinum3" },
	{ govTypes: "5", commodity: "computers", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "computers6" },
	{ govTypes: "5", commodity: "furs", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "furs3" },
	{ govTypes: "5", commodity: "alien_items", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "alien_items3" },
	{ govTypes: "5", commodity: "machinery", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "machinery6" },
	{ govTypes: "5", commodity: "gem_stones", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gem_stones1" },
	{ govTypes: "5,6,7", commodity: "food", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "food6" },
	{ govTypes: "6", commodity: "alien_items", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "alien_items4" },
	{ govTypes: "6", commodity: "furs", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "furs4" },
	{ govTypes: "6", commodity: "alloys", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "alloys2" },
	{ govTypes: "6", commodity: "liquor_wines", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "liquor_wines4" },
	{ govTypes: "6", commodity: "radioactives", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "radioactives3" },
	{ govTypes: "6", commodity: "textiles", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "textiles3" },
	{ govTypes: "6", commodity: "food", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "food7" },
	{ govTypes: "6", commodity: "gold,platinum", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gold,platinum4" },
	{ govTypes: "7", commodity: "alien_items", properties: "", permit: 1, scale: 1, period: 120, chance: 1, description: "alien_items5" },
	{ govTypes: "7", commodity: "gold,platinum", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gold,platinum5" },
	{ govTypes: "7", commodity: "alloys", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "alloys3" },
	{ govTypes: "7", commodity: "computers", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "computers7" },
	{ govTypes: "7", commodity: "radioactives", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "radioactives4" },
	{ govTypes: "7", commodity: "food", properties: "", permit: 1, scale: 1, period: 60, chance: 1, description: "food8" },
	{ govTypes: "7", commodity: "furs", properties: "", permit: 1, scale: 1, period: 90, chance: 1, description: "furs5" },
	{ govTypes: "6,7", commodity: "gem_stones", properties: "", permit: 1, scale: 1, period: 12, chance: 0.5, description: "gem_stones2" },
	{ govTypes: "0,1,2,3,4,5,6,7", commodity: "textiles", properties: "solar", permit: 1, scale: 1, period: 90, chance: 1, description: "textiles4" },
	{ govTypes: "0,1,2,3,4,5,6,7", commodity: "liquor_wines", properties: "brandy,brew,gargle blasters", permit: 1, scale: 1, period: 120, chance: 1, description: "liquor_wines5" },
	{ govTypes: "0,1,2,3,4,5,6,7", commodity: "food", properties: "cuisine,soup,burgers,meat,cutlet,steak", permit: 1, scale: 1, period: 90, chance: 1, description: "food9" },
	{ govTypes: "3,4,5,6,7", commodity: "firearms", properties: "", permit: 1, scale: 1, period: 120, chance: 0.5, description: "firearms4" }
];

// these commodities will be included when checking for illegal imports on docking, but as they are global defaults we won't add them via the normal mechanisms
// slaves are not included here, as they are being kept as illegal to export only. This is (a) because some escape pods get stored as slaves,
// and (b) to keep the Illegal Goods Tweak in the loop
this._defaultAdditions = ["narcotics"];

// extra commodityInfo
//		element 0 is permit costs
//		element 1 is % over base price when item is illegal
//		element 2 is % over base price for the black market price
this._commodityInfo = {
	"food": [2500, 6.5, 2.2],
	"textiles": [3000, 4.0, 2.0],
	"radioactives": [5000, 2.5, 1.3],
	"slaves": [4000, 0, 0],
	"liquor_wines": [5000, 2.2, 1.1],
	"luxuries": [6000, 2.1, 1.1],
	"narcotics": [8000, 0, 0],
	"computers": [6000, 2.1, 1.1],
	"machinery": [5000, 2.2, 1.1],
	"alloys": [3000, 2.2, 1.1],
	"firearms": [5000, 2.2, 1.1],
	"furs": [7000, 2.5, 1.4],
	"minerals": [3000, 6.0, 2.1],
	"gold": [8000, 1.3, 0.6],
	"platinum": [9000, 1.1, 0.6],
	"gem_stones": [8000, 1.8, 0.8],
	"alien_items": [6000, 2.8, 1.3]
};

// holds all illegal good definitions for this sector
this._illegalGoods = []; // illegal goods currently in play in sector
this._permits = []; // permits currently held by player
this._originalCredits = 0; // value to hold the players credits (used to refund the cost of the initial item)
this._display = 0;
this._curpage = 0;
this._direct = 0;
this._dataLoaded = false; // flag to indicate when data has been loaded
this._initialRun = true; // flag to indicate when the initial data set has been created
this._originating = -1;
this._startUpRunning = true;
this._markedSystems = [];
this._witchspaceDestination = -1;
this._done = false;
this._menuColor = "orangeColor";
this._itemColor = "yellowColor";
this._disabledColor = "darkGrayColor";
this._trueValues = ["yes", "1", 1, "true", true];
this._markerColors = ["whiteColor", "yellowColor", "orangeColor", "greenColor", "redColor", "blueColor", "purpleColor", "cyanColor"];
this._markerShapes = ["MARKER_X", "MARKER_PLUS", "MARKER_SQUARE", "MARKER_DIAMOND"];
this._firearmsColor = 2;
this._firearmsShape = 3;
this._mixedIllegalColor = 1;
this._mixedIllegalShape = 3;

// configuration settings for use in Lib_Config
this._siConfig = {
	Name: this.name,
	Alias: expandMissionText("smuggling_config_alias"),
	Display: expandMissionText("smuggling_config_display"),
	Alive: "_siConfig",
	Notify: "$markSystems",
	Bool: {
		B0: { Name: "_enableMarkedSystems", Def: true, Desc: expandMissionText("smuggling_config_marksystems") },
		Info: expandMissionText("smuggling_config_bool_info")
	},
	SInt: {
		S0: { Name: "_markedSystemRange", Def: 7, Min: 0, Max: 15, Desc: expandMissionText("smuggling_markedsys_range") },
		S1: { Name: "_firearmsColor", Def: 2, Min: 0, Max: 7, Desc: expandMissionText("smuggling_firearms_color") },
		S2: { Name: "_mixedIllegalColor", Def: 2, Min: 0, Max: 7, Desc: expandMissionText("smuggling_mixed_color") },
		S3: { Name: "_firearmsShape", Def: 3, Min: 0, Max: 3, Desc: expandMissionText("smuggling_firearms_shape") },
		S4: { Name: "_mixedIllegalShape", Def: 3, Min: 0, Max: 3, Desc: expandMissionText("smuggling_mixed_shape") },

		Info: expandMissionText("smuggling_config_sint_info")
	},
};

//-------------------------------------------------------------------------------------------------------------
this.$updateGeneralCommodityDefinition = function (goodDefinition, station, systemID) {
	var si = worldScripts.Smugglers_Illegal;
	if (goodDefinition.key == "slaves") {
		goodDefinition.legality_export = 1;
		goodDefinition.legality_import = 0;
		return goodDefinition;
	}
	if (goodDefinition.key == "narcotics") {
		goodDefinition.legality_export = 1;
		goodDefinition.legality_import = 0;
		goodDefinition.comment = expandDescription("[narcotics-legality-description]");
		return goodDefinition;
	}
	if (goodDefinition.key == "firearms") {
		goodDefinition.legality_export = 0;
		goodDefinition.legality_import = 0;
		goodDefinition.comment = expandDescription("[oolite-commodity-no-comment]");
	}
	if (si) {
		// initialise the data if it hasn't been already
		if (si._dataLoaded === false) si.$initialiseData();
		if ((station === 0 || station == null) && systemID && systemID != -1) {
			if (si._dataLoaded === true) {
				if (si.$isGoodIllegal(systemID, goodDefinition.key)) {
					goodDefinition.legality_import = si.$goodIllegalScale(systemID, goodDefinition.key);
				}
			}
		}
	}
	return goodDefinition;
}

//-------------------------------------------------------------------------------------------------------------
this.$updateLocalCommodityDefinition = function (goodDefinition, station, systemID) {
	var si = worldScripts.Smugglers_Illegal;
	if (si && station && station.allegiance === "galcop") {
		// apply price boost
		if (si.$isGoodIllegal(systemID, goodDefinition.key)) {
			// calculate the new price, based on the price limits in _commodityInfo
			var newprice = Math.round(goodDefinition.price * (si._commodityInfo[goodDefinition.key][1] + ((system.scrambledPseudoRandomNumber(systemID) * 0.6) - 0.3)), 2);
			// adjust the price property
			if (si._debug) log(this.name, "changing price on " + goodDefinition.key + " from " + goodDefinition.price + " to " + newprice);
			goodDefinition.price = newprice;
		}
	}
	return goodDefinition;
}

//-------------------------------------------------------------------------------------------------------------
this.startUp = function () {
	var msi = worldScripts.MarketScriptInterface_Main;
	msi.$addMarketInterface("tradegoods_general", "$updateGeneralCommodityDefinition", this.name);
	msi.$addMarketInterface("system_local", "$updateLocalCommodityDefinition", this.name, "lowest");

	if (this._disabled) {
		delete this.shipDockedWithStation;
		delete this.playerBoughtEquipment;
		delete this.shipWillEnterWitchspace;
		delete this.shipExitedWitchspace;
		delete this.playerEnteredNewGalaxy;
		delete this.playerWillSaveGame;
		delete this.guiScreenWillChange;
		delete this.guiScreenChanged;
		delete this.updateLocalCommodityDefinition;
		delete this.dayChanged;
		delete this.startUpComplete;
		delete this.startUp;
		return;
	}

	var core = worldScripts.Smugglers_CoreFunctions;
	this.$padTextRight = core.$padTextRight;
	this.$padTextLeft = core.$padTextLeft;
	this.$isBigGuiActive = core.$isBigGuiActive;
	this.$rand = core.$rand;

	if (missionVariables.Smuggling_Permits) {
		this._permits = JSON.parse(missionVariables.Smuggling_Permits);
		delete missionVariables.Smuggling_Permits;
	}
	this._originating = system.ID;
	delete this.startUp;
}

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {
	// register our settings, if Lib_Config is present
	if (worldScripts.Lib_Config) worldScripts.Lib_Config._registerSet(this._siConfig);

	// check for illegal goods tweak
	if (worldScripts["illegal_goods_tweak"]) this._IGTInstalled = true;

	// set up the interface screen, if required
	this.$initMainInterface(player.ship.dockedStation);
	this.$correctStartEndDates();
	this._startUpRunning = false;

	if (missionVariables.Smuggling_InitialRun) {
		this._initialRun = missionVariables.Smuggling_InitialRun;
	}

	if (missionVariables.Smuggling_MarkedSystems) {
		this._markedSystems = JSON.parse(missionVariables.Smuggling_MarkedSystems);
		delete missionVariables.Smuggling_MarkedSystems;
	}

	// add our repair items to the email systems list of repair equipment
	var w = worldScripts.EmailSystem;
	if (w) {
		var ga = worldScripts.GalCopAdminServices;
		for (var i = 1; i <= 20; i++) {
			ga._maint_repair_equip.push("EQ_SMUGGLING_COMPARTMENT_REPAIR_" + i);
		}
	}
	// only load these settings if Library is installed
	if (worldScripts.Lib_Config) {
		if (missionVariables.Smugglers_MarkSystems) this._enableMarkedSystems = (this._trueValues.indexOf(missionVariables.Smugglers_MarkSystems) >= 0 ? true : false);
		if (missionVariables.Smugglers_MarkSystemsRange) this._markedSystemRange = parseInt(missionVariables.Smugglers_MarkSystemsRange);
		if (missionVariables.Smugglers_FirearmsColor) this._firearmsColor = parseInt(missionVariables.Smugglers_FirearmsColor);
		if (missionVariables.Smugglers_MixedIllegalColor) this._mixedIllegalColor = parseInt(missionVariables.Smugglers_MixedIllegalColor);
		if (missionVariables.Smugglers_FirearmsShape) this._firearmsShape = parseInt(missionVariables.Smugglers_FirearmsShape);
		if (missionVariables.Smugglers_MixedIllegalShape) this._mixedIllegalShape = parseInt(missionVariables.Smugglers_MixedIllegalShape);
	}
	this.$markSystems();

	if (oolite.compareVersion("1.91") <= 0) {
		setExtraGuiScreenKeys(this.name, {
			guiScreen: "GUI_SCREEN_LONG_RANGE_CHART",
			registerKeys: { "igkey1": [{ key: "1", mod1: true }], "igkey2": [{ key: "2", mod1: true }] },
			callback: this.$jumpToIllegalsLRC.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen: "GUI_SCREEN_SHORT_RANGE_CHART",
			registerKeys: { "igkey1": [{ key: "1", mod1: true }], "igkey2": [{ key: "2", mod1: true }] },
			callback: this.$jumpToIllegalsSRC.bind(this)
		});

		if (worldScripts.ContextualHelp) {
			worldScripts.ContextualHelp.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_LONG_RANGE_CHART", "\n" + expandMissionText("smuggling_key_help"));
			worldScripts.ContextualHelp.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_SHORT_RANGE_CHART", "\n" + expandMissionText("smuggling_key_help"));
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function () {
	missionVariables.Smuggling_IllegalGoods = JSON.stringify(this._illegalGoods);
	missionVariables.Smuggling_Permits = JSON.stringify(this._permits);
	missionVariables.Smuggling_MarkedSystems = JSON.stringify(this._markedSystems);
	missionVariables.Smuggling_InitialRun = this._initialRun;
	// only save these settings if Library is installed
	if (worldScripts.Lib_Config) {
		missionVariables.Smugglers_MarkSystems = this._enableMarkedSystems;
		missionVariables.Smugglers_MarkSystemsRange = this._markedSystemRange;
		missionVariables.Smugglers_FirearmsColor = this._firearmsColor;
		missionVariables.Smugglers_MixedIllegalColor = this._mixedIllegalColor;
		missionVariables.Smugglers_FirearmsShape = this._firearmsShape;
		missionVariables.Smugglers_MixedIllegalShape = this._mixedIllegalShape;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipDockedWithStation = function (station) {
	this.$initMainInterface(station);
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillLaunchFromStation = function (station) {
	if (missionVariables.Smuggling_IllegalGoods) delete missionVariables.Smuggling_IllegalGoods;
	if (missionVariables.Smuggling_Permits) delete missionVariables.Smuggling_Permits;
	if (missionVariables.Smuggling_MarkedSystems) delete missionVariables.Smuggling_MarkedSystems;
}

//-------------------------------------------------------------------------------------------------------------
this.playerBoughtEquipment = function (equipmentKey) {
	var p = player.ship;

	if (equipmentKey === "EQ_IMPORT_PERMIT" || equipmentKey == "EQ_IMPORT_PERMIT_NC") {
		p.removeEquipment(equipmentKey);
		// give the player back the original amount, in case they decide not to purchase
		if (this._originalCredits != 0) {
			player.credits = this._originalCredits;
			this._originalCredits = 0;
		}
		// mission screen to ask how many TC to allocate
		this.$showPurchasePage();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillEnterWitchspace = function (cause, destination) {
	this._witchspaceDestination = destination;
	this._initialRun = true;
}

//-------------------------------------------------------------------------------------------------------------
this.shipExitedWitchspace = function () {
	this._originating = system.ID;
	this._done = false;
	this.$markSystems();
}

//-------------------------------------------------------------------------------------------------------------
this.playerEnteredNewGalaxy = function (galNumber) {
	// hopefully this is early enough in the sequence to establish illegal goods for the "smugglers_illegalmarket"
	this._illegalGoods = [];
	this._dataLoaded = false;
}

//-------------------------------------------------------------------------------------------------------------
this.dayChanged = function (newday) {
	this.$cleanup();
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenWillChange = function (to, from) {
	if (to === "GUI_SCREEN_MARKET") this.$updateDescriptions();
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged = function (to, from) {
	if (guiScreen === "GUI_SCREEN_EQUIP_SHIP") this._originalCredits = player.credits;

	// This code puts all the illegal goods on the F7 screen. However, with the description in the F8F8 screen this code is possibly unnecessary.
	// However, if there is feedback that says it would be useful I'll but it back in.
	if (from === "GUI_SCREEN_SYSTEM_DATA" && guiScreen != "GUI_SCREEN_SYSTEM_DATA") {
		// unhide the hud, if required
		if (player.ship.hudHidden != this._hudHidden) {
			player.ship.hudHidden = this._hudHidden;
		}
	}
	if (guiScreen === "GUI_SCREEN_SYSTEM_DATA" && from != "GUI_SCREEN_SYSTEM_DATA") {
		if (player.ship.isInSpace === false) {
			// hide the hud to give us more room for text
			this._hudHidden = player.ship.hudHidden;
			if (this.$isBigGuiActive() === false) {
				player.ship.hudHidden = true;
				// force an update in XenonUI (if installed)
				var xui = worldScripts.XenonUI;
				if (xui) xui.guiScreenChanged("", "");
				var xrui = worldScripts.XenonReduxUI;
				if (xrui) xrui.guiScreenChanged("", "");
			}
		}
	}
	if (guiScreen === "GUI_SCREEN_SYSTEM_DATA") {
		// load data onto screen at the end
		if (!this._dataLoadDelay) this._dataLoadDelay = new Timer(this, this.$dataLoadDelay, 0.2, 0);
	}
}

//-------------------------------------------------------------------------------------------------------------
this.infoSystemChanged = function (to, from) {
	if (guiScreen === "GUI_SCREEN_SYSTEM_DATA") {
		// load data onto screen at the end
		if (!this._dataLoadDelay) this._dataLoadDelay = new Timer(this, this.$dataLoadDelay, 0.2, 0);
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$dataLoadDelay = function $dataLoadDelay() {
	delete this._dataLoadDelay;
	// append any system-specific illegal goods to the display
	if (guiScreen != "GUI_SCREEN_SYSTEM_DATA") return;

	var sysID = player.ship.targetSystem;
	if (player.ship.hasOwnProperty("infoSystem")) sysID = player.ship.infoSystem;

	var concealment = System.infoForSystem(galaxyNumber, sysID).concealment;
	// if there is some form of concealment in effect, don't show anything
	if (concealment > 0) return;

	var goods = this.$illegalGoodsList(sysID);
	var dis_list = "";
	if (this._IGTInstalled === true) {
		dis_list = expandDescription("[illegal-goods-f7-igt" + (player.ship.isInSpace === false ? "" : "-inflight") + "]");
	} else {
		dis_list = expandDescription("[illegal-goods-f7-std" + (player.ship.isInSpace === false ? "" : "-inflight") + "]");
	}
	if (player.ship.isInSpace === false) {
		for (var i = 0; i < goods.length; i++) {
			dis_list += "\n" + expandMissionText("smuggling_illegal_import") + " -- " + this.$translateCommodityList(goods[i].commodity) + ": " + expandDescription("[smuggling-" + goods[i].description + "]");
		}
	} else {
		for (var i = 0; i < goods.length; i++) {
			dis_list += ", " + this.$translateCommodityList(goods[i].commodity);
		}
	}
	mission.addMessageText(dis_list);
}

//-------------------------------------------------------------------------------------------------------------
this.$initialiseData = function $initialiseData() {
	this._dataLoaded = true;

	// if the initial run flag isn't yet stored in the data, but there are other definitions, reset the flag to false
	// that should prevent any extreme prices.
	if (!missionVariables.Smuggling_InitialRun && missionVariables.Smuggling_IllegalGoods) {
		this._initialRun = false;
	}

	if (missionVariables.Smuggling_IllegalGoods) {
		this._illegalGoods = JSON.parse(missionVariables.Smuggling_IllegalGoods);
		this.$updateIllegalGoodsDescriptions();
		delete missionVariables.Smuggling_IllegalGoods;
	}
	if (this._illegalGoods.length == 0) {
		this.$initialSetup();
		this.$markSystems();
	}
}

//-------------------------------------------------------------------------------------------------------------
// change descriptions to tokens
this.$updateIllegalGoodsDescriptions = function $updateIllegalGoodsDescriptions() {
	if (this._illegalGoods[0].description.indexOf(this._illegalGoods[0].commodity) == 0) {
		return;
	}
	for (var i = 0; i < this._illegalGoods.length; i++) {
		var item = this._illegalGoods[i];
		for (var j = 1; j <= 10; j++) {
			var desc;
			try {
				desc = expandDescription("[smuggling-" + item.commodity + j.toString() + "]");
			} catch (ex) {
				break;
			}
			if (desc.substring(0, this._minLength).toLowerCase() == item.description.substring(0, this._minLength).toLowerCase()) {
				item.description = item.commodity + j.toString()
				break;
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// shows the installation page to select a compartment size
this.$showPurchasePage = function $showPurchasePage() {

	var curChoices = {};
	var p = player.ship;
	var stn = p.dockedStation;
	var rowAdjust = 0;

	var pagesize = 18;
	if (this.$isBigGuiActive() === true) pagesize = 24;

	var key = "[permit-purchase-header]";
	if (worldScripts["CargoTypeExtension"]) key = "[permit-purchase-header-nc]";

	var text = expandDescription(key, { cash: formatCredits(player.credits, true, true) });
	text += this.$padTextRight("Item", 20) + this.$padTextLeft("Cost", 10);

	var goods = this.$illegalGoodsList(system.ID);
	// remove any goods for which the player either already has a permit, or that no permit is allowed
	for (var i = goods.length - 1; i >= 0; i--) {
		if (goods[i].permit === 1) {
			// if the player already has a permit for this item, remove it.
			if (this.$playerHasPermit(system.ID, goods[i].commodity, false) === true) {
				goods.splice(i, 1);
			}
		} else {
			// if the item doesn't allow permits, remove it
			goods.splice(i, 1);
		}
	}

	for (var i = 0; i < goods.length; i++) {
		var list = goods[i].commodity.split(",");
		var cost = 0;
		for (var j = 0; j < list.length; j++) {
			cost += this._commodityInfo[list[j]][0] * (1 + (system.government + 1) / 8);
		}
		var menuitem = this.$padTextRight(this.$translateCommodityList(goods[i].commodity), 20) + this.$padTextLeft(formatCredits(cost, true, true), 10);
		curChoices["01_PERMIT~" + goods[i].commodity] = {
			text: menuitem,
			alignment: "LEFT",
			color: this._menuColor
		};
	}

	curChoices["97_SPACER"] = "";

	curChoices["99_EXIT"] = {
		text: "[exit-no-change]",
		color: this._itemColor
	};
	var def = "99_EXIT";

	for (var i = 0; i < ((pagesize - 4) - goods.length); i++) {
		curChoices["99_SPACER_" + i] = "";
	}

	var opts = {
		screenID: "oolite-smugglers-permit-map",
		titleKey: "smuggling_permit_purchase",
		allowInterrupt: true,
		exitScreen: "GUI_SCREEN_EQUIP_SHIP",
		overlay: {
			name: "stgu-form.png",
			height: 546
		},
		choices: curChoices,
		initialChoicesKey: def,
		message: text
	};

	mission.runScreen(opts, this.$purchaseScreenHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$purchaseScreenHandler = function $purchaseScreenHandler(choice) {

	if (!choice) return;

	if (choice.indexOf("01_PERMIT") >= 0) {
		var cmdty = choice.substring(choice.indexOf("~") + 1);
		var list = cmdty.split(",");
		var cost = 0;
		for (var j = 0; j < list.length; j++) {
			cost += this._commodityInfo[list[j]][0] * (1 + (system.government + 1) / 8);
		}

		this._permits.push({
			systemID: system.ID,
			commodity: cmdty,
			fake: 0
		});

		// charge the player
		player.credits -= cost;

		// make sure the right email is sent to the player
		var email = worldScripts.EmailSystem;
		if (email) {
			// send a notification email
			var msg = expandDescription("[permit-purchase-email]", {
				commodity: this.$translateCommodityList(cmdty).replace(", ", " and "),
				amount: formatCredits(cost, true, true)
			});

			email.$createEmail({
				sender: expandMissionText("smuggling_permit_authority"),
				subject: expandMissionText("smuggling_permit_subject"),
				date: clock.seconds,
				message: msg
			});
		}

		// apply the install time (15 minutes)
		clock.addSeconds(900);

	}

}

//-------------------------------------------------------------------------------------------------------------
// initialise the illegal goods F4 screen entries
this.$initMainInterface = function $initMainInterface(station) {
	station.setInterface(this.name, {
		title: expandMissionText("smuggling_illegal_goods_title"),
		category: expandMissionText("smuggling_interface_category"),
		summary: expandMissionText("smuggling_illegal_goods_summary"),
		callback: this.$initialIllegalGoodsOptions.bind(this)
	});
}

//-------------------------------------------------------------------------------------------------------------
this.$initialIllegalGoodsOptions = function $initialIllegalGoodsOptions() {
	this._display = 0;
	this._curpage = 0;
	this._direct = 0;
	this.$illegalGoodsOptions();
}

//-------------------------------------------------------------------------------------------------------------
this.$jumpToIllegalsLRC = function (key) {
	switch (key) {
		case "igkey1": this._display = 4; break;
		case "igkey2": this._display = 2; break;
	}
	this._curpage = 0;
	this._direct = 1;
	this.$illegalGoodsOptions();
}

//-------------------------------------------------------------------------------------------------------------
this.$jumpToIllegalsSRC = function (key) {
	switch (key) {
		case "igkey1": this._display = 4; break;
		case "igkey2": this._display = 2; break;
	}
	this._curpage = 0;
	this._direct = 2;
	this.$illegalGoodsOptions();
}

//-------------------------------------------------------------------------------------------------------------
this.$illegalGoodsOptions = function $illegalGoodsOptions() {
	function compareDate(a, b) {
		if (a.started < b.started) return -1;
		if (a.started > b.started) return 1;
		return 0;
	}

	function compareSystem(a, b) {
		if (a.systemID < b.systemID) return -1;
		if (a.systemID > b.systemID) return 1;
		return 0;
	}

	function compareSystemName(a, b) {
		if (a.name < b.name) return -1;
		if (a.name > b.name) return 1;
		return 0;
	}

	var curChoices = {};
	var text = "";
	var p = player.ship;
	var def = "";
	var sbm = worldScripts.Smugglers_BlackMarket;
	var col1 = 6;
	var col2 = 10;
	var pagesize = 20;

	if (this.$isBigGuiActive() === true) {
		pagesize = 26;
	}

	var govs = new Array();
	for (var i = 0; i < 8; i++)
		govs.push(String.fromCharCode(i));
	var spc = String.fromCharCode(31);

	// main menu
	if (this._display === 0) {
		text = "";
		var recent = [];

		for (var i = 0; i < this._illegalGoods.length; i++) {
			// list all new items (less that 5 days since start
			if (this._illegalGoods[i].started != 0 && ((clock.adjustedSeconds - this._illegalGoods[i].started) / 86400) < 7) {
				recent.push({
					name: System.systemNameForID(this._illegalGoods[i].systemID),
					systemID: this._illegalGoods[i].systemID,
					commodity: this.$translateCommodityList(this._illegalGoods[i].commodity),
					description: this._illegalGoods[i].description,
					started: this._illegalGoods[i].started
				});
			}
		}

		text = expandDescription("[recent-declarations]");
		var lines = 3;
		if (recent.length > 0) {
			recent.sort(compareDate);

			for (var i = 0; i < recent.length; i++) {
				var info = System.infoForSystem(galaxyNumber, recent[i].systemID);
				var rt = system.info.routeToSystem(info);
				if (rt) {
					var dist = rt.distance;
				} else {
					var dist = system.info.distanceToSystem(info);
				}
				var textitem = recent[i].name + " (" + govs[info.government] + spc + expandMissionText("smuggling_tl") + (info.techlevel + 1) + ", " + expandMissionText("smuggling_dist") + ": " + dist.toFixed(1) + expandMissionText("smuggling_ly") + ") -- " + recent[i].commodity + ": " + expandDescription("[smuggling-" + recent[i].description + "]");
				text += textitem + "\n";
				var w = defaultFont.measureString(textitem);
				lines += Math.ceil(w / 32);
				if (lines >= (pagesize - 6)) break;
			}

		} else {
			text += expandDescription("[recent-declarations-none]");
			lines += 1;
		}

		text += expandDescription("[offer-selections]");

		curChoices["01_INFO"] = {
			text: "[illegal-sector]",
			color: this._menuColor
		};
		curChoices["02_INRANGE"] = {
			text: "[illegal-7ly]",
			color: this._menuColor
		};
		curChoices["03_PLOT"] = {
			text: "[illegal-course]",
			color: this._menuColor
		};
		if (this._permits.length > 0) {
			curChoices["04_PERMITS"] = {
				text: "[list-permits]",
				color: this._menuColor
			};
			lines += 1;
		}
		if (sbm._waypoints.length > 0) {
			curChoices["05_WAYPOINTS"] = {
				text: "[list-waypoints]",
				color: this._menuColor
			};
			lines += 1;
		}

		curChoices["99_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		for (var i = 0; i < ((pagesize - 5) - lines); i++) {
			curChoices["99_SPACER_" + i] = "";
			//	text += "\n";
		}


		this._curpage = 0;

		def = "99_EXIT";

		var opts = {
			screenID: "oolite-smuggling-illegal-map",
			titleKey: "smuggling_illegal_goods_title",
			allowInterrupt: true,
			overlay: {
				name: "stgu-warning.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: def,
			message: text
		};
	}

	// illegal goods in sector
	if (this._display === 1) {

		var list = [];
		var checking = [];

		for (var i = 0; i < this._illegalGoods.length; i++) {
			if (checking.indexOf(this._illegalGoods[i].systemID) === -1) {
				list.push({
					systemID: this._illegalGoods[i].systemID,
					name: System.systemNameForID(this._illegalGoods[i].systemID)
				});
				checking.push(this._illegalGoods[i].systemID);
			}
		}

		list.sort(compareSystemName);

		var redux = 6;
		var maxpages = Math.ceil(list.length / (pagesize - redux));
		var pagestart = this._curpage * (pagesize - redux);
		var pageend = this._curpage * (pagesize - redux) + (pagesize - redux);
		if (pageend > list.length) pageend = list.length;

		text = expandMissionText("smuggling_illegal_notices", { page: (this._curpage + 1), max: (maxpages > 0 ? maxpages : "1") }) + "\n";
		if (this._IGTInstalled === true) {
			text += expandDescription("[notifications-extra-igt]");
		} else {
			text += expandDescription("[notifications-extra-std]");
		}

		for (var i = pagestart; i < pageend; i++) {
			var info = System.infoForSystem(galaxyNumber, list[i].systemID);
			var rt = system.info.routeToSystem(info);
			if (rt) {
				var dist = rt.distance;
			} else {
				var dist = system.info.distanceToSystem(info);
			}
			text += this.$padTextRight(info.name, col1) + this.$padTextRight("(" + govs[info.government] + spc + expandMissionText("smuggling_tl") + (info.techlevel + 1) + ", " + expandMissionText("smuggling_dist") + ": " + dist.toFixed(1) + expandMissionText("smuggling_ly") + ")", col2);

			var iglist = "";
			var goods = this.$illegalGoodsList(list[i].systemID);
			for (var j = 0; j < goods.length; j++) {
				var cmdty = goods[j].commodity;
				var subitems = cmdty.split(",");
				for (var k = 0; k < subitems.length; k++) {
					iglist += (iglist === "" ? "" : ", ") + displayNameForCommodity(subitems[k]);
					if (this.$playerHasPermit(list[i].systemID, subitems[k], false) === true) iglist += " (" + expandMissionText("smuggling_permit_mark") + ")";
				}
			}
			text += iglist + "\n";
		}

		if (this._curpage === maxpages - 1) {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._menuColor
			};
		}
		if (this._curpage === 0) {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._menuColor
			};
		}
		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-illegal-map",
			titleKey: "smuggling_illegal_goods_title",
			allowInterrupt: true,
			overlay: {
				name: "stgu-warning.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: this._lastChoice ? this._lastChoice : def,
			message: text
		};
	}

	// illegal goods on course
	if (this._display === 2) {
		var target = p.targetSystem;
		var source = system.ID;
		var myRoute = null;
		if (source >= 0) {
			if (p.hasEquipmentProviding("EQ_ADVANCED_NAVIGATIONAL_ARRAY")) {
				myRoute = System.infoForSystem(galaxyNumber, source).routeToSystem(System.infoForSystem(galaxyNumber, target), p.routeMode);
			} else {
				myRoute = {};
				myRoute.route = [];
				if (source !== target) myRoute.route.push(source);
				myRoute.route.push(target);
			}
		}
		if (myRoute != null) {
			var list = myRoute.route;

			var redux = 6;
			var maxpages = Math.ceil(list.length / (pagesize - redux));
			var pagestart = this._curpage * (pagesize - redux);
			var pageend = this._curpage * (pagesize - redux) + (pagesize - redux);
			if (pageend > list.length) pageend = list.length;

			text = expandMissionText("smuggling_illegal_on_course", { page: (this._curpage + 1), max: (maxpages == 0 ? "1" : maxpages) }) + "\n";
			if (this._IGTInstalled === true) {
				text += expandDescription("[notifications-extra-igt]");
			} else {
				text += expandDescription("[notifications-extra-std]");
			}

			for (var i = pagestart; i < pageend; i++) {
				var info = System.infoForSystem(galaxyNumber, list[i]);
				var rt = system.info.routeToSystem(info);
				if (rt && p.hasEquipmentProviding("EQ_ADVANCED_NAVIGATIONAL_ARRAY")) {
					var dist = rt.distance;
				} else {
					var dist = system.info.distanceToSystem(info);
				}
				text += this.$padTextRight(info.name, col1) + this.$padTextRight("(" + govs[info.government] + spc + expandMissionText("smuggling_tl") + (info.techlevel + 1) + ", " + expandMissionText("smuggling_dist") + ": " + dist.toFixed(1) + expandMissionText("smuggling_ly") + ")", col2);

				var iglist = "";
				var goods = this.$illegalGoodsList(list[i]);
				if (goods.length > 0) {
					for (var j = 0; j < goods.length; j++) {
						var cmdty = goods[j].commodity;
						var subitems = cmdty.split(",");
						for (var k = 0; k < subitems.length; k++) {
							iglist += (iglist === "" ? "" : ", ") + displayNameForCommodity(subitems[k]);
							if (this.$playerHasPermit(list[i], subitems[k], false) === true) iglist += " (" + expandMissionText("smuggling_permit_mark") + ")";
						}
					}
					text += iglist;
				} else {
					text += expandMissionText("smuggling_none");
				}
				text += "\n";
			}
		} else {
			var maxpages = 0;
			var list = [];
			if (system.ID === -1) {
				text = expandMissionText("smuggling_illegal_on_route") + "\n\n" + expandMissionText("smuggling_illegal_int") + "\n";
			} else {
				text = expandMissionText("smuggling_illegal_on_route") + "\n\n" + expandMissionText("smuggling_illegal_no_route") + "\n";
			}
		}

		if (this._curpage === maxpages - 1 || list.length === 0) {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._menuColor
			};
		}
		if (this._curpage === 0 || list.length === 0) {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._menuColor
			};
		}
		switch (this._direct) {
			case 0:
				curChoices["98_EXIT"] = {
					text: "[illegal-return]",
					color: this._itemColor
				};
				break;
			case 1:
			case 2:
				curChoices["99_EXIT"] = {
					text: "[illegal-return]",
					color: this._itemColor
				};
				break;
		}

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-illegal-map",
			titleKey: "smuggling_illegal_goods_title",
			allowInterrupt: true,
			overlay: {
				name: "stgu-warning.png",
				height: 546
			},
			exitScreen: (this._direct == 0 ? "GUI_SCREEN_INTERFACES" : (this._direct == 1 ? "GUI_SCREEN_LONG_RANGE_CHART" : "GUI_SCREEN_SHORT_RANGE_CHART")),
			choices: curChoices,
			initialChoicesKey: this._lastChoice ? this._lastChoice : def,
			message: text
		};
	}

	// purchased (legal or otherwise) permits
	if (this._display === 3) {

		var redux = 6;
		var maxpages = Math.ceil(this._permits.length / (pagesize - redux));
		if (maxpages === 0) maxpages = 1;

		text = expandMissionText("smuggling_permits_title", { page: (this._curpage + 1), max: (maxpages == 0 ? "1" : maxpages) }) + "\n\n";

		if (this._permits.length > 0) {
			var pagestart = this._curpage * (pagesize - redux);
			var pageend = this._curpage * (pagesize - redux) + (pagesize - redux);
			if (pageend > this._permits.length) pageend = this._permits.length;

			for (var i = pagestart; i < pageend; i++) {
				var info = System.infoForSystem(galaxyNumber, this._permits[i].systemID);
				text += "  " + expandMissionText("smuggling_permit_item", { commodity: this.$translateCommodityList(this._permits[i].commodity), sys: info.name }) + "\n";
			}
		} else {
			text += "  " + expandMissionText("smuggling_none");
		}

		if (this._curpage === maxpages - 1 || this._permits.length === 0) {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._menuColor
			};
		}
		if (this._curpage === 0 || this._permits.length === 0) {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._menuColor
			};
		}
		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-illegal-map",
			titleKey: "smuggling_illegal_goods_title",
			allowInterrupt: true,
			overlay: {
				name: "stgu-form.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: this._lastChoice ? this._lastChoice : def,
			message: text
		};

	}

	// illegal goods in range
	if (this._display === 4) {
		var list = system.info.systemsInRange(7);

		var redux = 6;
		var maxpages = Math.ceil(list.length / (pagesize - redux));
		var pagestart = this._curpage * (pagesize - redux);
		var pageend = this._curpage * (pagesize - redux) + (pagesize - redux);
		if (pageend > list.length) pageend = list.length;

		text = expandMissionText("smuggling_illegal_7ly", { page: (this._curpage + 1), max: (maxpages == 0 ? "1" : maxpages) }) + "\n";
		if (this._IGTInstalled === true) {
			text += expandDescription("[notifications-extra-igt]");
		} else {
			text += expandDescription("[notifications-extra-std]");
		}

		for (var i = pagestart; i < pageend; i++) {
			var rt = system.info.routeToSystem(list[i]);
			if (rt) {
				var dist = rt.distance;
			} else {
				var dist = system.info.distanceToSystem(list[i]);
			}
			text += this.$padTextRight(list[i].name, col1) + this.$padTextRight("(" + govs[list[i].government] + spc + expandMissionText("smuggling_tl") + (list[i].techlevel + 1) + ", " + expandMissionText("smuggling_dist") + ": " + dist.toFixed(1) + expandMissionText("smuggling_ly") + ")", col2);

			var iglist = "";
			var goods = this.$illegalGoodsList(list[i].systemID);
			if (goods.length > 0) {
				for (var j = 0; j < goods.length; j++) {
					var cmdty = goods[j].commodity;
					var subitems = cmdty.split(",");
					for (var k = 0; k < subitems.length; k++) {
						iglist += (iglist === "" ? "" : ", ") + displayNameForCommodity(subitems[k]);
						if (this.$playerHasPermit(list[i].systemID, subitems[k], false) === true) iglist += " (" + expandMissionText("smuggling_permit_mark") + ")";
					}
				}
				text += iglist;
			} else {
				text += expandMissionText("smuggling_none");
			}
			text += "\n";
		}

		if (this._curpage === maxpages - 1 || list.length === 0) {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._menuColor
			};
		}
		if (this._curpage === 0 || list.length === 0) {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._menuColor
			};
		}
		switch (this._direct) {
			case 0:
				curChoices["98_EXIT"] = {
					text: "[illegal-return]",
					color: this._itemColor
				};
				break;
			case 1:
			case 2:
				curChoices["99_EXIT"] = {
					text: "[illegal-return]",
					color: this._itemColor
				};
				break;
		}

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-illegal-map",
			titleKey: "smuggling_illegal_goods_title",
			allowInterrupt: true,
			overlay: {
				name: "stgu-warning.png",
				height: 546
			},
			exitScreen: (this._direct == 0 ? "GUI_SCREEN_INTERFACES" : (this._direct == 1 ? "GUI_SCREEN_LONG_RANGE_CHART" : "GUI_SCREEN_SHORT_RANGE_CHART")),
			choices: curChoices,
			initialChoicesKey: this._lastChoice ? this._lastChoice : def,
			message: text
		};
	}

	// rock hermit waypoints purchased
	if (this._display === 5) {
		var redux = 6;
		var maxpages = Math.ceil(sbm._waypoints.length / (pagesize - redux));
		if (maxpages === 0) maxpages = 1;

		text = expandMissionText("smuggling_waypoints", { page: (this._curpage + 1), max: (maxpages == 0 ? "1" : maxpages) }) + "\n\n";

		if (sbm._waypoints.length > 0) {
			var pagestart = this._curpage * (pagesize - redux);
			var pageend = this._curpage * (pagesize - redux) + (pagesize - redux);
			if (pageend > sbm._waypoints.length) pageend = sbm._waypoints.length;

			for (var i = pagestart; i < pageend; i++) {
				var info = System.infoForSystem(galaxyNumber, sbm._waypoints[i]);
				var rt = system.info.routeToSystem(info);
				if (rt) {
					var dist = rt.distance;
				} else {
					var dist = system.info.distanceToSystem(info);
				}
				text += info.name + " (" + govs[info.government] + spc + expandMissionText("smuggling_tl") + (info.techlevel + 1) + ", " + expandMissionText("smuggling_dist") + ": " + dist.toFixed(1) + expandMissionText("smuggling_ly") + ")\n";
			}
		} else {
			text += "  " + expandMissionText("smuggling_none");
		}

		if (this._curpage === maxpages - 1 || sbm._waypoints.length === 0) {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["10_NEXTPAGE"] = {
				text: "[illegal-nextpage]",
				color: this._menuColor
			};
		}
		if (this._curpage === 0 || sbm._waypoints.length === 0) {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._disabledColor,
				unselectable: true
			};
		} else {
			curChoices["11_PREVPAGE"] = {
				text: "[illegal-prevpage]",
				color: this._menuColor
			};
		}
		curChoices["98_EXIT"] = {
			text: "[illegal-return]",
			color: this._itemColor
		};

		def = "98_EXIT";

		var opts = {
			screenID: "oolite-smuggling-waypoints-map",
			titleKey: "smuggling_rockhermit_waypoints",
			allowInterrupt: true,
			overlay: {
				name: "stgu-map_marker.png",
				height: 546
			},
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: curChoices,
			initialChoicesKey: this._lastChoice ? this._lastChoice : def,
			message: text
		};

	}

	mission.runScreen(opts, this.$screenHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$screenHandler = function $screenHandler(choice) {

	delete this._lastChoice;

	switch (choice) {
		case "01_INFO":
			this._display = 1;
			break;
		case "02_INRANGE":
			this._display = 4;
			break;
		case "03_PLOT":
			this._display = 2;
			this._curpage = 0;
			break;
		case "04_PERMITS":
			this._display = 3;
			this._curpage = 0;
			break;
		case "05_WAYPOINTS":
			this._display = 5;
			this._curpage = 0;
			break;
		case "10_NEXTPAGE":
			this._curpage += 1;
			break;
		case "11_PREVPAGE":
			this._curpage -= 1;
			break;
		case "98_EXIT":
			this._display = 0;
			break;
	}

	this._lastChoice = choice;

	if (choice != "99_EXIT") {
		this.$illegalGoodsOptions();
	}

}

//-------------------------------------------------------------------------------------------------------------
// returns list of illegal goods for system id
this.$illegalGoodsList = function $illegalGoodsList(sysID, include) {

	var list = [];
	for (var i = 0; i < this._illegalGoods.length; i++) {
		if (this._illegalGoods[i].systemID === sysID) list.push(this._illegalGoods[i]);
	}
	// include default definitions when running a check on docking
	if (include && include === true) {
		for (var i = 0; i < this._defaultAdditions.length; i++) {
			list.push({
				systemID: sysID,
				commodity: this._defaultAdditions[i],
				started: 0,
				end: 0,
				description: "",
				permit: 0,
				scale: 2
			});
		}
	}
	return list;
}

//-------------------------------------------------------------------------------------------------------------
// returns list of illegal goods for system id, but just the commodity names (split out if necessary)
this.$illegalGoodsListCommodityOnly = function $illegalGoodsListCommodityOnly(sysID, include) {
	var list = [];
	for (var i = 0; i < this._illegalGoods.length; i++) {
		if (this._illegalGoods[i].systemID === sysID) {
			var splitlist = this._illegalGoods[i].commodity.split(",");
			for (var j = 0; j < splitlist.length; j++) {
				list.push(splitlist[j]);
			}
		}
	}
	// include default definitions when running a check on docking
	if (include && include === true) {
		for (var i = 0; i < this._defaultAdditions.length; i++) {
			list.push(this._defaultAdditions[i]);
		}
	}
	return list;
}

//-------------------------------------------------------------------------------------------------------------
// returns a particular illegal goods definition
this.$illegalGoodsDefinition = function $illegalGoodsDefinition(sysID, commodity) {
	for (var i = 0; i < this._illegalGoods.length; i++) {
		if (this._illegalGoods[i].systemID === sysID && this._illegalGoods[i].commodity.indexOf(commodity) >= 0) return this._illegalGoods[i];
	}
	return null;
}

//-------------------------------------------------------------------------------------------------------------
// returns true if a particular good is illegal in a particular system, otherwise false;
this.$isGoodIllegal = function $isGoodIllegal(sysID, commodity) {
	for (var i = 0; i < this._illegalGoods.length; i++) {
		if (this._illegalGoods[i].systemID === sysID && this._illegalGoods[i].commodity.indexOf(commodity) >= 0) return true;
	}
	return false;
}

//-------------------------------------------------------------------------------------------------------------
this.$goodIllegalScale = function $goodIllegalScale(sysID, commodity) {
	for (var i = 0; i < this._illegalGoods.length; i++) {
		if (this._illegalGoods[i].systemID === sysID && this._illegalGoods[i].commodity.indexOf(commodity) >= 0) return this._illegalGoods[i].scale;
	}
	return 0;
}

//-------------------------------------------------------------------------------------------------------------
// initial setup of illegal goods in sector
this.$initialSetup = function $initialSetup() {
	// first, add all default/permanent items
	for (var i = 0; i < this._possibilities.length; i++) {
		var item = this._possibilities[i];
		// work out the shortest description in all the possibilities
		var dl = expandDescription("[smuggling-" + item.description + "]").length;
		if (dl < this._minLength) this._minLength = dl;
		if (item.period === -1) {
			var govs = item.govTypes.split(",");
			for (var j = 0; j < govs.length; j++) {
				if (govs[j] != "") {
					var planetlist = this.$getSystemsByGovernment(parseInt(govs[j]), item.properties);
					for (var k = 0; k < planetlist.length; k++) {
						if (planetlist[k].sun_gone_nova) continue;
						this.$applyDefinition(planetlist[k].systemID, item, 0);
					}
				}
			}
		}
	}

	// next pick a random number of planets and apply an illegal good for them
	// add the definitions, between 30 and 60 ** TODO: work out if this is too much or too little
	if (!this.$rand) this.$rand = worldScripts.Smugglers_CoreFunctions.$rand; // make sure our rand function is available
	for (var i = 1; i <= (this.$rand(30) + 30); i++) {
		this.$addNewItem(this.$rand(20));
	}
}

//-------------------------------------------------------------------------------------------------------------
// add a new random item to the list
this.$addNewItem = function $addNewItem(startDays, excludeSystem) {

	var planetid = -1;
	var sys;
	var tries = 0;
	do {
		// pick a random planet
		planetid = this.$rand(256) - 1;
		sys = System.infoForSystem(galaxyNumber, planetid);
		// don't create items in nova systems
		if (sys.sun_gone_nova) planetid = -1;
		// don't create items in the excluded system
		if (planetid === excludeSystem) planetid = -1;

		if (planetid >= 0) {
			// check if this planet has any existing declarations
			var existing = this.$illegalGoodsList(planetid);
			var count = 0;
			for (var i = 0; i < existing.length; i++) {
				// count all the non-default items
				if (existing[i].end != 0) count += 1;
			}
			// if this planet already has an item, skip it.
			if (count > 0) planetid = -1;
		}
		tries += 1;
	} while (planetid === -1 && tries < 5);
	if (planetid === -1 && this._debug) log(this.name, "attempt to add illegal good failed");
	// if we didn't end up finding a planet, exit now
	if (planetid === -1) return;

	// get a list of possibilities
	var poss = this.$getPossibilitiesByGovernment(sys.government, sys.description);
	// pick one from the list
	var pick = this.$rand(poss.length) - 1;
	// add the definition
	this.$applyDefinition(planetid, poss[pick], startDays);
}

//-------------------------------------------------------------------------------------------------------------
// get a list of possible illegal goods for a particular government type (0-7)
this.$getPossibilitiesByGovernment = function $getPossibilitiesByGovernment(govType, planetDescription) {
	var list = [];
	for (var i = 0; i < this._possibilities.length; i++) {
		if (this._possibilities[i].period != -1 && this._possibilities[i].govTypes.indexOf(govType.toString()) != -1 &&
			(this._possibilities[i].properties === "" || this.$checkPlanetDescription(planetDescription, this._possibilities[i].properties) === true) &&
			Math.random() < this._possibilities[i].chance) {
			list.push(this._possibilities[i]);
		}
	}
	return list;
}

//-------------------------------------------------------------------------------------------------------------
// checks the planet description to see if the property text is somewhere in it.
// properties can be a single piece of text (eg "something"), or split with a "," character (eg "something,somethingelse")
this.$checkPlanetDescription = function $checkPlanetDescription(descr, properties) {
	if (properties.indexOf(",") === -1) {
		if (descr.indexOf(properties) >= 0) {
			return true;
		} else {
			return false;
		}
	} else {
		var proplist = properties.split(",");
		var result = false;
		for (var i = 0; i < proplist.length; i++) {
			if (descr.indexOf(proplist[i]) >= 0) result = true;
		}
		return result;
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns a list of planets of a particular government type (0-7) and (optionally) with a description that contains the "option" parameter (eg "civil war")
this.$getSystemsByGovernment = function $getSystemsByGovernment(govType, options) {
	var planets = SystemInfo.filteredSystems(this, function (other) {
		return (other.government === govType && ((options == null || options === "") || other.description.indexOf(options) >= 0));
	});
	return planets;
}

//-------------------------------------------------------------------------------------------------------------
// adds a definition to the active list
// sysID = planet ID
// def   = illegal goods definition
// startDays = number of days to take off the current date, to start this item in the past. This is to simulate an active galaxy, so everything doesn't start from now.
this.$applyDefinition = function $applyDefinition(sysID, def, startDays) {

	//log(this.name, "adding illegal good definition for system " + sysID + " for commodity " + def.commodity);

	// does this definition already exist for this planet?
	for (var i = 0; i < this._illegalGoods.length; i++) {
		if (this._illegalGoods[i].systemID === sysID && this._illegalGoods[i].commodity === def.commodity) return;
	}

	var stime = 0;
	var etime = 0;

	// if we haven't been passed a start time, just do it for sometime in the last day
	if (def.period != -1) {
		if (this._startUpRunning === false) {
			if (startDays == null || startDays === 0) {
				stime = clock.seconds - this.$rand(86400);
			} else {
				// otherwise, the starttime is subtracted from the current clock (as days), converted to seconds
				stime = clock.seconds - (startDays * 86400);
			}
			// set up the end point
			etime = stime + ((def.period + (this.$rand(10) - 5)) * 86400);
		} else {
			// if this is running during the startup routine, we need to reapply start and end times after it's completed
			if (startDays != 0) stime = (startDays * 86400) * -1;
			// set up the end point
			etime = ((def.period + (this.$rand(10) - 5)) * 86400);
		}
	} else {
		stime = 0;
		etime = 0;
	}
	if (def.period != -1 && this._debug) log(this.name, "adding illegal good definition for system " + sysID + " for commodity " + def.commodity + ", starting " + clock.clockStringForTime(stime) + ", ending " + clock.clockStringForTime(etime));

	this._illegalGoods.push({
		systemID: sysID,
		commodity: def.commodity,
		started: stime,
		end: etime,
		description: def.description,
		permit: def.permit,
		scale: def.scale
	});

}

//-------------------------------------------------------------------------------------------------------------
// update start/end times created during the startup process
this.$correctStartEndDates = function $correctStartEndDates() {
	for (var i = 0; i < this._illegalGoods.length; i++) {
		if (this._illegalGoods[i].started < 0) {
			var stime = clock.adjustedSeconds + this._illegalGoods[i].started;
			var etime = stime + this._illegalGoods[i].end;
			this._illegalGoods[i].started = stime;
			this._illegalGoods[i].end = etime;
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// look for any expired items and remove them from the array, and possibly add a new one
this.$cleanup = function $cleanup() {
	// remove expired items
	for (var i = this._illegalGoods.length - 1; i >= 0; i--) {
		if (this._illegalGoods[i].end != 0 && clock.adjustedSeconds > this._illegalGoods[i].end) {
			if (this._debug) log(this.name, "removing item: " + this._illegalGoods[i].systemID + ", " + this._illegalGoods[i].commodity + " ended: " + clock.clockStringForTime(this._illegalGoods[i].end))
			// do we have a permit for this one? If so, delete it
			for (var j = this._permits.length - 1; j >= 0; j--) {
				if (this._permits[j].systemID === this._illegalGoods[i].systemID && this._illegalGoods[i].commodity.indexOf(this._permits[j].commodity) >= 0) {
					this._permits.splice(j, 1);
				}
			}
			this._illegalGoods.splice(i, 1);
		}
	}

	// add new items
	var total = this.$rand(5) - this.$rand(3);
	if (total > 0) {
		if (this._debug) log(this.name, "going to try adding " + total + " new items");
		for (var i = 1; i <= total; i++) {
			// add new items, but not in the current system, because the illegal market script won't pick them up until the game is reloaded
			this.$addNewItem(1, system.ID);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// checks if the player has a permit for the system
// todo: add newcargos permit check
this.$playerHasPermit = function $playerHasPermit(sysID, cmdty, check) {
	for (var i = 0; i < this._permits.length; i++) {
		if (this._permits[i].systemID === sysID && this._permits[i].commodity === cmdty) {
			// are we doing a validity check
			if (check === true && this._permits[i].fake != 0) {
				// did it pass?
				if (Math.random() > this._permits[i].fake) {
					// oh dear, looks like it was discovered as fake
					return false;
				} else {
					// phew! passed!
					return true;
				}
			} else {
				// it's a bought one (or we're not bothering with how fake it is), so it's ok
				return true;
			}
		}
	}
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// removes a permits from the list
this.$removePermit = function $removePermit(sysID, cmdty) {
	for (var i = this._permits.length - 1; i >= 0; i--) {
		if (this._permits[i].systemID === sysID && this._permits[i].commodity === cmdty) {
			this._permits.splice(i, 1);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns display version of commodity list
this.$translateCommodityList = function $translateCommodityList(cmdty_list) {
	var list = cmdty_list.split(",");
	var ret = "";
	for (var i = 0; i < list.length; i++) {
		ret += (ret === "" ? "" : ", ") + displayNameForCommodity(list[i]);
	}
	return ret;
}

//-------------------------------------------------------------------------------------------------------------
// adds the F8F8 comment for illegal goods
this.$updateDescriptions = function $updateDescriptions() {
	// note: for some reason descriptions added via the illegal market script are not being displayed. So we're forcing the issue here.
	var nocomment = expandDescription("[oolite-commodity-no-comment]");
	if (player.ship.docked) {
		var sMarket = player.ship.dockedStation.market;
	} else {
		if (player.ship.target && player.ship.target.isStation) {
			var bcMarket = Ship.shipDataForKey(player.ship.target.dataKey)["market_broadcast"];
			if (!bcMarket) bcMarket = true;
			else {
				var trues = ["1", "true", "yes"];
				bcMarket = bcMarket.toLowerCase();
				if (trues.indexOf(bcMarket) !== -1) bcMarket = true;
				else bcMarket = false;
			}
			if (bcMarket) {
				var sMarket = player.ship.target.market;
			} else {
				var sMarket = system.mainStation.market;
			}
		} else {
			if (system.isInterstellarSpace) return;
			var sMarket = system.mainStation.market;
		}
	}
	var goods = [];
	if (system.ID != -1) {
		goods = this.$illegalGoodsList(system.ID, true);
	} else {
		return;
	}

	for (var cmdty in sMarket) {
		// work out what the comment should be for this commodity
		var cmt = nocomment;
		// special cases
		if (cmdty === "slaves") cmt = expandDescription("[oolite-commodity-illegal]");
		if (this._defaultAdditions.indexOf(cmdty) >= 0) cmt = expandDescription("[" + cmdty + "-legality-description]");
		// if this is an illegal commodity at this station go and grab the description for it from our data
		if (sMarket[cmdty].legality_import != 0 || sMarket[cmdty].legality_export != 0) {
			for (var i = 0; i < goods.length; i++) {
				if (goods[i].commodity.indexOf(cmdty) >= 0 && goods[i].description != "") cmt = expandDescription("[smuggling-" + goods[i].description + "]");
			}
		} else {
			// if it's legal here, but not legal at the main station, put up a warning message
			if (system.mainStation.market[cmdty].legality_import != 0 || system.mainStation.market[cmdty].legality_export != 0) {
				cmt = expandDescription("[main-station-illegal]");
			} else {
				// otherwise, just use the "no comment" comment.
				cmt = nocomment;
			}
		}
		// get the current comment
		var orig = manifest.comment(cmdty);
		// in most cases that's all we'll need
		var curr_cmt = orig;
		// but if another OXP has added their own comments, we'll need to get the first line
		// if the OXP adds comments without first adding a linebreak this won't work, but it should be good for most circumstances
		if (orig.indexOf("\n") >= 0) curr_cmt = orig.substring(0, orig.indexOf("\n"));
		// replace the current comment with our new one
		orig = orig.replace(curr_cmt, cmt);
		// plonk it back into the comment field for this commodity
		manifest.setComment(cmdty, orig);
	}
}

//-------------------------------------------------------------------------------------------------------------
// mark local systems on the galactic chart that have illegal goods
this.$markSystems = function $markSystems() {
	// clear out existing marks
	for (var i = 0; i < this._markedSystems.length; i++) {
		mission.unmarkSystem({
			system: this._markedSystems[i],
			name: "illegalgoods"
		});
	}
	// reset the array
	this._markedSystems.length = 0;

	if (this._enableMarkedSystems === false) return;
	// get all systems in 10 ly
	var list = system.info.systemsInRange(this._markedSystemRange);
	// mark systems that have illegal goods (other than narcs or slaves)
	// system: firearms only = orange, everything else yellow
	for (var i = 0; i < list.length; i++) {
		var goods = this.$illegalGoodsList(list[i].systemID);
		if (goods.length > 0) {
			this._markedSystems.push(list[i].systemID);
			if (goods.length > 1) {
				// if there's more than one illegal, mark it with yellow
				mission.markSystem({
					system: list[i].systemID,
					name: "illegalgoods",
					markerColor: this._markerColors[this._mixedIllegalColor],
					markerShape: this._markerShapes[this._mixedIllegalShape]
				});
			} else {
				// if the only illegal is firearms, mark it with orange
				if (goods[0].commodity === "firearms") {
					mission.markSystem({
						system: list[i].systemID,
						name: "illegalgoods",
						markerColor: this._markerColors[this._firearmsColor],
						markerShape: this._markerShapes[this._firearmsShape]
					});
				} else {
					// otherwise mark it with yellow
					mission.markSystem({
						system: list[i].systemID,
						name: "illegalgoods",
						markerColor: this._markerColors[this._mixedIllegalColor],
						markerShape: this._markerShapes[this._mixedIllegalShape]
					});
				}
			}
		}
	}
}