"use strict";

this.name = "vASC";
this.version = "1.2"; // August 18th, 2025

this._settings = {
    "COMPASS_MODE_BASIC": false, // is always false and not configurable
    "COMPASS_MODE_BEACONS": true,
    "COMPASS_MODE_PLANET": true,
    "COMPASS_MODE_STATION": true,
    "COMPASS_MODE_SUN": true,
    "COMPASS_MODE_TARGET": true,
    "COMPASS_POINT_APPROACH": true,
    "COMPASS_POINT_AWAY": true
};

/* ====================================================
      COMPASS TARGET CHANGED HANDLER
 worldScripts.vASC (very Advanced Space Compass)
 Automatically re-orients the player's ship toward the
 new target each time the compass target is changed.
======================================================= */

this.compassTargetChanged = function (whom, mode) {
    // BASIC mode should never react
    if (mode === "COMPASS_MODE_BASIC") {
        return;
    }
    
    // Check if this mode is enabled
    if (!this._settings[mode]) {
        return;
    }
    
    // Ensure the target exists and is valid
    if (!whom || !whom.isValid) {
        log(this.name, "Invalid target. Skipping re-orientation.");
        return;
    }

    // Get the player's current position and heading
    var playerPosition = player.ship.position;
    var playerHeading = player.ship.heading;

    // Calculate the vector from the player to the target
    var targetVector = whom.position.subtract(playerPosition);

    // Normalize the vector to get the direction
    var targetDirection = targetVector.direction();

    // Determine the view direction and calculate the required rotation
    try {
        if (player.ship.viewDirection === "VIEW_FORWARD" &&
            this._settings["COMPASS_POINT_APPROACH"]) {
            // Forward view: Rotate toward the target
            var angle = playerHeading.angleTo(targetDirection);
            var cross = playerHeading.cross(targetDirection).direction();
            player.ship.orientation = player.ship.orientation.rotate(cross, -angle);
        } else if (player.ship.viewDirection === "VIEW_AFT" &&
            this._settings["COMPASS_POINT_AWAY"]) {
            // Aft view: Rotate away from the target using the reversed heading
            var reversedHeading = playerHeading.multiply(-1);
            var angle = reversedHeading.angleTo(targetDirection);
            var cross = reversedHeading.cross(targetDirection).direction();
            player.ship.orientation = player.ship.orientation.rotate(cross, -angle);
        }
    } catch (error) {
        log(this.name, "Error during re-orientation: " + error.message);
        return;
    }

    // Log the re-orientation for debugging purposes
    // log(this.name, "Re-oriented player's ship toward target: " + whom.displayName);
}

/* ====================================================
      OTHER EVENT HANDLERS
======================================================= */

this.playerWillSaveGame = function () {
    missionVariables.vASC_settings = JSON.stringify(this._settings);
}

this.startUpComplete = function () {
    delete this.startUpComplete;
    if (missionVariables.vASC_settings) {
        this._settings = JSON.parse(missionVariables.vASC_settings);
    }
    if (player.ship.docked) {
        this.shipDockedWithStation(player.ship.dockedStation);
    }
}

this.shipDockedWithStation = function (station) {
    station.setInterface("vASC_config", {
        title: expandMissionText("wildefire_vasc_title"),
        category: expandDescription("[interfaces-category-ship-systems]"),
        summary: expandMissionText("wildefire_vasc_summary"),
        callback: this._showConfigScreen.bind(this)
    });
}

/* ====================================================
      CONFIGURATION SCREEN
======================================================= */

this._showConfigScreen = function () {
    mission.runScreen({
        screenID: "vASC_config",
        titleKey: "wildefire_vasc_title",
        messageKey: "wildefire_vasc_message",
        background: { name: "wildefire_vasc_background.png", height: 512 },
        exitScreen: "GUI_SCREEN_INTERFACES",
        choicesKey: "wildefire_vasc_choices",
        allowInterrupt: true,
        initialChoicesKey: "EXIT"
    }, function (toggle) {
        if (toggle == "EXIT") {
            return;
        }
        this._settings[toggle] = !this._settings[toggle];
        this._showConfigScreen();
    }
    );
    for (var mode in this._settings) {
        var status = this._settings[mode] ? "wildefire_vasc_enabled" : "wildefire_vasc_disabled";
        mission.addMessageText(expandMissionText("wildefire_vasc_" + mode) + ": " + expandMissionText(status));
    }
}

/* ====================================================
      THE END
======================================================= */