// worldScripts.retro_encabulator
"use strict";

this.name    = "retro_encabulator";
this.version = "0.7"; // October 6th, 2025
this.author  = "GLM-4.5";
this.license = "Public domain";

/* ====================== INITIALIZATION ===================== */

// Default prefix initialization
this._prefix = "retro";

// Background images array
this._backgrounds = ["GLM_retro_background.png", "GLM_turbo_background.png", "GLM_aristo_background.png"];

// Initialize journal array
this._journal = [];

// Maximum journal entries to prevent infinite growth
this._maxJournalEntries = 20;

// Mutation control variables
this._mutationAllowed = true;
this._lastMutationTime = 0;
this._lastDockedStation = null; // Track the last station where player docked

/* ====================== SAVE & LOAD ======================== */

// Save data to missionVariables when game saves
this.playerWillSaveGame = function () {
    // Save journal array
    missionVariables.retro_encabulator_journal = JSON.stringify(this._journal);

    // Save last mutation time
    missionVariables.retro_encabulator_lastMutation = this._lastMutationTime;
    
    // Save current prefix
    missionVariables.retro_encabulator_prefix = this._prefix;
}

// Initialize the F4 interface when the game starts
this.startUp = function () {
    delete this.startUp;
    
    // Load journal from missionVariables with JSON parse
    if (missionVariables.retro_encabulator_journal) {
        this._journal = JSON.parse(missionVariables.retro_encabulator_journal);

        // Ensure loaded journal doesn't exceed limit
        if (this._journal.length > this._maxJournalEntries) {
            // Keep only the most recent entries
            this._journal = this._journal.slice(-this._maxJournalEntries);
        }
    }

    // Load last mutation time
    if (missionVariables.retro_encabulator_lastMutation) {
        this._lastMutationTime = missionVariables.retro_encabulator_lastMutation;
    }
    
    // Load current prefix
    if (missionVariables.retro_encabulator_prefix) {
        this._prefix = missionVariables.retro_encabulator_prefix;
    }

    // Apply interface with the docked station
    this._applyInterface(player.ship.dockedStation);
}

/* ==================== INTERFACES SCREEN ==================== */

// Helper function to set up the F4 interface
this._applyInterface = function (station) {
    const reportTitle = this._capitalizePrefix(expandDescription("[" + this._prefix + "_prefix_display]")) +
        expandDescription("[retro_encabulator_report_title_suffix]");

    const reportSummary = expandDescription("[retro_encabulator_report_summary_prefix]") +
        expandDescription("[" + this._prefix + "_prefix_display]") +
        expandDescription("[retro_encabulator_report_summary_suffix]");

    station.setInterface("encabulator_report", {
        title:    reportTitle,
        category: expandDescription("[interfaces-category-ship-systems]"),
        summary:  reportSummary,
        callback: this._showEncabulator.bind(this)
    });

    // Add journal option after 3+ entries
    if (this._journal.length >= 3) {
        station.setInterface("encabulator_journal", {
            title:    expandDescription("[retro_encabulator_journal_interface_title]"),
            category: expandDescription("[interfaces-category-logs]"),
            summary:  expandDescription("[retro_encabulator_journal_interface_summary]"),
            callback: this._showJournal.bind(this)
        });
    }
}

// Initialize the F4 interface when docking
this.shipDockedWithStation = function (station) {
    // Only allow mutation if docking at a different station than last time
    if (this._lastDockedStation !== station) {
        this._mutationAllowed = true;
    }

    this._applyInterface(station);
}

// Clean up when undocking
this.shipWillLaunchFromStation = function (station) {
    station.setInterface("encabulator_report", null);
    station.setInterface("encabulator_journal", null);
    this._lastDockedStation = station; // Update the last docked station
}

// Allow mutation when day changes
this.dayChanged = function () {
    this._mutationAllowed = true;
}

/* ===================== MISSION SCREENS ===================== */

// Method to handle the F4 interface callback
this._showEncabulator = function () {
    // Get a new message (this updates this._prefix)
    const message = this._getMessage();

    // Select appropriate background based on prefix
    let backgroundIndex = 0; // Default to retro
    if (this._prefix === "turbo") backgroundIndex = 1;
    if (this._prefix === "aristo") backgroundIndex = 2;

    // Build parameters object
    const parameters = {
        title: expandDescription("[retro_encabulator_report_title]", { prefix: this._capitalizePrefix(expandDescription("[" + this._prefix + "_prefix_display]")) }),
        message: message,
        background: this._backgrounds[backgroundIndex],
        exitScreen: "GUI_SCREEN_INTERFACES"
    };

    // Add music only for retro prefix
    if (this._prefix === "retro") {
        parameters.music = "dt_is_that_a_retro_encabulator.ogg";
    }

    // Display the mission screen
    mission.runScreen(parameters);

    // Update the interface with the new prefix
    this._applyInterface(player.ship.dockedStation);
}

// Show journal via F4 interface
this._showJournal = function () {
    let journalText = expandDescription("[retro_encabulator_journal_header]") + "\n\n";

    if (this._journal.length === 0) {
        journalText += expandDescription("[retro_encabulator_journal_empty]");
    } else {
        // Add entry count indicator
        journalText += expandDescription("[retro_encabulator_journal_entries_prefix]") +
            this._journal.length +
            expandDescription("[retro_encabulator_journal_showing_recent]") +
            Math.min(this._journal.length, 8) + ")\n\n";

        // Display only the 8 most recent entries (newest first for better UX)
        const startIndex = Math.max(0, this._journal.length - 8);
        for (let i = this._journal.length - 1; i >= startIndex; i--) {
            var entry = this._journal[i];
            journalText += expandDescription("[retro_encabulator_journal_entry_prefix]") +
                entry.date + ": " + entry.note + "\n\n";
        }
    }

    // Build parameters object
    const parameters = {
        title: expandDescription("[retro_encabulator_journal_title]"),
        message: journalText,
        background: "GLM_journal_background.png",
        exitScreen: "GUI_SCREEN_INTERFACES"
    };

    // Random chance to play "Fascinating" sound (25% chance)
    if (Math.random() < 0.25) {
        parameters.music = "dt_fascinating.ogg";
    }

    // Display the mission screen
    mission.runScreen(parameters);
}

/* ==================== HELPER FUNCTIONS ===================== */

// Helper function to capitalize prefix for titles
this._capitalizePrefix = function (prefix) {
    return prefix.charAt(0).toUpperCase() + prefix.slice(1);
}

// Method to handle prefix selection and message generation
this._getMessage = function () {
  if (this._mutationAllowed) {
    // Mutation is allowed, so proceed with random selection
    const prefixes = ['turbo', 'retro', 'aristo'];
    const selectedPrefix = prefixes[Math.floor(Math.random() * prefixes.length)];

    // Check if prefix changed
    if (selectedPrefix !== this._prefix) {
        // Perform the mutation
        this._doMutation(this._prefix, selectedPrefix);
    }
  }
  // Generate description with current prefix
  return expandDescription("[" + this._prefix + "-description]");
}

// Perform mutation from old prefix to new prefix
this._doMutation = function (oldPrefix, newPrefix) {
    // Update mutation control
    this._mutationAllowed = false;
    this._lastMutationTime = clock.seconds;

    // Generate reaction
    const reaction = expandDescription("[retro_encabulator_" + oldPrefix + "_to_" + newPrefix + "]") || 
                     expandDescription("[retro_encabulator_mutation_fallback]");
    
    // Create journal entry
    const newEntry = {
        date: clock.days,
        oldPrefix: oldPrefix,
        newPrefix: newPrefix,
        note: reaction
    };

    // Add new entry to journal
    this._journal.push(newEntry);

    // Enforce journal size limit
    if (this._journal.length > this._maxJournalEntries) {
        this._journal.shift(); // Remove oldest entry
    }

    // Update current prefix
    this._prefix = newPrefix;
}