"use strict";
this.name = "market_tweak.js";
this.author = "Stranger, phkb";
this.copyright = "Creative Commons: attribution, non-commercial, sharealike with clauses - see readme.txt";
this.description = "Script for selecting market preset";
this.version = "3.10";

/*
    Tweaked by phkb
    All tweaks selectable/configurable via Library Config.
*/

// controlling prices/quantity of goods in harsh mining worlds (tl < 4, radius < 54250)
this._harshMining = {
    price_discount_factor: 3.0, // increase this value to make the price discount of mining goods greater
    quantity_increase_factor: 4.5, // increase this value to make the quantity boost of mining goods factor greater
    food_price_factor: 2.0, // increase this value to boost the price of food
    textiles_price_factor: 2.0, // increase this value to boost the price of textiles
    liquor_price_factor: 2.0, // increase this value to boost the price of liquor/wines
    furs_price_factor: 1.5, // increase this value to boost the price of furs
    medicine_price_factor: 2.0, // increase this value to boost the price of medicine
    oxygen_price_factor: 3.0, // increase this value to boost the price of oxygen
};

// Note: when changing settings that impact the current system, the change won't be seen until you jump out and back in again.
this._sweConfig = {
    Name: this.name,
    Alias: expandMissionText("sweconomy_config_alias"),
    Display: expandMissionText("sweconomy_config_display"),
    Alive: "_sweConfig",
    SInt: {
        S0: { Name: "_harshMining.price_discount_factor", Def: 3.0, Float: 1, Min: 1.0, Max: 10.0, Desc: expandMissionText("sweconomy_mining_discount") },
        S1: { Name: "_harshMining.quantity_increase_factor", Def: 4.5, Float: 1, Min: 1.0, Max: 10.0, Desc: expandMissionText("sweconomy_mining_quantity") },
        S2: { Name: "_harshMining.food_price_factor", Def: 2.0, Float: 1, Min: 0.1, Max: 5, Desc: expandMissionText("sweconomy_mining_food") },
        S3: { Name: "_harshMining.textiles_price_factor", Def: 2.0, Float: 1, Min: 0.1, Max: 5, Desc: expandMissionText("sweconomy_mining_textiles") },
        S4: { Name: "_harshMining.liquor_price_factor", Def: 2.0, Float: 1, Min: 0.1, Max: 5, Desc: expandMissionText("sweconomy_mining_liquor") },
        S5: { Name: "_harshMining.furs_price_factor", Def: 1.5, Float: 1, Min: 0.1, Max: 5, Desc: expandMissionText("sweconomy_mining_furs") },
        S6: { Name: "_harshMining.medicine_price_factor", Def: 2.0, Float: 1, Min: 0.1, Max: 5, Desc: expandMissionText("sweconomy_mining_medicine") },
        S7: { Name: "_harshMining.oxygen_price_factor", Def: 3.0, Float: 1, Min: 0.1, Max: 5, Desc: expandMissionText("sweconomy_mining_oxygen") },
        Info: expandMissionText("sweconomy_sint_info")
    }
};

this._miningWorlds = [];
this._labelsOn = false;

//-------------------------------------------------------------------------------------------------------------
this.startUp = function () {
    if (missionVariables.SWEconomy_HarshMining) this._harshMining = JSON.parse(missionVariables.SWEconomy_HarshMining);

    if (this._harshMining.hasOwnProperty("food_price_factor") == false) {
        this._harshMining.food_price_factor = 2.0;
        this._harshMining.textiles_price_factor = 2.0;
        this._harshMining.liquor_price_factor = 2.0;
        this._harshMining.furs_price_factor = 1.5;
    }
    if (this._harshMining.hasOwnProperty("medicine_price_factor") == false) {
        this._harshMining.medicine_price_factor = 2.0;
        this._harshMining.oxygen_price_factor = 3.0;
    }

    if (worldScripts.Lib_Config) worldScripts.Lib_Config._registerSet(this._sweConfig);

    var msi = worldScripts.MarketScriptInterface_Main;
    msi.$addMarketInterface("tradegoods_general", "$updateGeneralCommodityDefinition", this.name);
}

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {
    this.playerEnteredNewGalaxy();

    var xui = worldScripts.XenonUI;
    if (!xui) return;
    xui.$setF6OverlayLookup("economy", this.name, "$returnF6EconomyOverlay");
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function () {
    missionVariables.SWEconomy_HarshMining = JSON.stringify(this._harshMining);
}

//-------------------------------------------------------------------------------------------------------------
this.playerEnteredNewGalaxy = function () {
    this._miningWorlds.length = 0;
    for (var i = 0; i <= 255; i++) {
        var sys = System.infoForSystem(galaxyNumber, i);
        if (sys.techlevel < 4 && sys.radius < 5425) {
            this._miningWorlds.push(i);
        }
    }
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenWillChange = function (to, from) {
    // try to turn the labels off before we get to these screens
    if (to == "GUI_SCREEN_STATUS" || to == "GUI_SCREEN_SYSTEM_DATA") {
        if (this._labelsOn) this.$hideMiningLabels();
    }
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged = function (to, from) {
    if ((to == "GUI_SCREEN_LONG_RANGE_CHART" || to == "GUI_SCREEN_SHORT_RANGE_CHART") && this._labelsOn == false) {
        this.$showMiningLabels();
    }
    if (to != "GUI_SCREEN_LONG_RANGE_CHART" && to != "GUI_SCREEN_SHORT_RANGE_CHART" && this._labelsOn == true) {
        this.$hideMiningLabels();
    }
}

//-------------------------------------------------------------------------------------------------------------
this.$showMiningLabels = function () {
    this._labelsOn = true;
    for (var i = 0; i < this._miningWorlds.length; i++) {
        var s = System.infoForSystem(galaxyNumber, this._miningWorlds[i]);
        s.name = s.name + expandMissionText("sweconomy_mark_mining");
    }
}

//-------------------------------------------------------------------------------------------------------------
this.$hideMiningLabels = function () {
    for (var i = 0; i < this._miningWorlds.length; i++) {
        var s = System.infoForSystem(galaxyNumber, this._miningWorlds[i]);
        s.name = null; // reset to default
    }
    this._labelsOn = false;
}

//-------------------------------------------------------------------------------------------------------------
this.$updateGeneralCommodityDefinition = function (goodDefinition, station, systemID) {
    // does this system have any commodities that need adjusting?

    // only do this when the station parameter is null ie. system level
    // otherwise we would rerun the functions twice for the main station
    if (systemID && systemID != -1 && station == null) {
        var sys = System.infoForSystem(galaxyNumber, systemID);
        var industry_adjust = (25000 + sys.productivity) / 50000; // production of industrial goods vs GNP scaling
        var agro_adjust = (10000 + sys.productivity) / 25000; // production of agro goods vs GNP scaling

        if ((sys.techlevel + 2 * sys.government) > 12) { // illegal goods prohibited in (tech OR social) advanced worlds
            switch (goodDefinition.key) {
                case "firearms":
                case "slaves":
                case "narcotics":
                    goodDefinition.quantity = 0;
                    break;
            }
        }

        if (sys.economy > 4) { // items unavailable in agrarian worlds
            switch (goodDefinition.key) {
                case "computers":
                case "machinery":
                case "alloys":
                    goodDefinition.quantity = 0;
                    break;
            }
        }

        if (sys.economy < 3) { // items unavailable in industrial worlds
            switch (goodDefinition.key) {
                case "food":
                case "textiles":
                case "liquor_wines":
                case "furs":
                    goodDefinition.quantity = 0;
                    break;
            }
        }

        if (sys.techlevel < 4 && sys.radius < 5425) { // too harsh conditions to produce much by archaic methods
            switch (goodDefinition.key) {
                case "food":
                case "textiles":
                case "liquor_wines":
                case "furs":
                case "medicine":
                case "oxygen":
                    // reduce quantity
                    goodDefinition.quantity = 0; //parseInt(goodDefinition.quantity / this._harshMining.agri_quantity_reductor);
                    // increase prices
                    var factor = 1.0;
                    if (goodDefinition.key == "food") factor = this._harshMining.food_price_factor;
                    if (goodDefinition.key == "textiles") factor = this._harshMining.textiles_price_factor;
                    if (goodDefinition.key == "liquor_wines") factor = this._harshMining.liquor_price_factor;
                    if (goodDefinition.key == "furs") factor = this._harshMining.furs_price_factor;
                    if (goodDefinition.key == "medicine") factor = this._harshMining.medicine_price_factor;
                    if (goodDefinition.key == "oxygen") factor = this._harshMining.oxygen_price_factor;
                    // adjust this factor based on the underlying economy for the system
                    // eco 6 generally have slightly higher prices, so we don't want to increase those prices
                    // as much as we do for eco 7
                    factor = factor * (sys.economy / 7);
                    goodDefinition.price *= factor;
                    break;
            }
            if (goodDefinition.classes.indexOf("oolite-mining") >= 0) {
                // bump mining goods by factors relating to TL/radius/productivity
                // productivity plays the biggest role
                //log(this.name, "pre: " + goodDefinition.key + " price = " + goodDefinition.price + ", amt = " + goodDefinition.quantity);
                var prod_factor = (((sys.productivity / 4500) * 4 + (sys.techlevel / 3) + (sys.radius / 4500)) / 100) * this._harshMining.quantity_increase_factor;
                var price_factor = (((sys.productivity / 4500) * 3 + (sys.techlevel / 3) + (sys.radius / 4500)) / 100) * this._harshMining.price_discount_factor;
                //log(this.name, "prod = " + sys.productivity + ", tl = " + sys.techlevel + ", radius = " + sys.radius);
                //log(this.name, "prod_factor = " + prod_factor + ", price_factor = " + price_factor);
                var amt = goodDefinition.quantity;
                goodDefinition.quantity = parseInt(amt + amt * prod_factor);
                var price = goodDefinition.price;
                goodDefinition.price = parseInt(price - price * price_factor);
                //log(this.name, "post: " + goodDefinition.key + " price = " + goodDefinition.price + ", amt = " + goodDefinition.quantity);
            }
        }

        if ((sys.economy > 2) && (sys.techlevel > 3)) { // items available only in archaic tech OR industrial worlds    
            switch (goodDefinition.key) {
                case "minerals":
                case "radioactives":
                    goodDefinition.quantity = 0;
                    break;
            }
        }

        if (sys.techlevel < 7) { // items unavailable below median tech limit
            switch (goodDefinition.key) {
                case "computers":
                case "medicine":
                case "luxuries":
                    goodDefinition.quantity = 0;
                    break;
            }
        } else {
            if (goodDefinition.key == "computers") goodDefinition.quantity = Math.floor(goodDefinition.quantity * industry_adjust);
            if (goodDefinition.key == "medicine") goodDefinition.quantity = Math.floor(goodDefinition.quantity * agro_adjust);
            if (goodDefinition.key == "luxuries") goodDefinition.quantity = Math.floor(goodDefinition.quantity * industry_adjust);
        }

        if (sys.techlevel < 4) { // items unavailable in archaic tech worlds
            switch (goodDefinition.key) {
                case "machinery":
                case "alloys":
                case "oxygen":
                    goodDefinition.quantity = 0;
            }
        } else {
            switch (goodDefinition.key) {
                case "machinery":
                case "alloys":
                case "oxygen":
                    goodDefinition.quantity = Math.floor(goodDefinition.quantity * industry_adjust);
            }
        }

        // low tech production, but needs some investments 
        switch (goodDefinition.key) {
            case "liquor_wines":
            case "furs":
                goodDefinition.quantity = Math.floor(goodDefinition.quantity * agro_adjust);
                break;
        }
    }

    return goodDefinition;
}

//-------------------------------------------------------------------------------------------------------------
this.$returnF6EconomyOverlay = function(ratio, font) {
    return "f6-overlay-sw-economy_" + ratio + "_" + font + ".png";
}