"use strict";
this.name = "GalCopBB_SolarActivity";
this.author = "phkb";
this.copyright = "2017 phkb";
this.description = "Controls all solar activity-system mission settings (70-73)";
this.license = "CC BY-NC-SA 4.0";

/*
	*70 - Use scan equipment to take close up measurements of sun
			get to fuel scoop dist = ((system.sun.collisionRadius * system.sun.collisionRadius) / system.sun.position.squaredDistanceTo(player.ship.position)) > 0.75;
			prime scanning device, activate
			await completion
			If player retreats before scan is complete, scan will stop, but will restart automatically when in range again (ie it won't start from the beginning)
	*71 - Recover data cache in low orbit over sun
			find data cache
			transmit security code
			await data package
			put two nav beacons in low orbit around sun on system population, regardless of mission status. if mission is activated, turn on beacon labels
	72 - Launch monitoring devices (special missiles)
			must deploy 2 special missiles from within fuel scoop range, one on the planet side, the other on the far side of the sun.
			may need alternate launch mechanism because the player will be firing without a target and the standard controls don't allow for this.
			TODO 
			- how to file missile without a target?
			- plot out what happens if the player sells the solar missile
	*73 - Same as 70, but need to collect scanner from nearby system first, then perform scan
	*74 - Delivery solar scan data to hi-tech system for analysis

	Ideas for solar activity systems
		- create severe solar activity events, noted via Snoopers
		- during events, make it hard to jump into/out of system
			- jump out possible but only when distance from sun is 2 times planet-sun distance
			- jump in possible, but will occasionally fail and send the player to interstellar space. Warning could be given at start of jump sequence.
			- if player is tail-gating another ship, still apply the chance of interstellar space
		
*/

this._debug = false;
this._solarActivitySystems = []; // systems where there is solar activity (for missions 70-72)
this._scanTimer = null; // timer object used when scanning 
this._scanPosition = 0; // current position of the scan
this._outOfRangeCount = 0; // counts number of times scan attempted to start but was out of range
this._missEquipStore1 = ""; // stores equip key of first missile that was replaced for a special missile
this._missEquipStore2 = ""; // stores equip key of second missile that was replaced for a special missile
this._energy = null; // timer to deduct energy
this._energyReduction = 3; // amount of energy to deduct each quarter second
this._galJump = false;

//-------------------------------------------------------------------------------------------------------------
this.startUp = function () {
	this._solarActivitySystems = this.$findSolarActivitySystems();

	var gcm = worldScripts.GalCopBB_Missions;
	// add these mission types into the main control
	var list = [70, 71, 73, 74];
	gcm._availableMissionTypes = gcm._availableMissionTypes.concat(list);
	gcm._multiStageMissionTypes.push(73);
	this._debug = gcm._debug;
	// add our reputation entities
	var ent = worldScripts.GalCopBB_Reputation._entities;
	ent[expandDescription("[gcm_reputation_boh]")] = {
		scope: "system",
		display: true,
		getValueWS: "",
		getValueFN: "",
		maxValue: 7,
		minValue: 0,
		degrade: "0.05|30",
		rewardGrid: [{
			value: 0,
			description: ""
		},
		{
			value: 4,
			description: "",
			achievementWS: "GalCopBB_SolarActivity",
			achievementFN: "$beaconOfHopeReward"
		},
		{
			value: 6,
			description: "",
			achievementWS: "GalCopBB_SolarActivity",
			achievementFN: "$beaconOfHopeReward"
		},
		{
			value: 7,
			description: "",
			achievementWS: "GalCopBB_SolarActivity",
			achievementFN: "$beaconOfHopeReward"
		}
		]
	};
	// exclude some equipment items from being sold/stored by ShipConfig
	var sc = worldScripts.ShipConfiguration_Core;
	if (sc) {
		sc._noSell.push("EQ_GCM_SOLAR_SCANNER");
	}
}

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {
	if (missionVariables.GalCopBBMissions_MissileStore1) this._missEquipStore1 = missionVariables.GalCopBBMissions_MissileStore1;
	if (missionVariables.GalCopBBMissions_MissileStore2) this._missEquipStore2 = missionVariables.GalCopBBMissions_MissileStore2;
	worldScripts.BulletinBoardSystem.$registerBBEvent("GalCopBB_SolarActivity", "$custom_shipWillDockWithStation", "shipWillDockWithStation_start");
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function () {
	if (this._missEquipStore1 != "") {
		missionVariables.GalCopBBMissions_MissileStore1 = this._missEquipStore1;
	} else {
		delete missionVariables.GalCopBBMissions_MissileStore1;
	}
	if (this._missEquipStore2 != "") {
		missionVariables.GalCopBBMissions_MissileStore2 = this._missEquipStore2;
	} else {
		delete missionVariables.GalCopBBMissions_MissileStore2;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.systemWillPopulate = function () {
	if (this._solarActivitySystems.indexOf(system.ID) >= 0) {
		// put our two data gatherers in place regardless of whether there's an active mission or not
		// adds some ambience to systems. or something, anyway...
		var position1 = Vector3D(0, 0, 1.1).fromCoordinateSystem("sps");
		system.setPopulator("gcm-solarbeacon1", {
			callback: function (pos) {
				var checkShips = system.addShips("commsrelay", 1, pos, 100);
				var sm = null;
				if (checkShips) sm = checkShips[0];
				if (sm) {
					sm.displayName = expandDescription("[gcm_solar_monitor_alpha]");
					sm.energyRechargeRate = 500;
					sm.heatInsulation = 100;
					sm.script._missionID = 0;
					sm.script.keyID = "solar_alpha";
					sm.script.keyType = "solar_monitor";
					if (sm.script.shipDied) sm.script.$gcm_hold_shipDied = sm.script.shipDied;
					sm.script.shipDied = worldScripts.GalCopBB_Missions.$gcm_entity_shipDied;
				}
			}.bind(this),
			location: "COORDINATES",
			coordinates: position1
		});
		var position2 = Vector3D(0, 0, -1.1).fromCoordinateSystem("sps");
		system.setPopulator("gcm-solarbeacon2", {
			callback: function (pos) {
				var checkShips = system.addShips("commsrelay", 1, pos, 100);
				var sm = null;
				if (checkShips) sm = checkShips[0];
				if (sm) {
					sm.displayName = expandDescription("[gcm_solar_monitor_beta]");
					sm.energyRechargeRate = 500;
					sm.heatInsulation = 100;
					sm.script._missionID = 0;
					sm.script.keyID = "solar_beta";
					sm.script.keyType = "solar_monitor";
					if (sm.script.shipDied) sm.script.$gcm_hold_shipDied = sm.script.shipDied;
					sm.script.shipDied = worldScripts.GalCopBB_Missions.$gcm_entity_shipDied;
				}
			}.bind(this),
			location: "COORDINATES",
			coordinates: position2
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillExitWitchspace = function () {
	this._scanPosition = 0;
	this.$stopTimer();
	if (this._galJump === true) {
		this._galJump = false;
		this._solarActivitySystems = this.$findSolarActivitySystems();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillLaunchFromStation = function (station) {
	var gcm = worldScripts.GalCopBB_Missions;
	var list = gcm.$getListOfMissions(true, 71);
	for (var i = 0; i < list.length; i++) {
		if (system.ID === list[i].source &&
			list[i].data.targetQuantity > 0 &&
			list[i].data.quantity < list[i].data.targetQuantity &&
			list[i].expiry > clock.adjustedSeconds) {

			var targets = system.shipsWithPrimaryRole("commsrelay");
			for (var j = 0; j < targets.length; j++) {
				if (targets[j].script.keyType && targets[j].script.keyType == "solar_monitor") {
					targets[j].script._missionID = list[i].ID;
					var m_num = 1;
					targets[j].beaconCode = "M";
					if (targets[j].script.keyID == "solar_beta") m_num = 2;

					targets[j].beaconLabel = targets[j].displayName;
					// set up bradocast comms interface
					var bcc = worldScripts.BroadcastCommsMFD;
					if (bcc.$checkMessageExists("gcm_sm_transmit_seccode_" + m_num) === false) {
						bcc.$createMessage({
							messageName: "gcm_sm_transmit_seccode_" + m_num,
							callbackFunction: worldScripts.GalCopBB_DataCache.$transmitSecurityCode.bind(worldScripts.GalCopBB_DataCache),
							displayText: expandDescription("[gcm_transmit_securitycode]"),
							messageText: expandDescription("[gcm_transmitting_securitycode]"),
							ship: targets[j],
							transmissionType: "target",
							deleteOnTransmit: true,
							delayCallback: 5,
							hideOnConditionRed: true
						});
					}
				}
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.equipmentDamaged = function (equipmentKey) {
	if (equipmentKey === "EQ_GCM_UNPROCESSED_SCAN_DATA") {
		var list = worldScripts.GalCopBB_Missions.$getListOfMissions(true, 74);
		for (var i = 0; i < list.length; i++) {
			if (list[i].data.destroyedQuantity === 0) {
				list[i].data.destroyedQuantity = 1;
				list[i].payment /= 2;
				break;
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$custom_shipWillDockWithStation = function (station) {
	// do we have any 73 missions that will get their scanner here?
	var gcm = worldScripts.GalCopBB_Missions;
	if (gcm._simulator === true) return;
	var bb = worldScripts.BulletinBoardSystem;
	var list = gcm.$getListOfMissions(true, [73, 74]);
	for (var i = 0; i < list.length; i++) {
		if (list[i].data.missionType === 73 &&
			list[i].destination === system.ID &&
			station.isMainStation === true &&
			list[i].data.stage === 0) {

			// make sure there's enough room for the scanner
			if (player.ship.cargoSpaceAvailable === 0) gcm.$freeCargoSpace(1, "")

			// load up the scanner
			player.ship.awardEquipment("EQ_GCM_SOLAR_SCANNER");
			// add the arrival report text
			player.addMessageToArrivalReport(expandDescription("[missionType73_arrivalReport_pickup]"));
			// update the mission stage
			list[i].data.stage += 1;
			list[i].data.quantity += 1;
			list[i].destination = list[i].source;

			// force the chart marker to update
			bb.$revertChartMarker(list[i].ID)
			item.destination = list[i].source;
			bb.$updateBBMissionPercentage(list[i].ID, list[i].data.quantity / list[i].data.targetQuantity);

			gcm.$logMissionData(list[i].ID);
			player.consoleMessage(expandDescription("[goal_updated]"));
		}
		if (list[i].data.missionType === 74) {
			if (station.isMainStation && list[i].destination === system.ID) {
				if (list[i].expiry >= clock.adjustedSeconds) {
					if (player.ship.equipmentStatus("EQ_GCM_UNPROCESSED_SCAN_DATA") === "EQUIPMENT_OK") {
						list[i].arrivalReportText = expandDescription("[missionType74_arrivalReport_complete]", {
							payment: formatCredits(list[i].payment, false, true)
						});
					} else {
						list[i].arrivalReportText = expandDescription("[missionType74_arrivalReport_corrupt]", {
							payment: formatCredits(list[i].payment, false, true)
						});
					}
				} else {
					list[i].arrivalReportText = expandDescription("[missionType74_arrivalReport_late]");
				}

				list[i].data.quantity = 1;
				bb.$updateBBMissionPercentage(list[i].ID, 1);

				gcm.$logMissionData(list[i].ID);
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerEnteredNewGalaxy = function (galaxyNumber) {
	this._solarActivitySystems = this.$findSolarActivitySystems();
	this._galJump = true;
}

//-------------------------------------------------------------------------------------------------------------
this.shipDied = function (whom, why) {
	this.$stopTimer();
}

//-------------------------------------------------------------------------------------------------------------
this.mode = function () {

}

//-------------------------------------------------------------------------------------------------------------
this.activated = function () {
	worldScripts.GalCopBB_SolarActivity.$startScanner();
}

//-------------------------------------------------------------------------------------------------------------
this.$startScanner = function $startScanner() {
	if (this._scanTimer && this._scanTimer.isRunning) {
		player.consoleMessage(expandDescription("[gcm_solar_scan_stopped]"));
		worldScripts.GalCopBB_Missions.$playSound("stop");
		this.$stopTimer();
		return;
	}
	player.consoleMessage(expandDescription("[gcm_solar_scan_started]"));
	this._scanTimer = new Timer(this, this.$performScan, 1, 1);
	if (this._energy && this._energy.isRunning) this._energy.stop();
	this._energy = new Timer(this, this.$depleteEnergy, 0.25, 0.25);
	worldScripts.GalCopBB_Missions.$playSound("activate");
}

//-------------------------------------------------------------------------------------------------------------
this.$performScan = function $performScan() {
	var p = player.ship;
	if (p.equipmentStatus("EQ_GCM_SOLAR_SCANNER") === "EQUIPMENT_DAMAGED") {
		player.consoleMessage(expandDescription("[gcm_scanner_damaged]"));
		this.$stopTimer();
		return;
	}
	if (this.$isPlayerWithinRange() === false) {
		this._outOfRangeCount += 1;
		if (this._outOfRangeCount === 10) this._outOfRangeCount = 0;
		if (this._outOfRangeCount === 1) player.consoleMessage(expandDescription("[gcm_sun_out_of_range]"));
		return;
	}
	this._outOfRangeCount = 0;
	this._scanPosition += 1;
	if (parseInt(this._scanPosition / 5) === this._scanPosition / 5) {
		player.consoleMessage(expandDescription("[gcm_solar_scan_progress]", { complete: parseInt((this._scanPosition / 60) * 100) }));
	}
	// check for complete
	if (this._scanPosition >= 60) {
		this._scanPosition = 0;
		this.$stopTimer();
		var gcm = worldScripts.GalCopBB_Missions;
		var bb = worldScripts.BulletinBoardSystem;
		var list = gcm.$getListOfMissions(true, [70, 73]);
		for (var i = 0; i < list.length; i++) {
			if (system.ID === list[i].source &&
				list[i].expiry > clock.adjustedSeconds) {

				list[i].data.quantity += 1;
				bb.$updateBBMissionPercentage(list[i].ID, list[i].data.quantity / list[i].data.targetQuantity);

				gcm.$logMissionData(list[i].ID);
				player.consoleMessage(expandDescription("[goal_updated]"));
				break;
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$depleteEnergy = function $depleteEnergy() {
	var p = player.ship;
	p.energy -= this._energyReduction;
	if (p.energy < 64) {
		this.$stopTimer()
		player.consoleMessage(expandDescription("[gcm_insufficient_energy]"));
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns true if player is within fuel scoop range, otherwise false
this.$isPlayerWithinRange = function $isPlayerWithinRange() {
	return (((system.sun.collisionRadius * system.sun.collisionRadius) / system.sun.position.squaredDistanceTo(player.ship.position)) > 0.75);
}

//-------------------------------------------------------------------------------------------------------------
this.$stopTimer = function $stopTimer() {
	if (this._scanTimer && this._scanTimer.isRunning) {
		this._scanTimer.stop();
	}
	delete this._scanTimer;
	if (this._energy && this._energy.isRunning) {
		this._energy.stop();
	}
	delete this._energy;
}

//-------------------------------------------------------------------------------------------------------------
this.$beaconOfHopeReward = function $beaconOfHopeReward(entity, sysID, rep) {
	var reputation = worldScripts.GalCopBB_Reputation;
	if (rep >= 4 && rep < 6) {
		if (reputation.$checkForAward(entity, expandDescription("[gcm_beacon_of_support]"), 0) === false) {
			reputation.$addAward({
				title: expandDescription("[gcm_beacon_of_support]"),
				description: expandDescription("[gcm_beacon_support_description]"),
				entity: entity,
				system: 0,
				imageType: 4
			});
		}
	}
	if (rep >= 6 && rep < 7) {
		if (reputation.$checkForAward(entity, expandDescription("[gcm_beacon_of_courage]"), 0) === false) {
			reputation.$addAward({
				title: expandDescription("[gcm_beacon_of_courage]"),
				description: expandDescription("[gcm_beacon_courage_description]"),
				entity: entity,
				system: 0,
				imageType: 5
			});
		}
	}
	if (rep >= 7) {
		if (reputation.$checkForAward(entity, expandDescription("[gcm_reputation_boh]"), 0) === false) {
			reputation.$addAward({
				title: expandDescription("[gcm_reputation_boh]"),
				description: expandDescription("[gcm_beacon_hope_description]"),
				entity: entity,
				system: 0,
				imageType: 6
			});
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns a list of sysInfos where solar activity is present
this.$findSolarActivitySystems = function $findSolarActivitySystems() {
	return SystemInfo.filteredSystems(this, function (other) {
		return (other.description.indexOf(expandDescription("[gcm_desc_solar_activity]")) >= 0);
	});
}

//-------------------------------------------------------------------------------------------------------------
this.$acceptedMission = function $acceptedMission(missID) {
	var bb = worldScripts.BulletinBoardSystem;
	var item = bb.$getItem(missID);
	var gcm = worldScripts.GalCopBB_Missions;

	if (this._debug) log(this.name, "accepted mission id = " + missID);

	if (!item) {
		log(this.name, "!!ERROR: BB returned null value from $getItem on mission ID " + missID);
		return;
	}
	gcm.$updateLastMissionDate(item.source, item.data.missionType);
	gcm.$updateGeneralSettings(item);

	// give the player the solar scanner
	if (item.data.missionType === 70) {
		player.ship.awardEquipment("EQ_GCM_SOLAR_SCANNER");
	}
	// mission 72 is incomplete at the moment...
	if (item.data.missionType === 72) {
		// award the 2 missiles
		// if existing missiles need to be replaced, replace the cheapest ones first
		var flag = false;
		try {
			var miss = player.ship.missiles;
			flag = true;
		} catch (err) {
			if (this._debug) log(this.name, "!!ERROR: " + err);
		}
		// we're going to hold onto the missiles, though, and give them back when the mission is completed
		if (flag === true && (miss.length + 2) > player.ship.missileCapacity) {
			// get a list of installed missiles, sorted by price
			var checklist = [];
			for (var j = 0; j < miss.length; j++) {
				var misseq = EquipmentInfo.infoForKey(miss[j].equipmentKey);
				checklist.push({
					eq: miss[j].equipmentKey,
					price: misseq.price
				});
			}
			checklist.sort(compare);

			// work out what the player has got
			var mincost = 200000; // set a high value here so we are guaranteed to come in underneath it
			var key1 = "";
			var key2 = "";
			for (var j = 0; j < checklist.length; j++) {
				// see if the cost of this missile is less than or equal to the last mincost value we stored
				// we're doing less than or equal to because there could be more than 1 missile at the same price and we want both points loaded
				if (checklist[j].price <= mincost) {
					if (key1 != "") {
						key2 = key1;
					}
					mincost = checklist[j].price;
					key1 = checklist[j].eq;
				}
			}
			// only store the 2nd one if there are no slots left at all
			if (key2 != "" && (miss.length + 1) > player.ship.missileCapacity) {
				worldScripts.GalCopBBMissions_SolarActivity._missEquipStore2 = key2;
				// remove first instance of this key
				player.ship.removeEquipment(key2);
			}
			if (key1 != "") {
				worldScripts.GalCopBBMissions_SolarActivity._missEquipStore1 = key1;
				// remove first instance of this key
				player.ship.removeEquipment(key1);
			}
		}
		// give the player the missiles now
		player.ship.awardEquipment("EQ_GCM_SOLAR_MISSILE");
		player.ship.awardEquipment("EQ_GCM_SOLAR_MISSILE");
	}

	if (item.data.missionType === 74) {
		player.ship.awardEquipment("EQ_GCM_UNPROCESSED_SCAN_DATA");
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$completedMission = function $completedMission(missID) {
	var p = player.ship;
	var bb = worldScripts.BulletinBoardSystem;
	var item = bb.$getItem(missID);
	var gcm = worldScripts.GalCopBB_Missions;
	var sbm = worldScripts.Smugglers_BlackMarket;

	// update mission history
	gcm.$updateSuccessHistoryReputation(item);

	// *** type 70 - solar scanner
	if (item.data.missionType === 70 || item.data.missionType === 73) {
		p.removeEquipment("EQ_GCM_SOLAR_SCANNER");
		if (sbm) sbm.$removeSaleItem("EQ_GCM_SOLAR_SCANNER:" + missID);
	}
	// *** type 72 - give back the player's original missiles, if possible (they might have bought new ones)
	if (item.data.missionType === 72) {
		var sa = worldScripts.GalCopBB_SolarActivity;
		if (sa._missEquipStore1 != "") {
			p.awardEquipment(sa._missEquipStore1);
			sa._missEquipStore1 = "";
		}
		if (sa._missEquipStore2 != "") {
			p.awardEquipment(sa._missEquipStore2);
			sa._missEquipStore2 = "";
		}
		// make sure any remaining missiles are removed
		for (var i = 0; i < p.missileCapacity; i++) {
			p.removeEquipment("EQ_GCM_SOLAR_MISSILE");
		}
	}
	// *** type 74 - solar scan data
	if (item.data.missionType === 74) {
		p.removeEquipment("EQ_GCM_SOLAR_SCAN_DATA");
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$confirmCompleted = function $confirmCompleted(missID) {
	var p = player.ship;
	var result = "";
	var bb = worldScripts.BulletinBoardSystem;
	var item = bb.$getItem(missID);
	if (item) {
		if (item.data.missionType === 74) {
			var chk = p.equipmentStatus("EQ_GCM_UNPROCESSED_SCAN_DATA")
			if (chk !== "EQUIPMENT_OK" && chk !== "EQUIPMENT_DAMAGED") result += (result === "" ? "" : "\n") + expandDescription("[gcm_solar_scan_not_found]");
		}
	}
	return result;
}

//-------------------------------------------------------------------------------------------------------------
this.$terminateMission = function $terminateMission(missID) {
	var bb = worldScripts.BulletinBoardSystem;
	var item = bb.$getItem(missID);
	var sbm = worldScripts.Smugglers_BlackMarket;
	var gcm = worldScripts.GalCopBB_Missions;

	// adjust reputation only when the terminatePenalty flag is set to true 
	if (item.data.terminatePenalty === true) {
		// update mission history
		gcm.$updateFailedHistoryReputation(item);
	}

	// *** type 70/73 - solar scanner
	if (item.data.missionType === 70 || item.data.missionType === 73) {
		var eqsts = player.ship.equipmentStatus("EQ_GCM_SOLAR_SCANNER");
		if (eqsts === "EQUIPMENT_OK" || eqsts === "EQUIPMENT_DAMAGED") {
			player.ship.removeEquipment("EQ_GCM_SOLAR_SCANNER");
			if (sbm) sbm.$removeSaleItem("EQ_GCM_SOLAR_SCANNER:" + missID);
		}
	}
	if (item.data.missionType === 74) {
		player.ship.removeEquipment("EQ_GCM_UNPROCESSED_SCAN_DATA");
	}

}

//-------------------------------------------------------------------------------------------------------------
this.$failedMission = function $failedMission(missID) {
	var bb = worldScripts.BulletinBoardSystem;
	var sbm = worldScripts.Smugglers_BlackMarket;
	var gcm = worldScripts.GalCopBB_Missions;
	var eq = "";

	if (!sbm) {
		// if there's no blackmarket option, just remove the equipment
		this.$terminateMission(missID);
		return;
	}

	var item = bb.$getItem(missID);
	// update mission history
	gcm.$updateFailedHistoryReputation(item);

	// *** type 70/73 - solar scanner
	if (item.data.missionType === 70 || item.data.missionType === 73) {
		var eqsts = player.ship.equipmentStatus("EQ_GCM_SOLAR_SCANNER");
		if (eqsts === "EQUIPMENT_OK" || eqsts === "EQUIPMENT_DAMAGED") {
			eq = "EQ_GCM_SOLAR_SCANNER";
		}
	}

	if (eq != "") gcm._equipmentFromFailedMissions.push({
		missionType: item.data.missionType,
		source: item.source,
		equip: eq,
		date: clock.adjustedSeconds
	});
}

//-------------------------------------------------------------------------------------------------------------
this.$updateManifestEntry = function $updateManifestEntry(missID) {
	var gcm = worldScripts.GalCopBB_Missions;
	gcm.$updateManifestEntry(missID);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionAvailability = function $missionAvailability(missID, missType, origSysID) {
	return worldScripts.GalCopBB_Missions.$missionAvailability(missID, missType, origSysID);
}

//-------------------------------------------------------------------------------------------------------------
// 70/72 
this.$missionType7072_Values = function $missionType7072_Values(workTime, routeTime, routeDistance, destSysInfo) {
	var missValues = worldScripts.GalCopBB_CoreMissionValues;
	var result = {};
	result["quantity"] = 1;
	result["price"] = 100 + parseInt((8 - destSysInfo.government) * (Math.random() * 50));
	result["expiry"] = clock.adjustedSeconds + (3600 * 3); // 3 hours to complete
	result["penalty"] = result.price / 2;
	return result;
}

//-------------------------------------------------------------------------------------------------------------
// 71 
this.$missionType71_Values = function $missionType71_Values(workTime, routeTime, routeDistance, destSysInfo) {
	var missValues = worldScripts.GalCopBB_CoreMissionValues;
	var result = {};
	result["quantity"] = 2;
	result["price"] = 100 + parseInt((8 - destSysInfo.government) * (Math.random() * 50));
	result["expiry"] = clock.adjustedSeconds + (3600 * 3); // 3 hours to complete
	result["penalty"] = result.price / 2;
	return result;
}

//-------------------------------------------------------------------------------------------------------------
// 73
this.$missionType73_Values = function $missionType73_Values(workTime, routeTime, routeDistance, destSysInfo) {
	var missValues = worldScripts.GalCopBB_CoreMissionValues;
	var result = {};
	result["quantity"] = 2;
	result["price"] = 500 + parseInt((8 - destSysInfo.government) * (Math.random() * 90));
	result["expiry"] = clock.adjustedSeconds + routeTime + workTime;
	result["penalty"] = result.price / 2;
	return result;
}

//-------------------------------------------------------------------------------------------------------------
// 74
this.$missionType74_Values = function $missionType74_Values(workTime, routeTime, routeDistance, destSysInfo) {
	var missValues = worldScripts.GalCopBB_CoreMissionValues;
	var result = {};
	result["quantity"] = 1;
	result["price"] = 500 + parseInt((8 - destSysInfo.government) * (Math.random() * 90)) + missValues.$calcDistanceBonus(routeDistance, 10);
	result["expiry"] = clock.adjustedSeconds + routeTime + workTime;
	result["penalty"] = result.price / 2;
	return result;
}