"use strict";
this.name = "AsteroidHunter";
this.author = "phkb";
this.copyright = "2017 phkb";
this.description = "Adds a simple asteroid hunt mission to the Bulletin Board";
this.licence = "CC BY-NC-SA 3.0";

this._bb = null; // link to Bulletin Board worldScript

//-------------------------------------------------------------------------------------------------------------
this.startUp = function () {
	// establish a link to the bulletin board worldscript
	this._bb = worldScripts.BulletinBoardSystem;
}

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {
	this.$addLocalMissions()
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillExitWitchspace = function () {
	// adds new missions to the list
	this.$addLocalMissions();
}

//-------------------------------------------------------------------------------------------------------------
this.shipKilledOther = function (whom, damageType) {
	// mission type 1 - asteroid hunt - monitor when the player shoots an asteroid
	if (whom.hasRole("asteroid")) {
		var item = null;
		// cycle through the bulletin board list
		if (this._bb._data.length > 0) {
			for (var i = 0; i < this._bb._data.length; i++) {
				item = this._bb._data[i];
				// is this an accepted mission, and the target destination is the same as the current system...
				if (item.accepted === true && item.destination === system.ID) {
					// ..and there is some data on the item, and the data has a property of "type" and the type is 1 (asteroid hunt) 
					// and the number of destroyed asteroids is less than the target quantity
					if (item.hasOwnProperty("data") && item.data.hasOwnProperty("type") && item.data["type"] == 1 && item.data.quantity < item.data.targetQuantity) {
						// ...then update the quantity
						item.data.quantity += 1;
						// signal the bulletin board that we've had a change in the mission completion percentage
						this._bb.$updateBBMissionPercentage(item.missionID, (item.data.quantity / item.data.targetQuantity));

						// notify the player that we just did something that affected a mission
						player.consoleMessage("Mission goal updated");
					}
				}
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// handles the manifest entry callback from the bulletin board
this.$updateManifestEntry = function $updateManifestEntry(missID) {
	var item = this._bb.$getItem(missID);
	if (item.data.quantity == item.data.targetQuantity) {
		this._bb.$updateBBManifestText(missID,
			expandDescription("Mission to destroy asteroids complete. Return to the main station in [system] to claim payment.", {
				system: System.systemNameForID(item.source)
			})
		);
	} else {
		// otherwise, just update the description with what is currently expected
		this._bb.$updateBBManifestText(
			missID,
			expandDescription("Destroy [quantity] asteroids in [destination] within [expiry].", {
				quantity: (item.data.targetQuantity - item.data.quantity),
				destination: system.name,
				expiry: this._bb.$getTimeRemaining(item.expiry)
			})
		);
	}
}

//-------------------------------------------------------------------------------------------------------------
// adds the asteroid hunt mission to the bulletin board
this.$addLocalMissions = function $addLocalMissions() {
	// if we haven't got any missions in the available list, try to add one now
	// first, count up the number already in player
	var chk = 0;
	if (this._bb._data.length > 0) {
		for (var i = 0; i < this._bb._data.length; i++) {
			if (this._bb._data[i].destination === system.ID &&
				this._bb._data[i].hasOwnProperty("data") &&
				this._bb._data[i].data.hasOwnProperty("type") &&
				this._bb._data[i].data["type"] == 1)
				chk += 1;
		}
	}
	// if there's one there already, exit now
	if (chk > 0) return;

	// pick a random number of asteroid that need to be destroyed
	var qty = parseInt((Math.random() * 10) + 10);
	// calculate the payment
	var amt = qty * 10;
	// calculate the expiry time
	var expire = clock.adjustedSeconds + 21600; // 6 hours to complete.
	// add the mission to the bulletin board, and make a note of the id
	var id = this._bb.$addBBMission({
		source: system.ID,
		destination: system.ID,
		stationKey: "galcop",
		description: "Clean up the spaceways",
		details: expandDescription("Commanders are needed to help clean up the spacelanes by removing [quantity] asteroids in this system.", {
			quantity: qty
		}),
		payment: amt,
		expiry: expire,
		penalty: 0,
		model: "asteroid",
		modelPersonality: parseInt(Math.random() * 32767),
		manifestCallback: "$updateManifestEntry",
		worldScript: this.name,
		data: {
			type: 1,
			targetQuantity: qty,
			quantity: 0
		}
	});
}