/*

anarchies-sentinel-station-script.js

Script for turning Sentinel and Renegade Stations towards the planet.


Oolite
Copyright © 2004-2012 Giles C Williams and contributors

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
MA 02110-1301, USA.


Anarchies.oxp
Copyright © 2007-2012 "Commander McLane"

This work is licensed under the Creative Commons
Attribution-Noncommercial-Share Alike 3.0 Unported License.

To view a copy of this license, visit
http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter
to Creative Commons, 171 Second Street, Suite 300, San Francisco,
California, 94105, USA.

*/

"use strict";

this.name           = "anarchies-sentinel-station-script";
this.description    = "Ship script for Sentinel and Renegade Stations";
this.author         = "Commander McLane";
this.copyright      = "© 2012 Commander McLane";
this.license		= "CC-by-nc-sa 3.0";
this.version        = "2.9";


this.shipSpawned = function()
{
    // station is aligned along the z-axis and therefore towards the planet
    this.ship.orientation = [1, 0, 0, 0];
    // two variables for controlling the station's launch queue are initialised
    this.queuingDefenseShips = 0;
    this.queuingPatrolShips = 0;
}

this.stationLaunchedShip = function(ship)
{
    // if one of the defense ships is launched it gets subtracted from the queue counter
    if(ship.primaryRole === "defense_ship") this.queuingDefenseShips--;
    // same for the patrol ships
    if(ship.AI === "anarchiesSentinelStationPatrolAI.plist") this.queuingPatrolShips--;
    // if a trader has launched it gets a chance to head for the planet instead of jumping out
    if(ship.primaryRole === "trader" && Math.random() < 0.4) ship.switchAI("route1traderAI.plist");
}

this.shipTargetAcquired = function(target)
{
    // the station would never get a TARGET_LOST when its current target leaves scanner range
    // therefore we're tracing the target by a script timer in 2-second intervals once it's acquired
    // but not if it's not a piloted ship, or it already has a trace on it
    if(!target.isPiloted || (target === this.currentTarget && this.traceTargetTimer.isRunning)) return;
    // the current target is stored
    this.currentTarget = target;
    // if the timer already exists it is started
    if(this.traceTargetTimer)
    {
        this.traceTargetTimer.start();
    }
    // otherwise it is created
    else
    {
        this.traceTargetTimer = new Timer(this, this.$checkForTargetDistance, 2, 2);
    }
}

this.shipTargetLost = function(target)
{
    // after losing the target the timer is stopped
    this.traceTargetTimer.stop();
}

this.$checkForTargetDistance = function()
{
    // if the station's current target gets out of scanner range and then some
    // the station loses it
    if(this.ship.target && this.ship.target.position && this.ship.position.distanceTo(this.ship.target.position) > 28000)
    {
        this.ship.target = null;
    }
}

this.$launchDefenseShip = function()
{
    // custom defense ship launching, because we don't want all defense ships launched every time
    // five at a time is enough
    function $isMyDefenseShip(entity)
    {
        return entity.isShip && entity.primaryRole === "defense_ship" && entity.owner === this.ship
    }
    // we count the already launched defense ships and add those already in the launch queue
    // if less than five, another one gets ready for launch
    if(system.filteredEntities(this, $isMyDefenseShip, this.ship, 25600).length + this.queuingDefenseShips < 5)
    {
        // however, the actual launch may take a while
        this.ship.launchDefenseShip();
        // therefore we're adding the new defense ship to the queue counter
        this.queuingDefenseShips++;
    }
}

this.$launchPatrolShip = function()
{
    // custom patrol ship launching, we only want four at a time
    function $isMyPatrolShip(entity)
    {
        return entity.isShip && entity.primaryRole === "police" && entity.group.leader === this.ship
    }
    // we count the already launched patrol ships and add those already in the launch queue
    // if less than four, another one gets ready for launch
    if(system.filteredEntities(this, $isMyPatrolShip, this.ship, 35000).length + this.queuingPatrolShips < 4)
    {
        // first we raise the counter
        this.queuingPatrolShips++;
        // the ship gets ready for launch
        var patrolShip = this.ship.launchShipWithRole("police");
        // it is added to the station's group
        this.ship.group.addShip(patrolShip);
        patrolShip.group = this.ship.group;
        // and it gets a custom script and AI
        patrolShip.setScript("anarchies-sentinel-station-patrol-script.js");
        patrolShip.switchAI("anarchiesSentinelStationPatrolAI.plist");
    }
}

this.$launchScavenger = function()
{
    // we don't want too many scavengers at a time
    if(system.countShipsWithPrimaryRole("scavenger", this.ship, 25600) === 0) this.ship.launchScavenger();
}

this.$scanForRandomOffender = function()
{
    // as Sentinel Stations are only spawned in anarchy systems, the normal scan for offenders is useless, because it doesn't find any ship in an anarchy
    // therefore we do the scan by script
    function $isOffender(entity)
    {
        return entity.isShip && entity.scanClass !== "CLASS_ROCK" && entity.scanClass !== "CLASS_CARGO" && !entity.isCloaked && entity.bounty - randomBlindness > 0
    }
    // like the normal AI method, our scan isn't perfect; there is a random chance to evade it as long as you're offender
    var randomBlindness = Math.floor(Math.random() * 50);
    // all found ships are put in an array
    var offenders = system.filteredEntities(this, $isOffender, this.ship, 25600);
    // if it's empty, no offenders were found
    if(offenders.length === 0)
    {
        this.ship.sendAIMessage("NOTHING_FOUND");
    }
    // otherwise one of the found ships gets randomly chosen and targeted
    else
    {
        var targetedOffender = offenders[Math.floor(Math.random() * offenders.length)];
        this.ship.target = targetedOffender;
        this.ship.sendAIMessage("TARGET_FOUND");
    }
}

this.playerWillEnterWitchspace = this.shipDied = function()
{
    // the tracing timer is stopped and deleted when the player leaves the system
    if(this.traceTargetTimer)
    {
        this.traceTargetTimer.stop();
        delete this.traceTargetTimer;
    }
}
