"use strict";
this.name        = "EmailSystem";
this.author      = "phkb";
this.copyright   = "2017 phkb";
this.description = "Implements a simple email system via the interfaces screen.";
this.licence     = "CC BY-NC-SA 4.0";

// specific settings for this oxp
this._maxpage = 0;              		// total number of pages of inbox to display
this._curpage = 1;              		// the current page of the inbox being displayed
this._msRows = 18; //15					// rows to display on the mission screen
this._msCols = 32;						// columns to display on the mission screen
this._itemsOnPage = 0;					// number of emails being displayed on a page
this._displayType = 0;					// controls the view. 0 = inbox, 1 = email, 2 = trace
this._emailPage = 0;					// which page of the email are we viewing.
this._emailMaxPages = 0;				// total number of pages in the email we are viewing
this._emails = [];						// main array of emails
this._emailID = 0;						// id of the email we are viewing
this._disableEmailNotification = false;	// disables unread email notification when docking
this._notifyMainStationOnly = false; 	// unread email notification at main stations only
this._archiveLimit = 100;				// number of emails to keep before they start getting deleted automatically
this._updateRequired = true;			// flag to indicate the interface screen needs updating because a new email was added
this._newMailAlert = null				// timer for future dated new emails arriving
this._pageItems = [];
this._lastChoice = ["", "", ""];
this._itemColor = "yellowColor";
this._menuColor = "orangeColor";
this._exitColor = "yellowColor";
this._disabledColor = "darkGrayColor";
this._readColumn = 0.4;

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function() {
	// add the welcome email here. If this is the first time we've used the system, there won't be any data to load from mission variables
	// so it will not be overwritten by restoring saved values
	// once there are saved emails to restore, this email will disappear
	this.$createEmail({sender:"Xenon Industries",
		subject:"Welcome to your inbox!",
		date:clock.seconds - 100,
		message:"Xenon Industries is proud to welcome you to your new inbox. We hope you enjoy the facilities this system provides.\n\n"
			+ "Xenon Industries have been working behind the scenes for many years providing message facilities to spacers from "
			+ "all walks of life. However, it was only after GalCop regulation 3827B (Sub-section A) was passed that we "
			+ "have been allowed to provide a fully functional email system to any space going vessel. Even with this "
			+ "regulation, email facilities are only operational while dockside, as GalCop regulations make it clear than "
			+ "emailing while flying is a serious offence.\n\nDespite this, we know you will love all the benefits a good email "
			+ "client can provide, and we believe we have all of this and more.\n\n"
			+ "Thank you for using Xenon Industries's email system, and we trust it will perform flawlessly for you for years to come."}, false);

	if (defaultFont.measureString("!") > 0.4) this._readColumn = defaultFont.measureString("!") + 0.1;

	// restore saved data if any exists
	if (missionVariables.EmailSystem_Emails) {
        this._emails = JSON.parse(missionVariables.EmailSystem_Emails);
        delete missionVariables.EmailSystem_Emails;
    }

	this.$checkForFutureDatedEmails();

	var p = player.ship;
	if(p.docked) this.$initInterface(p.dockedStation);

	if (oolite.compareVersion("1.91") <= 0) {
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_EQUIP_SHIP", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_SHIPYARD", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_INTERFACES", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_STATUS", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_MANIFEST", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_SHORT_RANGE_CHART", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_LONG_RANGE_CHART", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_SYSTEM_DATA", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_MARKET", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});
		setExtraGuiScreenKeys(this.name, {
			guiScreen:"GUI_SCREEN_MARKETINFO", 
			registerKeys:{"key1":[{key:"e",mod1:true}]}, 
			callback:this.$showInbox.bind(this)
		});

		if (worldScripts.ContextualHelp) {
			var ch = worldScripts.ContextualHelp;
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_EQUIP_SHIP", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_SHIPYARD", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_INTERFACES", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_MANIFEST", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_STATUS", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_LONG_RANGE_CHART", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_SHORT_RANGE_CHART", "\nPress Ctrl-G on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_SYSTEM_DATA", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_MARKET", "\nPress Ctrl-E on this screen to open your email.");
			ch.$addHelpTextToGuiScreen(this.name, "GUI_SCREEN_MARKETINFO", "\nPress Ctrl-E on this screen to open your email.");
		}

	}
}

//=============================================================================================================
//public interfaces
//-------------------------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------------------
// Create a new email
// REQUIRED
// emailObj.sender					(text) Name of person sending email
// emailObj.subject					(text) Subject line of email
// emailObj.date					time in seconds
// OPTIONAL
// emailObj.message					(text) Body text of email
// emailObj.sentFrom				(int) ID of Planet that is the source of the email. Defaults to the current planet.
//										If the optional stopTrace flag is set, this ID will be the last planet in the trace.
// emailObj.isRead					(boolean) setting this to true will add the email to the inbox as if it's been read by the player.
//										this might be useful if you have displayed the message to the player using "addMessageToArrivalReport",
//										and want to add a corresponding email, which, because the player has seen it, should be flagged as read. Default false
// emailObj.stopTrace				(boolean) Indicates that the routing ticket is corrupt and a trace is only partial. The trace will terminate at the "sentFrom" planet. Default false.
// emailObj.traceRoute				(csv text) Allows the trace route information to be fully specified. If not specified, the route will be all planets between sentFrom and the current planet
//										using the fastest route.
// emailObj.expiryDate		    	time in seconds when email will expire
// emailObj.expiryDays				Number of days added to current date
// emailObj.expiryHours				Number of hours added to current date
// emailObj.expiryMinutes			Number of minutes added to current date
// emailObj.expirySeconds			Number of seconds added to current date
// emailObj.expiryText				Text to display in header when email expires
// emailObj.allowExpiryCancel		(boolean) indicates whether the expiry date of the email can be cancelled. (default true)
// emailObj.expiryOptions			(csv text) csv list of which options to display after expiry (possible values in list are 1,2,3 or 4) eg "3,4"
// emailObj.forceResponse			(boolean) Indicates that the player must select a response before the email can be closed. Default false
// emailObj.option1					(ResponseOption) Response option 1
// emailObj.option2					(ResponseOption) Response option 2
// emailObj.option3					(ResponseOption) Response option 3
// emailObj.option4					(ResponseOption) Response option 4
//
// Response options are an object:
// response.display					(text) Text to display on email. Will be slotted into "Send 'xxx' response"
// response.reply					(text) Text to be appended to the email as a reply.
// response.script					(text) the name of the worldScript where the callback function is located
// response.callback				(text) the name of the function to call when the reply is selected
// response.parameter				(object) object to pass to the function when it is called
this.$createEmail = function(emailObj) {
	// using parameters
	if (emailObj.hasOwnProperty("sender") === false || emailObj.sender === "") {
		throw "Invalid settings: sender must be supplied.";
	}
	if (emailObj.hasOwnProperty("subject") === false || emailObj.subject === "") {
		throw "Invalid settings: subject must be supplied.";
	}
	if (emailObj.hasOwnProperty("date") === false || emailObj.date === 0) {
		throw "Invalid settings: date must be supplied.";
	}
	if (emailObj.hasOwnProperty("sentFrom") === true) {
		if (emailObj.sentFrom < 0 || emailObj.sentFrom > 255) {
			throw "Invalid settings: sentFrom (planet ID) must be supplied and be in range 0-255.";
		}
	}

	var id = this.$nextID();

	var ebody = "";
	if (emailObj.hasOwnProperty("message") === true) ebody = emailObj.message;

	var planet = system.ID;
	if (emailObj.hasOwnProperty("sentFrom") === true) planet = emailObj.sentFrom;

	var stoptr = false;
	if (emailObj.hasOwnProperty("stopTrace") === true) stoptr = emailObj.stopTrace;

	var read = false;
	if (emailObj.hasOwnProperty("isRead") === true) read = emailObj.isRead;

	var trc = "";
	if (emailObj.hasOwnProperty("traceRoute") === true) {
		trc = emailObj.traceRoute;
	} else {
		trc = this.$populateTrace(planet);			// text: CSV list of planets in the trace
	}

	var forcersp = false;
	if (emailObj.hasOwnProperty("forceResponse") === true) forcersp = emailObj.forceResponse;

	var expDate = 0;
	if (emailObj.hasOwnProperty("expiryDate") === true) expDate = emailObj.expiryDate;
	if (emailObj.hasOwnProperty("expiryDays") === true || emailObj.hasOwnProperty("expiryHours") === true || emailObj.hasOwnProperty("expiryMinutes") === true || emailObj.hasOwnProperty("expirySeconds") === true) {
		var addsec = 0;
		// convert all values to seconds
		if (emailObj.hasOwnProperty("expiryDays") === true) addsec += emailObj.expiryDays * 24 * 60 * 60;
		if (emailObj.hasOwnProperty("expiryHours") === true) addsec += emailObj.expiryHours * 60 * 60;
		if (emailObj.hasOwnProperty("expiryMinutes") === true) addsec += emailObj.expiryMinutes * 60;
		if (emailObj.hasOwnProperty("expirySeconds") === true) addsec += emailObj.expirySeconds; // don't know why you'd want to, but just in case
		// add the expiry amount to the transmit date, so a future dated email with expiry will not expire before it's visible
		if (addsec > 0) expDate = emailObj.date + addsec;
	}

	var expText = "";
	if (emailObj.hasOwnProperty("expiryText") === true) expText = emailObj.expiryText;

	var expOptions = "";
	if (emailObj.hasOwnProperty("expiryOptions") === true) expOptions = emailObj.expiryOptions;
	if (expOptions != "" && expText === "") {
		throw "Invalid settings: expiryOptions have been defined but no expiryText has been set. Options will never be available to player.";
	}

	var expCancel = true;
	if (emailObj.hasOwnProperty("allowExpiryCancel") === true) expCancel = emailObj.allowExpiryCancel;

	var opt1 = {DisplayText:"", ReplyText:"", WorldScriptsName:"", CallbackFunction:"", FunctionParam:{}};
	if (emailObj.hasOwnProperty("option1") === true) {
		opt1.DisplayText = emailObj.option1.display;
		opt1.ReplyText = emailObj.option1.reply;
		opt1.WorldScriptsName = emailObj.option1.script;
		opt1.CallbackFunction = emailObj.option1.callback;
		if (emailObj.option1.hasOwnProperty("parameter") === true) opt1.FunctionParam = emailObj.option1.parameter;
	}
	var opt2 = {DisplayText:"", ReplyText:"", WorldScriptsName:"", CallbackFunction:"", FunctionParam:{}};
	if (emailObj.hasOwnProperty("option2") === true) {
		opt2.DisplayText = emailObj.option2.display;
		opt2.ReplyText = emailObj.option2.reply;
		opt2.WorldScriptsName = emailObj.option2.script;
		opt2.CallbackFunction = emailObj.option2.callback;
		if (emailObj.option2.hasOwnProperty("parameter") === true) opt2.FunctionParam = emailObj.option2.parameter;
	}
	var opt3 = {DisplayText:"", ReplyText:"", WorldScriptsName:"", CallbackFunction:"", FunctionParam:{}};
	if (emailObj.hasOwnProperty("option3") === true) {
		opt3.DisplayText = emailObj.option3.display;
		opt3.ReplyText = emailObj.option3.reply;
		opt3.WorldScriptsName = emailObj.option3.script;
		opt3.CallbackFunction = emailObj.option3.callback;
		if (emailObj.option3.hasOwnProperty("parameter") === true) opt3.FunctionParam = emailObj.option3.parameter;
	}
	var opt4 = {DisplayText:"", ReplyText:"", WorldScriptsName:"", CallbackFunction:"", FunctionParam:{}};
	if (emailObj.hasOwnProperty("option4") === true) {
		opt4.DisplayText = emailObj.option4.display;
		opt4.ReplyText = emailObj.option4.reply;
		opt4.WorldScriptsName = emailObj.option4.script;
		opt4.CallbackFunction = emailObj.option4.callback;
		if (emailObj.option4.hasOwnProperty("parameter") === true) opt4.FunctionParam = emailObj.option4.parameter;
	}

	if (expOptions != "" &&
		((expOptions.indexOf("1") >=0 && opt1.DisplayText === "") ||
		(expOptions.indexOf("2") >=0 && opt2.DisplayText === "") ||
		(expOptions.indexOf("3") >=0 && opt3.DisplayText === "") ||
		(expOptions.indexOf("4") >=0 && opt4.DisplayText === ""))) {
		throw "Invalid settings: expiryOptions have been defined but the response option has not been set.";
	}

	var msg = {ID:id,
			Sender:emailObj.sender,
			Subject:emailObj.subject,
			TransmitDate:emailObj.date,
			EmailBody:ebody,
			OriginatingPlanet:System.systemNameForID(planet),
			ReceivedPlanet:System.systemNameForID(system.ID),
			Read:read,
			CorruptTrace:stoptr,
			Trace:trc,
			ExpiryDate:expDate,
			ExpiryText:expText,
			ExpiryOptions:expOptions,
			AllowExpiryCancel:expCancel,
			ChosenOption:0,
			Marked:false,
			GalaxyNum:galaxyNumber,
			ForceResponse:forcersp,
			ResponseOption1:opt1,
			ResponseOption2:opt2,
			ResponseOption3:opt3,
			ResponseOption4:opt4};

	this._emails.push(msg);

	if (emailObj.date > clock.seconds && player.ship.docked === true) {
		// set a timer to ding when the email "appears" in the inbox - only while docked
		if (!this._newMailAlert || this._newMailAlert.isRunning === false) {
			this._newMailAlert = new Timer(this, this.$newMailArrived, (emailObj.date - clock.seconds), 0);
		}
	}

	// make sure we keep our array inside the bounds of the archive limit
	this.$deleteOldest();
	this._updateRequired = true;

	// return the msg id back to the caller, just in case they need it.
	return id;
}

//=============================================================================================================
// ship interfaces

//-------------------------------------------------------------------------------------------------------------
this.shipDockedWithStation = function(station) {

	if (this._disableEmailNotification === false && ((this._notifyMainStationOnly === true && station.isMainStation) || this._notifyMainStationOnly === false)) {
		var unread = this.$totalUnreadItemsRequiringResponse();
		if(unread != 0) {
			var addText = "emails";
			var respText = "require";
			if (unread === 1) { addText = "email"; respText = "requires"; }
			player.addMessageToArrivalReport("You have " + unread + " unread " + addText + " that " + respText + " a response.");
		}
	}

	this.$checkForFutureDatedEmails();
	this.$initInterface(station);
}

//-------------------------------------------------------------------------------------------------------------
this.shipLaunchedFromStation = function(station) {
	delete missionVariables.EmailSystem_Emails;
	if (this._newMailAlert && this._newMailAlert.isRunning) this._newMailAlert.stop();
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function() {
	// save email array
	missionVariables.EmailSystem_Emails = JSON.stringify(this._emails);
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged = function(to, from) {

	if (guiScreen === "GUI_SCREEN_INTERFACES" || this._updateRequired === true) {
		// update the interfaces screen
		this._updateRequired = false;
		var p = player.ship;
		if (p.dockedStation) this.$initInterface(p.dockedStation);
	}
}

//-------------------------------------------------------------------------------------------------------------
this.missionScreenEnded = function() {
	if (player.ship.hudHidden == true) player.ship.hudHidden = false;
}

//=============================================================================================================
// general functions

//-------------------------------------------------------------------------------------------------------------
// check for any future dated emails, and start a timer to play a ding noise
this.$checkForFutureDatedEmails = function() {
	// when will the next ding happen?
	if (player.ship.docked) {
		var next = 0;
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].TransmitDate > clock.seconds && (this._emails[i].TransmitDate - clock.seconds) > next) {
				next = (this._emails[i].TransmitDate - clock.seconds);
				// update the receipt planet and the trace of all future dated emails, so when they arrive
				// they have the correct values
				if (this._emails[i].GalaxyNum === galaxyNumber) {
					this._emails[i].ReceivedPlanet = System.systemNameForID(system.ID);
					this._emails[i].Trace = this.$populateTrace(System.systemIDForName(this._emails[i].OriginatingPlanet));
				}
			}
		}
		if (next > 0) {
			if (!this._newMailAlert || this._newMailAlert.isRunning === false)
				this._newMailAlert = new Timer(this, this.$newMailArrived, next, 0);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// play the new mail message noise
this.$newMailArrived = function $newMailArrived() {
	this._updateRequired = true;
	var mySound = new SoundSource;
	mySound.sound = "ding.ogg";
	mySound.loop = false;
	mySound.play();
	player.consoleMessage(expandDescription("[emailnewitem]"));
	// check for any more emails
	this.$checkForFutureDatedEmails();
}

//-------------------------------------------------------------------------------------------------------------
// returns the next ID number for new emails
this.$nextID = function() {
	var iMax = 0;
	if (this._emails != null && this._emails.length > 0) {
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].ID > iMax) iMax = this._emails[i].ID;
		}
	}
	return (iMax + 1);
}

//-------------------------------------------------------------------------------------------------------------
// this function should only be called when an email is created. Once created, it will be populated from the missionvariables store
this.$populateTrace = function(startPoint) {
	// get list of planets between start and end points
	var sysID = system.ID;
	if (system.ID === -1) sysID = player.ship.targetSystem;
	var myRoute = System.infoForSystem(galaxyNumber, sysID).routeToSystem(System.infoForSystem(galaxyNumber, startPoint), "OPTIMIZED_BY_TIME");
	// expand and reverse list for storage
	var ret = "";
	if (myRoute != null) {
		for (var i = 0; i < myRoute.route.length; i++) {
			if (i > 0) {
				ret += ",";
			}
			ret = ret + System.systemNameForID(myRoute.route[i]);
		}
	} else {
		ret = "<< Error: Trace unavailable >>";
	}
	return ret;
}

//-------------------------------------------------------------------------------------------------------------
// deletes oldest emails from array until the count is less than archive total
this.$deleteOldest = function() {
	function compare(a,b) {
		return a.TransmitDate - b.TransmitDate;
	}

	// look for and delete any expired emails first
	for (var i = this._emails.length - 1; i >= 0; i--) {
		if (this._emails[i].ExpiryDate != 0 && clock.seconds > this._emails[i].ExpiryDate && this._emails[i].ExpiryText === "") {
			this._emails.splice(i, 1);
		}
	}

	if (this._emails.length > this._archiveLimit) {
		// sort emails oldest to newest
		this._emails.sort(compare);
		var idx = 0;
		do {
			// only delete if there isn't a response required
			if (this.$requiresResponse(this._emails[idx]) === false) this._emails.splice(idx, 1);
			idx += 1;
			// if the player has 100 emails that require a response, we might still end up with an array greater than 100
			// so watch for that case here
			if (idx >= this._emails.length) break;

		} while (this._emails.length > this._archiveLimit);
	}

}

//-------------------------------------------------------------------------------------------------------------
// returns the email based on ID number
this.$getEmailByID = function(id) {
	for (var i = 0; i < this._emails.length; i++) {
		if (this._emails[i].ID === id) return this._emails[i];
	}
	return null;
}

//-------------------------------------------------------------------------------------------------------------
// returns the email based on ID number
this.$getEmailIndexByID = function(id) {
	for (var i = 0; i < this._emails.length; i++) {
		if (this._emails[i].ID === id) return i;
	}
	return -1;
}

//-------------------------------------------------------------------------------------------------------------
this.$findNextEmailID = function(id) {
	var ret = -1;
	for (var i = 0; i < this._emails.length; i++) {
		if (this._emails[i].ID === id && i < (this._emails.length - 1)) ret = this._emails[i + 1].ID;
	}
	return ret;
}

//-------------------------------------------------------------------------------------------------------------
// executes the callback function of the selected response
this.$executeCallback = function(id, optionNumber) {
	var email = this.$getEmailByID(id);

	switch (optionNumber) {
		case 1:
			if (email.ResponseOption1.CallbackFunction != "") {
				var w = worldScripts[email.ResponseOption1.WorldScriptsName];
				if (w) w[email.ResponseOption1.CallbackFunction](email.ResponseOption1.FunctionParam);

			}
			break;
		case 2:
			if (email.ResponseOption2.CallbackFunction != "") {
				var w = worldScripts[email.ResponseOption2.WorldScriptsName];
				if (w) w[email.ResponseOption2.CallbackFunction](email.ResponseOption2.FunctionParam);
			}
			break;
		case 3:
			if (email.ResponseOption3.CallbackFunction != null) {
				var w = worldScripts[email.ResponseOption3.WorldScriptsName];
				if (w) w[email.ResponseOption3.CallbackFunction](email.ResponseOption3.FunctionParam);
			}
			break;
		case 4:
			if (email.ResponseOption4.CallbackFunction != null) {
				var w = worldScripts[email.ResponseOption4.WorldScriptsName];
				if (w) w[email.ResponseOption4.CallbackFunction](email.ResponseOption4.FunctionParam);
			}
			break;
	}
}

//-------------------------------------------------------------------------------------------------------------
// counts the total number of emails viewable in the inbox
this.$totalItems = function() {
	var itms = 0;
	if (this._emails.length > 0) {
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].TransmitDate <= clock.seconds) itms += 1;
		}
	}
	return itms;
}

//-------------------------------------------------------------------------------------------------------------
// counts the total number of marked emails
this.$totalItemsMarked = function() {
	var itms = 0;
	if (this._emails.length > 0) {
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].Marked === true) itms += 1;
		}
	}
	return itms;
}

//-------------------------------------------------------------------------------------------------------------
// counts how many read items are in the inbox
this.$totalReadItems = function() {
	var itms = 0;
	if (this._emails != null && this._emails.length > 0) {
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].Read === true && this._emails[i].TransmitDate <= clock.seconds) itms += 1;
		}
	}
	return itms;
}

//-------------------------------------------------------------------------------------------------------------
// counts how many unread items are in the inbox
this.$totalUnreadItems = function() {
	var itms = 0;
	if (this._emails != null && this._emails.length > 0) {
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].Read === false && this._emails[i].TransmitDate <= clock.seconds) itms += 1;
		}
	}
	return itms;
}

//-------------------------------------------------------------------------------------------------------------
// counts how many unread items requiring a response are in the inbox
this.$totalUnreadItemsRequiringResponse = function() {
	var itms = 0;
	if (this._emails != null && this._emails.length > 0) {
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].Read === false && this._emails[i].TransmitDate <= clock.seconds && this._emails[i].ExpiryDate > clock.adjustedSeconds && this.$requiresResponse(this._emails[i]) === true) itms += 1;
		}
	}
	return itms;
}

//-------------------------------------------------------------------------------------------------------------
// marks all items read
this.$markAllItemsRead = function() {
	if (this._emails != null && this._emails.length > 0) {
		for (var i = 0; i < this._emails.length; i++) {
			if (this._emails[i].Read === false && this._emails[i].TransmitDate <= clock.seconds) this._emails[i].Read = true;
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// checks whether the current item is marked
this.$isItemMarked = function(id) {
	var email = this.$getEmailByID(id);
	if (email.Marked === true) {
		return true;
	} else {
		return false;
	}
}

//-------------------------------------------------------------------------------------------------------------
// marks the currently selected item
this.$markItem = function(id) {
	var email = this.$getEmailByID(id);
	email.Marked = true;
}

//-------------------------------------------------------------------------------------------------------------
// unmarks the currently selected item
this.$unmarkItem = function(id) {
	var email = this.$getEmailByID(id);
	email.Marked = false;
}

//-------------------------------------------------------------------------------------------------------------
// deletes the currently selected item
this.$deleteSelectedItem = function(id) {
	if (this._emails.length > 0 && id > 0) {
		var idx = this.$getEmailIndexByID(id);
		if (this.$requiresResponse(this._emails[idx]) === false || this._emails[idx].ExpiryDate < clock.adjustedSeconds) {
			this._emails.splice(idx, 1);
			this._maxpage = Math.ceil(this.$totalItems() / this._msRows);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// deletes all marked emails that don't have a pending response
this.$deleteMarkedItems = function() {
	if (this._emails.length > 0 && this.$totalItemsMarked() > 0) {
		for (var i = this._emails.length - 1; i >= 0; i--) {
			if (this._emails[i].Marked === true && this.$requiresResponse(this._emails[i]) === false) {
				this._emails.splice(i, 1);
			}
		}
		this._curpage = 0;
		this._maxpage = Math.ceil(this.$totalItems() / this._msRows);
	}
}

//-------------------------------------------------------------------------------------------------------------
// deletes all marked emails that don't have a pending response
this.$deleteReadItems = function() {
	if (this._emails.length > 0 && this.$totalReadItems() > 0) {
		for (var i = this._emails.length - 1; i >= 0; i--) {
			if (this._emails[i].Read === true && this.$requiresResponse(this._emails[i]) === false) {
				this._emails.splice(i, 1);
			}
		}
		this._curpage = 0;
		this._maxpage = Math.ceil(this.$totalItems() / this._msRows);
	}
}

//-------------------------------------------------------------------------------------------------------------
// deletes all emails that don't have a pending response.
this.$deleteAllItems = function() {
	for (var i = this._emails.length - 1; i >= 0; i--) {
		// only delete emails that are in the past
		if (this.$requiresResponse(this._emails[i]) === false && this._emails[i].TransmitDate <= clock.seconds) {
			this._emails.splice(i, 1);
		}
	}
	this._curpage = 0;
	this._maxpage = 1;
}

//-------------------------------------------------------------------------------------------------------------
// records the index of the response against the selected email
this.$recordResponse = function(id, option) {
	var email = this.$getEmailByID(id);
	email.ChosenOption = option;
	email.ResponseDate = clock.seconds;
}

//-------------------------------------------------------------------------------------------------------------
// Returns the header details of the email formatted for display
this.$header = function(email) {

	var text = "";

	text += this.$padTextRight("From:", 5) + email.Sender + "\n";
	text += this.$padTextRight("Sent:", 5) + clock.clockStringForTime(email.TransmitDate)
	// add the galaxy number if the email was received in a different galaxy to the one the player is in now.
	if (galaxyNumber != email.GalaxyNum) text += " (G" + (email.GalaxyNum + 1) + ")";
	text += "\n";
	// expiry date
	if (email.ExpiryDate != 0) {
		text += this.$padTextRight("Expires:", 5);
		if (email.ExpiryDate > clock.seconds) {
			text += clock.clockStringForTime(email.ExpiryDate) + "\n";
		} else {
			text += email.ExpiryText + "\n";
		}
	}

	text += this.$padTextRight("Subject:", 5) + email.Subject + "\n";
	text += this.$duplicate("-", 32) + "\n";

	return text;
}

//-------------------------------------------------------------------------------------------------------------
// checks whether a response is required for this email.
// Returns true if there is a response and none has been selected, otherwise false.
this.$requiresResponse = function(email) {
	var ret = false;
	if ((email.ResponseOption1.DisplayText != "" || email.ResponseOption2.DisplayText != "" || email.ResponseOption3.DisplayText != "" || email.ResponseOption4.DisplayText != "") && email.ChosenOption === 0) {
		ret = true;
	}
	return ret;
}

//-------------------------------------------------------------------------------------------------------------
// returns the inbox line for this email
this.$inboxDisplay = function(email) {

	var ret = "";

	if (email.Read === false) {
		ret += this.$padTextRight("!", this._readColumn);
	} else {
		ret += this.$padTextRight("", this._readColumn);
	}

	ret += this.$padTextRight(email.Sender, 10);
	ret += this.$padTextRight(this.$inboxTime(clock.clockStringForTime(email.TransmitDate)), 7.6);
	ret += this.$padTextRight(email.Subject, 14);

	return ret;
}

//-------------------------------------------------------------------------------------------------------------
// appends space to currentText to the specified length in 'em'
this.$padTextRight = function(currentText, desiredLength) {
	if (currentText == null) currentText = "";
	var hairSpace = String.fromCharCode(31);
	var currentLength = defaultFont.measureString(currentText);
	var hairSpaceLength = defaultFont.measureString(hairSpace);
	var ellip = "…";
	// calculate number needed to fill remaining length
	var padsNeeded = Math.floor((desiredLength - currentLength) / hairSpaceLength);
	if (padsNeeded < 1)	{
		// text is too long for column, so start pulling characters off
		var tmp = currentText;
		do {
			tmp = tmp.substring(0, tmp.length - 2) + ellip;
			if (tmp === ellip) break;
		} while (defaultFont.measureString(tmp) > desiredLength);
		currentLength = defaultFont.measureString(tmp);
		padsNeeded = Math.floor((desiredLength - currentLength) / hairSpaceLength);
		currentText = tmp;
	}
	if (padsNeeded < 0) padsNeeded = 0;
	// quick way of generating a repeated string of that number
	return currentText + new Array(padsNeeded).join(hairSpace);
}

//-------------------------------------------------------------------------------------------------------------
// works out whether a particular option should be displayed or not
this.$displayResponseOption = function(email, itemNumber) {

	var bShow = false;

	switch (itemNumber) {
		case 1:
			if (email.ResponseOption1 != null && email.ResponseOption1.DisplayText != "") bShow = true;
			// if the expiry is turned off but this option was included as an expiry option, don't show it
			if (email.ExpiryDate === 0 && email.ExpiryOptions.indexOf("1") >= 0) bShow = false;
			if (email.ExpiryDate != 0) {
				// don't show this option if it's an expiry option
				if (email.ExpiryDate > clock.seconds && email.ExpiryOptions.indexOf("1") >= 0) bShow = false;
				// don't show this option if the email has expired and it's not included in the expiry options
				if (email.ExpiryDate <= clock.seconds && email.ExpiryOptions.indexOf("1") < 0) bShow = false;
			}
			break;
		case 2:
			if (email.ResponseOption2 != null && email.ResponseOption2.DisplayText != "") bShow = true;
			// if the expiry is turned off but this option was included as an expiry option, don't show it
			if (email.ExpiryDate === 0 && email.ExpiryOptions.indexOf("2") >= 0) bShow = false;
			if (email.ExpiryDate != 0) {
				// don't show this option if it's an expiry option
				if (email.ExpiryDate > clock.seconds && email.ExpiryOptions.indexOf("2") >= 0) bShow = false;
				// don't show this option if the email has expired and it's not included in the expiry options
				if (email.ExpiryDate <= clock.seconds && email.ExpiryOptions.indexOf("2") < 0) bShow = false;
			}
			break;
		case 3:
			if (email.ResponseOption3 != null && email.ResponseOption3.DisplayText != "") bShow = true;
			// if the expiry is turned off but this option was included as an expiry option, don't show it
			if (email.ExpiryDate === 0 && email.ExpiryOptions.indexOf("3") >= 0) bShow = false;
			if (email.ExpiryDate != 0) {
				// don't show this option if it's an expiry option
				if (email.ExpiryDate > clock.seconds && email.ExpiryOptions.indexOf("3") >= 0) bShow = false;
				// don't show this option if the email has expired and it's not included in the expiry options
				if (email.ExpiryDate <= clock.seconds && email.ExpiryOptions.indexOf("3") < 0) bShow = false;
			}
			break;
		case 4:
			if (email.ResponseOption4 != null && email.ResponseOption4.DisplayText != "") bShow = true;
			// if the expiry is turned off but this option was included as an expiry option, don't show it
			if (email.ExpiryDate === 0 && email.ExpiryOptions.indexOf("4") >= 0) bShow = false;
			if (email.ExpiryDate != 0) {
				// don't show this option if it's an expiry option
				if (email.ExpiryDate > clock.seconds && email.ExpiryOptions.indexOf("4") >= 0) bShow = false;
				// don't show this option if the email has expired and it's not included in the expiry options
				if (email.ExpiryDate <= clock.seconds && email.ExpiryOptions.indexOf("4") < 0) bShow = false;
			}
			break;
	}

	return bShow;
}

//-------------------------------------------------------------------------------------------------------------
// duplicates text until it is just less than the desired length;
this.$duplicate = function(text, desiredLength) {
	var res = "";

	do {
		res += text;
	} while (defaultFont.measureString(res) < desiredLength);

	res = res.substring(1, res.length);
	return res;
}

//-------------------------------------------------------------------------------------------------------------
// adds a message to an array, splitting the lines based on a line width
this.$addMsgToArray = function(msg, ary, linewidth) {

	var prt = " ";
	if (msg.substring(0,1) === "~" || msg.substring(0,1) === "-") {
		// don't indent this one
		prt = "";
		// remove the "~" char
		if (msg.substring(0,1) === "~") msg = msg.substring(1, msg.length);
	}

	if (defaultFont.measureString(msg) > linewidth) {
		var words = msg.split(" ");
		var iPoint = 0
		var nextWord = "";
		do {
			// add the space in (after the first word)
			if (prt != "")  prt += " ";

			// add the word on
			prt = prt + words[iPoint];
			// get the next word in the array
			if (iPoint < words.length - 1) {
				nextWord = " " + words[iPoint + 1];
			} else {
				nextWord = ""
			}
			// check the width of the next with the nextword added on
			if (defaultFont.measureString(prt + nextWord) > linewidth) {
				// hit the max width - add the line and start again
				ary.push(prt);
				prt = "";
			}
			iPoint += 1;
			// check for the end of the array, and output remaining text
			if ((iPoint >= words.length) && (prt.trim() != "")) ary.push(prt);

		} while (iPoint < words.length);
	} else {
		ary.push(prt + msg);
	}
}

//=============================================================================================================
// screen interfaces
this.$initInterface = function(station) {

	var unread = "";
	var itms = this.$totalUnreadItems();
	if (itms > 0) unread = " (" + itms.toString() + " unread" + (this.$totalUnreadItemsRequiringResponse() > 0 ? " •" : "") + ")"

	station.setInterface(this.name,{
		title:"Email system" + unread,
		category:expandDescription("[interfaces-category-logs]"),
		summary:"Opens your email inbox",
		callback:this.$showInbox.bind(this)
	});

}

//=============================================================================================================
// inbox
this.$showInbox = function() {
	this._fromScreen = guiScreen;
	this._emailID = 0;
	this._emailIndex = 0;
	this._maxpage = Math.ceil(this.$totalItems() / this._msRows);
	this._curpage = 0;
	this._displayType = 0;
	this.$showPage();
}

//-------------------------------------------------------------------------------------------------------------
this.$showPage = function() {
	function compare(a,b) {
		return b.TransmitDate - a.TransmitDate;
	}

	if (this.$isBigGuiActive() === false) player.ship.hudHidden = true;

	var text = "";
	var opts;
	var curChoices = {};
	var def = "";

	//=============================================================================================================
	// inbox view
	if (this._displayType === 0) {
		// sort the inbox by date, newest to oldest
		this._emails.sort(compare);
		this._emailPage = 0;

		var expired = false;
		// look for and delete any expired emails
		for (var i = this._emails.length - 1; i >= 0; i--) {
			if (this._emails[i].ExpiryDate != 0 && clock.seconds > this._emails[i].ExpiryDate && this._emails[i].ExpiryText === "") {
				this._emails.splice(i, 1);
				expired = true;
			}
		}

		// adjust the max page if we remove any expired emails
		if (expired) this._maxpage = Math.ceil(this.$totalItems() / this._msRows);

		curChoices = this.$inboxPageChoices(this._curpage, this._msRows);

		// make sure the selected email stays selected, in case new emails arrived while we were viewing another email
		if (this._emailID != 0 && this._emailIndex != this.$getChoiceItemIndex(this._emailID)) {
			var old = this._emailIndex;
			this._emailIndex = this.$getChoiceItemIndex(this._emailID);
			if (this._emailIndex === 0 && old === this._msRows) {
				// we've been bumped to a new page, so switch to it
				this._curpage += 1;
				curChoices = this.$inboxPageChoices(this._curpage, this._msRows);
				this._emailIndex = this.$getChoiceItemIndex(this._emailID);
			}
			this._lastChoice[this._displayType] = "01_email-" + (this._emailIndex < 10 ? "0" : "") + this._emailIndex + "+" + this._emailID;
		}

		this._emailID = 0;

		text = this.$padTextRight("", this._readColumn + 0.15);
		text += this.$padTextRight("From", 10);
		text += this.$padTextRight("Date", 7.6);
		text += "Subject";
		if (this._itemsOnPage === 0) text += "\n\n(no items to display)";

		for (var i = 0; i < (this._msRows + 1) - this._itemsOnPage; i++) {
			curChoices["02_SPACER_" + i] = "";
		}

		var def = "99_EXIT";

		if (this._emails != null && this._emails.length != 0) {
			if (this._curpage < this._maxpage - 1) {
				curChoices["10_GOTONEXT"] = {text:"[emailopt_nextpage]", color:this._menuColor};
			} else {
				curChoices["10_GOTONEXT"] = {text:"[emailopt_nextpage]", color:this._disabledColor, unselectable:true};
			}
			if (this._curpage > 0) {
				curChoices["11_GOTOPREV"] = {text:"[emailopt_prevpage]", color:this._menuColor};
			} else {
				curChoices["11_GOTOPREV"] = {text:"[emailopt_prevpage]", color:this._disabledColor, unselectable:true};
			}
			if (this.$totalReadItems() > 0) {
				curChoices["42_MARKALLREAD"] = {text:"[emailopt_markallread]", color:this._menuColor};
				curChoices["45_DELREAD"] = {text:"[emailopt_deleteread]", color:this._menuColor};
			} else {
				curChoices["42_MARKALLREAD"] = {text:"[emailopt_markallread]", color:this._disabledColor, unselectable:true};
				curChoices["45_DELREAD"] = {text:"[emailopt_deleteread]", color:this._disabledColor, unselectable:true};
			}
			curChoices["50_DELALL"] = {text:"[emailopt_deleteall]", color:this._menuColor};
		} else {
			curChoices["10_GOTONEXT"] = {text:"[emailopt_nextpage]", color:this._disabledColor, unselectable:true};
			curChoices["11_GOTOPREV"] = {text:"[emailopt_prevpage]", color:this._disabledColor, unselectable:true};
			curChoices["42_MARKALLREAD"] = {text:"[emailopt_markallread]", color:this._disabledColor, unselectable:true};
			curChoices["45_DELREAD"] = {text:"[emailopt_deleteread]", color:this._disabledColor, unselectable:true};
			curChoices["50_DELALL"] = {text:"[emailopt_deleteall]", color:this._disabledColor, unselectable:true};
		}

		curChoices["99_EXIT"] = {text:"[emailopt_exit]", color:this._exitColor};

		var opts = {
			screenID: "oolite-emailsystem-main-map",
			title: "Inbox - page " + (this._curpage + 1).toString() + " of " + this._maxpage.toString(),
			allowInterrupt: true,
			exitScreen: this._fromScreen,
			overlay: {name:"email-message.png", height:546},
			choices: curChoices,
			initialChoicesKey: (this._lastChoice[this._displayType] ? this._lastChoice[this._displayType] : def),
			message: text
		};
	}

	//=============================================================================================================
	// email item
	if (this._displayType === 1) {
		var email = this.$getEmailByID(this._emailID);

		var headerlen = 4;
		if (email.ExpiryDate !=0 && email.ExpiryDate > clock.seconds) {
			headerlen += 1; // extra line if expiry date is still current
			if (email.allowExpiryCancel) headerlen +=1; // another line for the "Cancel expiry" option
		}

		email.Read = true;

		text += this.$header(email);

		var lines = email.EmailBody.split("\n");
		var dest = [];

		for (var i = 0; i < lines.length; i++) {
			this.$addMsgToArray(lines[i], dest, this._msCols);
		}

		// append any response to the body of the email
		if (email.ChosenOption > 0) {
			var resp = "";
			resp += "\n";
			resp += "~" + this.$duplicate("-", 32) + "\n";
			resp += "~" + this.$padTextRight("Reply sent:", 5) + clock.clockStringForTime(email.ResponseDate) + "\n";
			resp += "~Reply:\n\n";

			switch (email.ChosenOption) {
				case 1:
					resp += email.ResponseOption1.ReplyText;
					break;
				case 2:
					resp += email.ResponseOption2.ReplyText;
					break;
				case 3:
					resp += email.ResponseOption3.ReplyText;
					break;
				case 4:
					resp += email.ResponseOption4.ReplyText;
					break;
			}
			var respLines = resp.split("\n");
			for (var i = 0; i < respLines.length; i++) {
				this.$addMsgToArray(respLines[i], dest, this._msCols);
			}
		}

		var addText = "";
		var iPageHeight = 16 + (4 - headerlen);
		this._emailMaxPages = 0;

		// check if
		if (dest.length > iPageHeight && this.$requiresResponse(email) === true) {
			// how many options are there?
			var iRespCount = 0;
			if ($displayResponseOption(email, 1)) iRespCount += 1;
			if ($displayResponseOption(email, 2)) iRespCount += 1;
			if ($displayResponseOption(email, 3)) iRespCount += 1;
			if ($displayResponseOption(email, 4)) iRespCount += 1;

			// reduce the page height by the number of responses
			iPageHeight -= iRespCount;
		}

		if (dest.length > iPageHeight) {
			this._emailMaxPages = Math.ceil(dest.length / iPageHeight);
			addText = " (Page " + (this._emailPage + 1).toString() + " of " + this._emailMaxPages.toString() + ")";
		}
		var iStart = this._emailPage * iPageHeight; // when 0 then 0, when 1 then 16
		var iEnd = this._emailPage * iPageHeight + iPageHeight; // when 0 then 16, when 1 then 32 etc
		if (iEnd > dest.length) {
			iEnd = dest.length;
		}
		for (var i = iStart; i <= iEnd - 1; i++) {
			text += dest[i] + "\n";
		}

		def = "23_CLOSE";

		if (this._emailMaxPages != 0) {
			if (this._emailPage + 1 < this._emailMaxPages) {
				curChoices["21_NEXTPAGE"] = {text:"[emailitem_nextpage]", color:this._menuColor};
				def = "21_NEXTPAGE";
			} else {
				curChoices["21_NEXTPAGE"] = {text:"[emailitem_nextpage]", color:this._disabledColor, unselectable:true};
			}
			if (this._emailPage > 0) {
				curChoices["22_PREVPAGE"] = {text:"[emailitem_prevpage]", color:this._menuColor};
			} else {
				curChoices["22_PREVPAGE"] = {text:"[emailitem_prevpage]", color:this._disabledColor, unselectable:true};
			}
		}
		// if the force response option is turned on, disable all the close email options, so the player has to make a choice
		if (email.ForceResponse === true && email.ChosenOption === 0) {
			curChoices["23_CLOSE"] = {text:"[emailitem_close]", color:this._disabledColor, unselectable:true};
			curChoices["23A_CLOSEOPEN"] = {text:"[emailitem_closeopen]", color:this._disabledColor, unselectable:true};
			curChoices["24_CLOSEDEL"] = {text:"[emailitem_closedelete]", color:this._disabledColor, unselectable:true};
		} else {
			curChoices["23_CLOSE"] = {text:"[emailitem_close]", color:this._menuColor};
			if (this.$findNextEmailID(this._emailID) >= 0) {
				curChoices["23A_CLOSEOPEN"] = {text:"[emailitem_closeopen]", color:this._menuColor};
			} else {
				curChoices["23A_CLOSEOPEN"] = {text:"[emailitem_closeopen]", color:this._disabledColor, unselectable:true};
			}
			if (this.$requiresResponse(email) === true && email.ExpiryDate > clock.seconds && email.ChosenOption === 0) {
				curChoices["24_CLOSEDEL"] = {text:"[emailitem_closedelete]", color:this._disabledColor, unselectable:true};
			} else {
				curChoices["24_CLOSEDEL"] = {text:"[emailitem_closedelete]", color:this._menuColor};
			}
		}
		curChoices["25_TRACE"] = {text:"[emailitem_trace]", color:this._menuColor};
		if (email.ExpiryDate != 0 && email.ExpiryDate > clock.seconds && email.AllowExpiryCancel === true) {
			curChoices["25A_CANCELEXPIRY"] = {text:"[emailitem_cancelexpiry]", color:this._menuColor};
		}

		if (email.ChosenOption === 0) {
			if (this.$displayResponseOption(email, 1)) curChoices["26_OPT1"] = {text:"Send '" + email.ResponseOption1.DisplayText + "' response", color:this._menuColor};
			if (this.$displayResponseOption(email, 2)) curChoices["26_OPT2"] = {text:"Send '" + email.ResponseOption2.DisplayText + "' response", color:this._menuColor};
			if (this.$displayResponseOption(email, 3)) curChoices["26_OPT3"] = {text:"Send '" + email.ResponseOption3.DisplayText + "' response", color:this._menuColor};
			if (this.$displayResponseOption(email, 4)) curChoices["26_OPT4"] = {text:"Send '" + email.ResponseOption4.DisplayText + "' response", color:this._menuColor};
		}

		var opts = {
			screenID: "oolite-emailsystem-item-map",
			title: "Message" + addText,
			allowInterrupt: true,
			exitScreen: this._fromScreen,
			overlay: {name:"email-message_open.png", height:546},
			choices: curChoices,
			initialChoicesKey: (this._lastChoice[this._displayType] != "" ? this._lastChoice[this._displayType] : def),
			message: text
		};
	}

	//=============================================================================================================
	// trace
	if (this._displayType === 2) {
		var email = this.$getEmailByID(this._emailID);

		var initText = "- Received: ";

		text += this.$header(email);

		text += "Trace:\n\n";
		var planets = email.Trace.split(",");
		var bEnd = false;
		var bCrpt = false;
		var colHeight = 16;
		if (planets.length < colHeight) {
			for (var i = 0; i < planets.length; i++) {
				if (planets[i] != "") {
					text += initText + planets[i] + "\n";
					initText = "- Sent from: ";
				}
			}
		} else {
			for (var i = 0; i < colHeight; i++) {
				if (planets[i] != "") {
					text += this.$padTextRight(initText + planets[i], 10);
					initText = "- Sent from: ";
					if (i + colHeight < planets.length) {
						if (planets[i + colHeight] != "") {
							text += this.$padTextRight("- Sent from: " + planets[i + colHeight], 10);
						}
					}
					if (i + colHeight === planets.length) {
						if (email.CorruptTrace === true) {
							text += "- Routing ticket corrupt";
							bCrpt = true;
						}
					}
					if (i + (colHeight * 2) < planets.length) {
						if (planets[i + (colHeight * 2)] != "") {
							text += this.$padTextRight("- Sent from: " + planets[i + (colHeight * 2)], 10);
						}
					}
					if (i + (colHeight * 2) === planets.length) {
						if (email.CorruptTrace === true) {
							text += "- Routing ticket corrupt";
							bCrpt = true;
						}
					}
					text += "\n";
				}
			}
		}
		if (email.CorruptTrace === true) {
			if (bCrpt === false) {
				text += "- Routing ticket corrupt\n";
			}
			text += "\nTrace incomplete.";
		} else if (bEnd === false){
			text += "\nTrace complete.";
		}
		def = "27_CLOSE";

		curChoices["27_CLOSE"] = {text:"[emailtrace_close]", color:this._exitColor};

		var opts = {
			screenID: "oolite-emailsystem-trace-summary",
			title: expandDescription("[emailtrace_title]"),
			allowInterrupt: true,
			overlay: {name:"email-message_open.png", height:546},
			exitScreen: this._fromScreen,
			choices: curChoices,
			initialChoicesKey: (this._lastChoice[this._displayType] != "" ? this._lastChoice[this._displayType] : def),
			message: text
		};
		planets = [];
	}

	mission.runScreen(opts, this.$inboxHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$inboxHandler = function(choice) {

	var curdisplay = this._displayType;
	var newChoice = "";
	var newChoicePage = 0;
	
	this._lastChoice[this._displayType] = choice;

	if (!choice) {
		return; // launched while reading?
	} else if (choice.indexOf("01_email") >= 0) {
		this._emailID = parseInt(choice.substring(choice.indexOf("+") + 1));
		this._emailIndex = parseInt(choice.substring(choice.indexOf("-") + 1, choice.indexOf("+")));
		this._emailPage = 0;
		this._displayType = 1;
	} else if (choice === "11_GOTOPREV") {
		this._curpage -= 1;
		if (this._curpage === 0) {
			newChoice = "10_GOTONEXT";
			newChoicePage = this._displayType;
		}
	} else if (choice === "10_GOTONEXT") {
		this._curpage += 1;
		if (this._curpage === this._maxpage - 1) {
			newChoice = "11_GOTOPREV";
			newChoicePage = this._displayType;
		}
	} else if (choice === "42_MARKALLREAD") {
		this.$markAllItemsRead();
	} else if (choice === "45_DELREAD") {
		this.$deleteReadItems();
	} else if (choice === "50_DELALL") {
		this.$deleteAllItems();
	} else if (choice === "21_NEXTPAGE") {
		this._emailPage += 1;
		if ((this._emailPage - 1) === this._emailMaxPages) {
			newChoice = "22_PREVPAGE";
			newChoicePage = this._displayType;
		}
	} else if (choice === "22_PREVPAGE") {
		this._emailPage -= 1;
		if (this._emailPage === 0) {
			newChoice = "21_NEXTPAGE";
			newChoicePage = this._displayType;
		}
	} else if (choice === "23_CLOSE") {
		this._displayType = 0;
	} else if (choice === "23A_CLOSEOPEN") {
		this._emailPage = 0;
		// get next email id
		this._emailID = this.$findNextEmailID(this._emailID);
		// if this the last item on the page?
		if (this._emailIndex === this._msRows && (this._curpage + 1) < this._maxpage) {
			// we've flipped over to the next page
			this._curpage += 1;
			this.$inboxPageChoices(this._curpage, this._msRows);
		}
		this._emailIndex = this.$getChoiceItemIndex(this._emailID)
		newChoice = "01_email-" + (this._emailIndex < 10 ? "0" : "") + this._emailIndex + "+" + this._emailID;
		newChoicePage = 0;
	} else if (choice === "24_CLOSEDEL") {
		this._displayType = 0;
		this.$deleteSelectedItem(this._emailID);
	} else if (choice === "25_TRACE") {
		this._displayType = 2;
	} else if (choice === "25A_CANCELEXPIRY") {
		this._emails[this.$getEmailIndexByID(this._emailID)].ExpiryDate = 0;
	} else if (choice === "26_OPT1") {
		// make sure the email hasn't expired while being viewed
		if (this.$displayResponseOption(this.$getEmailByID(this._emailID), 1)) {
			this.$recordResponse(this._emailID, 1);
			this.$executeCallback(this._emailID, 1);
		}
	} else if (choice === "26_OPT2") {
		// make sure the email hasn't expired while being viewed
		if (this.$displayResponseOption(this.$getEmailByID(this._emailID), 2)) {
			this.$recordResponse(this._emailID, 2);
			this.$executeCallback(this._emailID, 2);
		}
	} else if (choice === "26_OPT3") {
		// make sure the email hasn't expired while being viewed
		if (this.$displayResponseOption(this.$getEmailByID(this._emailID), 3)) {
			this.$recordResponse(this._emailID, 3);
			this.$executeCallback(this._emailID, 3);
		}
	} else if (choice === "26_OPT4") {
		// make sure the email hasn't expired while being viewed
		if (this.$displayResponseOption(this.$getEmailByID(this._emailID), 4)) {
			this.$recordResponse(this._emailID, 4);
			this.$executeCallback(this._emailID, 4);
		}
	} else if (choice === "27_CLOSE") {
		this._displayType = 1;
	}
	
	if (newChoice != "") this._lastChoice[newChoicePage] = newChoice;
	
	if (choice != "99_EXIT") {
		this.$showPage();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$inboxPageChoices = function(cpage, lines) {
	//var output = "";

	var iStart = 0;
	var iEnd = 0;
	var offset = 0;
	var choices = {};

	// work out start point - start after any future dated emails
	for (var i = 0; i < this._emails.length; i++) {
		if (this._emails[i].TransmitDate <= clock.seconds) {
			offset = i;
			break;
		}
	}

	// set out initial end point
	iStart = offset;
	iEnd = iStart + lines;
	if (cpage != 0) {
		iStart = (cpage * lines) + offset;
		iEnd = iStart + lines;
	}
	if (iEnd > this._emails.length) {
		iEnd = this._emails.length;
	}

	this._itemsOnPage = 0;

	this._pageItems.length = 0;

	if (this.$totalItems() != 0) {
		for (var i = iStart; i < iEnd; i++) {
			// put an index in the key for sorting, and add the id for selecting
			this._itemsOnPage += 1;
			choices["01_email-" + (this._itemsOnPage < 10 ? "0" : "") + this._itemsOnPage + "+" + this._emails[i].ID] = {text:this.$inboxDisplay(this._emails[i]), alignment:"LEFT", color:this._itemColor};
			this._pageItems.push({ID:this._emails[i].ID, index:this._itemsOnPage});
		}
	}
	return choices;
}

//-------------------------------------------------------------------------------------------------------------
this.$getChoiceItemIndex = function(id) {
	var result = 0;
	if (this._pageItems && this._pageItems.length > 0) {
		for (var i = 0; i < this._pageItems.length; i++) {
			if (this._pageItems[i].ID === id) result = this._pageItems[i].index;
		}
	}
	return result;
}

//-------------------------------------------------------------------------------------------------------------
this.$inboxTime = function(emailTime) {
	return emailTime.substring(0, emailTime.length - 3);
}

//-------------------------------------------------------------------------------------------------------------
// returns true if a HUD with allowBigGUI is enabled, otherwise false
this.$isBigGuiActive = function() {
	if (oolite.compareVersion("1.83") <= 0) {
		return player.ship.hudAllowsBigGui;
	} else {
		var bigGuiHUD = ["XenonHUD.plist", "coluber_hud_ch01-dock.plist"]; 	// until there is a property we can check, I'll be listing HUD's that have the allow_big_gui property set here
		if (bigGuiHUD.indexOf(player.ship.hud) >= 0) {
			return true;
		} else {
			return false;
		}
	}
}
