"use strict";
this.name        = "GalCopAdminServices";
this.author      = "phkb";
this.copyright   = "2017 phkb";
this.description = "Transmits emails to player from GalCop admin relating to bounty notifications and other admin processes.";
this.licence     = "CC BY-NC-SA 4.0";

/*
	Todo:
	Check for docking fines on other station types (UPS Courier, Jaguar company, The collector, Aquatics, Resistance Commander, Lave Acedemy, FTZ, Planetfall);
	Check for purchase equip on Black Monks Monestary (need different voice for email).

	Ideas:
	Welcome to new galaxy emails
		- Tech Level 15 planets -- G1 Ceesxe, G2 Tezaeded, G3 none (14: LEZAER,BIRERA,LEORENDI,ORZAEDVE,TEEDUS,TIERA,CEDILE), G4 none (14: GEBIISSO,DICEBE),
			G5 Xevera, G6 Diesanen, G7 Quandixe and Maraus, G8 none (14: ESUSALE).
		- List of galactic regions
		- List of galactic routes
		- GalCop HQ's
		- RRS station HQ locations
		- Superhub locations
		- Gal Navy sector commands

	Constore emails about special deals
	Space bar emails
	Emails from revolutionists, saying "Here's the details of the conspiracy"
	Emails from Hognose Evangelists telling player about their religion
	Emails containing bargains for trade (ie "trade in Furs between X and Y for a super bargain. Offer won't last!" X will be system within 7 LY of player, Y will be within 10 LY of X.)
		maybe two different emails: 1 saying nnn commodity is really cheap at X, 1 saying demand for nnn commodity is high at Y
		have an opt in/opt out functionality
		-- seems to be similar to function in NewCargoes OXP, but might be useful as an alternative (ie. if you don't want NewCargos, you can still get market bargins)
		-- also similar to functionality in BlOomberg markets
*/

this._debug = true;
this._bountyEmailBody = "";										// holds details of next bounty email
this._bountyEmailTime = 0;										// the time of the last kill
this._bountyEmailTotalCR = 0;									// total amount of bounty
this._boughtEquipTimer = null;
this._killCount = 0;											// total number of kills
this._killAssortedCount = 0;									// total number of non-ship kills (eg missiles)
this._holdBountyEmail = [];										// holds bounty emails across hyperspace jumps
this._disableBounty = false;									// controls whether bounty emails are sent
this._disableDocking = false;									// controls whether docking violation emails are sent
this._disableEscapePods = false;								// controls whether escape pod rescue emails are sent
this._disableContracts = false;									// controls whether contract emails are sent
this._disableFines = false;										// controls whether fine processing emails are sent
this._disableEquipPurchase = false;								// controls whether equipment purchase emails are sent
this._disableMaintenance = false;								// controls whether repair and overhaul emails are sent
this._disableNewShip = false;									// controls whether new ship emails are sent
this._disableEliteFederation = false;							// controls whether changes to player rank are sent
this._defaultExpiryDays = 5;									// default expiry period for emails that expire
this._repairHullDamage = false;									// keeps track of hull damage email (Battle Damage OXP)
this._repairIronHide = false;									// keeps track of iron hide repair emails (IronHide OXP)
this._sendMaintEmail = false;									// keeps track of when we are sending maintenance emails (relates to IronHide OXP)
this._sendRepairEmail = false;									// keeps track of when we are sending repair emails (relates to IronHide OXP)
this._maintCost = 0;
this._maintCostInit = 0;
this._equipItems = [];											// list of equipment items, held in order to determine diff between purchase and repair
this._equipSalesRep = "";										// name of sales assistant for equipment
this._shipSalesRep = "";										// name of sales assistant for ships
this._maintRep = "";											// name of maintenance rep for repair or overhaul emails
this._dutyOfficer = "";											// station duty officer/flight controller (for docking fines)
this._galCopBountyOfficer = "";									// name of galcop bounty processor
this._insuranceOfficer = "";									// name of claims administrator (for escape pod insurance claims)
this._namesLocation = "";										// keeps track of the system/dock the names are associated with
this._galCopHQ = [7, 33, 26, 49, 102, 85, 202, 184];			// galcop hq locations in each sector
this._fedHQ = [131, 167, 58, 65, 230, 35, 181, 130];			// location of elite federation hq in each sector
this._playerEnteringShipyard = false;							// keeps track of when the player enters the shipyard at a station, so we can check for buying a new ship
this._licenceNumber = "";										// holds the pilot licence number, in case it's ever needed
this._maintItemNameReplacements = "";							// holds list of maint item replacement name keys
this._trueValues = ["yes", "1", 1, "true", true];
this._playerEjected = false;
this._maint_ignore_equip = []; // no longer used, but kept for backwards compatibility.
this._oldPassengerCount = 0;

/* equipment known to have description items for an overhaul email */
this._maint_known_equip = ["EQ_CARGO_BAY", "EQ_ECM", "EQ_FUEL_SCOOPS", "EQ_ESCAPE_POD", "EQ_ENERGY_UNIT", "EQ_NAVAL_ENERGY_UNIT", "EQ_DOCK_COMP", "EQ_GAL_DRIVE",
	"EQ_WEAPON_PULSE_LASER", "EQ_WEAPON_BEAM_LASER", "EQ_WEAPON_MINING_LASER", "EQ_WEAPON_MILITARY_LASER", "EQ_CLOAKING_DEVICE", "EQ_PASSENGER_BERTH", "EQ_FUEL_INJECTION",
	"EQ_SCANNER_SHOW_MISSILE_TARGET", "EQ_MULTI_TARGET", "EQ_ADVANCED_COMPASS", "EQ_ADVANCED_NAVIGATIONAL_ARRAY", "EQ_TARGET_MEMORY", "EQ_INTEGRATED_TARGETING_SYSTEM",
	"EQ_SHIELD_BOOSTER", "EQ_NAVAL_SHIELD_BOOSTER", "EQ_HEAT_SHIELD", "EQ_WORMHOLE_SCANNER", "EQ_WEAPON_TWIN_PLASMA_CANNON",
	"EQ_LMSS_FRONT", "EQ_LMSS_AFT", "EQ_LMSS_PORT", "EQ_LMSS_STARBOARD", "EQ_LMSS_ACTUATOR"];

/* equipment items devoted to repair (not equipment in themselves) */
this._maint_repair_equip = ["EQ_HULL_REPAIR", "EQ_IRONHIDE_REPAIR", "EQ_TURRET_RECOVER", "EQ_SHIP_VERSION_REPAIR", "EQ_SERVICE_LEVEL_SMALL_FIX_1",
	"EQ_SERVICE_LEVEL_SMALL_FIX_2", "EQ_SERVICE_LEVEL_SMALL_FIX_3", "EQ_SERVICE_LEVEL_SMALL_FIX_4"];

/* equipment items devoted to removal of equipment */
this._maint_remove_equip = ["EQ_PASSENGER_BERTH_REMOVAL", "EQ_MISSILE_REMOVAL", "EQ_WEAPON_NONE"];

/* equipment items to be ignored by the email system */
this._purchase_ignore_equip = ["EQ_FUEL", "EQ_MISSILE", "EQ_TRUMBLE", "EQ_RRS_FUEL", "EQ_SHIP_RESPRAY", "EQ_SHIP_RESPRAY_180", "EQ_SMUGGLING_COMPARTMENT", "EQ_IMPORT_PERMIT"];

// equipment items that will be immediately removed after purchase
this._purchase_removed = ["EQ_IRONHIDE_MIL", "EQ_REPAIRBOTS_RECHARGE_10", "EQ_REPAIRBOTS_RECHARGE_5"];

//======================================================================================================================
// Library config
this._galCopAdminConfig = {Name:this.name, Alias:"GalCop Admin Services", Display:"Config", Alive:"_galCopAdminConfig",
	Bool:{
		B0:{Name:"_disableBounty", Def:false, Desc:"Bounty emails"},
		B1:{Name:"_disableDocking", Def:false, Desc:"Docking fine emails"},
		B2:{Name:"_disableEscapePods", Def:false, Desc:"Escape pod emails"},
		B3:{Name:"_disableContracts", Def:false, Desc:"Contract emails"},
		B4:{Name:"_disableFines", Def:false, Desc:"Fine penalty emails"},
		B5:{Name:"_disableEquipPurchase", Def:false, Desc:"Equip purchase emails"},
		B6:{Name:"_disableNewShip", Def:false, Desc:"New ship emails"},
		B7:{Name:"_disableMaintenance", Def:false, Desc:"Maintenance emails"},
		B8:{Name:"_disableEliteFederation", Def:false, Desc:"Elite Fed emails"},
		Info:"Set item to true to disable those emails from being transmitted."},
};

//=============================================================================================================
// ship interfaces
//-------------------------------------------------------------------------------------------------------------
this.startUp = function() {
	// delete the missionScreenEnded routine if IronHide OXP is not installed
	if (!worldScripts["IronHide Armour Script"]) delete this.missionScreenEnded;
}

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function() {
	// register our settings, if Lib_Config is present
	if (worldScripts.Lib_Config) worldScripts.Lib_Config._registerSet(this._galCopAdminConfig);

	// load in the licence number if it's been saved
	if (missionVariables.GalCopAdmin_licenceNumber) this._licenceNumber = missionVariables.GalCopAdmin_licenceNumber;
	if (missionVariables.GalCopAdmin_DisableBounty) this._disableBounty = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableBounty) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableEscapePods) this._disableEscapePods = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableEscapePods) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableContracts) this._disableContracts = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableContracts) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableFines) this._disableFines = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableFines) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableEquipPurchase) this._disableEquipPurchase = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableEquipPurchase) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableNewShip) this._disableNewShip = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableNewShip) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableMaintenance) this._disableMaintenance = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableMaintenance) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableEliteFed) this._disableEliteFederation = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableEliteFed) >= 0 ? true : false);
	if (missionVariables.GalCopAdmin_DisableDocking) this._disableDocking = (this._trueValues.indexOf(missionVariables.GalCopAdmin_DisableDocking) >= 0 ? true : false);

	this._maintItemNameReplacements = expandDescription("[maint_itemname_list]");

	// send a late notice pilot licence email
	if (global.clock.clockStringForTime(global.clock.seconds).indexOf("2084004:20") === -1 && this._licenceNumber === "") {
		this._licenceNumber = this.$generateLicenceNumber();
		// pilot licence notification arrives shortly after the start
		var lgl = expandDescription("[legalstatuswarnings_" + expandDescription("[commander_legal_status]") + "]");
		var w = worldScripts.EmailSystem;
		w.$createEmail({sender:expandDescription("[new-licence-sender]"),
			subject:expandDescription("Re: New Pilot Registration"),
			date:global.clock.adjustedSeconds,
			message:expandDescription("[latenotice-licence]", {licenceno:this._licenceNumber, insertlegalstatus:lgl}),
			sentFrom:this._galCopHQ[galaxyNumber]
		});
	}

}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function() {
	// save the licence number in worldscripts
	//if (this._licenceNumber === "") this._licenceNumber = this.$generateLicenceNumber();
	missionVariables.GalCopAdmin_licenceNumber = this._licenceNumber;
	missionVariables.GalCopAdmin_DisableBounty = this._disableBounty;
	missionVariables.GalCopAdmin_DisableEscapePods = this._disableEscapePods;
	missionVariables.GalCopAdmin_DisableContracts = this._disableContracts;
	missionVariables.GalCopAdmin_DisableFines = this._disableFines;
	missionVariables.GalCopAdmin_DisableEquipPurchase = this._disableEquipPurchase;
	missionVariables.GalCopAdmin_DisableNewShip = this._disableNewShip;
	missionVariables.GalCopAdmin_DisableMaintenance = this._disableMaintenance;
	missionVariables.GalCopAdmin_DisableEliteFed = this._disableEliteFederation;
	missionVariables.GalCopAdmin_DisableDocking = this._disableDocking;
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillDockWithStation = function(station) {

	var w = worldScripts.EmailSystem;
	this.$setupRepNames(station);

	// have we changed ranking?
	var newrank = expandDescription("[commander_rank]");
	if (this._oldrank != newrank && newrank != "Harmless" && this._disableEliteFederation === false) {
		// send an email from the Elite Federation
		// get clean version of rank
		if (newrank.indexOf("Deadly") >=0) newrank = "Deadly";
		if (newrank.indexOf("E L I T E") >=0 || newrank.indexOf("ELITE") >= 0) newrank = "Elite";

		var pre = "a";
		if ("AEIOU".indexOf(player.ship.shipClassName.substring(0,1)) >= 0) {
			pre = "an";
		}
		w.$createEmail({sender:expandDescription("[elite-fed-sender]"),
			subject:expandDescription("[elite-fed-subject-" + newrank + "]"),
			date:global.clock.adjustedSeconds,
			message:expandDescription("[elite-fed-body-" + newrank + "]", {shipnamepre:pre, realshipname:player.ship.shipUniqueName, galnum:(galaxyNumber + 1)}),
			sentFrom:this._fedHQ[galaxyNumber]
		});
	}

	// do we have any pending bounty notifications?
	if (this._disableBounty === false && (this._bountyEmailBody != "" || (this._holdBountyEmail && this._holdBountyEmail.length > 0))) {
		var include = "";
		// have we got any bounty reports in the hold queue
		if (this._holdBountyEmail && this._holdBountyEmail.length > 0) {
			for (var i = 0; i < this._holdBountyEmail.length; i++) {
				if (this._holdBountyEmail[i].bountySystem === "Interstellar space") {
					include += "~In " + this._holdBountyEmail[i].bountySystem + ":\n" + this._holdBountyEmail[i].bountyEmail + "\n";
				} else {
					include += "~In the " + this._holdBountyEmail[i].bountySystem + " system:\n" + this._holdBountyEmail[i].bountyEmail + "\n";
				}
			}
			// any stuff left to add? add another system marker
			if (this._bountyEmailBody != "") {
				if (this._bountySystem === "Interstellar space") {
					include += "~In " + this._bountySystem + ":\n";
				} else {
					include += "~In the " + this._bountySystem + " system:\n";
				}
			}
			// clean up
			while(this._holdBountyEmail.length > 0) {
				this._holdBountyEmail.pop();
			}
		}
		// any emailbody holding info? add it to the include var
		if (this._bountyEmailBody != "") include += this._bountyEmailBody;

		// add warning for no-bounty kills
		if (this._bountyEmailTotalCR > 0 && this._bountyEmailBody.indexOf("(no bounty)") >= 0) include += "\nPlease note that GalCop does not condone the killing of innocent pilots.\n";

		// number of kills
		var kills = "A total of " + this._killCount + " ships were destroyed";
		if (this._killCount === 1) kills = "A single ship was destroyed";
		var killasrt = "";
		if (this._killAssortedCount > 0) {
			killasrt = " (plus " + this._killAssortedCount + " other items)";
			if (this._killAssortedCount === 1) killasrt = " (plus 1 other item)";
		}
		kills += killasrt;

		var emailType = "[galcop-bounty-admin-body]";
		// if we didn't have any bounty, switch to the no-bounty email type
		if (this._bountyEmailTotalCR === 0) emailType = "[galcop-nobounty-admin-body]";

		w.$createEmail({sender:expandDescription("[galcop-bounty-sender]"),
			subject:"Bounty confirmation",
			date:this._bountyEmailTime,
			message:expandDescription(emailType, {bounty:formatCredits(this._bountyEmailTotalCR, false, true), bountybody:include, totalkills:kills, sender:this._galCopBountyOfficer}),
			sentFrom:this._galCopHQ[galaxyNumber],
			expiryDays:this._defaultExpiryDays}
		);
		this._bountyEmailBody = "";
		this._bountyEmailTotalCR = 0;
		this._bountyEmailTime = 0;
		this._killCount = 0;
		this._killAssortedCount = 0;
	}

	// player ejected
	if (this._disableEscapePods === false && this._playerEjected === true) {
		var msg = expandDescription("[escapepod-rescue-body]", {sender:randomName() + " " + randomName()});
		w.$createEmail({sender:expandDescription("[escapepod-insurance-sender]"),
			subject:"Rescue",
			date:global.clock.adjustedSeconds,
			message:msg,
			expiryDays:this._defaultExpiryDays
		});
	}

	// docking fine
	if (this._disableDocking === false && station.requiresDockingClearance === true &&
		player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_GRANTED" && player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_TIMING_OUT" &&
		this._playerEjected === false && system.sun.isGoingNova === false) {

		// what type of station is this?
		var bSent = false;
		// calculate fine
		var fine = player.credits * 0.05;
		if (fine > 5000) fine = 5000;

		// main station
		if (station.isMainStation === true) {
			var msg = expandDescription("[galcop-docking-fine-body]", {fine:formatCredits(fine, true, true), stationname:station.displayName, sender:this._dutyOfficer});

			w.$createEmail({sender:expandDescription("[galcop-docking-fine-sender]"),
				subject:"Docking Penalty",
				date:global.clock.adjustedSeconds,
				message:msg,
				expiryDays:this._defaultExpiryDays
			});
			bSent = true;
		}
		// seedy space bar
		if (bSent === false && station.hasRole("random_hits_any_spacebar")) {
			var msg = expandDescription("[ssb-docking-fine-body]", {fine:formatCredits(fine, true, true), stationname:station.displayName, sender:this._dutyOfficer});

			w.$createEmail({sender:expandDescription("[ssb-docking-fine-sender]"),
				subject:"Docking Penalty",
				date:global.clock.adjustedSeconds,
				message:msg,
				expiryDays:this._defaultExpiryDays
			});
			bSent = true;
		}
		// blackmonk monestary
		if (bSent === false && station.hasRole("blackmonk_monastery")) {
			var msg = expandDescription("[bmm-docking-fine-body]", {fine:formatCredits(fine, true, true), stationname:station.displayName, sender:this._dutyOfficer});

			w.$createEmail({sender:expandDescription("[bmm-docking-fine-sender]"),
				subject:"Docking Penalty",
				date:global.clock.adjustedSeconds,
				message:msg,
				expiryDays:this._defaultExpiryDays
			});
			bSent = true;
		}
		// constore
		if (bSent === false && station.hasRole("constore")) {
			var msg = expandDescription("[constore-docking-fine-body]", {fine:formatCredits(fine, true, true), stationname:station.displayName, sender:this._dutyOfficer});

			w.$createEmail({sender:expandDescription("[constore-docking-fine-sender]"),
				subject:"Docking Penalty",
				date:global.clock.adjustedSeconds,
				message:msg,
				expiryDays:this._defaultExpiryDays
			});
			bSent = true;
		}
		// rescue station
		if (bSent === false && station.hasRole("rescue_station")) {
			var msg = expandDescription("[rs-docking-fine-body]", {fine:formatCredits(fine, true, true), stationname:station.displayName, sender:this._dutyOfficer});

			w.$createEmail({sender:expandDescription("[rs-docking-fine-sender]"),
				subject:"Docking Penalty",
				date:global.clock.adjustedSeconds,
				message:msg,
				expiryDays:this._defaultExpiryDays
			});
			bSent = true;
		}
		// other?
		if (bSent === false) {
			var msg = expandDescription("[generic-docking-fine-body]", {fine:formatCredits(fine, true, true), stationname:station.name, sender:this._dutyOfficer});

			w.$createEmail({sender:expandDescription("[generic-docking-fine-sender]"),
				subject:"Docking Penalty",
				date:global.clock.adjustedSeconds,
				message:msg,
				expiryDays:this._defaultExpiryDays
			});
		}
	}

	// general fine notification/payment
	if (this._disableFines === false && player.ship.markedForFines === true && station.isMainStation && station.suppressArrivalReports === false && system.sun.isGoingNova === false && player.bounty < (50 - (system.info.government * 6))) {
		// calculate what the fine is
		var gov = 0;
		var calc_fine = 0;
		if (global.system.ID === -1) {
			gov = 1;
		} else {
			gov = system.government;
		}
		calc_fine = 50 + ((gov < 2 || gov > 5) ? 50 : 0);
		calc_fine *= player.bounty;
		if (calc_fine > player.credits) {
			calc_fine = player.credits;
		}

		if (calc_fine > 0) {
			var msg = expandDescription("[marked-for-fines-intro]", {fine:formatCredits(calc_fine, true, true)});
			if (player.bounty >= 50) {
				msg += expandDescription("[marked-for-fines-fugitive]");
			}
			msg += expandDescription("[marked-for-fines-end]");
			w.$createEmail({sender:expandDescription("[marked-for-fines-sender]"),
				subject:"Fine payment",
				date:global.clock.adjustedSeconds,
				message:msg,
				expiryDays:this._defaultExpiryDays
			});
		}
	}

	this._playerEjected = false;
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged = function(to, from) {

	// make a note of when we reach the shipyafrds screen
	if (guiScreen === "GUI_SCREEN_SHIPYARD" && this._disableNewShip === false) {
		var p = player.ship;
		this._playerEnteringShipyard = true;
		this._oldcredits = player.credits;

		// calculate price of current ship
		// reproduction of core code
		this._storedShipCost = this.$cunningFee((((p.price - (p.price * 0.006 * this.$missingSubEntitiesAdjustment())) * 75 * p.serviceLevel) + 5000) / 10000 , 0.005);

		// calculate price of stock in hold
		this._storedManifestCost = 0;
		for (var i = 0; i < manifest.list.length; i++) {
			this._storedManifestCost += Math.round(manifest.list[i].quantity * (p.dockedStation.market[manifest.list[i].commodity].price / 10) * 100) / 100;
		}
	}
	if (guiScreen != "GUI_SCREEN_SHIPYARD" && guiScreen != "GUI_SCREEN_STATUS" && this._playerEnteringShipyard === true) {
		this._playerEnteringShipyard = false;
	}

	// sending an email at the start of a new game, welcome player to their just-purchased ship
	// note, because of the way Hardships works, it's hard to know when the player has finished looking at ships and testing them
	// so, if you have hardships installed, you won't get a welcome to your new ship email. Standard purchasing will still give an email
	if (guiScreen === "GUI_SCREEN_STATUS" && player.name === "Jameson" && !worldScripts.hardships) {
		if (global.clock.clockStringForTime(global.clock.seconds).indexOf("2084004:20") === 0 && !missionVariables.EmailSystem_newGame) {
			// this is a new game, send new ship email
			missionVariables.EmailSystem_newGame = 1;
			this.$sendNewShipEmail(player.ship);
		}
		if (global.clock.clockStringForTime(global.clock.seconds).indexOf("2084004:20") === -1) {
			// clean up
			delete missionVariables.EmailSystem_newGame;
		}
	}

	// send a pilot licence email
	if (guiScreen === "GUI_SCREEN_STATUS" && player.name === "Jameson") {
		if (global.clock.clockStringForTime(global.clock.seconds).indexOf("2084004:20") === 0 && this._licenceNumber === "") {
			this._licenceNumber = this.$generateLicenceNumber();
			// pilot licence notification arrives shortly after the start
			var w = worldScripts.EmailSystem;
			w.$createEmail({sender:expandDescription("[new-licence-sender]"),
				subject:expandDescription("New Pilot Registration"),
				date:global.clock.seconds + 40,
				message:expandDescription("[new-licence]", {licenceno:this._licenceNumber}),
				sentFrom:this._galCopHQ[galaxyNumber]
			});
		}
	}

	// make a note of the credit balance
	if(to && to === "GUI_SCREEN_EQUIP_SHIP") {
		this._oldcredits = player.credits;
		this._oldPassengerCount = player.ship.passengerCapacity;
		// if the Battle damage OXP is present, check for the HULL REPAIR item. If it's not on the player ship, there is hull damage to repair
		var w = worldScripts["Battle Damage"];
		if (w && missionVariables.BattleDamage_status === "DAMAGED") {
			// there is hull damage to repair
			this._repairHullDamage = true;
		}
		// if the Iron Hide OXP is present, check for damage so we can include an item in the email
		w = worldScripts["IronHide Armour Script"];
		if(w) {
			if ((player.ship.equipmentStatus("EQ_IRONHIDE") === "EQUIPMENT_OK" || player.ship.equipmentStatus("EQ_IRONHIDE_MIL") === "EQUIPMENT_OK") && missionVariables.ironHide_percentage < 100) {
				this._repairIronHide = true;
			}
		}

		// store equipment status of all items so we can tell when it's a repair job
		var p = player.ship;
		var eq = p.equipment;
		for (var i = 0; i < eq.length; i++) {
			var q = eq[i];
			this._equipItems.push({key:q.equipmentKey, status:p.equipmentStatus(q.equipmentKey)});
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// special case for IronHide
this.missionScreenEnded = function() {
	// if the ironhide armour gets repairs during the maintenance overhaul, it calls a mission screen
	// this will run when the mission screen ends
	if (this._sendMaintEmail === true) {
		this._sendMaintEmail = false;
		this._maintCost = this._maintCostInit - player.credits;
		this.$sendMaintenanceEmail()
	}
	// if the ironhide armour is repaired on its own, it still calls a mission screen
	// this will run when that screen ends
	if (this._sendRepairEmail === true) {
		this._sendRepairEmail = false;
		var msg = expandDescription("[purchase-maintenance]",
			{shipname:player.ship.shipClassName,
			maintenanceitems:"\n\n- Repair of IronHide armour",
			servicecost:formatCredits(this._maintCostInit - player.credits, true, true),
			costnote:"",
			sender:this._maintRep
		});

		var w = worldScripts.EmailSystem;
		w.$createEmail({sender:expandDescription("[purchase-maintenance-sender]"),
			subject:"Repair (Invoice #" + (this.$rand(500000) + 100000).toString() + ")",
			date:global.clock.adjustedSeconds,
			message:msg,
			expiryDays:this._defaultExpiryDays
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipLaunchedEscapePod = function(pod, pass) {
	this._playerEjected = true;
}

//-------------------------------------------------------------------------------------------------------------
this.playerBoughtEquipment = function(equipment) {
	if (this._debug) log(this.name, "equipment " + equipment);
	// start a timer to do all the bought equipment work
	// this is to ensure any OXP processes have completed (like refunding) before we do out calculations
	this._boughtKey = equipment
	this._boughtStn = player.ship.dockedStation;
	// make sure the timer isn't already running
	if (this._boughtEquipTimer && this._boughtEquipTimer.isRunning) this._boughtEquipTimer.stop();
	this._boughtEquipTimer = new Timer(this, this.$playerBoughtEquip, 1, 0);
}

//-------------------------------------------------------------------------------------------------------------
this.$playerBoughtEquip = function $playerBoughtEquip() {

	var equipment = this._boughtKey;
	if (this._debug) log(this.name, "bought key = " + equipment);
	// work out where the purchase occurred
	var stn = null;
	if (this._boughtStn) {
		stn = this._boughtStn;
	} else {
		if (player.ship.dockedStation) {
			stn = player.ship.dockedStation;
		} else {
			stn = system.mainStation;
		}
	}
	this._boughtStn = null;

	// don't do anything for these items
	if (this._purchase_ignore_equip.indexOf(equipment) >= 0) return;

	var w = worldScripts.EmailSystem;

	if (this._shipSalesRep === "") {
		this.$setupRepNames(stn);
	}

	if (equipment !== "EQ_RENOVATION") {
		// purchasing items
		var etype = "purchase";
		var subtype = 0;
		var eq = EquipmentInfo.infoForKey(equipment);

		// work out what sort of purchase this was: standard purchase, repair or removal

		// we could be repairing something so check the statuses we saved when we switched to the equip ship screen
		if (this._equipItems && this._equipItems.length > 0) {
			for (var i = 0; i < this._equipItems.length; i++) {
				if (this._equipItems[i].key === equipment && this._equipItems[i].status === "EQUIPMENT_DAMAGED") etype = "repair";
			}
		}
		// if the equipment key itself has "REPAIR" in it, switch to repair mode
		if (this._maint_repair_equip.indexOf(equipment) >= 0 || (equipment.indexOf("REPAIR") >= 0 && equipment.indexOf("REPAIRBOTS") === 0)) etype = "repair";
		// if the equipment key itself has "REMOVAL" in it, switch to removal mode
		var desc = eq.description.toLowerCase();
		if (this._maint_remove_equip.indexOf(equipment) >= 0 || 
			(desc.indexOf("remove") >= 0 || 
				(desc.indexOf("sell ") >= 0 && desc.indexOf("buy ") === -1) || 
				desc.indexOf("unmount") >= 0 || 
				desc.indexOf("undo ") >= 0 || 
				equipment.indexOf("REFUND") >= 0))
			{etype = "remove"; subtype = 1;}
		// if the player doesn't have the equipment anymore, it must be a removal
		// this should really only be triggered by an external pack calling the playerBoughtEquipment routine manually (see Ship Configuration for an example)
		if (etype === "purchase") {
			if (equipment != "EQ_PASSENGER_BERTH" && player.ship.equipmentStatus(equipment) !== "EQUIPMENT_OK" && this._purchase_removed.indexOf(equipment) === -1) etype = "remove";
		}

		var extra = "";
		var msg = "";
		var itemname = "";

		// see if there is a replacement description for this piece of equipment
		if (this._maintItemNameReplacements.indexOf(equipment) >= 0) {
			itemname = expandDescription("[maint_itemname_" + equipment + "]");
			if (!itemname || itemname === "" || itemname.indexOf("[maint_") >= 0) {
				itemname = eq.name;
			}
		} else {
			itemname = eq.name;
		}

		if (this._debug) {
			log(this.name, "etype = " + etype);
			log(this.name, "subtype = " + subtype);
			log(this.name, "old creds = " + this._oldcredits);
			log(this.name, "curr creds = " + player.credits);
		}

		switch (etype) {
			case "remove":
				if (this._disableEquipPurchase === false) {
					var cost = "";
					// were we charged something, or did we get a refund?
					if (eq.price != 0) {
						if ((this._oldcredits - player.credits) < 0) {
							if (subtype === 1) {
								cost = "The cost of this service was " + formatCredits(eq.price / 10, true, true) + ", but all together you were refunded " + formatCredits(player.credits - this._oldcredits, true, true) + ".";
							} else {
								cost = "In total you were refunded " + formatCredits(player.credits - this._oldcredits, true, true) + ".";
							}
						} else {
							cost = "You were charged " + formatCredits(this._oldcredits - player.credits, true, true) + ".";
						}
					} else {
						if ((this._oldcredits - player.credits) < 0) {
							cost = "In total you were refunded " + formatCredits(player.credits - this._oldcredits, true, true) + ".";
						} else {
							cost = "There was no charge for this service.";
						}
					}

					msg = expandDescription("[remove-equip]",
						{stationname:stn.displayName,
						equipname:itemname,
						equipcost:cost,
						sender:this._equipSalesRep
					});

					w.$createEmail({sender:expandDescription("[purchase-equip-sender]"),
						subject:"Removing equipment",
						date:global.clock.adjustedSeconds,
						message:msg,
						expiryDays:this._defaultExpiryDays
					});

					// special case for escape pod
					if (equipment === "EQ_ESCAPE_POD") {
						msg = expandDescription("[escapepod-sold-body]",
							{sender:randomName() + " " + randomName()
						});

						w.$createEmail({sender:expandDescription("[escapepod-insurance-sender]"),
							subject:"Contract termination",
							date:global.clock.adjustedSeconds + 150,
							message:msg,
							expiryDays:this._defaultExpiryDays
						});
					}
				}
				break;
			case "purchase":
				if (this._disableEquipPurchase === false) {
					var paid = ((eq.price / 10) * stn.equipmentPriceFactor);
					if (this._oldcredits != player.credits + ((eq.price / 10) * stn.equipmentPriceFactor)) {
						paid = (this._oldcredits - player.credits);
						// did we get a refund for an existing weapon?
						if (equipment.indexOf("EQ_WEAPON") >= 0) {
							extra = "(Note: You were refunded the cost of any laser that was removed in order to install your new one.)";
							// switch back to the direct equipment price, otherwise the email text gets confusing
							paid = ((eq.price / 10) * stn.equipmentPriceFactor);
						}
					}

					if (this._debug) log(this.name, "paid = " + paid);

					msg = expandDescription("[purchase-equip]",
						{stationname:stn.displayName,
						extranote:extra,
						equipname:itemname + ": " + eq.description,
						equipcost:"The cost of this item was " + formatCredits(paid, true, true),
						sender:this._equipSalesRep
					});

					// make the missle purchase email more reasonable.
					if (equipment.indexOf("MISSILE") >= 0) {
						msg = msg.replace("We hope you will enjoy many years of trouble-free use of this item.", "We hope this item performs flawlessly for you when you need it.");
					}

					w.$createEmail({sender:expandDescription("[purchase-equip-sender]"),
						subject:"Purchasing equipment",
						date:global.clock.adjustedSeconds,
						message:msg,
						expiryDays:this._defaultExpiryDays
					});

					// special case for escapepods - send insurance application notification
					if (equipment === "EQ_ESCAPE_POD") {
						msg = expandDescription("[escapepod-purchase-body]",
							{sender:randomName() + " " + randomName()
						});

						w.$createEmail({sender:expandDescription("[escapepod-insurance-sender]"),
							subject:"Insurance Application",
							date:global.clock.adjustedSeconds + 300,
							message:msg,
							expiryDays:this._defaultExpiryDays
						});
					}
				}
				break;

			case "repair":
				if (this._disableMaintenance === false) {
					if (equipment === "EQ_IRONHIDE_REPAIR") {
						this._sendRepairEmail = true;
						this._maintCostInit = this._oldcredits;
					} else {
						if ((this._oldcredits - player.credits) > 0) {
							var maint = "";
							if (eq.name.toLowerCase().indexOf("repair") >= 0) {
								maint = "\n\n- " + eq.name;
							} else {
								maint = "\n\n- Repair of " + eq.name;
							}

							// special cases - ShipVersion OXP
							if (equipment === "EQ_TURRET_RECOVER") maint = "\n\n- Recovery of destroyed turrets";
							if (equipment === "EQ_SHIP_VERSION_REPAIR") maint = "\n\n- Restoring the recharge bonuses provided by Ship Version equipment.";

							msg = expandDescription("[purchase-maintenance]",
								{shipname:player.ship.shipClassName,
								maintenanceitems:maint,
								servicecost:formatCredits(this._oldcredits - player.credits, true, true),
								costnote:"",
								sender:this._maintRep
							});

							var w = worldScripts.EmailSystem;
							w.$createEmail({sender:expandDescription("[purchase-maintenance-sender]"),
								subject:"Repair (Invoice #" + (this.$rand(500000) + 100000).toString() + ")",
								date:global.clock.adjustedSeconds,
								message:msg,
								expiryDays:this._defaultExpiryDays
							});
						}
					}
				}
				break;
		}

		if (equipment === "EQ_IRONHIDE_REPAIR") this._repairIronHide = false;
		if (equipment === "EQ_HULL_REPAIR") this._repairHullDamage = false;

	} else {
		if (this._disableMaintenance === false) {
			if (this._repairIronHide === false) {
				// if there's no ironhide armour repair, send the email immediately
				this._maintCost = this._oldcredits - player.credits;
				this.$sendMaintenanceEmail();
			} else {
				// otherwise set up params for sending email later
				this._sendMaintEmail = true;
				this._maintCostInit = this._oldcredits;
			}
		}
	}

	this._oldcredits = player.credits;
}

//-------------------------------------------------------------------------------------------------------------
this.playerBoughtNewShip = function(ship, price) {
	if (this._playerEnteringShipyard === true && this._disableNewShip === false) {
		this._newShipCost = (this._oldcredits + this._storedShipCost + this._storedManifestCost) - player.credits;
		this._playerEnteringShipyard = false;
		this.$sendNewShipEmail(ship);
	}
	this._oldcredits = player.credits;
}

//-------------------------------------------------------------------------------------------------------------
this.shipLaunchedFromStation = function(station) {
	// make a note of the rank we have on launch
	this._oldrank = expandDescription("[commander_rank]");
	// make a note of the system we launch in
	this._bountySystem = system.name;
}

//-------------------------------------------------------------------------------------------------------------
this.shipKilledOther = function(whom, damageType) {
	// don't record anything if this is a siumulator run
	if (this.$simulatorRunning()) return;
	if (whom.isMissile) {
		this._killAssortedCount += 1;
		this._bountyEmailTime = global.clock.seconds;
	}
	if (!whom.hasRole("escape-capsule") && (whom.isPiloted || whom.isDerelict || whom.hasRole("tharglet") || whom.hasRole("thargon"))) {
		// increment the kill counter
		this._killCount += 1;
		this._bountyEmailTime = global.clock.seconds;
		// but was their a bounty?
		if (whom.bounty > 0) {
			// add a new item to the bounty email body
			var true_bounty = whom.bounty;
			if (whom.isCargo || whom.scanClass == "CLASS_BUOY" || whom.scanClass == "CLASS_ROCK") true_bounty = whom.bounty / 10;
			this._bountyEmailBody += "~- Destruction of " + whom.displayName + " for " + formatCredits(true_bounty, false, true) + ".\n";
			this._bountyEmailTotalCR += true_bounty;
		} else {
			// add a no bounty item to the email body
			this._bountyEmailBody += "~- Destruction of " + whom.displayName + " (no bounty).\n";
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipExitedWitchspace = function() {
	// if we enter a system while there is some bounty email body hanging around, put it into an array so we can itemise it later
	if (this._bountyEmailBody != "") {
		if (!this._holdBountyEmail) {
			this._holdBountyEmail = [];
		}
		this._holdBountyEmail.push({bountyEmail:this._bountyEmailBody, bountySystem:this._bountySystem});
		this._bountyEmailBody = "";
	}
	this._bountySystem = system.name;
}

//-------------------------------------------------------------------------------------------------------------
// bounty on scooped escape capsule (1.81 only)
this.playerRescuedEscapePod = function(fee, reason, occupant) {

	var w = worldScripts.EmailSystem;

	var msg = "";
	var subj = "";
	var sndr = "";
	switch (reason) {
		case "insurance":
			sndr = expandDescription("[escapepod-insurance-sender]");
			subj = "Salvage fee for escape pod recovery";
			msg = expandDescription("[escapepod-insurance-body]",
				{occupant:occupant.name,
				description:occupant.description,
				fee:formatCredits(fee / 10, true, true),
				sender:this._insuranceOfficer
			});
			break;
		case "bounty":
			sndr = expandDescription("[galcop-bounty-sender]");
			subj = "Bounty for escape pod recovery";
			msg = expandDescription("[escapepod-bounty-body]",
				{occupant:occupant.name,
				description:occupant.description,
				fee:formatCredits(fee / 10, true, true),
				sentFrom:this._galCopHQ[galaxyNumber],
				sender:this._galCopBountyOfficer
			});
			break;
		//case "slave":
	}
	if (sndr != "" && this._disableEscapePods === false) {
		w.$createEmail({sender:sndr,
			subject:subj,
			date:global.clock.seconds,
			message:msg,
			expiryDays:this._defaultExpiryDays
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// send email to player when a contract is started. (1.81 only)
this.playerEnteredContract = function(type, contract) {

	var w = worldScripts.EmailSystem;

	var msg = "";
	var subj = "";
	var sndr = "";

	switch (type) {
		case "cargo":
			sndr = expandDescription("[cargo-contract-sender]");
			subj = "Accepted contract: " + contract.cargo_description;
			msg = expandDescription("[cargo-contract-start]",
				{description:contract.cargo_description,
				systemname:System.systemNameForID(contract.destination),
				time:global.clock.clockStringForTime(contract.arrival_time),
				fee:formatCredits(contract.fee, true, true)
			});
			break;
		case "passenger":
			sndr = expandDescription("[passenger-contract-sender]");
			subj = "Accepted contract: " + contract.name;
			msg = expandDescription("[passenger-contract-start]",
				{contractname:contract.name,
				systemname:System.systemNameForID(contract.destination),
				time:global.clock.clockStringForTime(contract.arrival_time),
				fee:formatCredits(contract.fee, true, true)
			});
			break;
		case "parcel":
			sndr = expandDescription("[parcel-contract-sender]");
			subj = "Accepted contract: " + contract.name;
			msg = expandDescription("[parcel-contract-start]",
				{contractname:contract.name,
				systemname:System.systemNameForID(contract.destination),
				time:global.clock.clockStringForTime(contract.arrival_time),
				fee:formatCredits(contract.fee, true, true)
			});
			break;
	}

	if (sndr != "" && this._disableContracts === false) {
		w.$createEmail({sender:sndr,
			subject:subj,
			date:global.clock.adjustedSeconds,
			message:msg
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// result of parcel/passenger/cargo contract (1.81 only)
this.playerCompletedContract = function(type, result, fee, contract) {

	var w = worldScripts.EmailSystem;

	if (this._disableContracts === false) {
		var msg = "";
		var subj = "";
		var sndr = "";

		sndr = expandDescription("[" + type + "-contract-sender]");
		msg = expandDescription("[" + type + "-contract-" + result + "]",
			{description:contract.cargo_description,
			contractname:contract.name,
			systemname:System.systemNameForID(contract.destination),
			time:global.clock.clockStringForTime(contract.arrival_time),
			fee:formatCredits(fee / 10, true, true)
		});
		subj = expandDescription("[" + type + "-contract-" + result + "-subject]");

		w.$createEmail({sender:sndr,
			subject:subj,
			date:global.clock.adjustedSeconds,
			message:msg,
			expiryDays:this._defaultExpiryDays
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// sends the maintenance email
this.$sendMaintenanceEmail = function() {

	var maint = "";
	var item = "";
	var count = 0;
	var hasLCB = false;

	var extra = "";
	if (this._repairIronHide === true) extra = " (includes cost of IronHide armour repair)";

	// add common items
	maint += "\n\n~General ship maintenance tasks\n";
	maint += expandDescription("[maintitem_all]");

	// add general items
	do {
		item = expandDescription("[maintitem_general]");
		if (item.trim() != "") {
			// because the first word of the item can be random, check if everything after the first word exists in our maintenance list yet
			// only add it if it doesn't exist
			if (maint.indexOf(item.substring(item.indexOf(" ")).trim()) === -1) {
				maint += "\n- " + item;
				count += 1;
			}
		}
	} while (count < system.info.techlevel);

	var p = player.ship;
	var itemname = "";
	var diagtype = expandDescription("[maintperform]");

	maint += "\n\n~Equipment specific maintenance tasks"
	// if the player ship hyperspace capable?
	if (p.hasHyperspaceMotor) {
		maint += "\n- " + diagtype + " diagnostics on Witchspace Drive";
		maint += "\n- " + expandDescription("[maint_Hyperspace]");
	}

	// get list of equipment keys to ignore (hull damage equipment item from the Battle Damage OXP, IronHide item, and any cats/dogs from Ships Cat OXP)
	// add equipment specific items;
	for (var i = 0; i < p.equipment.length; i++) {
		var e_item = p.equipment[i];
		if (this._maint_known_equip.indexOf(e_item.equipmentKey) >= 0) {
			// can this item be repaired in this system?
			if ((e_item.techLevel - 1) <= system.info.techlevel && e_item.isVisible) {
				// is this piece of equipment OK (don't do maintenance on damaged equipment)
				if (p.equipmentStatus(e_item.equipmentKey) === "EQUIPMENT_OK") {
					// add a diagnostic entry for all items
					itemname = e_item.name;

					// check if we have a large cargo bay - we'll use this later on
					if (e_item.equipmentKey === "EQ_CARGO_BAY") hasLCB = true;

					// special case for passenger berth, as its name has extraneous text in it
					if (e_item.equipmentKey === "EQ_PASSENGER_BERTH") itemname = "Passenger Berth";

					maint += "\n- " + diagtype + " diagnostics on " + itemname;

					// get a maintenance item for this equipment
					item = expandDescription("[maint_" + e_item.equipmentKey + "]");
					// do we have a real value? if so, add it to the list
					if (item.indexOf("[") === -1 && item.trim() != "") maint += "\n- " + item;
				} else if (p.equipmentStatus(e_item.equipmentKey) === "EQUIPMENT_DAMAGED") {
					// this item is damaged - just note it in the message
					// add a diagnostic entry for all items
					itemname = e_item.name;

					// special case for passenger berth, as its name has extraneous text in it
					if (e_item.equipmentKey === "EQ_PASSENGER_BERTH") itemname = "Passenger Berth";

					item = "Identified repairs required for " + itemname;
					maint += "\n- " + item;
				}
			}
		}
	}

	// did the player repair the ironhide armour?
	if ((p.equipmentStatus("EQ_IRONHIDE") === "EQUIPMENT_OK" || p.equipmentStatus("EQ_IRONHIDE_MIL") === "EQUIPMENT_OK") && missionVariables.ironHide_percentage === 100 && this._repairIronHide === true) {
		// player repaired ironhide armour
		maint += "\n- Repair IronHide armour";
		this._repairIronHide = false;
	}

	// for battle damage OXP, add an item to show that the damage has been repaired
	if (this._repairHullDamage === true) {
		maint += "\n- Repair hull damage";
		this._repairHullDamage = false;
	}

	// do we have any cargo space but no LCB?
	if (p.cargoSpaceCapacity > 0 && hasLCB === false) {
		maint += "\n- " + expandDescription("[maint_CargoBay]");
	}

	// do we have any missiles?
	if (p.missileCapacity > 0) {
		maint += "\n- " + diagtype + " diagnostics on missile system";
		maint += "\n- " + expandDescription("[maint_MissileSystem]");
	}

	// weapon equipment items don't appear in the ship.equipment array, so we need to do an individual check for each one
	if (p.forwardWeapon && p.forwardWeapon.equipmentKey != "EQ_WEAPON_NONE" && this._maint_known_equip.indexOf(p.forwardWeapon.equipmentKey) >= 0) {
		if (p.forwardWeapon.techLevel <= (system.info.techlevel - 1)) {
			maint += "\n- " + diagtype + " diagnostics on front " + p.forwardWeapon.name;
			item = expandDescription("[maint_" + p.forwardWeapon.equipmentKey + "]", {pos:"front"});
			maint += "\n- " + item;
		}
	}
	if (p.aftWeapon && p.aftWeapon.equipmentKey != "EQ_WEAPON_NONE" && this._maint_known_equip.indexOf(p.aftWeapon.equipmentKey) >= 0) {
		if (p.aftWeapon.techLevel <= (system.info.techlevel - 1)) {
			maint += "\n- " + diagtype + " diagnostics on aft " + p.aftWeapon.name;
			item = expandDescription("[maint_" + p.aftWeapon.equipmentKey + "]", {pos:"aft"});
			maint += "\n- " + item;
		}
	}
	if (p.portWeapon && p.portWeapon.equipmentKey != "EQ_WEAPON_NONE" && this._maint_known_equip.indexOf(p.portWeapon.equipmentKey) >= 0) {
		if (p.portWeapon.techLevel <= (system.info.techlevel - 1)) {
			maint += "\n- " + diagtype + " diagnostics on port " + p.portWeapon.name;
			item = expandDescription("[maint_" + p.portWeapon.equipmentKey + "]", {pos:"port"});
			maint += "\n- " + item;
		}
	}
	if (p.starboardWeapon && p.starboardWeapon.equipmentKey != "EQ_WEAPON_NONE" && this._maint_known_equip.indexOf(p.starboardWeapon.equipmentKey) >= 0) {
		if (p.starboardWeapon.techLevel <= (system.info.techlevel - 1)) {
			maint += "\n- " + diagtype + " diagnostics on starboard " + p.starboardWeapon.name;
			item = expandDescription("[maint_" + p.starboardWeapon.equipmentKey + "]", {pos:"starboard"});
			maint += "\n- " + item;
		}
	}

	maint += "\n- Ship detailing and valet service\n- Parts and labour";

	var msg = expandDescription("[purchase-maintenance]",
		{shipname:p.shipClassName,
		maintenanceitems:maint,
		servicecost:formatCredits(this._maintCost, true, true),
		costnote:extra,
		sender:this._maintRep
	});

	var w = worldScripts.EmailSystem;
	w.$createEmail({sender:expandDescription("[purchase-maintenance-sender]"),
		subject:"Overhaul (Invoice #" + (this.$rand(500000) + 100000).toString() + ")",
		date:global.clock.adjustedSeconds,
		message:msg,
		expiryDays:this._defaultExpiryDays
	});
}

//-------------------------------------------------------------------------------------------------------------
// sends new ship email
this.$sendNewShipEmail = function(ship) {

	// make sure we have created the rep names
	if (this._shipSalesRep === "") this.$setupRepNames();

	var equip = "";
	var specs = "";
	var inj = 7;
	if (0 >= oolite.compareVersion("1.81")) inj = ship.injectorSpeedFactor;

	var wpn = "\n     Front - ";
	if (ship.forwardWeapon && ship.forwardWeapon.equipmentKey != "EQ_WEAPON_NONE") {
		wpn += ship.forwardWeapon.name;
	} else {
		wpn += "None";
	}

	if (" 3 7 11 15 ".indexOf(" " + ship.weaponFacings.toString() + " ") >= 0) {
		wpn += "\n     Aft - ";
		if (ship.aftWeapon && ship.aftWeapon.equipmentKey != "EQ_WEAPON_NONE") {
			wpn += ship.aftWeapon.name;
		} else {
			wpn += "None";
		}
	}
	if (" 5 7 13 15 ".indexOf(" " + ship.weaponFacings.toString() + " ") >= 0) {
		wpn += "\n     Port - ";
		if (ship.portWeapon && ship.portWeapon.equipmentKey != "EQ_WEAPON_NONE") {
			wpn += ship.portWeapon.name;
		} else {
			wpn += "None";
		}
	}
	if (" 9 11 13 15 ".indexOf(" " + ship.weaponFacings.toString() + " ") >= 0) {
		wpn += "\n     Starboard - ";
		if (ship.starboardWeapon && ship.starboardWeapon.equipmentKey != "EQ_WEAPON_NONE") {
			wpn += ship.starboardWeapon.name;
		} else {
			wpn += "None";
		}
	}

	specs = "- Max speed/thrust: " + (ship.maxSpeed / 1000).toFixed(3) + "/" + (ship.maxThrust / 1000).toFixed(3) + " LS\n" +
		"- Injector speed: " + ((ship.maxSpeed * inj) / 1000).toFixed(3) + " LS\n" +
		"- Max pitch/roll/yaw: " + ship.maxPitch.toFixed(2) + "/" + ship.maxRoll.toFixed(2) + "/" + ship.maxYaw.toFixed(2) + "\n" +
		"- Laser mounts: " + wpn + "\n" +
		"- Missile pylons: " + ship.missileCapacity + "\n" +
		"- Cargo capacity: " + ship.cargoSpaceCapacity + " tons\n" +
		"- Max energy: " + (ship.maxEnergy) + " units\n" +
		"- Energy recharge rate: " + ship.energyRechargeRate.toFixed(2) + " units/sec\n" +
		"- Hyperspace capable: " + ship.hasHyperspaceMotor + "\n";

	if (ship.equipment.length > 0) {
		equip = "Your ship also came equipped with the following items:\n";
		var added = false;
		var e = ship.equipment;
		for (var i = 0; i < e.length; i++) {
			var q = e[i];
			if ("EQ_HULL_REPAIR".indexOf(q.equipmentKey) === -1) {
				if (q.isPortableBetweenShips === false && q.isVisible === true) {
					equip += "- " + q.name + "\n";
					added = true
				}
			}
		}
		if (added === false) {
			equip = "";
		} else {
			equip += "\n";
		}
	}

	var cost = ship.price;
	var extra = "";

	if (this._newShipCost) {
		cost = this._newShipCost;
		extra = " As part of this transaction your old ship was traded for " + formatCredits(this._storedShipCost, false, true);
		if (this._storedManifestCost > 0) extra += ", and the content of your hold was traded for " + formatCredits(this._storedManifestCost, true, true);
		extra += ".";
	}

	var msg = expandDescription("[purchase-ship]",
		{stationname:player.ship.dockedStation.displayName,
		shipclass:ship.shipClassName,
		shipspecs:specs,
		shipequip:equip,
		shipcost:formatCredits(cost, false, true),
		extracost:extra,
		rndsystem:System.systemNameForID(this.$rand(256)-1),
		sender:this._shipSalesRep
	});

	var w = worldScripts.EmailSystem;
	w.$createEmail({sender:expandDescription("[purchase-ship-sender]"),
		subject:"Purchase of new ship",
		date:global.clock.adjustedSeconds,
		message:msg
	});

	// transfer of ownership email arrives shortly after the first one.
	w.$createEmail({sender:"GalCop Ship Registry",
		subject:"Transfer of ownership",
		date:global.clock.adjustedSeconds + 25,
		message:expandDescription("[new-ship-transfer]")
	});
}

//-------------------------------------------------------------------------------------------------------------
// return a random number between 1 and max
this.$rand = function(max) {
	return Math.floor((Math.random() * max) + 1)
}

//-------------------------------------------------------------------------------------------------------------
// generates a random licence number in the format "GPL000000-XX000-XX0000"
this.$generateLicenceNumber = function() {

	var ln = "GPL";

	ln += (this.$rand(800000) + 100000);
	ln += "-" + expandDescription("[licencecodes]");
	ln += (this.$rand(500) + 200);
	ln += "-" + expandDescription("[licencecodes]");
	ln += (this.$rand(8000) + 1000);

	return ln;
}

//-------------------------------------------------------------------------------------------------------------
// set up and store various names for inclusion in emails
this.$setupRepNames = function(station) {
	if (!station && !player.ship.dockedStation) station = system.mainStation;
	if (!station && player.ship.dockedStation) station = player.ship.dockedStation;

	// if we just redocked at the same station, don't change anything
	if (this._namesLocation === system.name + "-" + station.displayName) return;

	this._namesLocation = system.name + "-" + station.displayName;

	this._shipSalesRep = randomName() + " " + randomName();
	this._equipSalesRep = randomName() + " " + randomName();
	this._maintRep = randomName() + " " + randomName();
	this._dutyOfficer = randomName() + " " + randomName();
	this._insuranceOfficer = randomName() + " " + randomName();
	// bounty processing occurs in a central location, so only update the officer occasionally
	if (this._galCopBountyOfficer === "") {
		this._galCopBountyOfficer = randomName() + " " + randomName();
	} else {
		if (this.$rand(20) > 15) this._galCopBountyOfficer = randomName() + " " + randomName();
	}
}

//-------------------------------------------------------------------------------------------------------------
// reproductions of core code to try and work out an accurate trade in value for the player's ship
this.$cunningFee = function(value, precision) {
	var fee = value;
	var superfee = 100000.0;
	var max = 1 + precision;
	var min = 1 - precision;
	var rounded_fee = superfee * Math.floor(0.5 + fee / superfee);
	if (rounded_fee === 0) rounded_fee = 1;
	var ratio = fee / parseFloat(rounded_fee);
	while ((ratio < min || ratio > max) && superfee > 1)
	{
		rounded_fee = superfee * Math.floor(0.5 + fee / superfee);
		if (rounded_fee === 0) rounded_fee = 1;
		ratio = fee / parseFloat(rounded_fee);
		superfee /= 10.0;
	}
	if (ratio > min && ratio < max) fee = rounded_fee;
	return fee;
}

//-------------------------------------------------------------------------------------------------------------
this.$missingSubEntitiesAdjustment = function() {
	// each missing subentity depreciates the ship by 5%, up to a maximum of 35% depreciation.
	var percent = 5 * (player.ship.subEntityCapacity - player.ship.subEntities.length);
	return (percent > 35 ? 35 : percent);
}

//-------------------------------------------------------------------------------------------------------------
this.$simulatorRunning = function() {
	var w = worldScripts["Combat Simulator"];
	if (w && w.$checkFight && w.$checkFight.isRunning) return true;
	return false;
}
