"use strict";

// worldScripts.world_builder

this.name = "world_builder";

this.infoSystemChanged = this.shipWillExitWitchspace = function () {
    this._applyInterface();
};

this.startUp = function () {
    this._applyInterface();
    delete this.startUp;
};

this._applyInterface = function () {
    if (system.mainStation) {
        system.mainStation.setInterface("world_builder", {
            title: "World Builder",
            category: "Uncategorised",
            summary: "Improve a world, any world. Currently improving " + System.infoForSystem(galaxyNumber, player.ship.infoSystem).name,
            callback: this._changeWorld.bind(this)
        });
    }
};

this.propertyToChange = null;

this._changeWorld = function () {
    mission.runScreen({
        title: "World Builder (Step 1 of 2)",
        message: "Enter the property to change (case-sensitive). Below are some of the available properties:\n\n" +
                 "1. name\n" +
                 "2. economy\n" +
                 "3. economy_description\n" +
                 "   - Changing one may not automatically update the other. Ensure consistency when modifying these values.\n" +
                 "4. government\n" +
                 "5. government_description\n" +
                 "   - Similar to economy, changing one may not update the other. Keep them in sync manually.\n" +
                 "6. techlevel\n" +
                 "7. population\n" +
                 "8. population_description\n" +
                 "9. inhabitants\n" +
                 "10. productivity\n" +
                 "11. radius\n" +
                 "12. description",
        screenID: "world_builder",
        exitScreen: "GUI_SCREEN_SYSTEM_DATA",
        allowInterrupt: true,
        music: "dt_manipulating_property_values.ogg",
        textEntry: true
    }, this._changeWorld2.bind(this));
};

this._changeWorld2 = function (propertyToChange) {
    if (propertyToChange && propertyToChange !== "") {
        // Check if the input is a positive integer
        var numericInput = parseInt(propertyToChange, 10);
        if (!isNaN(numericInput) && numericInput > 0) {
            // Map the numeric input to the corresponding property name
            switch (numericInput) {
                case 1:
                    this.propertyToChange = "name";
                    break;
                case 2:
                    this.propertyToChange = "economy";
                    break;
                case 3:
                    this.propertyToChange = "economy_description";
                    break;
                case 4:
                    this.propertyToChange = "government";
                    break;
                case 5:
                    this.propertyToChange = "government_description";
                    break;
                case 6:
                    this.propertyToChange = "techlevel";
                    break;
                case 7:
                    this.propertyToChange = "population";
                    break;
                case 8:
                    this.propertyToChange = "population_description";
                    break;
                case 9:
                    this.propertyToChange = "inhabitants";
                    break;
                case 10:
                    this.propertyToChange = "productivity";
                    break;
                case 11:
                    this.propertyToChange = "radius";
                    break;
                case 12:
                    this.propertyToChange = "description";
                    break;
                default:
                    // Invalid numeric input
                    mission.runScreen({
                        title: "World Builder",
                        message: "Invalid numeric input. Please enter a valid property name or number.",
                        exitScreen: "GUI_SCREEN_SYSTEM_DATA",
                        allowInterrupt: true,
                        screenID: "world_builder"
                    });
                    return; // Exit early to avoid proceeding with invalid input
            }
        } else {
            // Use the input as-is (assuming it's a property name)
            this.propertyToChange = propertyToChange;
        }
    }

    // Proceed to Step 2: Enter new value for the property
    var parameters = {
        title: "World Builder (Step 2 of 2)",
        message: "Select new value for property.",
        screenID: "world_builder",
        exitScreen: "GUI_SCREEN_SYSTEM_DATA",
        allowInterrupt: true,
        textEntry: true
    }

    // Provide property-specific help messages
    switch (this.propertyToChange) {
        case "name":
            parameters.message = "Enter a new value for 'name'.\n" +
                                 "- A text string naming the planet.\n" +
                                 "- Example: 'Alpha Centauri', 'Tionisla'.";
            break;

        case "economy":
            parameters.message = "Enter a new value for 'economy'.\n" +
                                 "- Valid range: 0 to 7 (integer).\n" +
                                 "- Represents the economic type of the system.\n" +
                                 "- Default descriptions, if economy_description is not set:\n" +
                                 "- 0: Rich Industrial\n" +
                                 "- 1: Average Industrial\n" +
                                 "- 2: Poor Industrial\n" +
                                 "- 3: Mainly Industrial\n" +
                                 "- 4: Mainly Agricultural\n" +
                                 "- 5: Rich Agricultural\n" +
                                 "- 6: Average Agricultural\n" +
                                 "- 7: Poor Agricultural";
            break;

        case "economy_description":
            parameters.message = "Enter a new value for 'economy_description'.\n" +
                                 "- A free-form text string describing the economy.\n" +
                                 "- Example: 'Rich Industrial', 'Average Agricultural'.";
            break;

        case "government":
            parameters.message = "Enter a new value for 'government'.\n" +
                                 "- Valid range: 0 to 7 (integer).\n" +
                                 "- Represents the government type of the system.\n" +
                                 "- Example: 0 = Anarchy, 7 = Corporate State.\n" +
                                 "- Default descriptions, if government_description is not set:\n" +
                                 "- 0: Anarchy\n" +
                                 "- 1: Feudal\n" +
                                 "- 2: Multi-Government\n" +
                                 "- 3: Dictatorship\n" +
                                 "- 4: Communist\n" +
                                 "- 5: Confederacy\n" +
                                 "- 6: Democracy\n" +
                                 "- 7: Corporate State";
            break;

        case "government_description":
            parameters.message = "Enter a new value for 'government_description'.\n" +
                                 "- A free-form text string describing the government.\n" +
                                 "- Example: 'Corporate State', 'Feudal'.";
            break;

        case "techlevel":
            parameters.message = "Enter a new value for 'techlevel'.\n" +
                                 "- Valid range: 0 to 15 (integer).\n" +
                                 "- Represents the technological sophistication of the planet.\n" +
                                 "- On F7, this is displayed as techLevel + 1 (e.g., setting 0 displays as 1).";
            break;

        case "population":
            parameters.message = "Enter a new value for 'population'.\n" +
                                 "- Valid range: 0 to 100 (integer).\n" +
                                 "- Represents the population in hundreds of millions.\n" +
                                 "- On F7, this is displayed as 'Population: X.X Billion' (e.g., population = 14 becomes 'Population: 1.4 Billion').";
            break;

        case "population_description":
            parameters.message = "Enter a new value for 'population_description'.\n" +
                                 "- Overrides the default text used to describe the population.\n" +
                                 "- Example: '7 Castaways', '13 Billion'.";
            break;

        case "inhabitant":
            parameters.message = "Enter a new value for 'inhabitant' (singular word).\n" +
                                 "- A free-form text string describing the inhabitants.\n" +
                                 "- Example: 'Human Colonist', 'Rodent-Ruined Reptile'.\n" +
                                 "- Remember to stay aligned with inhabitantsDescription (plural words).";
            break;

        case "inhabitants":
            parameters.message = "Enter a new value for 'inhabitants' (plural words).\n" +
                                 "- A free-form text string describing the inhabitants.\n" +
                                 "- Example: 'Human Colonists', 'Rodent-Ruined Reptiles'.\n" +
                                 "- Remember to also update inhabitantDescription (singular word).";
            break;

        case "productivity":
            parameters.message = "Enter a new value for 'productivity'.\n" +
                                 "- A whole number representing gross productivity in MegaCredits (MCr).\n" +
                                 "- On F7, this is displayed as 'Gross Productivity: X MCr'.\n" +
                                 "- Its role in game mechanics is unclear; it may be purely cosmetic.";
            break;

        case "radius":
            parameters.message = "Enter a new value for 'radius'.\n" +
                                 "- Valid range: 3000 to 7000 (integer). (Limits unconfirmed.)\n" +
                                 "- The radius of the planet. On F7, this is displayed as kilometres.\n" +
                                 "- The radius in game is 1/100th of this.";
            break;

        case "description":
            parameters.message = "Enter a new value for 'description'.\n" +
                                 "- A free-form text string describing the planet.\n" +
                                 "- Use this field creatively to add flavour to your worlds.\n" +
                                 "- Example: 'This is a dull planet.', 'The Lord Chamberlain visited this planet on stardate x.'";
            break;

        default:
            parameters.message = "You are setting the '" + this.propertyToChange + "'property.\n" +
                                 "Sorry, no out-of-range validation will be done - you should know what you are doing.";
            break;
    }

    mission.runScreen(parameters, this._changeWorld3.bind(this));
};

this._changeWorld3 = function (valueToChange) {
    if (valueToChange && valueToChange !== "") {
        const sys = System.infoForSystem(galaxyNumber, player.ship.infoSystem);
        sys[this.propertyToChange.toString()] = valueToChange;
    }
    mission.runScreen({
        title: "World Builder",
        message: "Exit to view result.",
        exitScreen: "GUI_SCREEN_SYSTEM_DATA",
        allowInterrupt: true,
        screenID: "world_builder"
    });
};