(function () {
	"use strict";
	this.name = "XenonReduxUI";
	this.author = "phkb";
	this.copyright = "2016 phkb";
	this.description = "Simplified versions of the Xenon UI backgrounds (no side panels).";
	this.license = "CC BY-NC-SA 3.0";

	this._marketObserverInstalled = false; // flag to indicate when market observer is installed (for wide F8 display)
	this._disableMissionScreen = []; // array of mission "screenID's" that will have the background disabled
	this._chartExitScreenTimer = null;
	this._missionOverlay = {}; // array of mission screen ID's and image names for overlays
	this._backgroundOverride = []; // array of mission screen ID's we've forcefully added a background to
	this._disableOverlays = false; // flag that indicates whether mission overlays will be applied to mission screens
	this._enableLibGUIInt = true; // flag to indicate whether integration with LibGUI is on or off. true means it is on.
	this._activeMode = true;
	this._guiActive = ""; // monitors changes in the currently selected GUI (when Library is installed)
	this._useBGS = "";
	this._trueValues = ["yes", "1", 1, "true", true];
	this._hdbgActive = false; // indicates that the HG Backgrounds OXP is active
	this._addedID = {}; // keeps track of screen ID's we have added to Library GUI, so we can remove them if another GUI is selected
	this._xn = null; // link to Xenon UI (if installed)
	this._defaultBG = {};
	this._amber = false;  // indicates whether blue (false) or amber (true) backgrounds will be used

	// configuration settings for use in Lib_Config
	this._xenonReduxUIConfig = {
		Name: this.name,
		Alias: "Xenon Redux UI",
		Display: "Config",
		Alive: "_xenonReduxUIConfig",
		Notify: "$changeSettings",
		Bool: {
			B0: {
				Name: "_disableOverlays",
				Def: false,
				Desc: "Disable overlays"
			},
			B1: {
				Name: "_activeMode",
				Def: true,
				Desc: "Active mode"
			},
			B2: {
				Name: "_amber",
				Def: false,
				Desc: "Amber color"
			},
			Info: "0 - Disables the mission screen overlay images.\n1 - XenonReduxUI will actively attempt to apply backgrounds to all mission screens, except for those added as exceptions.\n2 - Switch amber color on."
		},
	};

	/*
		This OXP replaces all the backgrounds of all UI screens with new versions using a hi-tech-like design theme.
		In most cases any individual backgrounds supplied by individual OXP's on their mission "runScreen" will overridden by these ones (but see below).

		This OXP is designed to be compatible with the BackGroundSet (BGS). However, on some systems some BGS images may still be shown,
		depending on the load order of the OXP's. If this OXP loads after BGS, these images will be used. If BGS loads after this OXP, it's images may be used.

		If your OXP has backgrounds you want to keep, and you want to still use this OXP, use the "$addMissionScreenException" to your startUpComplete script:
		Example:
			var w = worldScripts.XenonReduxUI;
			if (w) w.$addMissionScreenException("mymissionscreenid");

	*/

	//-------------------------------------------------------------------------------------------------------------
	// adds a screen ID to the list of mission screen ID's for which this OXP will not override
	this.$addMissionScreenException = function $addMissionScreenException(missionScreenID) {
		var lib = worldScripts.Lib_GUI;
		if (this._disableMissionScreen.indexOf(missionScreenID) === -1) this._disableMissionScreen.push(missionScreenID);
		if (lib && this._enableLibGUIInt && (!this._addedID[missionScreenID] || this._addedID[missionScreenID] === 2) && lib.$cur === "XenonReduxUI") {
			if (!lib.$IDRules[missionScreenID]) {
				this._addedID[missionScreenID] = 2;
				this.$passConfig(missionScreenID);
				lib.$IDRules[missionScreenID] = {
					mpic: 0
				};
			}
			//lib.$IDRules[missionScreenID] = {
			//	mpic: 0
			//};
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// function to add an overlay image to a mission screen
	this.$addMissionOverlay = function $addMissionOverlay(missionScreenID, imagename) {
		if (!this._missionOverlay[missionScreenID] && imagename == null) imagename = "";
		if (imagename != null) this._missionOverlay[missionScreenID] = imagename;
		var lib = worldScripts.Lib_GUI;
		// lib_GUI stuff
		if (lib && this._enableLibGUIInt) {
			if (imagename != null && imagename != "") {
				lib.$IDs.XenonReduxUI[missionScreenID] = {
					pic: {
						name: (this._amber === true ? "amber" : "xenon") + "_redux.png",
						height: 546
					},
					picBig: {
						name: (this._amber === true ? "amber" : "xenon") + "_redux_nohud.png",
						height: 546
					},
					ov: {
						name: imagename,
						height: 546
					},
					sndRef: this._useBGS
				};
				if (lib.$cur === "XenonReduxUI") {
					if (!this._addedID[missionScreenID] || this._addedID[missionScreenID] === 2) {
						if (!lib.$IDRules[missionScreenID]) {
							this._addedID[missionScreenID] = 2;
							this.$passConfig(missionScreenID);
							if (this._disableOverlays === true) {
								lib.$IDRules[missionScreenID] = {
									pic: 1,
									ov: 0
								};
							} else {
								lib.$IDRules[missionScreenID] = {
									pic: 1,
									ov: 1
								};
							}
						}
						/*if (this._disableOverlays === true) {
							lib.$IDRules[missionScreenID] = {
								pic: 1,
								ov: 0
							};
						} else {
							lib.$IDRules[missionScreenID] = {
								pic: 1,
								ov: 1
							};
						}*/
					}
				}
			} else {
				if (imagename === "") {
					lib.$IDs.XenonReduxUI[missionScreenID] = {
						pic: {
							name: (this._amber === true ? "amber" : "xenon") + "_redux.png",
							height: 546
						},
						picBig: {
							name: (this._amber === true ? "amber" : "xenon") + "_redux_nohud.png",
							height: 546
						},
						sndRef: this._useBGS
					};
					if (lib.$cur === "XenonReduxUI" && (!this._addedID[missionScreenID] || this._addedID[missionScreenID] === 2)) {
						if (!lib.$IDRules[missionScreenID]) {
							this._addedID[missionScreenID] = 2;
							this.$passConfig(missionScreenID);
							lib.$IDRules[missionScreenID] = {
								pic: 1
							};
						}
						//lib.$IDRules[missionScreenID] = {
						//	pic: 1
						//};
					}
				}
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.startUp = function () {
		this._xn = worldScripts.XenonUI;

		var disable = false;
		if (worldScriptNames.indexOf("Xenon Redux UI Resource Pack") === -1) {
			log(this.name, "ERROR! The Xenon Redux UI Resource pack is not installed. Xenon Redux UI cannot function without it.");
			disable = true;
		}

		if (disable === true || (worldScripts.XenonUI && (!worldScripts.Lib_GUI || this._enableLibGUIInt === false))) {
			// disable this OXP if XenonUI is installed along with it, but Library GUI isn't there to mediate.
			delete this.startUpComplete;
			delete this.guiScreenChanged;
			delete this.playerWillSaveGame;
			delete this.missionScreenEnded;
			delete this.gameResumed;
			delete this.startUp;
			delete worldScripts.XenonReduxUI_HeightFix.missionScreenOpportunity;
			delete worldScripts.XenonReduxUI_HeightFix.guiScreenChanged;
			return;
		}

		var lib = worldScripts.Lib_GUI;

		if (missionVariables.XenonReduxUI_ActiveMode) this._activeMode = (this._trueValues.indexOf(missionVariables.XenonReduxUI_ActiveMode) >= 0 ? true : false);
		if (missionVariables.XenonReduxUI_DisableOverlays) this._disableOverlays = (this._trueValues.indexOf(missionVariables.XenonReduxUI_DisableOverlays) >= 0 ? true : false);
		// only reload the amber setting if lib config is present - otherwise we'll just take the preset value
		if (lib) {
			if (missionVariables.XenonReduxUI_Amber) this._amber = (this._trueValues.indexOf(missionVariables.XenonReduxUI_Amber) >= 0 ? true : false);
			this._lastAmber = this._amber;
		}
		if (!lib || lib.$cur === "XenonReduxUI") this.$updateGuiColorSettings();

		if (worldScripts["BGS"]) this._useBGS = "BGS";
		if (worldScripts["market_observer3"]) this._marketObserverInstalled = true;
		if (worldScripts["hdbg"]) this._hdbgActive = true;

		if (lib && this._enableLibGUIInt === true) {
			// flag some screens as readonly
			this._addedID["oolite-contracts-parcels-none"] = 1;
			this._addedID["oolite-contracts-parcels-details"] = 1;
			this._addedID["oolite-contracts-parcels-summary"] = 1;
			this._addedID["oolite-contracts-passengers-none"] = 1;
			this._addedID["oolite-contracts-passengers-details"] = 1;
			this._addedID["oolite-contracts-passengers-summary"] = 1;
			this._addedID["oolite-contracts-cargo-none"] = 1;
			this._addedID["oolite-contracts-cargo-details"] = 1;
			this._addedID["oolite-contracts-cargo-summary"] = 1;
			this._addedID["oolite-primablemanager"] = 1;
			this._addedID["oolite-register"] = 1;
			this._addedID["Lib_Config"] = 1;

			// disable our standard functions so LibGUI can take over
			this.$previous_guiScreenChanged = this.guiScreenChanged;
			delete this.guiScreenChanged;
			delete this.missionScreenEnded;
			delete this.gameResumed;
			delete this.startUp;
			this.$libGUISetup();
		}

		// if we don't have library to orchestrate backgrounds, disable "better screens" so xenon UI has priority
		// otherwise, we end up with a clash of different background sets
		if (!lib && worldScripts["smivs_screen-backgrounds_worldscript"]) {
			worldScripts["smivs_screen-backgrounds_worldscript"].hide_guiScreenChanged = worldScripts["smivs_screen-backgrounds_worldscript"].guiScreenChanged;
			delete worldScripts["smivs_screen-backgrounds_worldscript"].guiScreenChanged;
		}

		this.$addStandardMissionScreenOverlays();
		this.$addStandardMissionScreenExceptions();

		// make Ship's library compatible with XenonUI
		if (worldScripts["Ships Library"] && !this._xn) {
			var sl = worldScripts["Ships Library"];
			sl._showPage = this._showPage;
		}
		if (worldScripts.sothis_tc && !this._xn) {
			var stc = worldScripts.sothis_tc;
			stc.missionScreenOpportunity = this._sothis_missionScreenOpportunity;
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.startUpComplete = function () {
		// register our settings, if Lib_Config is present
		if (worldScripts.Lib_Config) {
			worldScripts.Lib_Config._registerSet(this._xenonReduxUIConfig);
			if (!worldScripts.XenonUI) this.$setBGSDefaultBackgrounds();
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.guiScreenChanged = function (to, from) {

		var p = player.ship;

		var imagetype = "redux";
		var redalert = "";
		var nohud = "";
		var imagename = "";
		var startTimer = false;
		var overlay = "";

		if (this._marketObserverInstalled) {
			if (from == "GUI_SCREEN_MARKET" || from == "GUI_SCREEN_MARKETINFO" || to == "GUI_SCREEN_MARKET" || to == "GUI_SCREEN_MARKETINFO")
				// Because market observer uses a specialised HUD a slightly incorrect background image was being selected.
				// This was unnoticable if the "allow_big_gui" was not enabled on the default HUD, but quite noticeable on the Xenon HUD.
				// So we start a timer to force the correct background after Market Observer has set or reset the HUD.
				startTimer = true;
		}

		// fix for the issue where exiting the ship library screen could sometimes result in the wrong background being displayed
		// this would only be apparent with HUD's that don't implement the "allow_big_gui" setting.
		if (from == "GUI_SCREEN_SHIPLIBRARY" && p && p.docked) startTimer = true;

		if (p.alertCondition == 3) redalert = "_redalert";
		if (p.hudHidden) nohud = "_nohud";
		if (this.$isBigGuiActive() == true) nohud = "_nohud";

		switch (guiScreen) {
			case "GUI_SCREEN_MAIN":
				return;
			case "GUI_SCREEN_KEYBOARD":
				if (this._disableOverlays === false) overlay = "xrui-keyboard.png";
				break;
			case "GUI_SCREEN_STICKMAPPER":
				if (this._disableOverlays === false) overlay = "xrui-gamepad.png";
				break;
			case "GUI_SCREEN_OPTIONS":
				if (this._disableOverlays === false) overlay = "xrui-settings.png";
				break;
			case "GUI_SCREEN_LOAD":
				if (this._disableOverlays === false) overlay = "xrui-load.png";
				break;
			case "GUI_SCREEN_SAVE":
			case "GUI_SCREEN_SAVE_OVERWRITE":
				if (this._disableOverlays === false) overlay = "xrui-save.png";
				break;
			case "GUI_SCREEN_EQUIP_SHIP":
				if (this._disableOverlays === false) overlay = "xrui-equipship.png";
				break;
			case "GUI_SCREEN_INTERFACES":
				if (this._disableOverlays === false) overlay = "xrui-interfaces.png";
				break;
			case "GUI_SCREEN_STATUS":
				// override BGS status and nocrowd image, if installed.
				if (worldScripts["BGS-M"]) {
					worldScripts["BGS-M"].bgsI.status = {
						name: "xenon_" + imagetype + "_" + nohud + redalert + ".png",
						height: 546
					};
					worldScripts["BGS-M"].bgsI.noCrowd = {
						name: "xenon_" + imagetype + "_" + nohud + redalert + ".png",
						height: 546
					};
				}
				/*if (this._disableOverlays === false) {
					if (this.$playerHasDamagedEquipment() === true) {
						overlay = "xrui-warning.png";
					} else {
						overlay = "xrui-ok.png";
					}
				}*/
				break;
			case "GUI_SCREEN_MANIFEST":
				if (this._disableOverlays === false) overlay = "xrui-clipboard.png";
				break;
			case "GUI_SCREEN_SHORT_RANGE_CHART":
				break;
			case "GUI_SCREEN_LONG_RANGE_CHART":
				// override BGS lrc image, if installed.
				if (worldScripts["BGS-M"]) {
					worldScripts["BGS-M"].bgsI.lrc = {
						name: "xenon_" + imagetype + nohud + redalert + ".png",
						height: 546
					};
				}
				break;
			case "GUI_SCREEN_SYSTEM_DATA":
				var sysinf = System.infoForSystem(galaxyNumber, p.targetSystem);
				if (sysinf["sun_gone_nova"] || sysinf["sun_going_nova"]) imagename += "_nova";
				break;
			case "GUI_SCREEN_MARKET":
				if (this._marketObserverInstalled) imagename += "_wide";
				if (this._disableOverlays === false) overlay = "xrui-market.png";
				break;
			case "GUI_SCREEN_MARKETINFO":
				if (this._marketObserverInstalled) imagename += "_wide";
				if (this._disableOverlays === false) overlay = "xrui-tag.png";
				break;
			case "GUI_SCREEN_REPORT":
				if (worldScripts.hdbg) return;
				if (this._disableOverlays === false) overlay = "xrui-report.png";
				if (this._hdbgActive === true) return;
				break;
			case "GUI_SCREEN_MISSION":
				// look for any exceptions and jump out if found
				if (this._disableMissionScreen.indexOf(mission.screenID) != -1) imagename = "disable";
				// check if we have an overlay to apply to this mission screen
				if (this._disableOverlays == false && this._disableMissionScreen.indexOf(mission.screenID) === -1) {
					if (this._missionOverlay[mission.screenID]) overlay = this._missionOverlay[mission.screenID];
				}
				break;
		}

		// build the filename if it hasn't been disabled
		if (imagename != "disable") {
			imagename = (this._amber === true ? "amber" : "xenon") + "_" + imagetype + imagename + nohud + redalert + ".png";
			// load the image to the background
			setScreenBackground({
				name: imagename,
				height: 546
			});
		}
		if (overlay != "") setScreenOverlay({
			name: overlay,
			height: 546
		});

		// start a timer to force the background, if required
		if (startTimer) {
			if (this._chartExitScreenTimer && this._chartExitScreenTimer.isRunning) this._chartExitScreenTimer.stop();
			this._chartExitScreenTimer = new Timer(this, this.$addBackground, 0.25, 0);
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.playerWillSaveGame = function () {
		missionVariables.XenonReduxUI_ActiveMode = this._activeMode;
		missionVariables.XenonReduxUI_DisableOverlays = this._disableOverlays;
		missionVariables.XenonReduxUI_Amber = this._amber;
	}

	//-------------------------------------------------------------------------------------------------------------
	this.missionScreenEnded = function () {
		this.guiScreenChanged("", "");
	}

	//-------------------------------------------------------------------------------------------------------------
	this.gameResumed = function () {
		this.guiScreenChanged("", "");
	}

	//-------------------------------------------------------------------------------------------------------------
	// adds overlays to the standard mission screens
	this.$addStandardMissionScreenOverlays = function $addStandardMissionScreenOverlays() {
		this.$addMissionOverlay("oolite-contracts-parcels-none", "xrui-briefcase.png");
		this.$addMissionOverlay("oolite-contracts-parcels-summary", "xrui-briefcase.png");
		this.$addMissionOverlay("oolite-contracts-passengers-none", "xrui-boardingpass.png");
		this.$addMissionOverlay("oolite-contracts-passengers-summary", "xrui-boardingpass.png");
		this.$addMissionOverlay("oolite-contracts-cargo-none", "xrui-cargo.png");
		this.$addMissionOverlay("oolite-contracts-cargo-summary", "xrui-cargo.png");
		this.$addMissionOverlay("oolite-primablemanager", "xrui-switch.png");
		this.$addMissionOverlay("oolite-register", "xrui-register.png");
	}

	//-------------------------------------------------------------------------------------------------------------
	// adds map screen exceptions for known screen ID's
	this.$addStandardMissionScreenExceptions = function $addStandardMissionScreenExceptions() {
		// each of these items can't be set using setScreenBackground - needs to be done via the plist file
		this.$addMissionScreenException("oolite-contracts-parcels-details");
		this.$addMissionScreenException("oolite-contracts-passengers-details");
		this.$addMissionScreenException("oolite-contracts-cargo-details");
		this.$addMissionScreenException("rrs_mission_map");
		this.$addMissionScreenException("blackmonks-map");
		this.$addMissionScreenException("ups-map");
		this.$addMissionScreenException("escort-contracts");
		if (worldScripts.Random_Hits) {
			var ver = worldScripts.Random_Hits.version.split(".");
			if (parseInt(ver[0]) > 1 ||
				(parseInt(ver[0]) === 1 && parseInt(ver[1])) > 11 ||
				(parseInt(ver[0]) === 1 && parseInt(ver[1]) === 11 && parseInt(ver[2]) > 2)) {
				this.$addMissionScreenException("random_hits_map");
			} else {
				this.$addMissionScreenException("random_hits");
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// callback handler for when settings are changed
	this.$changeSettings = function $changeSettings() {
		var lib = worldScripts.Lib_GUI;
		if (this._amber != this._lastAmber) {
			this.$libGUISetup();
			this._lastAmber = this._amber;
		}
		if (lib.$cur === "XenonReduxUI") {
			if (global.setScreenBackgroundForKey) this.$setBackgroundDefaults(true);
			this.$updateGuiColorSettings();
		}
		/*if (this._disableOverlays === true) {
			if (this._enableLibGUIInt === true && lib && lib.$cur === "XenonReduxUI") {
				var keys = Object.keys(this._missionOverlay);
				for (var i = 0; i < keys.length; i++) {
					if (!this._addedID[keys[i]] || this._addedID[keys[i]] === 2) {
						if (!lib.$IDRules[keys[i]]) {
							this._addedID[keys[i]] = 2;
							this.$passConfig(keys[i]);
						}
						lib.$IDRules[keys[i]] = {
							pic: 1,
							ov: 0
						};
					}
				}
			}
		} else {
			if (this._enableLibGUIInt === true && lib && lib.$cur === "XenonReduxUI") {
				var keys = Object.keys(this._missionOverlay);
				for (var i = 0; i < keys.length; i++) {
					if (!this._addedID[keys[i]] || this._addedID[keys[i]] === 2) {
						if (!lib.$IDRules[keys[i]]) {
							this._addedID[keys[i]] = 2;
							this.$passConfig(keys[i]);
						}
						lib.$IDRules[keys[i]] = {
							pic: 1,
							ov: 1
						};
					}
				}
			}
		}*/
		// remove any background that we added forcefully
		/*if (lib.$cur === "XenonReduxUI" && this._activeMode === false) {
			for (var i = 0; i < this._backgroundOverride.length; i++) {
				//delete lib.$IDs.XenonReduxUI[this._backgroundOverride[i]];
				if (!this._addedID[this._backgroundOverride[i]] || this._addedID[this._backgroundOverride[i]] === 2) {
					delete lib.$IDRules[this._backgroundOverride[i]];
				}
			}
		}*/
		// reset our monitoring array regardless of which gui is active
		if (this._activeMode === false) this._backgroundOverrride = [];

		this.$setupGeneralOverlays();
	}

	//-------------------------------------------------------------------------------------------------------------
	// sets up the LibGUI parameters
	this.$libGUISetup = function $libGUISetup() {

		var lib = worldScripts.Lib_GUI;
		// setup some event callbacks
		this.guiScreenChanged = this.$guiScreenChanged;
		//this.equipmentDamaged = this.$equipmentDamaged;
		//this.equipmentRepaired = this.$equipmentRepaired;
		//this.equipmentAdded = this.$equipmentAdded;
		//this.equipmentRemoved = this.$equipmentRemoved;
		this.missionScreenEnded = this.$missionScreenEnded;

		var style = (this._amber === true ? "amber" : "xenon");

		lib.$guis.XenonReduxUI = {
			GUI_SCREEN_GAMEOPTIONS: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_LOAD: {
				pic: {
					name: style + "_redux.png",
					height: 546
				}
			},
			GUI_SCREEN_SAVE: {
				pic: {
					name: style + "_redux.png",
					height: 546
				}
			},
			GUI_SCREEN_SAVE_OVERWRITE: {
				pic: {
					name: style + "_redux.png",
					height: 546
				}
			},
			GUI_SCREEN_EQUIP_SHIP: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_INTERFACES: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_KEYBOARD: {
				pic: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux_nohud.png",
					height: 546
				}
			},
			GUI_SCREEN_MANIFEST: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_MARKET: {
				pic: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picBig: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + "_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picRed: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + "_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_MARKETINFO: {
				pic: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picBig: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + "_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picRed: {
					name: style + "_redux" + (this._marketObserverInstalled === true ? "_wide" : "") + "_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_OPTIONS: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_SHIPLIBRARY: {
				pic: {
					name: style + "_redux_nohud.png",
					height: 546
				}
			},
			GUI_SCREEN_SHIPYARD: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_STATUS: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_STICKMAPPER: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_STICKPROFILE: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_SYSTEM_DATA: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picNova: {
					name: style + "_redux_nova.png",
					height: 546
				},
				picNovaBig: {
					name: style + "_redux_nova_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				picNovaRed: {
					name: style + "_redux_nova_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_SHORT_RANGE_CHART: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_LONG_RANGE_CHART: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
		};

		if (this._hdbgActive === false) {
			lib.$guis.XenonReduxUI.generic = {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				sndRef: this._useBGS
			};
			lib.$guis.XenonReduxUI.GUI_SCREEN_REPORT = {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			};
		}

		this.$setupVariableScreens();
		this.$setupGeneralOverlays();

		// mission screen IDs
		lib.$IDs.XenonReduxUI = {
			generic: {
				pic: {
					name: style + "_redux.png",
					height: 546
				},
				picBig: {
					name: style + "_redux_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_redux.png",
					height: 546
				},
				picRed: {
					name: style + "_redux_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			}
		};

	}

	//-------------------------------------------------------------------------------------------------------------
	// adjust background images on F5 screen based on availability/status of equipment
	this.$setupVariableScreens = function $setupVariableScreens() {
		/*if (this._disableOverlays === false) {
			if (this.$playerHasDamagedEquipment() === true) {
				worldScripts.Lib_GUI.$guis.XenonReduxUI.GUI_SCREEN_STATUS["ov"] = {name:"xrui-warning.png", height:546};
			} else {
				worldScripts.Lib_GUI.$guis.XenonReduxUI.GUI_SCREEN_STATUS["ov"] = {name:"xrui-ok.png", height:546};
			}
		}*/
	}

	//-------------------------------------------------------------------------------------------------------------
	// adjust overlay images on general (non-mission) screens
	this.$setupGeneralOverlays = function $setupGeneralOverlays() {

		var lib = worldScripts.Lib_GUI.$guis.XenonReduxUI;

		if (this._disableOverlays) {
			delete lib.GUI_SCREEN_GAMEOPTIONS["ov"];
			delete lib.GUI_SCREEN_LOAD["ov"];
			delete lib.GUI_SCREEN_SAVE["ov"];
			delete lib.GUI_SCREEN_SAVE_OVERWRITE["ov"];
			delete lib.GUI_SCREEN_EQUIP_SHIP["ov"];
			delete lib.GUI_SCREEN_INTERFACES["ov"];
			delete lib.GUI_SCREEN_KEYBOARD["ov"];
			delete lib.GUI_SCREEN_MANIFEST["ov"];
			delete lib.GUI_SCREEN_MARKET["ov"];
			delete lib.GUI_SCREEN_MARKETINFO["ov"];
			delete lib.GUI_SCREEN_OPTIONS["ov"];
			delete lib.GUI_SCREEN_REPORT["ov"];
			//delete lib.GUI_SCREEN_STATUS["ov"];
			delete lib.GUI_SCREEN_STICKMAPPER["ov"];
		} else {
			lib.GUI_SCREEN_GAMEOPTIONS["ov"] = {
				name: "xrui-settings.png",
				height: 546
			};
			lib.GUI_SCREEN_LOAD["ov"] = {
				name: "xrui-load.png",
				height: 546
			};
			lib.GUI_SCREEN_SAVE["ov"] = {
				name: "xrui-save.png",
				height: 546
			};
			lib.GUI_SCREEN_SAVE_OVERWRITE["ov"] = {
				name: "xrui-question.png",
				height: 546
			};
			lib.GUI_SCREEN_EQUIP_SHIP["ov"] = {
				name: "xrui-equipship.png",
				height: 546
			};
			lib.GUI_SCREEN_INTERFACES["ov"] = {
				name: "xrui-interfaces.png",
				height: 546
			};
			lib.GUI_SCREEN_KEYBOARD["ov"] = {
				name: "xrui-keyboard.png",
				height: 546
			};
			lib.GUI_SCREEN_MANIFEST["ov"] = {
				name: "xrui-clipboard.png",
				height: 546
			};
			lib.GUI_SCREEN_MARKET["ov"] = {
				name: "xrui-market.png",
				height: 546
			};
			lib.GUI_SCREEN_MARKETINFO["ov"] = {
				name: "xrui-tag.png",
				height: 546
			};
			lib.GUI_SCREEN_OPTIONS["ov"] = {
				name: "xrui-settings.png",
				height: 546
			};
			if (this._hdbgActive === false) lib.GUI_SCREEN_REPORT["ov"] = {
				name: "xrui-report.png",
				height: 546
			};
			lib.GUI_SCREEN_STICKMAPPER["ov"] = {
				name: "xrui-gamepad.png",
				height: 546
			};
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// alternate version for use with Lib_GUI
	this.$missionScreenEnded = function $missionScreenEnded() {
		var lib = worldScripts.Lib_GUI;
		if (this._guiActive != lib.$cur && (this._guiActive === "XenonReduxUI" || lib.$cur === "XenonReduxUI")) {
			if (lib.$cur === "XenonReduxUI") {
				// if we've got it available (v1.85/86) then force the default screen background to be Xenon Redux
				if (global.setScreenBackgroundForKey) this.$setBackgroundDefaults(true);
				this.$updateGuiColorSettings();
				for (var i = 0; i < this._disableMissionScreen.length; i++) {
					if (!this._addedID[this._disableMissionScreen[i]] || this._addedID[this._disableMissionScreen[i]] === 2) {
						if (!lib.$IDRules[this._disableMissionScreen[i]]) {
							this._addedID[this._disableMissionScreen[i]] = 2;
							this.$passConfig(this._disableMissionScreen[i]);
							lib.$IDRules[this._disableMissionScreen[i]] = {
								mpic: 0
							};
						}
						/*lib.$IDRules[this._disableMissionScreen[i]] = {
							mpic: 0
						};*/
					}
				}
				var keys = Object.keys(this._missionOverlay);
				for (var i = 0; i < keys.length; i++) {
					// update Lib_GUI with our settings for mission screens
					if (!this._addedID[keys[i]] || this._addedID[keys[i]] === 2) {
						if (!lib.$IDRules[keys[i]]) {
							this._addedID[keys[i]] = 2;
							this.$passConfig(keys[i]);
							if (this._missionOverlay[keys[i]] != "") {
								if (this._disableOverlays === true) {
									lib.$IDRules[keys[i]] = {
										pic: 1,
										ov: 0
									};
								} else {
									lib.$IDRules[keys[i]] = {
										pic: 1,
										ov: 1
									};
								}
							} else {
								lib.$IDRules[keys[i]] = {
									pic: 1
								};
							}
						}
						/*if (this._missionOverlay[keys[i]] != "") {
							if (this._disableOverlays === true) {
								lib.$IDRules[keys[i]] = {
									pic: 1,
									ov: 0
								};
							} else {
								lib.$IDRules[keys[i]] = {
									pic: 1,
									ov: 1
								};
							}
						} else {
							lib.$IDRules[keys[i]] = {
								pic: 1
							};
						}*/
					}
				}
			} else {
				// if we've got it available (v1.85/86) then reset the default screen backgrounds to be what it was previously
				if (global.setScreenBackgroundForKey) this.$setBackgroundDefaults(false);
				/*var items = Object.keys(this._addedID);
				for (var i = 0; i < items.length; i++) {
					if (this._addedID[items[i]] === 2) {
						delete lib.$IDRules[items[i]];
					}
				}*/
			}
			this._guiActive = lib.$cur;
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	// gets any unrecorded mission ScreenID's add add them to the list of rules to use Xenon
	this.$guiScreenChanged = function $guiScreenChanged(to, from) {
		if (guiScreen === "GUI_SCREEN_STATUS") {
			// this bit should only be run once after load
			if (this._guiActive === "") {
				// force all the screen rego stuff to happen if xenon was set after loading
				if (worldScripts.Lib_GUI.$cur === "XenonReduxUI") this.missionScreenEnded();
				// make sure our local copy gets updated
				this._guiActive = worldScripts.Lib_GUI.$cur;
			}
		}

		if (guiScreen === "GUI_SCREEN_MISSION" && this._guiActive === "XenonReduxUI" && this._activeMode === true) {
			if (mission.screenID && mission.screenID != "" && this._disableMissionScreen.indexOf(mission.screenID) === -1) {
				this.$addMissionOverlay(mission.screenID, null);
				// make a note of any screen we add this way
				if (this._backgroundOverride.indexOf(mission.screenID) === -1) this._backgroundOverride.push(mission.screenID);
				// force an update
				worldScripts.Lib_GUI.guiScreenChanged();
			}
			// we'll try to override LibGUI if a mission screen is displayed without a screenID
			if (!mission.screenID) {
				this.$previous_guiScreenChanged(to, from);
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentAdded = function $equipmentAdded(equipmentKey) {
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentRemoved = function $equipmentRemoved(equipmentKey) {
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentDamaged = function $equipmentDamaged(equipmentKey) {
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentRepaired = function $equipmentRepaired(equipmentKey) {
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// forces a background after the timer has expired
	// used when mission screens exit to the short or long range chart.
	this.$addBackground = function $addBackground() {
		this.guiScreenChanged("", "");
	}

	//-------------------------------------------------------------------------------------------------------------
	// returns true if a HUD with allowBigGUI is enabled, otherwise false
	this.$isBigGuiActive = function $isBigGuiActive() {
		if (oolite.compareVersion("1.83") <= 0) {
			return player.ship.hudAllowsBigGui;
		} else {
			var bigGuiHUD = ["XenonHUD.plist", "coluber_hud_ch01-dock.plist"]; // until there is a property we can check, I'll be listing HUD's that have the allow_big_gui property set here
			if (bigGuiHUD.indexOf(player.ship.hud) >= 0) {
				return true;
			} else {
				return false;
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// returns true if any of the visible equipment items are damaged
	this.$playerHasDamagedEquipment = function $playerHasDamagedEquipment() {
		var p = player.ship;
		var eq = p.equipment;
		for (var i = 0; i < eq.length; i++) {
			var itm = eq[i];
			if (itm.isVisible === true && p.equipmentStatus(itm.equipmentKey) === "EQUIPMENT_DAMAGED") return true;
		}
		return false;
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$setBackgroundDefaults = function $setBackgroundDefaults(onOff) {
		if (onOff === true) {
			this._defaultBG["long_range_chart_mission"] = getScreenBackgroundForKey("long_range_chart_mission");
			this._defaultBG["short_range_chart_mission"] = getScreenBackgroundForKey("short_range_chart_mission");
			this._defaultBG["custom_chart_mission"] = getScreenBackgroundForKey("custom_chart_mission");
			this._defaultBG["long_range_chart"] = getScreenBackgroundForKey("long_range_chart");
			this._defaultBG["short_range_chart"] = getScreenBackgroundForKey("short_range_chart");
			this._defaultBG["keyboardsettings"] = getScreenBackgroundForKey("keyboardsettings");
			this._defaultBG["load_save"] = getScreenBackgroundForKey("load_save");
			this._defaultBG["mount_weapon"] = getScreenBackgroundForKey("mount_weapon");
			this._defaultBG["newgame"] = getScreenBackgroundForKey("newgame");
			this._defaultBG["settings"] = getScreenBackgroundForKey("settings");
			this._defaultBG["shiplibrary"] = getScreenBackgroundForKey("shiplibrary");

			var nohud = "";
			if (this.$isBigGuiActive() === true) nohud = "_nohud";

			var style = (this._amber === true ? "amber" : "xenon");

			var imagename = style + "_redux" + nohud + ".png";
			setScreenBackgroundForKey("short_range_chart_mission", {
				name: imagename,
				height: 546
			});
			setScreenBackgroundForKey("long_range_chart_mission", {
				name: imagename,
				height: 546
			});
			setScreenBackgroundForKey("short_range_chart", {
				name: imagename,
				height: 546
			});
			setScreenBackgroundForKey("long_range_chart", {
				name: imagename,
				height: 546
			});

			setScreenBackgroundForKey("keyboardsettings", {
				name: style + "_redux_nohud.png",
				height: 546
			});
			setScreenBackgroundForKey("load_save", {
				name: style + "_redux.png",
				height: 546
			});
			setScreenBackgroundForKey("mount_weapon", {
				name: style + "_redux.png",
				height: 546
			});
			setScreenBackgroundForKey("newgame", {
				name: style + "_redux_nohud.png",
				height: 546
			});
			setScreenBackgroundForKey("settings", {
				name: style + "_redux.png",
				height: 546
			});
			setScreenBackgroundForKey("shiplibrary", {
				name: style + "_redux_nohud.png",
				height: 546
			});
		} else {
			var keys = Object.keys(this._defaultBG);
			for (var i = 0; i < keys.length; i++) {
				var bg = getScreenBackgroundForKey(keys[i]);
				// are xenon's backgrounds still set?
				if (bg.hasOwnProperty("name") === true && bg.name.indexOf(style + "_redux") >= 0) {
					// ok, we can reset things now
					setScreenBackgroundForKey(keys[i], this._defaultBG[keys[i]]);
				}
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$updateGuiColorSettings = function $updateGuiColorSettings() {
		if (global.setGuiColorSettingForKey) {
			global.setGuiColorSettingForKey("screen_divider_color", (this._amber === false ? "0 74 127 200" : "127 74 0 200"));
			global.setGuiColorSettingForKey("screen_title_color", (this._amber === false ? "0 148 255 200" : "255 148 0 200"));
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$setBGSDefaultBackgrounds = function $setBGSDefaultBackgrounds() {
		// until BGS has something similar
		if (worldScripts.Lib_GUI.$cur == "BGS" && global.setGuiColorSettingForKey) {
			setGuiColorSettingForKey("screen_title_color", "0.9 0.9 0.8");
			setGuiColorSettingForKey("screen_divider_color", "0.0 0.0 0.0 0.0");

			setScreenBackgroundForKey("paused_docked_overlay", "bgs_ov_paused.png");
			setScreenBackgroundForKey("paused_overlay", "bgs_ov_paused.png");
			setScreenBackgroundForKey("intro", "bgs_intro.png");
			setScreenBackgroundForKey("keyboardsettings", "bgs_fullscr.png");
			setScreenBackgroundForKey("load_save", "bgs_options.png");
			setScreenBackgroundForKey("newgame", "bgs_intro.png");
			setScreenBackgroundForKey("oxz-manager", "bgs_fullscr.png");
			setScreenBackgroundForKey("settings", "bgs_options.png");
			setScreenBackgroundForKey("shiplibrary", "bgs_fullscr.png");
			setScreenBackgroundForKey("long_range_chart_mission", "lib_black.png");
			setScreenBackgroundForKey("short_range_chart_mission", "lib_black.png");
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// used to allow XenonUI and XenonReduxUI to co-exist
	this.$xenonCrossConfig = function $xenonCrossConfig(screenID) {
		this._addedID[screenID] = 2;
	}
	this.$passConfig = function $passConfig(screenID) {
		if (this._xn) this._xn.$xenonCrossConfig(screenID);
	}

	//-------------------------------------------------------------------------------------------------------------
	// to make Ships Library compatible with XenonUI
	this._showPage = function() {
		var chapter = this.$contents[this.$chapter];
	
		var text = this._textFromOffset(this.$chapter,this.$offset,this.$msStore,this.$msRows,this.$msCols);
		
		var opts = {
			screenID: "ships-library",
			titleKey: chapter.key+"-title",
			allowInterrupt: true,
			exitScreen: "GUI_SCREEN_INTERFACES",
			choices: {
				"01_PREV":expandMissionText("ships-library-page-back"),
				"09_NEXT":expandMissionText("ships-library-page-next"),
				"10_CONTENTS":expandMissionText("ships-library-contents"),
				"99_EXIT":expandMissionText("ships-library-exit"),
			},
			initialChoicesKey: this.$lastchoice?this.$lastchoice:"09_NEXT",
			message: text
		};
		if (chapter.opts) {
			var extras = Object.keys(chapter.opts)
			for (var k=0;k<extras.length;k++) {
				opts[extras[k]] = chapter.opts[extras[k]];
			}
		}
		if (chapter.backgrounds) {
			var page = this._pageOfOffset(chapter,this.$offset,this.$msStore);
			if (page < chapter.backgrounds.length) {
				var backg = chapter.backgrounds[page];
			} else {
				var backg = chapter.backgrounds[chapter.backgrounds.length-1];
			}
			if (backg != "") {
				opts.overlay = backg;
			}
		}
	
		mission.runScreen(opts,this._manualHandler,this);
		this.$fcbM = addFrameCallback(this._moveShip.bind(this));
	
	}

	//-------------------------------------------------------------------------------------------------------------
	// to make Sothis TC compatible with XenonUI
	this._sothis_missionScreenOpportunity = function() {
		if (!this.$showWelcome) return;
		var messText = expandDescription("[STC_welcome]");
		if (this.$new_cargoes)
			messText = messText + expandDescription("[STC_new_cargoes]");	
		var tcbgpic = "OOmap_G"+(galaxyNumber+1)+".png";
		mission.runScreen({
			title: "Sothis Trade Center",
			message: messText,
			overlay: tcbgpic
		});
		this.$showWelcome = false;
	}

}).call(this);