"use strict";
this.name        = "Autodock_Core";
this.author      = "phkb";
this.copyright   = "2015 phkb";
this.description = "Main routine for handling autodock option";
this.licence     = "CC BY-NC-SA 4.0";

this._payment = 10;				// the cost of an autodock. (should be an even number so the refund price doesn't have decimals)
this._minTechLevel = 0;			// the minimum techlevel required for an autodock (this matches main logic of being 1 less that the actual TL. ie entering 6 here means TL7)
this._mainStationOnly = true;	// flag that indicates whether autodock is provided at the main station only (true), or all stations (false)
this._abort = false;			// used if the player has the autodock, but credit went below payment amount during flight, then tries to autodock
								// docking will be cancelled, and this flag prevents the player from being refunded as no payment would have been made.


// configuration settings for use in Lib_Config
this._adConfig = {Name:this.name, Display:"Settings", Alias:"Autodock", Alive:"_adConfig", 
	Bool:{B0:{Name:"_mainStationOnly", Def:true, Desc:"Autodock at main stns"},Info:"Autodock only available at main stations."},
	SInt:{
		S0:{Name:"_payment", Def:10, Desc:"Payment Amount", Min:1, Max:100},
		S1:{Name:"_minTechLevel", Def:0, Desc:"Minimum tech level", Min:0, Max:14},
		Info:"0 - Cost of autodock, in credits\n1 - Minimum tech level for autodock"}
};
this._trueValues = ["yes", "1", 1, "true", true];

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function() {
	// register our settings, if Lib_Config is present
	if (worldScripts.Lib_Config) worldScripts.Lib_Config._registerSet(this._adConfig);
	if (missionVariables.Autodock_Payment) this._payment = parseInt(missionVariables.Autodock_Payment);
	if (missionVariables.Autodock_TechLevel) this._minTechLevel = parseInt(missionVariables.Autodock_TechLevel);
	if (missionVariables.Autodock_MainStation) this._mainStationOnly = (this._trueValues.indexOf(missionVariables.Autodock_MainStation) >= 0 ? true : false);
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function() {
	missionVariables.Autodock_Payment = this._payment;
	missionVariables.Autodock_TechLevel = this._minTechLevel;
	missionVariables.Autodock_MainStation = this._mainStationOnly;
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillDockWithStation = function(station) {
	// make sure the equipment is removed on docking
	var p = player.ship;
	// check for an fast-dock
	var diff = (global.clock.adjustedSeconds - global.clock.seconds);
	if (diff > 1000 && diff <= 1200) {
		if (p.equipmentStatus("EQ_NDC_AUTODOCK") == "EQUIPMENT_OK") {
			this.$notifyAndDeduct();
		}
	}
	p.removeEquipment("EQ_NDC_AUTODOCK");
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillLaunchFromStation = function(station) {
	// auto install after launching (to cope with scenario where player docks at another station, or the same one even)
	if ((station.isMainStation && this._mainStationOnly == true) || this._mainStationOnly == false) {
		this.$installAutoDock();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipEnteredStationAegis = function(station) {
	var p = player.ship;
	if (this._mainStationOnly == true) this.$installAutoDock();
	// if we have less that payment amount when we hit the aegis, remove the autodock item so the player can't fast-dock
	if (player.credits < this._payment && p.equipmentStatus("EQ_NDC_AUTODOCK") == "EQUIPMENT_OK") {
		p.removeEquipment("EQ_NDC_AUTODOCK");
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipExitedStationAegis = function(station) {
	if (this._mainStationOnly == true) {
		var p = player.ship;
		p.removeEquipment("EQ_NDC_AUTODOCK");
	}
}

//-------------------------------------------------------------------------------------------------------------
this.shipExitedWitchspace = function() {
	var p = player.ship;
	// remove the equipment, then attempt to reinstall it
	// this allows for the new system not having a high enough tech level
	p.removeEquipment("EQ_NDC_AUTODOCK");
	// only instal the autodock at this point if the mainstationonly flag is set to false (meaning all stations)
	if (this._mainStationOnly == false) this.$installAutoDock();
}

//-------------------------------------------------------------------------------------------------------------
this.playerStartedAutoPilot = function() {
	var p = player.ship;
	if (p.equipmentStatus("EQ_NDC_AUTODOCK") === "EQUIPMENT_OK"){
		if (player.credits >= this._payment) {
			this.$notifyAndDeduct();
		} else {
			// in the off chance the player has dropped below payment amount between installing the autodock and arriving in a new system
			// (eg bribing ships via the Broadcast comms MFD)
			this._abort = true;
			p.disengageAutopilot();
			player.consoleMessage("Insufficient credit available for autodock. Manual docking required.");
			p.removeEquipment("EQ_NDC_AUTODOCK");

		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerCancelledAutoPilot = function() {
	var p = player.ship;
	if (p.equipmentStatus("EQ_NDC_AUTODOCK") == "EQUIPMENT_OK" && this._abort == false) {
		// notify player
		player.consoleMessage("Control returned to pilot...depositing partial refund of " + formatCredits(this._payment / 2, false, true) + " to your account.", 5);
		// refund player
		player.credits += (this._payment / 2);
		// make sure the player still has enough cash for an autodock, and remove it if they don't
		if (player.credits < this._payment) p.removeEquipment("EQ_NDC_AUTODOCK");
	}
	this._abort = false;
}

//-------------------------------------------------------------------------------------------------------------
this.equipmentDamaged = function(equipment) {
	// if the dock computer gets damaged, add the autodock equipment
	if (equipment == "EQ_DOCK_COMP") {
		// if we're only doing this at main stations, check if we're within the aegis when the docking computer was damaged
		if (this._mainStationOnly == true && player.ship.withinStationAegis == false) {
			// if not, return at this point, as the autodocker will be installed when we hit the aegis
			return;
		}
		this.$installAutoDock();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.equipmentRepaired = function(equipment){
	// check for equipment being repaired in flight
	if (equipment == "EQ_DOCK_COMP") {
		var p = player.ship;
		// if the dock computer gets repaired in flight, remove the autodock
		if (p.status == "STATUS_IN_FLIGHT") p.removeEquipment("EQ_NDC_AUTODOCK");
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$installAutoDock = function() {
	var p = player.ship;
	// give the player the autodock equipment if the tech level is greater than/equal to min tech level and the player doesn't have a working dock comp and has more than required credit
	if (system.info.techlevel >= this._minTechLevel && player.credits >= this._payment) {
		if (p.equipmentStatus("EQ_DOCK_COMP") != "EQUIPMENT_OK") {
			p.awardEquipment("EQ_NDC_AUTODOCK");
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.$notifyAndDeduct = function() {
	player.consoleMessage("Transferring flight control to station for docking procedure...deducting " + formatCredits(this._payment, false, true) + " from your account.", 5);
	player.credits -= this._payment;
}
