(function () {
	"use strict";
	this.name = "XenonUI";
	this.author = "phkb";
	this.copyright = "2016 phkb";
	this.description = "New backgrounds for UI screens. Needs one of the Xenon UI Resource packs (A through N) to be installed as well.";
	this.license = "CC BY-NC-SA 3.0";

	this._marketObserverInstalled = false; // flag to indicate when market observer is installed (for wide F8 display)
	this._disableMissionScreen = []; // array of mission "screenID's" that will have the background disabled
	this._chartExitScreenTimer = null;
	this._missionOverlay = {}; // array of mission screen ID's and image names for overlays
	this._backgroundOverride = []; // array of mission screen ID's we've forcefully added a background to
	this._disableOverlays = false; // flag that indicates whether mission overlays will be applied to mission screens
	this._enableLibGUIInt = true; // flag to indicate whether integration with LibGUI is on or off. true means it is on.
	this._activeMode = true;
	this._guiActive = ""; // monitors changes in the currently selected GUI (when Library is installed)
	this._useBGS = "";
	this._trueValues = ["yes", "1", 1, "true", true];
	this._hdbgActive = false; // indicates that the HG Backgrounds OXP is active
	this._addedID = {}; // keeps track of screen ID's we have added to Library GUI, so we can remove them if another GUI is selected
	this._xr = null; // link to Xenon Redux UI (if installed)
	this._defaultBG = {};
	this._amber = false; // indicates whether blue (false) or amber (true) backgrounds will be used
	this._ratio = "";
	this._font = "default";
	this._f6Overlay = {
		"economy":null,
		"government":null,
		"suncolor":null,
		"techlevel":null
	};
	// configuration settings for use in Lib_Config
	this._xenonConfig = {
		Name: this.name,
		Alias: "Xenon UI",
		Display: "Config",
		Alive: "_xenonConfig",
		Notify: "$changeSettings",
		Bool: {
			B0: {
				Name: "_disableOverlays",
				Def: false,
				Desc: "Disable overlays"
			},
			B1: {
				Name: "_activeMode",
				Def: true,
				Desc: "Active mode"
			},
			B2: {
				Name: "_amber",
				Def: false,
				Desc: "Amber color"
			},
			Info: "0 - Disables the various screen overlay images.\n1 - XenonUI will actively attempt to apply backgrounds to all mission screens, except for those added as exceptions.\n2 - Switch amber color on."
		},
	};

	/*
		This OXP replaces all the backgrounds of all UI screens with new versions using a hi-tech-like design theme.
		In most cases any individual backgrounds supplied by individual OXP's on their mission "runScreen" will overridden by these ones (but see below).

		A resource pack is required to complete the system. Depending on the font you are using in Oolite, you may want to use a specific resource pack to match.
		The available packs are:
			Pack name		Font
			--------------- -----------------------------
			Pack A (16:9)	OCR A Extended
			Pack B (16:10)	OCA A Extended
			Pack C (16:9)	Xolonium
			Pack D (16:10)	Xolonium
			Pack E (16:9)	Dangerous Square
			Pack F (16:10)	Dangerous Square
			Pack G (16:9)	Default font (Helvetica Bold)
			Pack H (16:10)	Default font (Helvetica Bold)
			Pack I (16:9)	Dosis font
			Pack J (16:10)	Dosis font
			Pack K (16:9)	Discognate font
			Pack L (16:10)	Discognate font
			Pack M (16:9)	NovaSquare font
			Pack N (16:10)	NovaSquare font
			
		More packs may become available as new fonts are released for Oolite.

		This OXP is designed to be compatible with the BackGroundSet (BGS). However, on some systems some BGS images may still be shown,
		depending on the load order of the OXP's. If this OXP loads after BGS, these images will be used. If BGS loads after this OXP, it's images may be used.

		To disable the BGS images you will need to edit BGS, and rename the "screenbackgrounds.plist" file in the "Config" folder to something like "xcreenbackgrounds.plist".

		If your OXP has backgrounds you want to keep, and you want to still use this OXP, use the "$addMissionScreenException" to your startUpComplete script:
		Example:
			var w = worldScripts.XenonUI;
			if (w) w.$addMissionScreenException("mymissionscreenid");

		Note: When mission screens exit to the short or long range charts, the backgrounds are not being set. So a timer is in place to force the background on.

	*/

	//-------------------------------------------------------------------------------------------------------------
	// adds a screen ID to the list of mission screen ID's this OXP will not override
	this.$addMissionScreenException = function $addMissionScreenException(missionScreenID) {
		var lib = worldScripts.Lib_GUI;
		if (this._disableMissionScreen.indexOf(missionScreenID) === -1) this._disableMissionScreen.push(missionScreenID);
		if (lib && this._enableLibGUIInt && (!this._addedID[missionScreenID] || this._addedID[missionScreenID] === 2) && lib.$cur === "XenonUI") {
			if (!lib.$IDRules[missionScreenID]) {
				this._addedID[missionScreenID] = 2;
				this.$passConfig(missionScreenID);
				// we can only add now...
				lib.$IDRules[missionScreenID] = {
					mpic: 0
				};
			}
			//lib.$IDRules[missionScreenID] = {
			//	mpic: 0
			//};
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// function to add an overlay image to a mission screen
	this.$addMissionOverlay = function $addMissionOverlay(missionScreenID, imagename) {
		if (!this._missionOverlay[missionScreenID] && imagename == null) imagename = "";
		if (imagename != null) this._missionOverlay[missionScreenID] = imagename;
		var lib = worldScripts.Lib_GUI;
		// lib_GUI stuff
		if (lib && this._enableLibGUIInt) {
			if (imagename != null && imagename != "") {
				lib.$IDs.XenonUI[missionScreenID] = {
					pic: {
						name: (this._amber === true ? "amber" : "xenon") + "_standard_f0.png",
						height: 546
					},
					picBig: {
						name: (this._amber === true ? "amber" : "xenon") + "_standard_f0_nohud.png",
						height: 546
					},
					ov: {
						name: imagename,
						height: 546
					},
					sndRef: this._useBGS
				};
				if (lib.$cur === "XenonUI") {
					if (!this._addedID[missionScreenID] || this._addedID[missionScreenID] === 2) {
						if (!lib.$IDRules[missionScreenID]) {
							this._addedID[missionScreenID] = 2;
							this.$passConfig(missionScreenID);
							// we can only add now...
							if (this._disableOverlays === true) {
								lib.$IDRules[missionScreenID] = {
									pic: 1,
									ov: 0
								};
							} else {
								lib.$IDRules[missionScreenID] = {
									pic: 1,
									ov: 1
								};
							}
						}
						/*if (this._disableOverlays === true) {
							lib.$IDRules[missionScreenID] = {
								pic: 1,
								ov: 0
							};
						} else {
							lib.$IDRules[missionScreenID] = {
								pic: 1,
								ov: 1
							};
						}*/
					}
				}
			} else {
				if (imagename === "") {
					lib.$IDs.XenonUI[missionScreenID] = {
						pic: {
							name: (this._amber === true ? "amber" : "xenon") + "_standard_f0.png",
							height: 546
						},
						picBig: {
							name: (this._amber === true ? "amber" : "xenon") + "_standard_f0_nohud.png",
							height: 546
						},
						sndRef: this._useBGS
					};
					if (lib.$cur === "XenonUI" && (!this._addedID[missionScreenID] || this._addedID[missionScreenID] === 2)) {
						if (!lib.$IDRules[missionScreenID]) {
							this._addedID[missionScreenID] = 2;
							this.$passConfig(missionScreenID);
							// we can only add now...
							lib.$IDRules[missionScreenID] = {
								pic: 1
							};
						}
						//lib.$IDRules[missionScreenID] = {
						//	pic: 1
						//};
					}
				}
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$setF6OverlayLookup = function(section, ws, fnName) {
		var check = ["economy","government","suncolor","techlevel"];
		if (check.indexOf(section) == -1) {
			log(this.name, "Exception! Invalid section defined. Must be either 'economy', 'government', 'suncolor' or 'techlevel' - tried to use " + section);
			return false;
		}
		if (!worldScripts[ws][fnName]) {
			log(this.name, "Exception! WorldScript/Function not found! " + ws + "." + fnName);
			return false;
		}
		this._f6Overlay[section] = {worldscript:ws, functionname:fnName};
		return true;
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$clearF6OverlayLookup = function(section) {
		var check = ["economy","government","suncolor","techlevel"];
		if (check.indexOf(section) == -1) {
			log(this.name, "Exception! Invalid section defined. Must be either 'economy', 'government', 'suncolor' or 'techlevel' - tried to use " + section);
			return false;
		}
		this._f6Overlay[section] = null;
		return true;
	}

	//-------------------------------------------------------------------------------------------------------------
	this.startUp = function () {
		this._xr = worldScripts.XenonReduxUI;

		// check to make sure we have resources (with some scope for new packs)
		var res = ["Xenon UI Resource Pack A (16:9)", "Xenon UI Resource Pack B (16:10)",
			"Xenon UI Resource Pack C (16:9)", "Xenon UI Resource Pack D (16:10)",
			"Xenon UI Resource Pack E (16:9)", "Xenon UI Resource Pack F (16:10)",
			"Xenon UI Resource Pack G (16:9)", "Xenon UI Resource Pack H (16:10)",
			"Xenon UI Resource Pack I (16:9)", "Xenon UI Resource Pack J (16:10)",
			"Xenon UI Resource Pack K (16:9)", "Xenon UI Resource Pack L (16:10)",
			"Xenon UI Resource Pack M (16:9)", "Xenon UI Resource Pack N (16:10)"
		]

		var found = false;
		for (var i = 0; i < res.length; i++) {
			if (worldScriptNames.indexOf(res[i]) >= 0) {
				found = true;
				if (res[i].indexOf("16:9") > 0) this._ratio = "16_9"; else this._ratio = "16_10";
				break;
			}
		}
		if (found === false) {
			log(this.name, "ERROR! No Xenon UI Resource packs installed. Xenon UI cannot function without one resource pack installed.");
			// turn off the OXP here
			delete this.guiScreenChanged;
			delete this.missionScreenEnded;
			delete this.gameResumed;
			delete this.startUp;
			delete this.startUpComplete;
			delete this.infoSystemChanged;
			return;
		}

		var rp = oolite.resourcePaths; 
		var i = rp.length;
		while (i--) {
			if (rp[i].indexOf("Font") >= 0) {
				if (rp[i].indexOf("NovaSquare") > 0) {this._font = "novasquare"; break;}
				if (rp[i].indexOf("Discognate") > 0) {this._font = "discognate"; break;}
				if (rp[i].indexOf("Xolonium") > 0) {this._font = "xolonium"; break;}
				if (rp[i].indexOf("Dangerous") > 0) {this._font = "dangeroussquare"; break;}
				if (rp[i].indexOf("Dosis") > 0) {this._font = "dosis"; break;}
				if (rp[i].indexOf("OCR") > 0) {this._font = "ocraextended"; break;}
			}
		}

		var lib = worldScripts.Lib_GUI;

		if (missionVariables.XenonUI_ActiveMode) this._activeMode = (this._trueValues.indexOf(missionVariables.XenonUI_ActiveMode) >= 0 ? true : false);
		if (missionVariables.XenonUI_DisableOverlays) this._disableOverlays = (this._trueValues.indexOf(missionVariables.XenonUI_DisableOverlays) >= 0 ? true : false);
		// only reload the amber setting if lib config is present - otherwise we'll just take the preset value
		if (lib) {
			if (missionVariables.XenonUI_Amber) this._amber = (this._trueValues.indexOf(missionVariables.XenonUI_Amber) >= 0 ? true : false);
			this._lastAmber = this._amber;
		}
		if (!lib || lib.$cur === "XenonUI") {
			this.$updateGuiColorSettings();
		}

		if (worldScripts["BGS"]) this._useBGS = "BGS";
		if (worldScripts["market_observer3"]) this._marketObserverInstalled = true;
		if (worldScripts["hdbg"]) this._hdbgActive = true;

		if (!lib) this._enableLibGUIInt = false;
		if (lib && this._enableLibGUIInt === true) {
			// flag some screens as readonly
			this._addedID["oolite-contracts-parcels-none"] = 1;
			this._addedID["oolite-contracts-parcels-details"] = 1;
			this._addedID["oolite-contracts-parcels-summary"] = 1;
			this._addedID["oolite-contracts-passengers-none"] = 1;
			this._addedID["oolite-contracts-passengers-details"] = 1;
			this._addedID["oolite-contracts-passengers-summary"] = 1;
			this._addedID["oolite-contracts-cargo-none"] = 1;
			this._addedID["oolite-contracts-cargo-details"] = 1;
			this._addedID["oolite-contracts-cargo-summary"] = 1;
			this._addedID["oolite-primablemanager"] = 1;
			this._addedID["oolite-register"] = 1;
			this._addedID["Lib_Config"] = 1;

			// disable our standard functions so LibGUI can take over
			this.$previous_guiScreenChanged = this.guiScreenChanged;
			delete this.guiScreenChanged;
			var ver = lib.version.split(".");
			if (ver.length >= 2) {
				// only delete the infoSystemChanged if Lib GUI is greater than version 1.1
				if (parseInt(ver[0]) > 1 || parseInt(ver[1]) > 1 || ver.length > 2)
					delete this.infoSystemChanged;
			}
			delete this.missionScreenEnded;
			delete this.gameResumed;
			delete this.startUp;
			this.$libGUISetup();
		}
		// if we don't have library to orchestrate backgrounds, disable "better screens" so xenon UI has priority
		// otherwise, we end up with a clash of different background sets
		if (!lib && worldScripts["smivs_screen-backgrounds_worldscript"]) {
			worldScripts["smivs_screen-backgrounds_worldscript"].hide_guiScreenChanged = worldScripts["smivs_screen-backgrounds_worldscript"].guiScreenChanged;
			delete worldScripts["smivs_screen-backgrounds_worldscript"].guiScreenChanged;
		}

		this.$addStandardMissionScreenOverlays();
		this.$addStandardMissionScreenExceptions();

		if (worldScripts["BGS-M"] && (!lib || this._enableLibGUIInt === false)) {
			// bgs does a frame callback to apply screen overlays to the LRC
			// however, with 1.80 the overlays don't work anymore
			// unfortunately, the frame callback is still running which then removes the XenonUI F6 overlay
			// to prevent this, we're monkey patching the framecallback destination function to be an empty function
			// that appears to be the only way to stop BGS from doing the screen overlay thing, without actually getting a separate fix released for it.
			var bgs = worldScripts["BGS-M"];
			if (!bgs.x_doBGSChartsTimer) {
				bgs.x_doBGSChartsTimer = bgs.doBGSChartsTimer;
				bgs.doBGSChartsTimer = function (delta) {};
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.startUpComplete = function () {
		// register our settings, if Lib_Config is present
		if (worldScripts.Lib_Config) {
			worldScripts.Lib_Config._registerSet(this._xenonConfig);
			this.$setBGSDefaultBackgrounds();
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.playerWillSaveGame = function () {
		missionVariables.XenonUI_ActiveMode = this._activeMode;
		missionVariables.XenonUI_DisableOverlays = this._disableOverlays;
		missionVariables.XenonUI_Amber = this._amber;
	}

	//-------------------------------------------------------------------------------------------------------------
	this.chartHighlightModeChanged = function (newMode) {
		if (this._enableLibGUIInt === false) {
			this.guiScreenChanged("", "");
		} else {
			var lib = worldScripts.Lib_GUI;
			var f6ovr = "f6-overlay.png";
			if (player.ship.hasOwnProperty("chartHighlightMode") === true) {
				switch (player.ship.chartHighlightMode) {
					case "OOLRC_MODE_SUNCOLOR":
						var lookup = this._f6Overlay.suncolor;
						if (lookup) {
							f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
						} else {
							f6ovr = "f6-overlay-suncolor.png";
						}
						break;
					case "OOLRC_MODE_TECHLEVEL":
						var lookup = this._f6Overlay.techlevel;
						if (lookup) {
							f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
						} else {
							f6ovr = "f6-overlay-techlevel.png";
						}
						break;
					case "OOLRC_MODE_ECONOMY":
						var lookup = this._f6Overlay.economy;
						if (lookup) {
							f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
						} else {
							f6ovr = "f6-overlay-economy.png";
						}
						break;
					case "OOLRC_MODE_GOVERNMENT":
						var lookup = this._f6Overlay.government;
						if (lookup) {
							f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
						} else {
							f6ovr = "f6-overlay-government.png";
						}
						break;
				}
			}
			lib.$guis.XenonUI.GUI_SCREEN_SHORT_RANGE_CHART.ov = {
				name: f6ovr,
				height: 546
			};
			lib.$guis.XenonUI.GUI_SCREEN_LONG_RANGE_CHART.ov = {
				name: f6ovr,
				height: 546
			};
			lib.guiScreenChanged();
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.infoSystemChanged = function (to, from) {
		var lib = worldScripts.Lib_GUI;
		if (lib && this._enableLibGUIInt === true) {
			// force lib gui to do a screen update
			lib.guiScreenChanged();
			return;
		}
		this.guiScreenChanged(guiScreen, guiScreen);
	}

	//-------------------------------------------------------------------------------------------------------------
	this.guiScreenChanged = function (to, from) {

		var p = player.ship;
		var startTimer = false;
		var imagetype = "standard";
		var redalert = "";
		var nohud = "";
		var imagename = "";
		var ana = "";
		var overlay = "";
		var overlayHeight = 0;

		if (this._marketObserverInstalled) {
			if (from === "GUI_SCREEN_MARKET" || from === "GUI_SCREEN_MARKETINFO" || to === "GUI_SCREEN_MARKET" || to === "GUI_SCREEN_MARKETINFO")
				// Because market observer uses a specialised HUD a slightly incorrect background image was being selected.
				// This was unnoticable if the "allow_big_gui" was not enabled on the default HUD, but quite noticeable on the Xenon HUD.
				// So we start a timer to force the correct background after Market Observer has set or reset the HUD.
				startTimer = true;
		}

		// fix for the issue where exiting the ship library screen could sometimes result in the wrong background being displayed
		// this would only be apparent with HUD's that don't implement the "allow_big_gui" setting.
		if (from === "GUI_SCREEN_SHIPLIBRARY" && p && p.docked) startTimer = true;

		if (p.alertCondition === 3) redalert = "_redalert";
		if (p.hudHidden) nohud = "_nohud";
		if (this.$isBigGuiActive() === true) nohud = "_nohud";
		if (p.isInSpace) {
			imagetype = "inflight";
			// going to assume that nohud versions of inflight guis are not that important
			nohud = "";
		}

		switch (guiScreen) {
			case "GUI_SCREEN_MAIN":
				return;
			case "GUI_SCREEN_SHIPLIBRARY":
				imagename = "f0";
				nohud = "_nohud";
				break;
			case "GUI_SCREEN_KEYBOARD":
				imagename = "nomenu";
				nohud = "_nohud";
				if (this._disableOverlays === false) overlay = "xui-keyboard.png";
				break;
			case "GUI_SCREEN_STICKPROFILE":
				imagename = "f2";
				nohud = "";
				break;
			case "GUI_SCREEN_STICKMAPPER":
				imagename = "f2";
				nohud = "";
				if (this._disableOverlays === false) overlay = "xui-gamepad.png";
				break;
			case "GUI_SCREEN_OPTIONS":
				imagename = "f2";
				if (this._disableOverlays === false) overlay = "xui-settings.png";
				break;
			case "GUI_SCREEN_LOAD":
				imagename = "f2";
				if (this._disableOverlays === false) overlay = "xui-load.png";
				break;
			case "GUI_SCREEN_SAVE":
				imagename = "f2";
				if (this._disableOverlays === false) overlay = "xui-save.png";
				break;
			case "GUI_SCREEN_SAVE_OVERWRITE":
				imagename = "f2";
				if (this._disableOverlays === false) overlay = "xui-question.png";
				break;
			case "GUI_SCREEN_EQUIP_SHIP":
				imagename = "f3-1";
				if (this._disableOverlays === false) overlay = "xui-equipship.png";
				break;
			case "GUI_SCREEN_SHIPYARD":
				imagename = "f3-2";
				break;
			case "GUI_SCREEN_INTERFACES":
				imagename = "f4";
				if (this._disableOverlays === false) overlay = "xui-interfaces.png";
				break;
			case "GUI_SCREEN_STATUS":
				imagename = "f5-1";
				// override BGS status and nocrowd image, if installed.
				if (worldScripts["BGS-M"]) {
					worldScripts["BGS-M"].bgsI.status = {
						name: (this._amber === true ? "amber" : "xenon") + "_" + imagetype + "_" + imagename + nohud + redalert + ".png",
						height: 546
					};
					worldScripts["BGS-M"].bgsI.noCrowd = {
						name: (this._amber === true ? "amber" : "xenon") + "_" + imagetype + "_" + imagename + nohud + redalert + ".png",
						height: 546
					};
				}
				/*if (this._disableOverlays === false) {
					if (this.$playerHasDamagedEquipment() === true) {
						overlay = "xui-warning.png";
					} else {
						overlay = "xui-ok.png";
					}
				}*/
				break;
			case "GUI_SCREEN_MANIFEST":
				imagename = "f5-2";
				if (this._disableOverlays === false) overlay = "xui-clipboard.png";
				break;
			case "GUI_SCREEN_SHORT_RANGE_CHART":
			case "GUI_SCREEN_LONG_RANGE_CHART":
				if (guiScreen == "GUI_SCREEN_SHORT_RANGE_CHART") imagename = "f6";
				if (guiScreen == "GUI_SCREEN_LONG_RANGE_CHART") {
					imagename = "f6-2";
					if (worldScripts["BGS-M"]) worldScripts["BGS-M"].bgsI.lrc = {
						name: (this._amber === true ? "amber" : "xenon") + "_" + imagetype + "_" + imagename + ana + nohud + redalert + ".png",
						height: 546
					};
				}
				ana = "_ana";
				if (p.equipmentStatus("EQ_ADVANCED_NAVIGATIONAL_ARRAY") != "EQUIPMENT_OK") ana = "_no_ana";
				if (p.hasOwnProperty("chartHighlightMode") === true) {
					switch (p.chartHighlightMode) {
						case "OOLRC_MODE_SUNCOLOR":
							var lookup = this._f6Overlay.suncolor;
							if (lookup) {
								overlay = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
							} else {
								overlay = "f6-overlay-suncolor.png";
							}
							break;
						case "OOLRC_MODE_TECHLEVEL":
							var lookup = this._f6Overlay.techlevel;
							if (lookup) {
								overlay = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
							} else {
								overlay = "f6-overlay-techlevel.png";
							}
							break;
						case "OOLRC_MODE_ECONOMY":
							var lookup = this._f6Overlay.economy;
							if (lookup) {
								overlay = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
							} else {
								overlay = "f6-overlay-economy.png";
							}
							break;
						case "OOLRC_MODE_GOVERNMENT":
							var lookup = this._f6Overlay.government;
							if (lookup) {
								overlay = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
							} else {
								overlay = "f6-overlay-government.png";
							}
							break;
					}
				} else {
					overlay = "f6-overlay.png";
				}
				break;
			case "GUI_SCREEN_SYSTEM_DATA":
				imagename = "f7";
				var sysinf = System.infoForSystem(galaxyNumber, p.targetSystem);
				if (sysinf["sun_gone_nova"] || sysinf["sun_going_nova"]) imagename += "_nova";
				break;
			case "GUI_SCREEN_MARKET":
				imagename = "f8";
				if (this._disableOverlays === false) overlay = "xui-market.png";
				if (this._marketObserverInstalled) imagename += "_wide";
				break;
			case "GUI_SCREEN_MARKETINFO":
				imagename = "f8-2";
				if (this._marketObserverInstalled) imagename += "_wide";
				if (this._disableOverlays === false) overlay = "xui-tag.png";
				break;
			case "GUI_SCREEN_REPORT":
				imagename = "f0";
				if (this._disableOverlays === false) overlay = "xui-report.png";
				if (this._hdbgActive === true) return;
				break;
			case "GUI_SCREEN_MISSION":
				imagename = "f0";
				// look for any exceptions and jump out if found
				if (this._disableMissionScreen.indexOf(mission.screenID) != -1) imagename = "disable";
				// check if we have an overlay to apply to this mission screen
				if (this._disableOverlays === false && this._disableMissionScreen.indexOf(mission.screenID) === -1) {
					if (this._missionOverlay[mission.screenID]) overlay = this._missionOverlay[mission.screenID];
				}
				break;
		}

		// build the file filename if it hasn't been disabled
		if (imagename != "disable") {
			imagename = (this._amber === true ? "amber_" : "xenon_") + imagetype + "_" + imagename + ana + nohud + redalert + ".png";
			// load the image to the background
			setScreenBackground({
				name: imagename,
				height: 546
			});
		}
		// apply the overlay if one was found
		if (overlay != "") setScreenOverlay({
			name: overlay,
			height: 546
		});

		// start a timer to force the background, if required
		if (startTimer) {
			if (this._chartExitScreenTimer && this._chartExitScreenTimer.isRunning) this._chartExitScreenTimer.stop();
			this._chartExitScreenTimer = new Timer(this, this.$addBackground, 0.25, 0);
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.missionScreenEnded = function () {
		this.guiScreenChanged("", "");
	}

	//-------------------------------------------------------------------------------------------------------------
	this.gameResumed = function () {
		this.guiScreenChanged("", "");
	}

	//-------------------------------------------------------------------------------------------------------------
	// adds overlays to the standard mission screens
	this.$addStandardMissionScreenOverlays = function $addStandardMissionScreenOverlays() {
		this.$addMissionOverlay("oolite-contracts-parcels-none", "xui-briefcase.png");
		this.$addMissionOverlay("oolite-contracts-parcels-summary", "xui-briefcase.png");
		this.$addMissionOverlay("oolite-contracts-passengers-none", "xui-boardingpass.png");
		this.$addMissionOverlay("oolite-contracts-passengers-summary", "xui-boardingpass.png");
		this.$addMissionOverlay("oolite-contracts-cargo-none", "xui-cargo.png");
		this.$addMissionOverlay("oolite-contracts-cargo-summary", "xui-cargo.png");
		this.$addMissionOverlay("oolite-primablemanager", "xui-switch.png");
		this.$addMissionOverlay("oolite-register", "xui-register.png");
	}

	//-------------------------------------------------------------------------------------------------------------
	// adds map screen exceptions for known screen ID's
	this.$addStandardMissionScreenExceptions = function $addStandardMissionScreenExceptions() {
		// each of these items can't be set using setScreenBackground - needs to be done via the plist file
		this.$addMissionScreenException("oolite-contracts-parcels-details");
		this.$addMissionScreenException("oolite-contracts-passengers-details");
		this.$addMissionScreenException("oolite-contracts-cargo-details");
		this.$addMissionScreenException("rrs_mission_map");
		this.$addMissionScreenException("blackmonks-map");
		this.$addMissionScreenException("ups-map");
		this.$addMissionScreenException("escort-contracts");
		if (worldScripts.Random_Hits) {
			var ver = worldScripts.Random_Hits.version.split(".");
			if (parseInt(ver[0]) > 1 ||
				(parseInt(ver[0]) === 1 && parseInt(ver[1])) > 11 ||
				(parseInt(ver[0]) === 1 && parseInt(ver[1]) === 11 && parseInt(ver[2]) > 2)) {
				this.$addMissionScreenException("random_hits_map");
			} else {
				this.$addMissionScreenException("random_hits");
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// callback handler for when settings are changed
	this.$changeSettings = function $changeSettings() {
		var lib = worldScripts.Lib_GUI;
		if (this._amber != this._lastAmber) {
			this.$libGUISetup();
			this._lastAmber = this._amber;
		}
		if (lib.$cur === "XenonUI") {
			if (global.setScreenBackgroundForKey) this.$setBackgroundDefaults(true);
			this.$updateGuiColorSettings();
		}
		// because of changes to library gui (from version 1.7), overlays can no longer be changed dynamically and will always be on/off
		// until the game is saved and reloaded
		/*if (this._disableOverlays === true) {
			if (this._enableLibGUIInt === true && lib && lib.$cur === "XenonUI") {
				var keys = Object.keys(this._missionOverlay);
				for (var i = 0; i < keys.length; i++) {
					if (!this._addedID[keys[i]] || this._addedID[keys[i]] === 2) {
						if (!lib.$IDRules[keys[i]]) {
							this._addedID[keys[i]] = 2;
							this.$passConfig(keys[i]);
						}
						lib.$IDRules[keys[i]] = {
							pic: 1,
							ov: 0
						};
					}
				}
			}
		} else {
			if (this._enableLibGUIInt === true && lib && lib.$cur === "XenonUI") {
				var keys = Object.keys(this._missionOverlay);
				for (var i = 0; i < keys.length; i++) {
					if (!this._addedID[keys[i]] || this._addedID[keys[i]] === 2) {
						if (!lib.$IDRules[keys[i]]) {
							this._addedID[keys[i]] = 2;
							this.$passConfig(keys[i]);
						}
						lib.$IDRules[keys[i]] = {
							pic: 1,
							ov: 1
						};
					}
				}
			}
		}*/
		// reset our monitoring array regardless of which gui is active
		if (this._activeMode === false) this._backgroundOverride = [];

		this.$setupGeneralOverlays();
	}

	//-------------------------------------------------------------------------------------------------------------
	// sets up the LibGUI parameters
	this.$libGUISetup = function $libGUISetup() {

		// setup some event callbacks
		this.guiScreenChanged = this.$guiScreenChanged;
		this.equipmentDamaged = this.$equipmentDamaged;
		this.equipmentRepaired = this.$equipmentRepaired;
		this.equipmentAdded = this.$equipmentAdded;
		this.equipmentRemoved = this.$equipmentRemoved;
		this.missionScreenEnded = this.$missionScreenEnded;

		var style = (this._amber === true ? "amber" : "xenon");

		var lib = worldScripts.Lib_GUI;
		lib.$guis.XenonUI = {
			GUI_SCREEN_GAMEOPTIONS: {
				pic: {
					name: style + "_standard_f2.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f2_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f2.png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f2_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_LOAD: {
				pic: {
					name: style + "_standard_nomenu.png",
					height: 546
				}
			},
			GUI_SCREEN_SAVE: {
				pic: {
					name: style + "_standard_nomenu.png",
					height: 546
				}
			},
			GUI_SCREEN_SAVE_OVERWRITE: {
				pic: {
					name: style + "_standard_nomenu.png",
					height: 546
				}
			},
			GUI_SCREEN_EQUIP_SHIP: {
				pic: {
					name: style + "_standard_f3-1.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f3-1_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_INTERFACES: {
				pic: {
					name: style + "_standard_f4.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f4_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_KEYBOARD: {
				pic: {
					name: style + "_standard_nomenu_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_standard_nomenu_nohud.png",
					height: 546
				}
			},
			GUI_SCREEN_MANIFEST: {
				pic: {
					name: style + "_standard_f5-2.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f5-2_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f5-2.png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f5-2_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_MARKET: {
				pic: {
					name: style + "_standard_f8" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f8" + (this._marketObserverInstalled === true ? "_wide" : "") + "_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f8" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f8" + (this._marketObserverInstalled === true ? "_wide" : "") + "_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_MARKETINFO: {
				pic: {
					name: style + "_standard_f8-2" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f8-2" + (this._marketObserverInstalled === true ? "_wide" : "") + "_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f8-2" + (this._marketObserverInstalled === true ? "_wide" : "") + ".png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f8-2" + (this._marketObserverInstalled === true ? "_wide" : "") + "_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_OPTIONS: {
				pic: {
					name: style + "_standard_f2.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f2_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f2.png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f2_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_SHIPLIBRARY: {
				pic: {
					name: style + "_standard_f0_nohud.png",
					height: 546
				}
			},
			GUI_SCREEN_SHIPYARD: {
				pic: {
					name: style + "_standard_f3-2.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f3-2_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_STATUS: {
				pic: {
					name: style + "_standard_f5-1.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f5-1_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f5-1.png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f5-1_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_STICKMAPPER: {
				pic: {
					name: style + "_standard_f2.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f2.png",
					height: 546
				},
				picFlight: {
					name: style + "_flight_f2.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_STICKPROFILE: {
				pic: {
					name: style + "_standard_f2.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f2.png",
					height: 546
				},
				picFlight: {
					name: style + "_flight_f2.png",
					height: 546
				},
				sndRef: this._useBGS
			},
			GUI_SCREEN_SYSTEM_DATA: {
				pic: {
					name: style + "_standard_f7.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f7_nohud.png",
					height: 546
				},
				picNova: {
					name: style + "_standard_f7_nova.png",
					height: 546
				},
				picNovaBig: {
					name: style + "_standard_f7_nova_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f7.png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f7_redalert.png",
					height: 546
				},
				picNovaRed: {
					name: style + "_inflight_f7_nova_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			},
		};

		if (this._hdbgActive === false) {
			lib.$guis.XenonUI.generic = {
				pic: {
					name: style + "_standard_nomenu.png",
					height: 546
				},
				sndRef: this._useBGS
			};
			lib.$guis.XenonUI.GUI_SCREEN_REPORT = {
				pic: {
					name: style + "_standard_f0.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f0_nohud.png",
					height: 546
				},
				sndRef: this._useBGS
			};
		}

		this.$setupVariableScreens();
		this.$setupGeneralOverlays();

		// mission screen IDs
		lib.$IDs.XenonUI = {
			generic: {
				pic: {
					name: style + "_standard_f0.png",
					height: 546
				},
				picBig: {
					name: style + "_standard_f0_nohud.png",
					height: 546
				},
				picFlight: {
					name: style + "_inflight_f0.png",
					height: 546
				},
				picRed: {
					name: style + "_inflight_f0_redalert.png",
					height: 546
				},
				sndRef: this._useBGS
			}
		};

	}

	//-------------------------------------------------------------------------------------------------------------
	// adjust background images on F5/F6 screen based on availability/status of equipment
	this.$setupVariableScreens = function $setupVariableScreens() {
		var ana = "_ana";
		if (player.ship.equipmentStatus("EQ_ADVANCED_NAVIGATIONAL_ARRAY") != "EQUIPMENT_OK") ana = "_no_ana";

		var lib = worldScripts.Lib_GUI;
		var f6ovr = "f6-overlay.png";
		if (player.ship.hasOwnProperty("chartHighlightMode") === true) {
			switch (player.ship.chartHighlightMode) {
				case "OOLRC_MODE_SUNCOLOR":
					var lookup = this._f6Overlay.suncolor;
					if (lookup) {
						f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
					} else {
						f6ovr = "f6-overlay-suncolor.png";
					}
					break;
				case "OOLRC_MODE_TECHLEVEL":
					var lookup = this._f6Overlay.techlevel;
					if (lookup) {
						f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
					} else {
						f6ovr = "f6-overlay-techlevel.png";
					}
					break;
				case "OOLRC_MODE_ECONOMY":
					var lookup = this._f6Overlay.economy;
					if (lookup) {
						f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
					} else {
						f6ovr = "f6-overlay-economy.png";
					}
					break;
				case "OOLRC_MODE_GOVERNMENT":
					var lookup = this._f6Overlay.government;
					if (lookup) {
						f6ovr = worldScripts[lookup.worldscript][lookup.functionname](this._ratio, this._font);
					} else {
						f6ovr = "f6-overlay-government.png";
					}
					break;					
			}
		}

		var style = (this._amber === true ? "amber" : "xenon");

		lib.$guis.XenonUI.GUI_SCREEN_SHORT_RANGE_CHART = {
			pic: {
				name: style + "_standard_f6" + ana + ".png",
				height: 546
			},
			picBig: {
				name: style + "_standard_f6" + ana + "_nohud.png",
				height: 546
			},
			picFlight: {
				name: style + "_inflight_f6" + ana + ".png",
				height: 546
			},
			picRed: {
				name: style + "_inflight_f6" + ana + "_redalert.png",
				height: 546
			},
			ov: {
				name: f6ovr,
				height: 546
			},
			sndRef: this._useBGS
		};
		lib.$guis.XenonUI.GUI_SCREEN_LONG_RANGE_CHART = {
			pic: {
				name: style + "_standard_f6-2" + ana + ".png",
				height: 546
			},
			picBig: {
				name: style + "_standard_f6-2" + ana + "_nohud.png",
				height: 546
			},
			picFlight: {
				name: style + "_inflight_f6-2" + ana + ".png",
				height: 546
			},
			picRed: {
				name: style + "_inflight_f6-2" + ana + "_redalert.png",
				height: 546
			},
			ov: {
				name: f6ovr,
				height: 546
			},
			sndRef: this._useBGS
		};

	}

	//-------------------------------------------------------------------------------------------------------------
	// adjust overlay images on general (non-mission) screens
	this.$setupGeneralOverlays = function $setupGeneralOverlays() {

		var lib = worldScripts.Lib_GUI.$guis.XenonUI;

		if (this._disableOverlays) {
			delete lib.GUI_SCREEN_GAMEOPTIONS["ov"];
			delete lib.GUI_SCREEN_LOAD["ov"];
			delete lib.GUI_SCREEN_SAVE["ov"];
			delete lib.GUI_SCREEN_SAVE_OVERWRITE["ov"];
			delete lib.GUI_SCREEN_EQUIP_SHIP["ov"];
			delete lib.GUI_SCREEN_INTERFACES["ov"];
			delete lib.GUI_SCREEN_KEYBOARD["ov"];
			delete lib.GUI_SCREEN_MANIFEST["ov"];
			delete lib.GUI_SCREEN_MARKET["ov"];
			delete lib.GUI_SCREEN_MARKETINFO["ov"];
			delete lib.GUI_SCREEN_OPTIONS["ov"];
			if (this._hdbgActive === false) delete lib.GUI_SCREEN_REPORT["ov"];
			//delete lib.GUI_SCREEN_STATUS["ov"];
			delete lib.GUI_SCREEN_STICKMAPPER["ov"];
		} else {
			lib.GUI_SCREEN_GAMEOPTIONS["ov"] = {
				name: "xui-settings.png",
				height: 546
			};
			lib.GUI_SCREEN_LOAD["ov"] = {
				name: "xui-load.png",
				height: 546
			};
			lib.GUI_SCREEN_SAVE["ov"] = {
				name: "xui-save.png",
				height: 546
			};
			lib.GUI_SCREEN_SAVE_OVERWRITE["ov"] = {
				name: "xui-question.png",
				height: 546
			};
			lib.GUI_SCREEN_EQUIP_SHIP["ov"] = {
				name: "xui-equipship.png",
				height: 546
			};
			lib.GUI_SCREEN_INTERFACES["ov"] = {
				name: "xui-interfaces.png",
				height: 546
			};
			lib.GUI_SCREEN_KEYBOARD["ov"] = {
				name: "xui-keyboard.png",
				height: 546
			};
			lib.GUI_SCREEN_MANIFEST["ov"] = {
				name: "xui-clipboard.png",
				height: 546
			};
			lib.GUI_SCREEN_MARKET["ov"] = {
				name: "xui-market.png",
				height: 546
			};
			lib.GUI_SCREEN_MARKETINFO["ov"] = {
				name: "xui-tag.png",
				height: 546
			};
			lib.GUI_SCREEN_OPTIONS["ov"] = {
				name: "xui-settings.png",
				height: 546
			};
			if (this._hdbgActive === false) lib.GUI_SCREEN_REPORT["ov"] = {
				name: "xui-report.png",
				height: 546
			};
			lib.GUI_SCREEN_STICKMAPPER["ov"] = {
				name: "xui-gamepad.png",
				height: 546
			};
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// alternate version for use with Lib_GUI
	this.$missionScreenEnded = function $missionScreenEnded() {
		var lib = worldScripts.Lib_GUI;
		if (this._guiActive != lib.$cur && (this._guiActive === "XenonUI" || lib.$cur === "XenonUI")) {
			if (lib.$cur === "XenonUI") {
				// if we've got it available (v1.85/86) then force the default screen background to be Xenon's
				if (global.setScreenBackgroundForKey) this.$setBackgroundDefaults(true);
				this.$updateGuiColorSettings();
				for (var i = 0; i < this._disableMissionScreen.length; i++) {
					if (!this._addedID[this._disableMissionScreen[i]] || this._addedID[this._disableMissionScreen[i]] === 2) {
						if (!lib.$IDRules[this._disableMissionScreen[i]]) {
							this._addedID[this._disableMissionScreen[i]] = 2;
							this.$passConfig(this._disableMissionScreen[i]);
							// we can only add now...
							lib.$IDRules[this._disableMissionScreen[i]] = {
								mpic: 0
							};
						}
						//lib.$IDRules[this._disableMissionScreen[i]] = {
						//	mpic: 0
						//};
					}
				}
				var keys = Object.keys(this._missionOverlay);
				for (var i = 0; i < keys.length; i++) {
					// update Lib_GUI with our settings for mission screens
					if (!this._addedID[keys[i]] || this._addedID[keys[i]] === 2) {
						if (!lib.$IDRules[keys[i]]) {
							this._addedID[keys[i]] = 2;
							this.$passConfig(keys[i]);
							if (this._missionOverlay[keys[i]] != "") {
								if (this._disableOverlays === true) {
									lib.$IDRules[keys[i]] = {
										pic: 1,
										ov: 0
									};
								} else {
									lib.$IDRules[keys[i]] = {
										pic: 1,
										ov: 1
									};
								}
							} else {
								lib.$IDRules[keys[i]] = {
									pic: 1
								};
							}
						}
						/*if (this._missionOverlay[keys[i]] != "") {
							if (this._disableOverlays === true) {
								lib.$IDRules[keys[i]] = {
									pic: 1,
									ov: 0
								};
							} else {
								lib.$IDRules[keys[i]] = {
									pic: 1,
									ov: 1
								};
							}
						} else {
							lib.$IDRules[keys[i]] = {
								pic: 1
							};
						}*/
					}
				}
			} else {
				// if we've got it available (v1.85/86) then reset the default screen backgrounds to be what it was previously
				if (global.setScreenBackgroundForKey) this.$setBackgroundDefaults(false);
			}
			this._guiActive = lib.$cur;
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	// gets any unrecorded mission ScreenID's add add them to the list of rules to use Xenon
	this.$guiScreenChanged = function $guiScreenChanged(to, from) {
		if (guiScreen === "GUI_SCREEN_STATUS") {
			// this bit should only be run once after load
			if (this._guiActive === "") {
				// force all the screen rego stuff to happen if xenon was set after loading
				if (worldScripts.Lib_GUI.$cur === "XenonUI") {
					this.missionScreenEnded();
					this.$setBackgroundDefaults(true);
				}
				// make sure our local copy gets updated
				this._guiActive = worldScripts.Lib_GUI.$cur;
			}
		}

		if (guiScreen === "GUI_SCREEN_MISSION" && this._guiActive === "XenonUI" && this._activeMode === true) {
			if (mission.screenID && mission.screenID != "" && this._disableMissionScreen.indexOf(mission.screenID) === -1) {
				this.$addMissionOverlay(mission.screenID, null);
				// make a note of any screen we add this way
				if (this._backgroundOverride.indexOf(mission.screenID) === -1) this._backgroundOverride.push(mission.screenID);
				// force an update
				worldScripts.Lib_GUI.guiScreenChanged();
			}
			// we'll try to override LibGUI if a mission screen is displayed without a screenID
			if (!mission.screenID) {
				this.$previous_guiScreenChanged(to, from);
			}
		}

		// force an update so the right overlay is used
		if (to == "GUI_SCREEN_SHORT_RANGE_CHART" || to == "GUI_SCREEN_LONG_RANGE_CHART") {
			this.chartHighlightModeChanged();
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentAdded = function $equipmentAdded(equipmentKey) {
		if (equipmentKey != "EQ_ADVANCED_NAVIGATIONAL_ARRAY") return;
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentRemoved = function $equipmentRemoved(equipmentKey) {
		if (equipmentKey != "EQ_ADVANCED_NAVIGATIONAL_ARRAY") return;
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentDamaged = function $equipmentDamaged(equipmentKey) {
		if (equipmentKey != "EQ_ADVANCED_NAVIGATIONAL_ARRAY") return;
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// activated when Library GUI is in use
	this.$equipmentRepaired = function $equipmentRepaired(equipmentKey) {
		if (equipmentKey != "EQ_ADVANCED_NAVIGATIONAL_ARRAY") return;
		this.$setupVariableScreens();
	}

	//-------------------------------------------------------------------------------------------------------------
	// forces a background after the timer has expired
	this.$addBackground = function $addBackground() {
		this.guiScreenChanged("", "");
	}

	//-------------------------------------------------------------------------------------------------------------
	// returns true if a HUD with allowBigGUI is enabled, otherwise false
	this.$isBigGuiActive = function $isBigGuiActive() {
		if (oolite.compareVersion("1.83") <= 0) {
			return player.ship.hudAllowsBigGui;
		} else {
			var bigGuiHUD = ["XenonHUD.plist", "coluber_hud_ch01-dock.plist"]; // until there is a property we can check, I'll be listing HUD's that have the allow_big_gui property set here
			if (bigGuiHUD.indexOf(player.ship.hud) >= 0) {
				return true;
			} else {
				return false;
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// returns true if any of the visible equipment items are damaged
	this.$playerHasDamagedEquipment = function $playerHasDamagedEquipment() {
		var p = player.ship;
		var eq = p.equipment;
		for (var i = 0; i < eq.length; i++) {
			var itm = eq[i];
			if (itm.isVisible === true && p.equipmentStatus(itm.equipmentKey) === "EQUIPMENT_DAMAGED") return true;
		}
		return false;
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$setBackgroundDefaults = function $setBackgroundDefaults(onOff) {
		if (onOff === true) {
			this._defaultBG["long_range_chart_mission"] = getScreenBackgroundForKey("long_range_chart_mission");
			this._defaultBG["short_range_chart_mission"] = getScreenBackgroundForKey("short_range_chart_mission");
			this._defaultBG["custom_chart_mission"] = getScreenBackgroundForKey("custom_chart_mission");
			this._defaultBG["long_range_chart"] = getScreenBackgroundForKey("long_range_chart");
			this._defaultBG["short_range_chart"] = getScreenBackgroundForKey("short_range_chart");
			this._defaultBG["keyboardsettings"] = getScreenBackgroundForKey("keyboardsettings");
			this._defaultBG["load_save"] = getScreenBackgroundForKey("load_save");
			this._defaultBG["mount_weapon"] = getScreenBackgroundForKey("mount_weapon");
			this._defaultBG["newgame"] = getScreenBackgroundForKey("newgame");
			this._defaultBG["settings"] = getScreenBackgroundForKey("settings");
			this._defaultBG["shiplibrary"] = getScreenBackgroundForKey("shiplibrary");
			this._defaultBG["paused_docked_overlay"] = getScreenBackgroundForKey("paused_docked_overlay");
			this._defaultBG["paused_overlay"] = getScreenBackgroundForKey("paused_overlay");
			this._defaultBG["oxz-manager"] = getScreenBackgroundForKey("oxz-manager");

			var p = player.ship;
			var nohud = "";
			var ana = "_ana";
			var style = (this._amber === true ? "amber" : "xenon");
			if (this.$isBigGuiActive() === true) nohud = "_nohud";
			if (p.equipmentStatus("EQ_ADVANCED_NAVIGATIONAL_ARRAY") != "EQUIPMENT_OK") ana = "_no_ana";

			var imagename = style + "_standard_f0" + nohud + ".png";
			setScreenBackgroundForKey("short_range_chart_mission", {
				name: imagename,
				height: 546
			});
			setScreenBackgroundForKey("long_range_chart_mission", {
				name: imagename,
				height: 546
			});

			imagename = style + "_standard_f6" + ana + nohud + ".png";
			setScreenBackgroundForKey("short_range_chart", {
				name: imagename,
				height: 546
			});
			imagename = style + "_standard_f6-2" + ana + nohud + ".png";
			setScreenBackgroundForKey("long_range_chart", {
				name: imagename,
				height: 546
			});

			setScreenBackgroundForKey("keyboardsettings", {
				name: style + "_standard_nomenu_nohud.png",
				height: 546
			});
			setScreenBackgroundForKey("load_save", {
				name: style + "_standard_nomenu.png",
				height: 546
			});
			setScreenBackgroundForKey("mount_weapon", {
				name: style + "_standard_f3-1.png",
				height: 546
			});
			setScreenBackgroundForKey("newgame", {
				name: style + "_standard_nomenu_nohud.png",
				height: 546
			});
			setScreenBackgroundForKey("settings", {
				name: style + "_inflight_f2.png",
				height: 546
			});
			setScreenBackgroundForKey("shiplibrary", {
				name: style + "_standard_f0_nohud.png",
				height: 546
			});
			setScreenBackgroundForKey("paused_docked_overlay", {
				name: "pause_overlay.png",
				height: 546
			});
			setScreenBackgroundForKey("paused_overlay", {
				name: "pause_overlay.png",
				height: 546
			});
			imagename = "xenon_oxzmgr.png";
			setScreenBackgroundForKey("oxz-manager", {
				name: imagename,
				height: 546
			});
		} else {
			var keys = Object.keys(this._defaultBG);
			for (var i = 0; i < keys.length; i++) {
				var bg = getScreenBackgroundForKey(keys[i]);
				// are xenon's backgrounds still set?
				if (bg.hasOwnProperty("name") === true && (bg.name.indexOf(style + "_standard") >= 0 || bg.name.indexOf("pause_overlay") >= 0)) {
					// ok, we can reset things now
					setScreenBackgroundForKey(keys[i], this._defaultBG[keys[i]]);
				}
			}
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$updateGuiColorSettings = function $updateGuiColorSettings() {
		if (global.setGuiColorSettingForKey) {
			setGuiColorSettingForKey("screen_divider_color", (this._amber === false ? "0 74 127 200" : "127 74 0 200"));
			setGuiColorSettingForKey("screen_title_color", (this._amber === false ? "0 148 255 200" : "255 148 0 200"));
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	this.$setBGSDefaultBackgrounds = function $setBGSDefaultBackgrounds() {
		// until BGS has something similar
		if (worldScripts.Lib_GUI.$cur == "BGS" && global.setGuiColorSettingForKey) {
			setGuiColorSettingForKey("screen_title_color", "0.9 0.9 0.8");
			setGuiColorSettingForKey("screen_divider_color", "0.0 0.0 0.0 0.0");

			setScreenBackgroundForKey("paused_docked_overlay", "bgs_ov_paused.png");
			setScreenBackgroundForKey("paused_overlay", "bgs_ov_paused.png");
			setScreenBackgroundForKey("intro", "bgs_intro.png");
			setScreenBackgroundForKey("keyboardsettings", "bgs_fullscr.png");
			setScreenBackgroundForKey("load_save", "bgs_options.png");
			setScreenBackgroundForKey("newgame", "bgs_intro.png");
			setScreenBackgroundForKey("oxz-manager", "bgs_fullscr.png");
			setScreenBackgroundForKey("settings", "bgs_options.png");
			setScreenBackgroundForKey("shiplibrary", "bgs_fullscr.png");
			setScreenBackgroundForKey("long_range_chart_mission", "lib_black.png");
			setScreenBackgroundForKey("short_range_chart_mission", "lib_black.png");
		}
	}

	//-------------------------------------------------------------------------------------------------------------
	// used to allow XenonUI and XenonReduxUI to co-exist
	this.$xenonCrossConfig = function $xenonCrossConfig(screenID) {
		this._addedID[screenID] = 2;
	}
	this.$passConfig = function $passConfig(screenID) {
		if (this._xr) this._xr.$xenonCrossConfig(screenID);
	}

	
}).call(this);