"use strict";
this.name = "Smugglers_Contracts";
this.author = "phkb and cim";
this.description = "Adds smuggling contracts via an interface screen, with code borrowed heavily from the cargo contracts system, authored by cim.";
this.licence = "CC BY-NC-SA 3.0";

this._disabled = false;

/*
Note: The smuggling contract system is independent of the standard cargo contract system. The reason for this is that
smuggled cargo might exist in a number of different places (smuggling compartment, or hidden under a manifest relabel)
This means the normal cargo contract system cannot pick up the presence of hidden cargo.

The end result is that a lot of the standard cargo contract core code has been reproduced here as JS.

TODO: add "Negotiate" option to haggle for the fee
*/

//=============================================================================================================
/**** Configuration options and API ****/

/* OXPs which wish to add a background to the summary pages should
   set this value */
this._smugglingSummaryPageBackground = "";
/* OXPs which wish to add an overlay to the parcel mission screens
   should set this value */
this._smugglingPageOverlay = "";
this._smugglingPageOverlayHeight = 0;

this._suspendedDestination = null;
this._suspendedHUD = false;
this._smugglingContracts = [];
this._smugglingRepGood = 0;
this._smugglingRepBad = 0;
this._smugglingRepUnknown = 0;
this._maxRep = 70;
this._checkForCargo = false;
this._marketrnd = 0;

//=============================================================================================================
/* this.$addSmugglingContractToSystem(cargo)
 * This function adds the defined smuggling contract to the black market contract list.
 * A contract definition is an object with the following parameters, all required:
 *
 * destination: system ID of destination system
 * commodity:   the cargo type
 * size:        the number of units of cargo
 * deadline:    the deadline for delivery, in clock seconds
 * payment:     the payment for delivery on time, in credits
 *
 * and optionally, the following parameters:
 *
 * deposit:     the deposit payment required by the player (default 0)
 * route:       a route object generated with system.info.routeToSystem
 *              describing the route between the source and destination
 *              systems.
 *
 * If this is not specified, it will be generated automatically.
 *
 * The function will return true if the contract can be added, false
 * otherwise.
 */
this.$addSmugglingContractToSystem = function $addSmugglingContractToSystem(cargo) {
	if (cargo.destination < 0 || cargo.destination > 255) {
		log(this.name, "Rejected contract: destination missing or invalid");
		return false;
	}
	if (cargo.deadline <= clock.adjustedSeconds) {
		log(this.name, "Rejected contract: deadline invalid");
		return false;
	}
	if (cargo.payment < 0) {
		log(this.name, "Rejected contract: payment invalid");
		return false;
	}
	if (!cargo.size || cargo.size < 1) {
		log(this.name, "Rejected contract: size invalid");
		return false;
	}
	if (!cargo.commodity) {
		log(this.name, "Rejected contract: commodity unspecified");
		return false;
	}
	if (!system.mainStation.market[cargo.commodity]) {
		log(this.name, "Rejected contract: commodity invalid");
		return false;
	}
	if (!cargo.route) {
		var destinationInfo = System.infoForSystem(galaxyNumber, cargo.destination);
		cargo.route = system.info.routeToSystem(destinationInfo);
		if (!cargo.route) {
			log(this.name, "Rejected contract: route invalid");
			return false;
		}
	}
	if (!cargo.deposit) {
		cargo.deposit = 0;
	} else if (cargo.deposit >= cargo.payment) {
		log(this.name, "Rejected contract: deposit higher than total payment");
		return false;
	}

	this._contracts.push(cargo);
	return true;
}

//=============================================================================================================
/* Event handlers */

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {
	if (this._disabled === true) {
		delete this.shipExitedWitchspace;
		delete this.playerWillSaveGame;
		delete this.shipWillLaunchFromStation;
		delete this.guiScreenWillChange;
		delete this.guiScreenChanged;
		delete this.startUp;
		return;
	}

	this._helper = worldScripts["oolite-contracts-helpers"];
	var core = worldScripts.Smugglers_CoreFunctions;
	this.$padTextRight = core.$padTextRight;
	this.$padTextLeft = core.$padTextLeft;
	this.$isBigGuiActive = core.$isBigGuiActive;
	this.$rand = core.$rand;

	if (missionVariables.Smuggling_MarketRnd) this._marketrnd = missionVariables.Smuggling_MarketRnd;
	if (missionVariables.Smuggling_RepGood) {
		this._smugglingRepGood = missionVariables.Smuggling_RepGood;
		this._smugglingRepBad = missionVariables.Smuggling_RepBad;
		this._smugglingRepUnknown = missionVariables.Smuggling_RepUnknown;
	}
	this.$normaliseSmugglingReputation();

	if (missionVariables.Smuggling_ActiveContracts) {
		this._smugglingContracts = JSON.parse(missionVariables.Smuggling_ActiveContracts);
	}
	// stored contents of systems's cargo contract list
	if (missionVariables.Smuggling_Contracts) {
		this._contracts = JSON.parse(missionVariables.Smuggling_Contracts);
	} else {
		this.$initialiseSmugglingContractsForSystem();
	}

	// its unclear whether this or ContractsOnBB will get to it's startUpComplete first, so we'll force the issue here
	if (worldScripts.ContractsOnBB) {
		worldScripts.ContractsOnBB.$convertSmugglingContracts();
	}

	var xui = worldScripts.XenonUI;
	if (xui) xui.$addMissionScreenException("oolite-smuggling-contracts-details"); // smuggling-contracts-details
	var xrui = worldScripts.XenonReduxUI;
	if (xrui) xrui.$addMissionScreenException("oolite-smuggling-contracts-details");
}

//-------------------------------------------------------------------------------------------------------------
this.shipExitedWitchspace = function () {
	if (!system.isInterstellarSpace && !system.sun.hasGoneNova && system.stations.length > 1) {
		// must be a regular system with at least one station other than the main one
		this._marketrnd = this.$rand(256) - 1;
		this.$erodeSmugglingReputation();
		this.$initialiseSmugglingContractsForSystem();
		this.$adjustMarket();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function () {
	// encode the contract list to a string for storage in the savegame
	missionVariables.Smuggling_Contracts = JSON.stringify(this._contracts);
	missionVariables.Smuggling_RepGood = this._smugglingRepGood;
	missionVariables.Smuggling_RepBad = this._smugglingRepBad;
	missionVariables.Smuggling_RepUnknown = this._smugglingRepUnknown;
	missionVariables.Smuggling_ActiveContracts = JSON.stringify(this._smugglingContracts);
	missionVariables.Smuggling_MarketRnd = this._marketrnd;
}

//-------------------------------------------------------------------------------------------------------------
// when the player exits the mission screens, reset their destination
// system and HUD settings, which the mission screens may have
// affected.
this.shipWillLaunchFromStation = function () {
	this.$resetViews();
}

//-------------------------------------------------------------------------------------------------------------
this.shipDockedWithStation = function (station) {
	if (player.ship.dockedStation.allegiance === "galcop") this._checkForCargo = true;
}

//-------------------------------------------------------------------------------------------------------------
this.missionScreenOpportunity = function () {
	if (this._checkForCargo === true) {
		this._checkForCargo = false;
		this.$checkForCompletedContracts();
	}
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenWillChange = function (to, from) {
	this.$resetViews();
	if (to === "GUI_SCREEN_MANIFEST") this.$updateManifest();
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged = function (to, from) {
	if (to != "GUI_SCREEN_MISSION") this.$resetViews();
}

//=============================================================================================================
/* Interface functions */

//-------------------------------------------------------------------------------------------------------------
// runs through all smuggling contracts looking for any completed ones
this.$checkForCompletedContracts = function $checkForCompletedContracts() {

	if (this._smugglingContracts.length === 0) return;

	var sc = worldScripts.Smugglers_Equipment;
	var sdm = worldScripts.Smugglers_DockMaster;
	var p = player.ship;

	var list = [];

	for (var i = this._smugglingContracts.length - 1; i >= 0; i--) {
		var m = this._smugglingContracts[i];
		var errorCode = "";
		var new_payment = 0;
		var percent_delivered = 0;

		// are we in the right system
		if (m.destination === system.ID && m.destinationName === system.name) {
			if (clock.adjustedSeconds <= m.deadline) {
				var onhand = 0;

				onhand += sdm.$getTotalUnlabelled(m.commodity); // add any cargo currently hiding under a relabel
				onhand += p.manifest[m.commodity]; // add any visible cargo in the hold
				onhand -= sdm.$getTotalRelabelled(m.commodity); // take away any cargo that might be masked by a label (ie. cargo that looks like the commodity but really isn't)
				onhand += sc.$getCargoQuantity(m.commodity); // add any cargo in the smuggling compartment
				onhand += sc.$checkHyperCargo(m.commodity); // add any cargo in hyper cargo

				if (onhand >= m.size) {
					errorCode = "success";
					player.credits += m.fee;
					this.$removeCargo(m.commodity, m.size);
					player.setPlayerRole("trader-smuggler");
				} else {
					percent_delivered = 100.0 * (onhand / m.size);
					var acceptable_ratio = 100.0 - 10.0 * system.ID / 256; // down to 90%
					if (percent_delivered >= acceptable_ratio) {
						var shortfall = 100 - percent_delivered;
						new_payment = percent_delivered * (m.fee) / 100.0;
						player.credits += new_payment;
						errorCode = "short";
						this.$removeCargo(m.commodity, m.size);
						player.setPlayerRole("trader-smuggler");
					} else {
						errorCode = "refused_short";
					}
				}
			} else {
				errorCode = "late";
			}
		} else {
			if (clock.adjustedSeconds > m.deadline) {
				errorCode = "failed";
			}
		}

		var desc = "";
		// adjust the fee and reputation based on the result
		switch (errorCode) {
			case "success":
				// build a contact
				var contactHome = this.$rand(256) - 1;
				var contact = "a " + System.infoForSystem(galaxyNumber, contactHome).inhabitant.toLowerCase() + " from " + System.systemNameForID(contactHome);
				desc = expandDescription("[smuggled-cargo-delivered-okay]", {
					cargo: this.$descriptionForGoods(m),
					fee: formatCredits(m.fee, true, true),
					contactdesc: contact
				});
				this.$increaseSmugglingReputation(10);
				this.$contractCompletedEmail(errorCode, m.fee, m);
				if (this.$anyContractsLeftForSystem(i, m.destination) === false)
					mission.unmarkSystem({
						system: m.destination,
						name: "smuggling_contract_" + m.destination
					});
				break;
			case "late":
				desc = expandDescription("[smuggled-cargo-delivered-late]", {
					cargo: this.$descriptionForGoods(m)
				});
				this.$decreaseSmugglingReputation(10);
				this.$contractCompletedEmail(errorCode, 0, m);
				if (this.$anyContractsLeftForSystem(i, m.destination) === false)
					mission.unmarkSystem({
						system: m.destination,
						name: "smuggling_contract_" + m.destination
					});
				break;
			case "failed":
				desc = expandDescription("[smuggled-cargo-failed]", {
					cargo: this.$descriptionForGoods(m)
				});
				this.$decreaseSmugglingReputation(10);
				this.$contractCompletedEmail(errorCode, 0, m);
				if (this.$anyContractsLeftForSystem(i, m.destination) === false)
					mission.unmarkSystem({
						system: m.destination,
						name: "smuggling_contract_" + m.destination
					});
				break;
			case "short":
				desc = expandDescription("[smuggled-cargo-short]", {
					cargo: this.$descriptionForGoods(m),
					fee: formatCredits(new_payment, true, true),
					percent: percent_delivered
				});
				this.$contractCompletedEmail(errorCode, new_payment, m);
				if (this.$anyContractsLeftForSystem(i, m.destination) === false)
					mission.unmarkSystem({
						system: m.destination,
						name: "smuggling_contract_" + m.destination
					});
				break;
			case "refused_short":
				desc = expandDescription("[smuggled-cargo-refused-short]", {
					cargo: this.$descriptionForGoods(m)
				});
				// reset the error so we can skip this one - the player still has time to complete the contract
				errorCode = "";
				break;
		}
		if (desc != "") list.push(desc);

		// if we got to this point with a result of some kind, remove the contract
		if (errorCode != "") {
			if (worldScripts.ContractsOnBB) {
				worldScripts.ContractsOnBB.$removeSmugglingContract(i);
			}
			this._smugglingContracts.splice(i, 1);
		}
	}

	if (list.length > 0) {
		// add all contract summary items to the arrival report
		var text = "";
		for (var i = 0; i < list.length; i++) {
			text += list[i] + "\n\n";
		}
		mission.runScreen({
			titleKey: "smuggling-contracts-title-summary",
			message: text,
			exitScreen: "GUI_SCREEN_STATUS"
		});
	}
}

//-------------------------------------------------------------------------------------------------------------
// removes a specific amount of the commodity, from various locations in this order:
// 1. relabelled commodities, 2. standard hold, 3. smuggling compartmnt, 4. hypercargo
// returns quantity not found (should be zero in most circumstances)
this.$removeCargo = function $removeCargo(commodity, amount) {

	var se = worldScripts.Smugglers_Equipment;
	var sdm = worldScripts.Smugglers_DockMaster;
	var p = player.ship;
	var sum = amount;

	// is there any relabelled cargo?
	if (sdm.$getTotalRelabelled(commodity) > 0) {
		// unlabel the cargo so the next bit will pick it up
		var relbl_remain = sdm.$removeRealCargoQuantity(commodity, sum);
	}

	// do we have the cargo anywhere? check normal hold
	if (p.manifest[commodity] >= sum) {
		p.manifest[commodity] -= sum;
		sum = 0;
	} else if (p.manifest[commodity] > 0) {
		sum -= p.manifest[commodity];
		p.manifest[commodity] = 0;
	}
	// smuggling compartment
	if (sum > 0 && se.$getCargoQuantity(commodity) > 0) {
		var remain = se.$removeCargo(commodity, sum);
		sum = remain;
	}
	// and hypercargo
	if (sum > 0 && se.$checkHyperCargo(commodity) > 0) {
		var remain = se.$removeHyperCargo(commodity, sum);
		sum = remain;
	}

	return sum;
}

//-------------------------------------------------------------------------------------------------------------
// resets HUD and jump destination
this.$resetViews = function $resetViews() {
	if (this._suspendedHUD !== false) {
		player.ship.hudHidden = false;
		this._suspendedHUD = false;
	}
	if (this._suspendedDestination !== null) {
		player.ship.targetSystem = this._suspendedDestination;
		this._suspendedDestination = null;
	}
}

//-------------------------------------------------------------------------------------------------------------
// initialise a new smuggling contract list for the current system
this.$initialiseSmugglingContractsForSystem = function $initialiseSmugglingContractsForSystem() {
	// clear list
	this._contracts = [];

	var checklist = [];

	var si = worldScripts.Smugglers_Illegal;
	var destlist = [];

	if (si._illegalGoods.length > 0) {
		for (var i = 0; i < si._illegalGoods.length; i++) {
			if (destlist.indexOf(si._illegalGoods[i].systemID) === -1)
				destlist.push(si._illegalGoods[i].systemID);
		}
	} else {
		// no illegal goods defined - shouldn't happen, but just in case
		return;
	}

	var numContracts = Math.floor(5 * Math.random() + 5 * Math.random() + 5 * Math.random() + (this.$getSmugglingReputationPrecise() * Math.random()));
	if (this.$getSmugglingReputationPrecise() >= 0 && numContracts < 5) {
		numContracts += 5;
	}
	if (numContracts > 16) {
		numContracts = 16;
	} else if (numContracts < 0) {
		numContracts = 0;
	}

	for (var i = 0; i < numContracts; i++) {
		var cargo = new Object;

		// pick a random system to take the goods to
		var destination = destlist[this.$rand(destlist.length) - 1];

		// discard if chose the current system
		if (destination === system.ID) continue;
		if (typeof destination === "undefined") continue;

		// get the SystemInfo object for the destination
		var destinationInfo = System.infoForSystem(galaxyNumber, destination);

		if (destinationInfo.sun_gone_nova) continue;

		var daysUntilDeparture = 1 + (Math.random() * (7 + this.$getSmugglingReputationPrecise() - destinationInfo.government));
		if (daysUntilDeparture <= 0) {
			// loses some more contracts if reputation negative
			continue;
		}

		var commodities = si.$illegalGoodsListCommodityOnly(destination, true);

		var attempts = 0;
		do {
			var remotePrice = 0;
			attempts++;
			var commodity = commodities[Math.floor(Math.random() * commodities.length)];

			// sub-tc contracts only available for top rep
			if (system.mainStation.market[commodity]["quantity_unit"] != 0 && this.$getSmugglingReputationPrecise() < 6.5) {

			} else {
				remotePrice = this.$priceForCommodity(commodity, destinationInfo);
			}
			//log(this.name, "contract " + i + "--attempt " + attempts + ": " + commodity + " remoteprice = " + remotePrice + ", sourcepriceadj = " + system.mainStation.market[commodity].price / 20);
		} while (remotePrice < system.mainStation.market[commodity].price / 20 && attempts < 10);
		// failed to find a good one.
		if (attempts === 10) continue;

		// don't overload the player with all the same type of contracts
		var checkcount = 0;
		for (var j = 0; j < checklist.length; j++) {
			if (checklist[j] === commodity) checkcount += 1;
		}
		if (checkcount >= 3) continue;

		cargo.commodity = commodity;

		var amount = 0;
		// larger unit sizes for kg/g commodities
		switch (parseInt(system.mainStation.market[commodity]["quantity_unit"])) {
			case 1:
				amount = this.$rand(20) + 10;
				break;
			case 2:
				amount = this.$rand(40) + 20;
				break;
			case 0:
				amount = this.$rand(10);
				if (amount > 5 && Math.random() > 0.5) amount += (this.$rand(5) - 3);
				break;
		}
		cargo.size = amount;

		// adjustment to prices based on quantity (larger = more profitable)
		var discount = Math.min(10 + Math.floor(amount / 10), 35);
		var unitPrice = system.mainStation.market[commodity].price * (100 - discount) / 1000;
		var localValue = Math.floor(unitPrice * amount);
		remotePrice = remotePrice * (200 + discount) / 200;
		var remoteValue = Math.floor(remotePrice * amount);
		var profit = remoteValue - localValue;

		// skip if unprofitable
		if (profit <= 100) continue;

		// check that a route to the destination exists
		// route calculation is expensive so leave this check to last
		var routeToDestination = system.info.routeToSystem(destinationInfo);

		// if the system cannot be reached, ignore this contract
		if (!routeToDestination) continue;

		// we now have a valid destination, so generate the rest of
		// the parcel details
		cargo.destination = destination;
		// we'll need this again later, and route calculation is slow
		cargo.route = routeToDestination;

		// higher share for transporter for longer routes, less safe systems
		var share = 100 + destinationInfo.government - (10 * routeToDestination.route.length);
		if (share < 10) share = 10;
		share = 100 - share;

		// safety: now multiply the fee by 2 compared with 1.76 contracts
		// prevents exploit discovered by Mad Hollander at
		// http://bb.oolite.space/viewtopic.php?p=188127
		localValue *= 2;
		// this may need to be raised further

		// absolute value of profit remains the same
		var fee = (localValue + Math.floor(profit * (share / 100))) * 2;

		fee -= fee % 20; // round to nearest 20 credits;

		cargo.payment = fee;
		cargo.deposit = localValue - (localValue % 20);
		// rare but not impossible; last safety check
		if (cargo.deposit >= cargo.payment) continue;

		// time allowed for delivery is time taken by "fewest jumps"
		// route, plus timer above. Higher reputation makes longer
		// times available.
		cargo.deadline = clock.adjustedSeconds + Math.floor(daysUntilDeparture * 86400) + (cargo.route.time * 3600);

		// make sure the illegal good will be illegal when the player arrives, based on the route time
		var def = si.$illegalGoodsDefinition(destination, commodity);
		if (def && def.end < cargo.deadline) continue;

		// add it to the check list so we can make sure we don't offer too many of the same type of contract
		checklist.push(commodity);

		// add parcel to contract list
		this.$addSmugglingContractToSystem(cargo);
	}
}

//-------------------------------------------------------------------------------------------------------------
// if the interface is activated, this function is run.
this.$smugglingContractsScreens = function $smugglingContractsScreens(interfaceKey) {
	// the interfaceKey parameter is not used here, but would be useful if
	// this callback managed more than one interface entry

	this.$validateSmugglingContracts();

	// set up variables used to remember state on the mission screens
	this._suspendedDestination = null;
	this._suspendedHUD = false;
	this._contractIndex = 0;
	this._routeMode = "LONG_RANGE_CHART_SHORTEST";
	this._lastOptionChosen = "06_EXIT";

	// start on the summary page if more than one contract is available
	var summary = (this._contracts.length > 1);

	this.$smugglingContractsDisplay(summary);
}


//-------------------------------------------------------------------------------------------------------------
// this function is called after the player makes a choice which keeps
// them in the system, and also on initial entry to the system
// to select the appropriate mission screen and display it
this.$smugglingContractsDisplay = function $smugglingContractsDisplay(summary) {

	// Again. Has to be done on every call to this function, but also
	// has to be done at the start.
	this.$validateSmugglingContracts();

	// if there are no contracts (usually because the player has taken
	// the last one) display a message and quit.
	if (this._contracts.length === 0) {
		var missionConfig = {
			titleKey: "smuggling-contracts-none-available-title",
			messageKey: "smuggling-contracts-none-available-message",
			allowInterrupt: true,
			screenID: "oolite-smuggling-contracts-none",
			exitScreen: "GUI_SCREEN_INTERFACES"
		};
		if (this._smugglingSummaryPageBackground != "") {
			missionConfig.background = this._smugglingSummaryPageBackground;
		}
		if (this._smugglingPageOverlay != "") {
			if (this._smugglingPageOverlayHeight != 0) {
				missionConfig.overlay = {
					name: this._smugglingPageOverlay,
					height: this._smugglingPageOverlayHeight
				};
			} else {
				missionConfig.overlay = this._smugglingPageOverlay;
			}
		}
		mission.runScreen(missionConfig);
		// no callback, just exits contracts system
		return;
	}

	// make sure that the 'currently selected contract' pointer
	// is in bounds
	if (this._contractIndex >= this._contracts.length) {
		this._contractIndex = 0;
	} else if (this._contractIndex < 0) {
		this._contractIndex = this._contracts.length - 1;
	}
	// sub functions display either summary or detail screens
	if (summary) {
		this.$smugglingContractSummaryPage();
	} else {
		this.$smugglingContractSinglePage();
	}

}

//-------------------------------------------------------------------------------------------------------------
// display the mission screen for the summary page
this.$smugglingContractSummaryPage = function $smugglingContractSummaryPage() {
	var playerrep = this._helper._playerSkill(this.$getSmugglingReputationPrecise());
	// column 'tab stops'
	var columns = [9, 15, 21, 26];
	var anyWithSpace = false;

	// column header line
	var headline = expandMissionText("smuggling-contracts-column-goods");
	// pad to correct length to give a table-like layout
	headline += this._helper._paddingText(headline, columns[0]);
	headline += expandMissionText("smuggling-contracts-column-destination");
	headline += this._helper._paddingText(headline, columns[1]);
	headline += expandMissionText("smuggling-contracts-column-within");
	headline += this._helper._paddingText(headline, columns[2]);
	headline += expandMissionText("smuggling-contracts-column-deposit");
	headline += this._helper._paddingText(headline, columns[3]);
	headline += expandMissionText("smuggling-contracts-column-fee");
	// required because of way choices are displayed.
	headline = " " + headline;

	// setting options dynamically; one contract per line
	var options = new Object;
	var i;
	for (i = 0; i < this._contracts.length; i++) {
		// temp variable to simplify following code
		var cargo = this._contracts[i];
		// write the description, padded to line up with the headers
		var optionText = this.$descriptionForGoods(cargo);
		optionText += this._helper._paddingText(optionText, columns[0]);
		optionText += System.infoForSystem(galaxyNumber, cargo.destination).name;
		optionText += this._helper._paddingText(optionText, columns[1]);
		optionText += this._helper._timeRemaining(cargo);
		optionText += this._helper._paddingText(optionText, columns[2]);
		// right-align the fee so that the credits signs line up
		var priceText = formatCredits(cargo.deposit, false, true);
		priceText = this._helper._paddingText(priceText, 3.5) + priceText;
		optionText += priceText
		optionText += this._helper._paddingText(optionText, columns[3]);
		// right-align the fee so that the credits signs line up
		priceText = formatCredits(cargo.payment - cargo.deposit, false, true);
		priceText = this._helper._paddingText(priceText, 3.5) + priceText;
		optionText += priceText

		// need to pad the number in the key to maintain alphabetical order
		var istr = i;
		if (i < 10) {
			istr = "0" + i;
		}
		// needs to be aligned left to line up with the heading
		options["01_CONTRACT_" + istr] = {
			text: optionText,
			alignment: "LEFT"
		};

		// check if there's space for this contract
		if (!this.$hasSpaceFor(cargo)) {
			options["01_CONTRACT_" + istr].color = "darkGrayColor";
		} else {
			anyWithSpace = true;
			// if there doesn't appear to be sufficient time remaining
			if (this._helper._timeRemainingSeconds(cargo) < this._helper._timeEstimateSeconds(cargo)) {
				options["01_CONTRACT_" + istr].color = "orangeColor";
			}
		}
	}
	// if we've come from the detail screen, make sure the last
	// contract shown there is selected here
	var icstr = this._contractIndex;
	if (icstr < 10) icstr = "0" + this._contractIndex;

	var initialChoice = "01_CONTRACT_" + icstr;
	// if none of them have any space...
	if (!anyWithSpace) initialChoice = "06_EXIT";

	// next, an empty string gives an unselectable row
	options["02_SPACER"] = "";

	// now need to add further spacing to fill the remaining rows, or
	// the options will end up at the bottom of the screen.
	var rowsToFill = 21;
	if (player.ship.hudHidden || this.$isBigGuiActive() === true) rowsToFill = 27;

	for (i = 4 + this._contracts.length; i < rowsToFill; i++) {
		// each key needs to be unique at this stage.
		options["07_SPACER_" + i] = "";
	}

	// numbered 06 to match the option of the same function in the other branch
	options["06_EXIT"] = expandMissionText("smuggling-contracts-command-quit");

	var missionConfig = {
		titleKey: "smuggling-contracts-title-summary",
		message: headline,
		allowInterrupt: true,
		screenID: "oolite-smuggling-contracts-summary",
		exitScreen: "GUI_SCREEN_INTERFACES",
		choices: options,
		initialChoicesKey: initialChoice
	};

	if (this._smugglingSummaryPageBackground != "") {
		missionConfig.background = this._smugglingSummaryPageBackground;
	}
	if (this._smugglingPageOverlay != "") {
		if (this._smugglingPageOverlayHeight != 0) {
			missionConfig.overlay = {
				name: this._smugglingPageOverlay,
				height: this._smugglingPageOverlayHeight
			};
		} else {
			missionConfig.overlay = this._smugglingPageOverlay;
		}
	}

	// now run the mission screen
	mission.runScreen(missionConfig, this.$processSmugglingChoice, this);
}

//-------------------------------------------------------------------------------------------------------------
// display the mission screen for the contract detail page
this.$smugglingContractSinglePage = function $smugglingContractSinglePage() {

	var playerrep = this._helper._playerSkill(this.$getSmugglingReputationPrecise());

	// temp variable to simplify code
	var cargo = this._contracts[this._contractIndex];

	// This mission screen uses the long range chart as a backdrop.
	// This means that the first 18 lines are taken up by the chart,
	// and we can't put text there without overwriting the chart.
	// We therefore need to hide the player's HUD, to get the full 27
	// lines.

	if (!player.ship.hudHidden && this.$isBigGuiActive() === false) {
		this._suspendedHUD = true; // note that we hid it, for later
		player.ship.hudHidden = true;
	}

	// We also set the player's witchspace destination temporarily
	// so we need to store the old one in a variable to reset it later
	this._suspendedDestination = player.ship.targetSystem;

	// That done, we can set the player's destination so the map looks
	// right.
	player.ship.targetSystem = cargo.destination;

	// start with 18 blank lines, since we don't want to overlap the chart
	var message = new Array(18).join("\n");

	message += expandMissionText("smuggling-contracts-long-description", {
		"smuggling-contracts-longdesc-goods": this.$descriptionForGoods(cargo),
		"smuggling-contracts-longdesc-destination": this._helper._systemName(cargo.destination),
		"smuggling-contracts-longdesc-deadline": this._helper._timeRemaining(cargo),
		"smuggling-contracts-longdesc-time": this._helper._timeEstimate(cargo),
		"smuggling-contracts-longdesc-payment": formatCredits(cargo.payment, false, true),
		"smuggling-contracts-longdesc-deposit": formatCredits(cargo.deposit, false, true)
	});

	// use a special background
	var backgroundSpecial = "LONG_RANGE_CHART";

	// the available options will vary quite a bit, so this rather
	// than a choicesKey in missiontext.plist
	var options = new Object;
	// this is the only option which is always available
	options["06_EXIT"] = expandMissionText("smuggling-contracts-command-quit");

	// if the player has sufficient space
	if (this.$hasSpaceFor(cargo)) {
		options["05_ACCEPT"] = {
			text: expandMissionText("smuggling-contracts-command-accept")
		};

		// if there's not much time left, change the option colour as a warning!
		if (this._helper._timeRemainingSeconds(cargo) < this._helper._timeEstimateSeconds(cargo)) {
			options["05_ACCEPT"].color = "orangeColor";
		}
	} else {
		options["05_UNAVAILABLE"] = {
			text: expandMissionText("smuggling-contracts-command-unavailable"),
			color: "darkGrayColor",
			unselectable: true
		};
	}

	// if the ship has a working advanced nav array, can switch
	// between 'quickest' and 'shortest' routes
	// (and also upgrade the special background)
	if (player.ship.hasEquipmentProviding("EQ_ADVANCED_NAVIGATIONAL_ARRAY")) {
		backgroundSpecial = this._routeMode;
		if (this._routeMode === "LONG_RANGE_CHART_SHORTEST") {
			options["01_MODE"] = expandMissionText("smuggling-contracts-command-ana-quickest");
		} else {
			options["01_MODE"] = expandMissionText("smuggling-contracts-command-ana-shortest");
		}
	}
	// if there's more than one, need options for forward, back, and listing
	if (this._contracts.length > 1) {
		options["02_BACK"] = expandMissionText("smuggling-contracts-command-back");
		options["03_NEXT"] = expandMissionText("smuggling-contracts-command-next");
		options["04_LIST"] = expandMissionText("smuggling-contracts-command-list");
	} else {
		// if not, we may need to set a different choice
		// we never want 05_ACCEPT to end up selected initially
		if (this._lastChoice === "02_BACK" || this._lastChoice === "03_NEXT" || this._lastChoice === "04_LIST") {
			this._lastChoice = "06_EXIT";
		}
	}

	var title = expandMissionText("smuggling-contracts-title-detail", {
		"smuggling-contracts-title-detail-number": this._contractIndex + 1,
		"smuggling-contracts-title-detail-total": this._contracts.length
	});

	// finally, after all that setup, actually create the mission screen

	var missionConfig = {
		title: title,
		message: message,
		allowInterrupt: true,
		screenID: "oolite-smuggling-contracts-details",
		exitScreen: "GUI_SCREEN_INTERFACES",
		backgroundSpecial: backgroundSpecial,
		choices: options,
		initialChoicesKey: this._lastChoice
	};

	if (this._smugglingPageOverlay != "") {
		if (this._smugglingPageOverlayHeight != 0) {
			missionConfig.overlay = {
				name: this._smugglingPageOverlay,
				height: this._smugglingPageOverlayHeight
			};
		} else {
			missionConfig.overlay = this._smugglingPageOverlay;
		}
	}

	mission.runScreen(missionConfig, this.$processSmugglingChoice, this);
}

//-------------------------------------------------------------------------------------------------------------
this.$processSmugglingChoice = function $processSmugglingChoice(choice) {
	this.$resetViews();
	// can occur if ship launches mid mission screen
	if (choice == null) return;

	// now process the various choices
	if (choice.match(/^01_CONTRACT_/)) {
		// contract selected from summary page
		// set the index to that contract, and show details
		var index = parseInt(choice.slice(12), 10);
		this._contractIndex = index;
		this._lastChoice = "04_LIST";
		this.$smugglingContractsDisplay(false);

	} else if (choice === "01_MODE") {
		// advanced navigation array mode flip
		this._routeMode = (this._routeMode === "LONG_RANGE_CHART_SHORTEST") ? "LONG_RANGE_CHART_QUICKEST" : "LONG_RANGE_CHART_SHORTEST";
		this._lastChoice = "01_MODE";
		this.$smugglingContractsDisplay(false);

	} else if (choice === "02_BACK") {
		// reduce contract index (parcelContractsDisplay manages wraparound)
		this._contractIndex--;
		this._lastChoice = "02_BACK";
		this.$smugglingContractsDisplay(false);

	} else if (choice === "03_NEXT") {
		// increase contract index (parcelContractsDisplay manages wraparound)
		this._contractIndex++;
		this._lastChoice = "03_NEXT";
		this.$smugglingContractsDisplay(false);

	} else if (choice === "04_LIST") {
		// display the summary page
		this.$smugglingContractsDisplay(true);

	} else if (choice === "05_ACCEPT") {
		this.$acceptContract();
		// do not leave the setting as accept for the next contract!
		this._lastChoice = "03_NEXT";
		this.$smugglingContractsDisplay(false);
	}
	// if we get this far without having called smugglingContractsDisplay
	// that means either 'exit' or an unrecognised option was chosen
	if (choice === "06_EXIT") {
		var sbm = worldScripts.Smugglers_BlackMarket;
		sbm.$initialBlackMarket();
	}
}

//-------------------------------------------------------------------------------------------------------------
// move goods from the contracts list to the player's ship (if possible)
this.$acceptContract = function $acceptContract() {
	var cargo = this._contracts[this._contractIndex];

	if (cargo.deposit > player.credits) {
		this._helper._soundFailure();
		return;
	}

	// give the cargo to the player
	var result = this.$awardContract(cargo.size, cargo.commodity, system.ID, cargo.destination, cargo.deadline, cargo.payment, cargo.deposit, cargo.route);

	if (result) {
		// pay the deposit
		player.credits -= cargo.deposit;
		// give the player the merchandise
		player.ship.manifest[cargo.commodity] += cargo.size;
		mission.markSystem({
			system: cargo.destination,
			name: "smuggling_contract_" + cargo.destination,
			markerShape: "MARKER_PLUS"
		});

		// remove the contract from the station list
		this._contracts.splice(this._contractIndex, 1);

		this._helper._soundSuccess();

	} else {
		// else must have had manifest change recently
		// (unlikely, but another OXP could have done it)
		this._helper._soundFailure();
	}
}

//-------------------------------------------------------------------------------------------------------------
// store the contract
this.$awardContract = function $awardContract(quantity, commodity, source, destination, deadline, payment, deposit, route) {
	var cargo = {};
	cargo.size = quantity;
	cargo.commodity = commodity;
	cargo.start = source;
	cargo.startName = System.systemNameForID(source);
	cargo.destination = destination;
	cargo.destinationName = System.systemNameForID(destination);
	cargo.deadline = deadline;
	cargo.fee = payment;
	cargo.premium = deposit;
	cargo.route = route;

	this._smugglingContracts.push(cargo);
	this.$contractStartedEmail(cargo);

	return true;
}

//-------------------------------------------------------------------------------------------------------------
// removes any expired parcels
this.$validateSmugglingContracts = function $validateSmugglingContracts() {
	var c = this._contracts.length - 1;
	var removed = false;
	// iterate downwards so we can safely remove as we go
	for (var i = c; i >= 0; i--) {
		// if the time remaining is less than 1/3 of the estimated
		// delivery time, even in the best case it's probably not
		// going to get there.
		if (this._helper._timeRemainingSeconds(this._contracts[i]) < this._helper._timeEstimateSeconds(this._contracts[i]) / 3) {
			if (worldScripts.ContractsOnBB) {
				worldScripts.ContractsOnBB.$reindexContracts(34000, i);
			}
			// remove it
			this._contracts.splice(i, 1);
			removed = true;
		}
	}
}


//=============================================================================================================
/* Utility functions */

//-------------------------------------------------------------------------------------------------------------
// lower-cases the initial letter of the package contents
this.$formatPackageName = function $formatPackageName(name) {
	return name.charAt(0).toLowerCase() + name.slice(1);
}

//-------------------------------------------------------------------------------------------------------------
// calculates a sample price for a commodity in a distant system
this.$priceForCommodity = function $priceForCommodity(commodity, systeminfo) {
	//sample price returns decicredits, need credits
	var si = worldScripts.Smugglers_Illegal;
	var goods = si.$illegalGoodsListCommodityOnly(systeminfo.systemID);
	if (goods.indexOf(commodity) >= 0) {
		//return Math.round((si._commodityInfo[commodity][1] - (Math.random() * 15)), 2)
		return (systeminfo.samplePrice(commodity) * si._commodityInfo[commodity][1]) / 10;
	} else {
		return systeminfo.samplePrice(commodity) / 10;
	}
}

//-------------------------------------------------------------------------------------------------------------
// description of the cargo
this.$descriptionForGoods = function $descriptionForGoods(cargo) {
	var unit = expandMissionText("smuggling_tons");
	var stn = system.mainStation;
	if (!stn && system.stations.length > 0) stn = system.stations[0];
	if (!stn) return "";
	if (stn.market[cargo.commodity]["quantity_unit"] === "1") {
		unit = expandMissionText("smuggling_kilos");
	} else if (stn.market[cargo.commodity]["quantity_unit"] === "2") {
		unit = expandMissionText("smuggling_grams");
	}
	return cargo.size + expandDescription("[cargo-" + unit + "-symbol]") + " " + displayNameForCommodity(cargo.commodity);
}

//-------------------------------------------------------------------------------------------------------------
// check if player's ship has space for the cargo and can afford the deposit
this.$hasSpaceFor = function $hasSpaceFor(cargo) {
	if (cargo.deposit > player.credits) {
		return false;
	}
	var amountInTC = cargo.size;
	if (system.mainStation.market[cargo.commodity]["quantity_unit"] === "1") {
		var spareSafe = 499 - (player.ship.manifest[cargo.commodity] % 1000);
		amountInTC -= spareSafe;
		amountInTC = Math.ceil(amountInTC / 1000);
		if (amountInTC < 0) {
			amountInTC = 0;
		}
	} else if (system.mainStation.market[cargo.commodity]["quantity_unit"] === "2") {
		var spareSafe = 499999 - (player.ship.manifest[cargo.commodity] % 1000000);
		amountInTC -= spareSafe;
		amountInTC = Math.ceil(amountInTC / 1000000);
		if (amountInTC < 0) {
			amountInTC = 0;
		}
	}

	return (amountInTC <= player.ship.cargoSpaceAvailable);
}

//-------------------------------------------------------------------------------------------------------------
// returns true if another contracts exists for the given system ID, otherwise false
this.$anyContractsLeftForSystem = function $anyContractsLeftForSystem(idx_ignore, sysID) {
	for (var i = 0; i < this._smugglingContracts.length; i++) {
		if (i != idx_ignore && this._smugglingContracts[i].destination === sysID) return true;
	}
	return false;
}

//-------------------------------------------------------------------------------------------------------------
// update the manifest screen with any contracts
this.$updateManifest = function $updateManifest() {

	// don't try an update in interstellar space
	if (system.isInterstellarSpace) return;

	if (this._smugglingContracts.length > 0) {
		var list = [];
		list.push(expandMissionText("smuggling-contracts-title-summary") + ":");

		for (var i = 0; i < this._smugglingContracts.length; i++) {
			var cargo = this._smugglingContracts[i];
			var message = expandMissionText("smuggling-contracts-long-descr-manifest", {
				"smuggling-contracts-longdesc-goods": this.$descriptionForGoods(cargo),
				"smuggling-contracts-longdesc-destination": cargo.destinationName,
				"smuggling-contracts-longdesc-deadline": this.$formatTravelTime(this._helper._timeRemainingSeconds(cargo)),
				"smuggling-contracts-longdesc-time": this._helper._timeEstimate(cargo),
				"smuggling-contracts-longdesc-payment": formatCredits(cargo.fee, false, true),
				"smuggling-contracts-longdesc-deposit": formatCredits(cargo.deposit, false, true)
			});
			list.push(message);
		}
		mission.setInstructions(list, this.name);
	} else {
		mission.setInstructions(null, this.name);
	}
}

//=============================================================================================================
// Smuggling reputation functions
// essentially a JS recreation of the same rep functions in core code to give the player a separate reputation for smuggling

//-------------------------------------------------------------------------------------------------------------
this.$getSmugglingReputation = function $getSmugglingReputation() {
	return parseInt(this.$smugglingReputation() / 10);
}

//-------------------------------------------------------------------------------------------------------------
this.$getSmugglingReputationPrecise = function $getSmugglingReputationPrecise() {
	return this.$smugglingReputation() / 10;
}

//-------------------------------------------------------------------------------------------------------------
this.$smugglingReputation = function $smugglingReputation() {
	var good = this._smugglingRepGood;
	var bad = this._smugglingRepBad;
	var unknown = this._smugglingRepUnknown;

	if (unknown > 0) {
		unknown = this._maxRep - (((2 * unknown) + (this._marketrnd % unknown)) / 3);
	} else {
		unknown = this._maxRep;
	}

	return (good + unknown - 3 * bad) / 2;
}

//-------------------------------------------------------------------------------------------------------------
this.$decreaseSmugglingReputation = function $decreaseSmugglingReputation(amount) {

	var good = this._smugglingRepGood;
	var bad = this._smugglingRepBad;
	var unknown = this._smugglingRepUnknown;

	for (var i = 0; i < amount; i++) {
		if (good > 0) {
			// shift a bean from good to bad
			good--;
			if (bad < this._maxRep) bad++;
		} else {
			// shift a bean from unknown to bad
			if (unknown > 0) unknown--;
			if (bad < this._maxRep) bad++;
		}
	}
	this._smugglingRepGood = good;
	this._smugglingRepBad = bad;
	this._smugglingRepUnknown = unknown;
}

//-------------------------------------------------------------------------------------------------------------
this.$increaseSmugglingReputation = function $increaseSmugglingReputation(amount) {
	var good = this._smugglingRepGood;
	var bad = this._smugglignRepBad;
	var unknown = this._smugglingRepUnknown;

	for (var i = 0; i < amount; i++) {
		if (bad > 0) {
			// shift a bean from bad to unknown
			bad--;
			if (unknown < this._maxRep) unknown++;
		} else {
			// shift a bean from unknown to good
			if (unknown > 0) unknown--;
			if (good < this._maxRep) good++;
		}
	}
	this._smugglingRepGood = good;
	this._smugglingRepBad = bad;
	this._smugglingRepUnknown = unknown;
}

//-------------------------------------------------------------------------------------------------------------
this.$erodeSmugglingReputation = function $erodeSmugglingReputation() {
	var good = this._smugglingRepGood;
	var bad = this._smugglingRepBad;
	var unknown = this._smugglingRepUnknown;
	if (unknown < this._maxRep) {
		if (bad > 0) {
			bad--;
		} else {
			if (good > 0) good--;
		}
		unknown++;
	}
	this._smugglingRepGood = good;
	this._smugglingRepBad = bad;
	this._smugglingRepUnknown = unknown;
}

//-------------------------------------------------------------------------------------------------------------
this.$normaliseSmugglingReputation = function $normaliseSmugglingReputation() {
	var good = this._smugglingRepGood;
	var bad = this._smugglingRepBad;
	var unknown = this._smugglingRepUnknown;

	var c = good + bad + unknown;
	if (c === 0) {
		unknown = this._maxRep;
	} else if (c != this._maxRep) {
		good = good * this._maxRep / c;
		bad = bad * this._maxRep / c;
		unknown = this._maxRep - good - bad;
	}
	this._smugglingRepGood = good;
	this._smugglingRepBad = bad;
	this._smugglingRepUnknown = unknown;
}

//-------------------------------------------------------------------------------------------------------------
// format the travel time
this.$formatTravelTime = function $formatTravelTime(seconds) {
	// this function uses an hours-only format
	// but provides enough information to use a days&hours format if
	// oolite-contracts-time-format in missiontext.plist is overridden

	// extra minutes are discarded
	var hours = Math.floor(seconds / 3600);

	var days = Math.floor(hours / 24);
	var spareHours = hours % 24;

	return expandMissionText("smuggling-time-format", {
		"smuggling-time-format-hours": hours,
		"smuggling-time-format-days": days,
		"smuggling-time-format-spare-hours": spareHours
	});
}

//-------------------------------------------------------------------------------------------------------------
// routine to zero out quantities of any illegal goods under contract
// so if the player arrives short, he can't just buy one, launch, redock and complete the contract
this.$adjustMarket = function $adjustMarket() {
	var stn = system.mainStation;
	if (stn) {
		for (var i = this._smugglingContracts.length - 1; i >= 0; i--) {
			var m = this._smugglingContracts[i];
			// are we in the right system
			if (m.destination === system.ID && m.destinationName === system.name) {
				// set the market quantity to zero
				stn.setMarketQuantity(m.commodity, 0);
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
// send email to player when a contract is started. (1.81 only)
this.$contractStartedEmail = function $contractStartedEmail(contract) {

	var w = worldScripts.EmailSystem;
	if (!w) return;

	var ga = worldScripts.GalCopAdminServices;
	if (ga._disableContracts === true) return;

	var msg = "";
	var subj = "";
	var sndr = "";

	sndr = expandDescription("[smuggling-contract-sender]");
	subj = expandMissionText("smuggling_accepted_contract") + ": " + this.$descriptionForGoods(contract);
	msg = expandDescription("[smuggling-contract-start]", {
		description: this.$descriptionForGoods(contract),
		systemname: System.systemNameForID(contract.destination),
		time: global.clock.clockStringForTime(contract.deadline),
		fee: contract.fee.toFixed(2)
	});

	w.$createEmail({
		sender: sndr,
		subject: subj,
		date: global.clock.seconds,
		message: msg
	});
}

//-------------------------------------------------------------------------------------------------------------
this.$contractCompletedEmail = function $contractCompletedEmail(result, fee, contract) {

	var w = worldScripts.EmailSystem;
	if (!w) return;

	var ga = worldScripts.GalCopAdminServices;
	if (ga._disableContracts === true) return;

	var msg = "";
	var subj = "";
	var sndr = "";

	sndr = expandDescription("[smuggling-contract-sender]");
	msg = expandDescription("[smuggling-contract-" + result + "]", {
		description: this.$descriptionForGoods(contract),
		systemname: System.systemNameForID(contract.destination),
		time: global.clock.clockStringForTime(contract.deadline),
		fee: (fee / 10).toFixed(2)
	});
	subj = expandDescription("[smuggling-contract-" + result + "-subject]");

	w.$createEmail({
		sender: sndr,
		subject: subj,
		date: global.clock.seconds,
		message: msg,
		expiryDays: ga._defaultExpiryDays
	});
}