// JavaScript for Oolite (ECMAScript 5)
"use strict";

    this.license = "CC-BY-NC-SA 4.0";
    this.author  = "Wildeblood";
    this.version = "1.4";
    this.name    = "dynooverse";

// Other scripts can change the trends at any time, by setting e.g.
// worldScripts.dynooverse.biases.agriculture = 0.6;

this.biases = {
    agriculture: 0.5,
    industry: 0.5,
    government: 0.5,
    techlevel: 0.6,  // Biased toward incrementing
    population: 0.4, // Biased toward decrementing
    productivity: 0.5
};

this.DESCRIPTION_EXPIRY_DAYS = 14; // Days until changed descriptions revert to default.
this.POPULATION_MAX = 100;        // Starting population ranges from 8 to 64.
this.PRODUCTIVITY_MAX = 100000;  // Starting productivity ranges from 768 to 56320.
this.sendEmails = false;        // Flag to determine if emails are available.


this.dayChanged = function (newDay) {

    // Randomly select a valid system using the helper function
    const sysID = this._selectVisibleSystem();
    if (!sysID) {
        return;
    }
    const sys = System.infoForSystem(galaxyNumber, sysID);

    // Randomly select a system property to modify
    const randomBranch = Math.floor(Math.random() * 6);
    switch (randomBranch) {

        case 0: // Economy
            let economy = sys.economy;
            if (economy >= 0 && economy <= 3) {
                sys.economy = this._modifyBoundedValue(sys.economy, 0, 3, this.biases.industry);
            } else if (economy >= 4 && economy <= 7) {
                sys.economy = this._modifyBoundedValue(sys.economy, 4, 7, this.biases.agriculture);
            }
            this._sendMessage(sysID, sys.name, "Economy", economy, sys.economy);
            break;

        case 1: // Government
            let government = sys.government;
            sys.government = this._modifyBoundedValue(sys.government, 0, 7, this.biases.government); 
            this._sendMessage(sysID, sys.name, "Government", government, sys.government);
            break;

        case 2: // Tech Level
            let techlevel = sys.techlevel;
            sys.techlevel = this._modifyBoundedValue(sys.techlevel, 0, 15, this.biases.techlevel);
            this._sendMessage(sysID, sys.name, "Tech level", techlevel, sys.techlevel);
            break;

        case 3: // Population
            let population = sys.population;
            sys.population = this._modifyBoundedValue(sys.population, 0, this.POPULATION_MAX, this.biases.population); 
            this._sendMessage(sysID, sys.name, "Population", population, sys.population);
            break;

        case 4: // Productivity
            let productivity = sys.productivity;
            const randomPercentage = this._getRandomPercentage(0.05, 0.10);
         // const changeDirection = Math.random() < 0.5 ? -1 : 1;
            const changeDirection = Math.random() < this.biases.productivity ? 1 : -1; // Use bias for direction
            let modifiedProductivity = sys.productivity + changeDirection * Math.floor(sys.productivity * randomPercentage);
            sys.productivity = Math.min(this.PRODUCTIVITY_MAX, Math.max(1, modifiedProductivity));
            this._sendMessage(sysID, sys.name, "Productivity", productivity, sys.productivity);
            break;

        case 5: // Description
            let oldDescription = sys.description;
            sys.description = this._viceRegalNotices(sys);
            this._sendMessage(sysID, sys.name, "Description", oldDescription, sys.description);
            sys.dynoo_description_expiry = newDay + this.DESCRIPTION_EXPIRY_DAYS;
            break;

        default:
            log(this.name, "Error: Unexpected branch value");
    }

    // Remove expired description messages
    for (let i = 0; i < 256; i++) {
        const sysInfo = System.infoForSystem(galaxyNumber, i);
        if (sysInfo.dynoo_description_expiry && sysInfo.dynoo_description_expiry <= newDay) {
            sysInfo.description = null;
            sysInfo.dynoo_description_expiry = null;
        }
    }
};

this.startUp = function () {
    // Restore biases from missionVariables
    if (missionVariables.Dynooverse_biases) {
        try {
            var savedBiases = JSON.parse(missionVariables.Dynooverse_biases);
            if (savedBiases && typeof savedBiases === "object") {
                this.biases = savedBiases;
                log(this.name, "Restored biases from missionVariables.");
            } else {
                log(this.name, "Error: Saved biases are invalid. Using default biases.");
            }
        } catch (e) {
            log(this.name, "Error parsing saved biases:", e.message);
            log(this.name, "Using default biases.");
        }
    } else {
        log(this.name, "No saved biases found. Using default biases.");
        // Assume first run
        if (worldScripts.dynooverse_stats) {
            worldScripts.dynooverse_stats.playerEnteredNewGalaxy(galaxyNumber);
        }
    }

    // Check if the email system is available
    if (worldScripts.EmailSystem &&
        worldScripts.EmailSystem.$createEmail) {
        this.sendEmails = true;
        log(this.name, "Email system is available.");
    } else {
        this.sendEmails = false;
        log(this.name, "Email system is not available. Falling back to player.commsMessage.");
    }

    delete this.startUp;
};

// Save biases to missionVariables before saving the game
this.playerWillSaveGame = function () {
    missionVariables.Dynooverse_biases = JSON.stringify(this.biases);
    log(this.name, "Saved biases to missionVariables.");
};

// Helper function to format and deliver messages
this._sendMessage = function (sysID, sysName, property, oldValue, newValue) {
    // Validate inputs
    if (!sysName || typeof sysName !== "string" ||
        !property || typeof property !== "string") {
        log(this.name, "Invalid parameters provided to _sendMessage.");
        return;
    }

    // Construct the message dynamically
    const message = property + " of " + sysName + " changed from " + oldValue + " to " + newValue;
    const subject = property + " of " + sysName;
    const sender = 
        property === "Population" || property === "Tech level" ? "GalCop Bureau of Statistics" :
        property === "Economy" || property === "Productivity" ? "Wildefire Investor Advisory" :
        sysName + " News Service";

    // Send the message via email or player.commsMessage
    if (this.sendEmails) {
        worldScripts.EmailSystem.$createEmail({
            sender: sender,
            subject: subject,
            date: clock.adjustedSeconds,
            message: message,
            sentFrom: sysID // Use sysID as the sender's location
        });
    } else {
        player.commsMessage(message);
    }
};

// Helper function to randomly generate vice-regal notices
this._viceRegalNotices = function (sys) {
    const notices = [
        "The Lord Chamberlain, Cholmondely, made an official visit to [SYSTEM] on stardate [DATE].",
        "Governor-General [NAME] issued a decree strengthening trade regulations in [SYSTEM].",
        "A royal envoy arrived in [SYSTEM], announcing new subsidies for agricultural development.",
        "The High Chancellor visited [SYSTEM] to oversee the expansion of industrial facilities.",
        "A gala was held in [SYSTEM] to celebrate the centennial of its founding.",
        "Diplomatic envoys from [OTHER_SYSTEM] arrived in [SYSTEM] to negotiate trade agreements.",
        "The regional governor declared a day of celebration in honor of [EVENT] in [SYSTEM]."
    ];

    // Replace placeholders with dynamic values
    const randomNotice = notices[Math.floor(Math.random() * notices.length)];
    const systemName = "[SYSTEM]";
    const date = "[DATE]";
    const otherSystem = "[OTHER_SYSTEM]";
    const event = "[EVENT]";

    return randomNotice
        .replace(systemName, sys.name)
        .replace(date, (2084004 + clock.days).toString())
        .replace(otherSystem, this._getRandomSystemName())
        .replace(event, this._getRandomEvent());
};

// Helper function to get a random system name
this._getRandomSystemName = function () {
    const randomSysID = Math.floor(Math.random() * 256);
    const randomSys = System.infoForSystem(galaxyNumber, randomSysID);
    return randomSys ? randomSys.name : "a nearby system";
};

// Helper function to get a random event
this._getRandomEvent = function () {
    const events = [
        "the discovery of a new asteroid belt",
        "the anniversary of GalCop's founding",
        "the completion of a major infrastructure project",
        "the signing of a peace treaty",
        "the launch of a new trade route"
    ];
    return events[Math.floor(Math.random() * events.length)];
};

// Helper function to randomly select a valid sysID for a visible system
this._selectVisibleSystem = function () {
    let attempts = 0;
    const MAX_ATTEMPTS = 1000; // Arbitrary limit to prevent infinite loops

    while (attempts < MAX_ATTEMPTS) {
        const sysID = Math.floor(Math.random() * 256);
        const sys = System.infoForSystem(galaxyNumber, sysID);

        if (sys && sys.population > 0 && (!sys.concealment || sys.concealment < 100)) {
            return sysID;
        }

        attempts++;
    }

    log(this.name, "Failed to find a valid system after " + MAX_ATTEMPTS + " attempts.");
    return null; // Indicate failure
};

// Helper Functions
this._modifyBoundedValue = function (value, lowerBound, upperBound, bias) {
    if (bias === undefined) {
        bias = 0.5;
    }
    if (value === lowerBound) {
        return value + 1; // Always move upward from the lower bound
    } else if (value === upperBound) {
        return value - 1; // Always move downward from the upper bound
    } else {
        // Use the bias to determine whether to increment or decrement
        const change = Math.random() < bias ? 1 : -1;
        return Math.min(upperBound, Math.max(lowerBound, value + change));
    }
};

this._getRandomPercentage = function (minPercent, maxPercent) {
    return minPercent + Math.random() * (maxPercent - minPercent);
};
