"use strict";
this.name = "ExternalDockSystem";
this.author = "phkb";
this.description = "System for add external docks to stations";
this.copyright = "2024 phkb";
this.license = "CC BY-NC-SA 4.0";

this._fcb = null;
this._externalDocks = [];
this._checked = false;

//----------------------------------------------------------------------------------------
this.startUpComplete = function() {
    if (missionVariables.ExternalDockSystem_Docked) {
        var key = missionVariables.ExternalDockSystem_Docked;
        var pos = missionVariables.ExternalDockSystem_Position;
        var realPos = missionVariables.ExternalDockSystem_RealPosition;
     
        var ed = this._externalDocks;
        for (var i = 0; i < ed.length; i++) {
            if (ed[i].station.dataKey == key && player.ship.dockedStation.dataKey == key && (!ed[i].relativePos || JSON.stringify(ed[i].relativePos) == pos) && (ed[i].relativePos || !ed[i].realPos || JSON.stringify(ed[i].realPos) == realPos)) {
                player.ship.dockedStation._dockedViaED = ed[i];
                break;
            }
        }
        delete missionVariables.ExternalDockSystem_Docked;
        delete missionVariables.ExternalDockSystem_Position;
        delete missionVariables.ExternalDockSystem_RealPosition;
    }
}

//----------------------------------------------------------------------------------------
this.playerWillSaveGame = function() {
	var p = player.ship;
	var stn = p.dockedStation;
	if (stn.hasOwnProperty("_dockedViaED") && stn._dockedViaED.allowLaunch) {
		missionVariables.ExternalDockSystem_Docked = stn.dataKey;
        missionVariables.ExternalDockSystem_Position = JSON.stringify(stn._dockedViaED.relativePos);
        missionVariables.ExternalDockSystem_RealPosition = JSON.stringify(stn._dockedViaED.realPos);
	} else {
        delete missionVariables.ExternalDockSystem_Docked;
        delete missionVariables.ExternalDockSystem_Position;
        delete missionVariables.ExternalDockSystem_RealPosition;
    }
}

//----------------------------------------------------------------------------------------
this.shipWillEnterWitchspace = function() {
    this.$stopFCB();
    this._externalDocks.length = 0;
}

//----------------------------------------------------------------------------------------
this.shipWillDockWithStation = this.shipDied = function() {
    this.$stopFCB();
}

//----------------------------------------------------------------------------------------
this.shipExitedWitchspace = function() {
    if (this._externalDocks.length > 0) {
        for (var i = 0; i < this._externalDocks.length; i++) {
            this._externalDocks[i].externalDock._speedWarning = false;
            this._externalDocks[i].externalDock._clearanceWarning = false;
        }
        this.$startFCB();
    }
}

//----------------------------------------------------------------------------------------
this.shipWillLaunchFromStation = function(station) {
    var p = player.ship;
    if (station.hasOwnProperty("_dockedViaED")) {
        var ed = station._dockedViaED;
        var dve = ed.externalDock;
        var subIndex = ed.launchSubEntityIndex;
        dve._speedWarning = true;
        dve._clearanceWarning = false;
        p.position = dve.position.add(station.orientation.multiply(station.subEntities[subIndex].orientation).vectorForward().multiply(p.boundingBox.z + 50 + ed.launchDistanceBoost));
        p.orientation = station.orientation.multiply(station.subEntities[subIndex].orientation);
        p.velocity = p.vectorForward.multiply(p.speed);
        delete station._dockedViaED;
    }
    if (this._externalDocks.length > 0) this.$startFCB();
}

//----------------------------------------------------------------------------------------
this.$addExternalDock = function(obj) {

    if (!obj.station.isStation) throw "Invalid settings: target entity is not a station.";
    if (obj.hasOwnProperty("position")) {
        if (!Array.isArray(obj.position)) throw "Invalid settings: position property is not an array";
        if (obj.position.length != 3) throw "Invalid settings: position property has incorrect number of elements";
    }
    if (obj.hasOwnProperty("realPosition")) {
        if (!obj.realPosition.hasOwnProperty("x") || !obj.realPosition.hasOwnProperty("y") || !obj.realPosition.hasOwnProperty("z")) throw "Invalid settings: realPosition property is not an valid Vector3D";
    }
    if (!obj.hasOwnProperty("position") && !obj.hasOwnProperty("realPosition")) throw "Invalid settings: position or realPosition property missing";
    if (obj.scale && isNaN(obj.scale)) throw "Invalid settings: scale property is no a valid number";
    //stn, position, scale, allowLaunch, launch_orientation, predock_callback, postdock_callback
    if (obj.hasOwnProperty("allowLaunch") && obj.allowLaunch == true && (!obj.launchSubEntityIndex || isNaN(obj.launchSubEntityIndex))) "Invalid settings: launch subEntity index missing or invalid";

    if (!obj.hasOwnProperty("allowLaunch")) {obj.allowLaunch = false; obj.launchSubEntityIndex = -1;}

    if (obj.position) {
        var dve = system.addVisualEffect("external_dock_ve_blank", 
            obj.station.position.add(obj.station.vectorRight.multiply(obj.position[0])).add(obj.station.vectorUp.multiply(obj.position[1])).add(obj.station.vectorForward.multiply(obj.position[2])));
    }
    if (obj.realPosition) {
        var dve = system.addVisualEffect("external_dock_ve_blank", obj.realPosition);
    }
    if (!obj.scale || obj.scale == 0) obj.scale = 1;
    if (obj.scale && obj.scale != 1) dve.scale(obj.scale);

    obj.station._savedPosition = obj.station.position;
    obj.station._savedOrientation = obj.station.orientation;
    obj.station._isMoving = false; // we'll change this to true if the position or orientation changes.

    this._externalDocks.push({
        station:obj.station, 
        externalDock:dve, 
        relativePos:(obj.hasOwnProperty("position") ? obj.position : null), 
        realPos:(obj.hasOwnProperty("realPosition") ? obj.realPosition : null),
        scale:obj.scale,
        dockRange:(obj.hasOwnProperty("dockRange") && !isNaN(obj.dockRange) ? obj.dockRange : 150),
        allowLaunch:obj.allowLaunch,
        launchDistanceBoost:(obj.hasOwnProperty("launchDistanceBoost") && !isNaN(obj.launchDistanceBoost) ? obj.launchDistanceBoost : 0),
        launchSubEntityIndex:obj.launchSubEntityIndex,
        preDockCallback:obj.preDockCallback, 
        postDockCallback:obj.postDockCallback
    });

    return dve;
}

//----------------------------------------------------------------------------------------
this.$startFCB = function() {
    if (!this._fcb || !isValidFrameCallback(this._fcb)) {
        this._fcb = addFrameCallback(this.$checkExternalDocks);
    }
}

//----------------------------------------------------------------------------------------
this.$stopFCB = function() {
    if (this._fcb && isValidFrameCallback(this._fcb)) removeFrameCallback(this._fcb);
    this._fcb = null;
}

//----------------------------------------------------------------------------------------
this.$checkExternalDocks = function _checkExternalDocks(delta) {
    // inline function to reposition docks on stations if they move
    function reposition(obj) {
        //log("external dock testing", "we are repositioning the dock point because the station (" + obj.station + ") has moved");
        var dve = obj.externalDock;
        var stn = obj.station;
        var pos = obj.relativePos;
        if (pos) {
            dve.position = stn.position.add(stn.vectorRight.multiply(pos[0])).add(stn.vectorUp.multiply(pos[1])).add(stn.vectorForward.multiply(pos[2]));
        } else {
            dve.position = obj.realPos;
        }
        
        stn._savedPosition = stn.position;
        stn._savedOrientation = stn.orientation;
    }

    var that = _checkExternalDocks;
    var _p = (that._p = that._p || player.ship);
    var _ed = (that._ed = that._ed || worldScripts.ExternalDockSystem._externalDocks);
    var i = _ed.length;
    while (i--) {
        if (!_p || !_p.isValid) break;
        var ed = _ed[i].externalDock;
        var stn = _ed[i].station;
        var scale = _ed[i].scale;
        var dockRange = _ed[i].dockRange;

        if (!stn.isValid) continue; // something has happened to the station - just continue the loop

        // check to see if station has moved
        // if so, perform function to adjust location of dock flashers
        if (stn._isMoving || (stn.position.angleTo(stn._savedPosition) != 0 || stn.orientation.dot(stn._savedOrientation) != 1)) {
            stn._isMoving = true;
            reposition(_ed[i]);
        }

        // check player distance to dock
        var dist = ed.position.distanceTo(_p);
        var deviation = _p.vectorForward.angleTo(ed.position.subtract(_p.position));
        //if (dist < 2000) log("testing", "dist = " + dist + ", dev = " + deviation);
        if (dist > (2800 * scale) && ed._speedWarning) ed._speedWarning = false;
        if (dist > (6500 * scale) && ed._clearanceWarning) ed._clearanceWarning = false;
        if (dist <= (2000 * scale) && _p.speed < 100 && ed._speedWarning) ed._speedWarning = false;
        if (dist <= (2000 * scale) && _p.speed >= 100 && !ed._speedWarning && deviation < 0.06) {
            ed._speedWarning = true;
            stn.commsMessage("Reduce speed to less than 100m/s for docking.", _p);
        }
        if (dist <= (5000 * scale) && !ed._clearanceWarning && deviation < 0.06 && stn.requiresDockingClearance && (player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_GRANTED" && player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_REQUESTED" && player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_TIMING_OUT")) {
            ed._clearanceWarning = true;
            stn.commsMessage(expandDescription("[oolite-station-docking-requires-clearance]"), _p);
        }
        if (dist <= (5000 * scale) && ed._clearanceWarning && deviation < 0.06 && stn.requiresDockingClearance && (player.dockingClearanceStatus == "DOCKING_CLEARANCE_STATUS_GRANTED" || player.dockingClearanceStatus == "DOCKING_CLEARANCE_STATUS_REQUESTED" || player.dockingClearanceStatus == "DOCKING_CLEARANCE_STATUS_TIMING_OUT")) {
            ed._clearanceWarning = false;
        }
        if (dist < dockRange && _p.speed < 100 && deviation < 0.09) {
            if (_ed[i].allowLaunch) {
                stn._dockedViaED = _ed[i];
            }
            if (_ed[i].preDockCallback) {
                try {
                    _ed[i].preDockCallback(stn);
                } catch (ex) {
                    log("external docking", "ERROR calling pre-dock callback: " + ex);
                }
            }
            // check if the player has requested docking clearance (and the station requires it)
            // if they haven't requested it, fine the player
            if (stn.requiresDockingClearance && (player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_GRANTED" && player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_REQUESTED" && player.dockingClearanceStatus != "DOCKING_CLEARANCE_STATUS_TIMING_OUT")) {
                var fine = parseInt(player.credits * 0.05);
                if (fine > 5000) fine = 5000;
                player.credits -= fine;
                var txt = expandDescription("[station-docking-clearance-fined-@-cr]").replace("%@", formatCredits(fine, false, true));
                player.addMessageToArrivalReport(txt);
            }   
            stn.dockPlayer();
            if (_ed[i].postDockCallback) {
                try {
                    _ed[i].postDockCallback(stn);
                } catch (ex) {
                    log("external docking", "ERROR calling post-dock callback: " + ex);
                }
            }
            return;
        }
    }
}