"use strict";
this.name = "GalCopBB_MissionDetails";
this.author = "phkb";
this.copyright = "2017 phkb";
this.description = "Gets the PhraseGen mission details version for each mission";
this.license = "CC BY-NC-SA 4.0";

/*
    Notes for translators: Many of the mission descriptions have been put together using PhraseGen, a sentence 
    construction tool in GNN. However, the way the data is structured to make use of the tool makes the task of 
    translation into other languages complicated.

    So, there are two options. Option 1 is to rewrite the data structures in your translation language. To do that, 
    you would need to make wholesale changes to the galcopbb_missiondetails.js file.

    Option 2 is to avoid using the tool altogether. I have added prebuilt samples to the "*_details" entry of each mission
    definition. To use these, change every instance of '_phraseGen = "1";' to '_phraseGen = "0";' in descriptions.plist. 
    At that point, you need only translate the visible sentences in that file and can ignore galcopbb_missiondetails.js 
    altogether.
*/

/*
    Notes: 
    It appears that phraseGen has a problem with apostrophes inside dice roll (Dnn=) functions.

    Remember to prefix all [ and ] characters with a double backslash "\\". This ensures they won't get 
    translated by PhraseGen. Also remember to have a trailing space after the closing bracket, even with "," and ".".

    Be careful of including actual numbers in the text (eg "50km"). These should be added via one of the replacement 
    elements, rather than in the sentence text.

    Generally descriptions should hint at system or galactic events, rather than being explicit. 
    For instance, talking about relations cooling or heating up between two systems is OK as there doesn't need to be 
    any additional game content for it to be "true" in the player's mind. Talking about outright war between two 
    systems is harder, because now the war has to be (or should be) simulated with ships, the length of the war needs 
    to be calculated, and the repercussions of who won/lost need to be added. By keeping it hinted at only, there is 
    no need for additional resources.
    The exception to this rule is when a particular system or galactic event is part of the mission spec (eg medical 
    missions)
*/

this._gcmPG = {
    "abandon": ["abandon", "dump", "drop", "leave behind"],
    "ability": ["ability", "skills", "capability", "power", "wherewithal"],
    "accept": ["accept", "take on", "agree to", "sign up for"],
    "achievable": ["achievable", "attainable", "reachable", "feasible", "viable", "realistic"],
    "agents": ["agents", "security agents", "undercover agents", "deep-cover agents", "security personnel", "deep-cover security personnel", "operatives", "security operatives", "undercover operatives", "undercover security operatives", "deep-cover operatives", "deep-cover security operatives"],
    "approximately": ["approximately", "about", "around"],
    "are_expensive": ["are expensive", "are quite expensive", "require considerable financial resources", "aren't cheap", "require a significant financial outlay"],
    "are_willing_to": ["are willing to", "are able to", "can", "are prepared to", "are ready to"],
    "assistance": ["help", "assistance", "aid", "support", "backing"],
    "attacked": ["attacked", "assaulted", "molested", "hit", "struck", "battered", "waylaid", "ambushed"],
    "attention": ["It has come to our attention", "We have reason to believe", "We have it on good authority"],
    "backbone": ["who have the ", "who have the backbone and ", "who have the fortitude and ", "who have the courage and ", "who have the grit and determination, and just the ", "who have the strength of character and "],
    "battle": ["a battle", "a skirmish", "a fight", "a mêlée", "a scuffle", "a fracas", "an altercation"],
    "be_given": ["be given", "be issued with", "be supplied with", "receive", "get", "get issued with", "get supplied with", "be provided with"],
    "be_sufficient": ["suffice", "do", "be enough", "be adequate", "be sufficient"],
    "broadcasting": ["giving off", "emitting", "broadcasting"],
    "broken_down": ["broken down", "been suspended", "been put on ice", "been put on hold", "stalled"],
    "calling_for": ["calling for", "asking for", "seeking", "requesting", "looking for"],
    "care_for": ["help", "assist", "care for", "look after"],
    "collect": ["collect", "recover", "retrieve", "pick up"],
    "concerning": ["concerning", "regarding", "about", "in relation to", "with regard to", "in connection with"],
    "constant": ["constant", "continual", "continuous", "regular"],
    "containers": ["canisters", "containers", "pods"],
    "contract": ["contract", "mission", "assignment", "commission"],
    "crime": ["cultural insensitivity", "inappropriate text messages", "a poorly-timed and innuendo-riddled comedy sketch", "bad breath", "a political minefield involving small, furry animals", "a bad case of food poisoning", "disagreement over the price of our chief import and export commodities", "an unfortunate incident involving a paper-clip and a writing stylus", "a dispute over land-rights", "a dispute over legal representation", "a dispute over political machinations", "a dispute over seating arrangements at the negotiation table", "the content of an internally leaked memo", "alcoholic beverages", "some very bad background muzak", "a dispute over the qualities of some TV sitcoms", "a dispute over the quality of imported Trumble meat", "the perceived lack of quality donuts during trade meetings", "the death of the primary negotiator", "the death of the substitute negotiator", "the inconvenience caused by a bomb detonating in the primary meeting room"],
    "desires": ["desires", "is keen to have", "needs", "wants", "wishes for"],
    "destroy": ["annihilate$", "batter$", "burn$", "crush$", "destroy$", "eradicate$", "exterminate$", "pulverise$", "terminate$", "vaporise$"],
    "destroyed": ["annihilated", "battered", "burned", "crushed", "destroyed", "eradicated", "exterminated", "obliterated", "pulverised", "terminated", "vaporised"],
    "destroying": ["annihilating", "battering", "burning", "crushing", "destroying", "eradicating", "exterminating", "obliterating", "pulverising", "terminating", "vaporising"],
    "deteriorated": ["deteriorated", "worsened", "declined", "gone downhill", "weakened"],
    "donation": ["donation", "contribution", "gift", "bequest", "endowment"],
    "due_to": ["due to", "due in no small part to", "thanks to", "due largely to", "largely due to", "owing to", "largely owing to", "mostly due to", "mainly due to", "in no small part due to", "largely because of", "largely thanks to", "because of", "largely because of", "mostly because of", "mainly because of"],
    "enable_us": ["enable us", "allow us", "help us"],
    "forced": ["forced", "compelled", "duty-bound", "obliged"],
    "grown_exponentially": ["has grown exponentially", "is very clear and growing each day", "grows continually", "is growing exponentially", "is growing continually", "is building every day"],
    "happen": ["happen", "take place", "transpire", "occur"],
    "happened": ["happened", "taken place", "occurred", "transpired"],
    "happening": ["happening", "taking place", "transpiring", "occurring"],
    "hard": ["hard", "difficult", "complicated", "tricky"],
    "harmful": ["harmful", "damaging", "dangerous", "unsafe", "detrimental", "toxic"],
    "helpful_reason": ["external arbitration", "mood lighting in the various meeting rooms", "fragrant candles in meetings", "a popular rock band for lunch-time entertainment during trade talks", "a broad selection of donuts and pastries during negotiations", "nicely pitched ambient background music during trade talks"],
    "imperative": ["imperative", "vital", "crucial", "essential"],
    "important": ["important", "critical", "valuable", "crucial", "vital", "significant", "imperative"],
    "in_search_of": ["in search of", "searching for", "looking for", "trying to find", "trying to locate", "in need of", "seeking"],
    "inexperienced": ["poor", "inadequate", "badly trained", "inexperienced", "opinionated", "loud-mouthed", "idiotic"],
    "incursions": ["incursions", "invasions", "raids", "attacks", "forays", "sorties"],
    "information": ["information", "data", "evidence", "material", "intelligence"],
    "information_about": ["information about", "data concerning", "evidence of", "material relating to", "intelligence about"],
    "is_willing_to": ["is willing to", "is able to", "can", "is prepared to", "is ready to"],
    "large": ["high", "large", "considerable", "significant", "substantial", "sizeable"],
    "many_different": ["many different types", "a wide variety", "several types", "a large array", "a complex array", "a diverse array", "a wide array"],
    "need_for": ["demand for", "need for", "necessity of"],
    "on_route_to": ["on route to", "on their way to", "heading for"],
    "operating_mantra": ["operating mantra", "reason for being", "stated goal", "simple goal", "fundamental drive", "goal and purpose"],
    "pilot": ["pilot", "commander", "captain"],
    "pirates": ["pirates", "brigands", "bandits"],
    "plenty_of": ["plenty of", "many", "quite a number of", "a large number of", "any number of"],
    "received": ["received", "uncovered", "discovered", "unearthed", "obtained", "collected", "gotten hold of"],
    "repercussions": ["repercussions", "consequences", "ramifications", "impact"],
    "set_up": ["set up", "established", "created"],
    "sexy": ["sexy", "glamorous", "that much fun", "trendy", "exciting", "fashionable", "thrilling"],
    "short": ["short", "brief", "limited"],
    "steal": ["steal", "pirate", "pinch", "pilfer", "filch", "appropriate", "thieve", "pocket", "nick", "make off with"],
    "stealing": ["stealing", "pirating", "pinching", "pilfering", "filching", "appropriating", "thieving", "pocketing", "nicking", "making off with"],
    "struggling": ["struggling", "fighting", "battling", "striving"],
    "travel_there": ["head there", "go there", "fly there", "travel there", "head over there", "fly over there", "journey there"],
    "travel_to": ["travel to", "enter", "go to", "fly to", "head to", "venture to", "head over to", "journey to", "journey into"],
    "venture_into": ["venture into", "enter", "head into", "go into", "fly into"],
    "vile": ["vile", "abhorrent", "abominable", "despicable", "disgusting", "hideous", "revolting", "toxic"],
    "war": ["war", "struggle", "fight", "battle"],
    "we_believe": ["we believe", "we think", "we estimate", "in our estimation", "it's our opinion", "we are of the opinion", "it's our considered opinion"],
};

this._debug = true;
this._setDef = null;
this._phraseGen = null;

//-------------------------------------------------------------------------------------------------------------
this.startUp = function () {
    this._phraseGen = worldScripts.GNN_PhraseGen;
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType1 = function $missionType1() {
    var def = {
        fieldA: ["danger is lurking in our system", "a hidden danger (D50=lurks|lies in wait) in our system", "A serious threat exists on our spacelanes"],
        fieldB: ["are needed", "are being sought", "are being called(D50= on|)", "are required"],
        fieldC: ["to help", "to assist", "to lend a hand"],
        fieldD: ["in cleaning up the spacelanes", "in making our system safer", "in improving the safety of all ships entering out system"],
        fieldE: ["going out", "heading out", "flying out"],
        fieldF: [].concat(this._gcmPG.destroying),
        fieldG: ["in this system", "in our system space", "in our transit lanes", "between the witchpoint and the main planet, or between the planet and the sun"],
        fieldH: [],
        fieldI: ["task involves", "job involves", "mission involves", "undertaking involves", "operational parameters are"],

        fieldJ: ["(D50=[5]|)a recent asteroid storm", "(D50=[5]|)an unfortunate mining incident", "(D50=[5]|)an accident involving mining personnel", "(D50=[5]|)a recent series of accidents involving asteroids"],
        fieldK: ["a number of", "quite a few", "several"],
        fieldL: ["threaten the safety of ships entering our system", "are creating havoc for ships on our spacelanes", "are causing consternation for all pilots in our system", "are impeding the flow of traffic into and out of our system", "has prompted local authorities to take action", "are lying around in our system"],
        fieldM: ["need your (D50=assistance|help)", "are asking for (D50=assistance|help)", "are requesting your (D50=assistance|help)", "are asking for your (D50=assistance|help)", "are [8] [9]s who can help"],
        fieldN: ["A strike by local miners, combined with "],
        fieldO: [].concat(this._gcmPG.hard),
        fieldP: [].concat(this._gcmPG.sexy),
        fieldQ: [].concat(this._gcmPG.in_search_of),
        fieldR: [].concat(this._gcmPG.pilot),
        sentences: [
            ">{9 }2 3 4 by 5 and #6 \\[target\\] asteroids (D50=anywhere you can find them |)7.",
            "(D50=The Spacelane Safety Commission is|We are) {8 >9 }3 4. The 9 5 and 6 \\[target\\] asteroids (D50=anywhere you can find them |)7.",
            "1. {1 has left behind 2 (D50=stray |)rocks that 3. We(D50=, the Spacelane Safety Commission,|) 4 in }6 \\[target\\] asteroids (D50=anywhere you can find them|wherever they might be hiding).",
            "{9! Want to earn a few extra credits? Prepared to do whatever it takes? The Spacelane Safety Commission is 8 anyone who is capable of hunting down and }6 \\[target\\] asteroids. (D50=Yes|We know), it's not (D50=particularly |){6, or (D50=even |)7, but it pays. Do you have what it takes?"
        ],
    };

    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType2 = function $missionType2() {
    var def = {
        fieldA: [].concat(this._gcmPG.incursions),
        fieldB: ["increasing", "on the increase", "escalating", "intensifying"],
        fieldC: [].concat(this._gcmPG.in_search_of),
        fieldD: [].concat(this._gcmPG.backbone),
        fieldE: [].concat(this._gcmPG.ability),
        fieldF: [].concat(this._gcmPG.travel_there),
        fieldG: [].concat(this._gcmPG.destroy),
        fieldH: ["to assist with making", "to help with making", "to help us make"],
        fieldI: ["solutions(D50= to the problem|)", "answers(D50= to the problem|)", "resolutions(D50= to the problem|)", "ways to fix the problem"],
        fieldJ: ["helping", "assisting", "coming to the aid of", "providing assistance to"],
        fieldK: [" for all space-farer's", " for everyone", " for all ships that come here"],
        fieldL: ["plagued", "inundated", "overwhelmed", "beset"],
        fieldM: [].concat(this._gcmPG.pirates),
        fieldN: [" recently", " in recent months", " of late", " lately"],
        fieldO: ["vermin", "pests", "parasites", "bottom-dwellers", "scum", "poachers"],
        fieldP: [].concat(this._gcmPG.contract),
        fieldQ: ["struggled", "found it increasingly difficult", "found it hard"],
        fieldR: [].concat(this._gcmPG.pilot),
        sentences: [
            "Pirate 1 from nearby \\[destination\\] have been 2{5. We are (D50=now |)}3 >{9 }45 to 6 and 7 \\[target\\] pirate ships 8 our system safer. (D50=By doing this you will receive, aside from the monetary benefits, the unending gratitude of all the inhabitants of \\[system\\].|)",
            "With pirate 1 from nearby \\[destination\\] 2, we are (D50=now |)3 9. The conclusion we reached was that we need >{9 }45 to head to \\[destination\\] (D50=and|to engage and) 7 \\[target\\] pirate ships. By doing so you will be {1 local authorities in making our system safer2.",
            "Our system has been {3 by 4 from \\[destination\\]5, and we have 8 to (D50=cope|deal) (D50=effectively |)with the onslaught. We are (D50=now |)}3 any >{9 }45 to enter \\[destination\\] space (D50=and|to engage and) 7 \\[target\\] of the {6. Succeeding in this 7 will bring safety and (D50=security|stability) to our spacelanes(D50=, and gratitude from all the inhabitants of \\[system\\]|).",
            "{4 from nearby \\[destination\\] have5 become bolder in their }1 on our spacelanes, and local authorities have {8 to (D50=cope|deal) (D50=effectively |)with the onslaught. So we are }3 >{9 }45 to 6 and take them on. We need \\[target\\] of the {6 ^}7. If you succeed in this {7 you will be bringing (D50=peace and safety|safety and security) to our spacelanes.",
            "A recent (D50=series|wave) of pirate attacks from nearby \\[destination\\], with the associated loss of life and property, has (D50=prompted|driven) local authorities to authorise }1 into their space. The (D50=goal|outcome) of this {7 is simple: to (D50=|engage and )}7 \\[target\\] pirate ships. Aside from (D50=any|the) monetary (D50=rewards|benefits), (D50=and the gratitude of all the inhabitants of \\[system\\], |)completing this {7 will help (D50=to |)bring (D50=safety|peace) and security to our spacelanes."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType3 = function $missionType3() {
    var def = {
        fieldA: [].concat(this._gcmPG.due_to),
        fieldB: [].concat(this._gcmPG.crime),
        fieldC: ["drastic", "severe", "extreme", "harsh"],
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.backbone),
        fieldF: [].concat(this._gcmPG.ability),
        fieldG: [].concat(this._gcmPG.destroy),
        fieldH: [],
        fieldI: ["lengthy negotiations", "our best efforts"],
        fieldJ: [].concat(this._gcmPG.helpful_reason),
        fieldK: [].concat(this._gcmPG.broken_down),
        fieldL: [].concat(this._gcmPG.are_willing_to),
        fieldM: [].concat(this._gcmPG.travel_to),
        fieldN: [].concat(this._gcmPG.incursions),
        fieldO: ["regret", "sorrow", "grief", "deep sorrow"],
        fieldP: ["delegates", "agents", "envoys", "emissaries", "ambassadors"],
        fieldQ: [].concat(this._gcmPG.destroying),
        fieldR: [].concat(this._gcmPG.pilot),
        sentences: [
            "A (D50=recent |)breakdown in trade relations between \\[system\\] and \\[destination\\], 1 2, has led to the 3 action of attempting to put a limit on their trade. We are 4 >{9 }56, who will risk the wrath of local authorities, to 7 \\[target\\] trader (D50=ships|vessels).",
            "Despite 9, and the presence of {1, trade relations between \\[system\\] and \\[destination\\] have 2. While regrettable, we are pursuing every possible action to bring \\[destination\\] back to the negotiation table, and so we are looking to put a limit on their trade. We are }4 >{9 }56, who {3 4 \\[destination\\] and }7 \\[target\\] trade (D50=ships|vessels).",
            "Trade relations with \\[destination\\] have {2 }1 2. In order to bring them back to the negotiation table we want to disrupt the flow of goods into their system. We are 4 >{9 }56, who {3 4 \\[destination\\] and }7 \\[target\\] trade (D50=ships|vessels).",
            "With trade talks stalled, }1 2, our leaders has been left with no alternative but to authorise {5 into \\[destination\\] to disrupt their trade. We are }4 >{9 }56, who {3 risk GalCop penalties by flying to \\[destination\\] and 8 \\[target\\] trade (D50=ships|vessels).",
            "It is with {6 that we have come to this point, but 7 from \\[destination\\] have refused to budge on key trade policies, which have forced us into the (D50=unfortunate|unenviable) position of having to resort to a more forceful response. We are authorising 5 into their space with the (D50=goal|sole purpose) of 8 \\[target\\] of their trade (D50=ships|vessels)."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType4 = function $missionType4() {
    var def = {
        fieldA: [].concat(this._gcmPG.incursions),
        fieldB: ["make any sort of impact on", "get any sort of traction with", "have any impact on"],
        fieldC: ["sailing into and out of the system", "brazenly chugging along the spacelanes without fear", "entering the system without fear", "able to enter the system almost without fear"],
        fieldD: ["we haven't generated one news story or public announcement", "we don't even rate a mention on their nightly news broadcasts", "we've been relegated to an also-ran behind the weather and sport on their news tabloids", "we are considered something of a has-been by the people there"],
        fieldE: ["know the truth about", "have their eyes opened about", "see the reality of the atrocities performed by"],
        fieldF: ["we now need help", "we are in need of some help", "we need some help", "we are now looking for some help"],
        fieldG: [].concat(this._gcmPG.in_search_of),
        fieldH: [].concat(this._gcmPG.destroy),
        fieldI: [],
        fieldJ: ["as a way of drawing attention to our cause", "as a way of bringing our cause back into the spotlight", "to once again instil (D50=terror|fear) into the hearts of traders in that system"],
        fieldK: ["had a pretty big impact on our ability to", "really messed up our illegal operations, to the point where we are struggling to", "drained the profits out of a lot of our operations, so that it is becoming extremely difficult to"],
        fieldL: ["send a message to", "let", "broadcast a message to", "broadcast over every comm channel, to"],
        fieldM: ["risking damage to their legal status", "dealing with the potential legal backlash", "leaving their moral high-ground at the door"],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [].concat(this._gcmPG.pilot),
        sentences: [
            "Our 1 in \\[destination\\] have been struggling to 2 the government there. More traders than we care to admit are 3, and 4. We want the people of \\[destination\\] to 5 their government. But 6. We are 7 >{9 who can risk the wrath of GalCop Vipers and }8 \\[target\\] trader (D50=ships|vessels), {1.",
            "A recent push by GalCop and local government forces has {2 keep our various lines of business operating smoothly. We want to 3 GalCop and anyone else that we will not go quietly into the night. We will survive. But }6. We are 7 >{9 who can(D50=, for a short time at least,|) defy GalCop by joining (D50=with us|our ranks) and 4 by going to \\[destination\\] to }8 \\[target\\] trader (D50=ships|vessels)(D50= from anywhere in the system|).",
            "With GalCop and local government forces spreading their version of 'peace' in \\[destination\\], traders have been 3. We want them to 5 their government and to return fear to the minds of all traders. We are 7 >{9 who are OK with 4, flying into \\[destination\\] and ruthlessly }8 \\[target\\] trade (D50=ships|vessels)(D50= from anywhere in the system|)."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType5 = function $missionType5() {
    var def = {
        fieldA: [].concat(this._gcmPG.incursions),
        fieldB: [].concat(this._gcmPG.in_search_of),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.backbone),
        fieldE: [].concat(this._gcmPG.ability),
        fieldF: ["increasing", "on the increase", "escalating", "intensifying"],
        fieldG: [].concat(this._gcmPG.venture_into),
        fieldH: [].concat(this._gcmPG.destroy),
        fieldI: ["combating", "fighting", "addressing", "contending with", "opposing", "battling"],
        fieldJ: ["recently", "in recent months", "of late", "lately"],
        fieldK: ["an increased number", "an [5] in the number", "a [4][3] in the number"],
        fieldL: ["increase", "upsurge", "escalation", "rise", "growth"],
        fieldM: ["steady ", "sharp ", "distinct "],
        fieldN: ["increase", "upsurge", "escalation"],
        fieldO: ["led us to believe", "brought us to the conclusion", "allowed us to determine"],
        fieldP: [].concat(this._gcmPG.contract),
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Ongoing 1 by Thargoid warships have necessitated the recruitment of (D50=non-military|civilian) forces in 9 the threat. Long range (D50=scans|telemetry data) of the Thargoid arrival vector has {6 that their (D50=staging area|point of origin) is somewhere between \\[system\\] and \\[destination\\]. We are }2 >3 45 to 7 interstellar space between \\[system\\] and \\[destination\\] (D50=and|to engage and) 8 \\[target\\] Thargoid warships.",
            "We have been seeing {2 of Thargoid invasion vessels entering our system 1. Long range (D50=scans|telemetry data) taken from their arrival vector has 6 the Thargoids are (D50=coming from|staging) somewhere between \\[destination\\] and \\[system\\]. To combat this threat we are }2 (D50=non-military|civilian) >3 45 to 7 interstellar space (D50=and|to engage and) 8 \\[target\\] Thargoid warships.",
            "Thargoid warships have been invading our space with a disturbing frequency {1. Long-range (D50=scans|telemetry data) taken from their arrival vector has 6 they are (D50=coming from|staging) somewhere between \\[destination\\] and \\[system\\]. Our plan in }9 this threat is simple: we are 2 (D50=non-military|civilian) >3 45 to 7 interstellar space (D50=and|to engage and) 8 \\[target\\] Thargoid warships.",
            "Thargoid 1 have been 6 {1. While their ultimate purpose in our space is still unknown, their on-going presence has impacted on our trade. Long-range (D50=scans|telemetry data) taken from their arrival vector has 6 they are (D50=coming from|staging) somewhere between \\[destination\\] and \\[system\\], so to combat this threat we are }2 (D50=non-military|civilian) >3 45 to 7 interstellar space (D50=and|to engage and) 8 \\[target\\] Thargoid warships.",
            "Our military (D50=leaders|commanders) are planning a major (D50=battle|assault) against the Thargoids, but (D50=prior to this|before this happens) they want some non-military >3 45 to 7 interstellar space between \\[system\\] and \\[destination\\] to (D50=perform some diversionary tactics|act as a diversion). The military has authorised this {7, for the destruction of \\[target\\] warships, on top of the usual bounty given for Thargoid warship destruction."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType6 = function $missionType6() {
    var def = {};
    if (Math.random() > 0.4) {
        def = {
            fieldA: [].concat(this._gcmPG.containers),
            fieldB: ["stash", "supply", "hoard", "stockpile"],
            fieldC: ["abandoned", "dumped", "dropped", "discarded", "ditched"],
            fieldD: [].concat(this._gcmPG.in_search_of),
            fieldE: ["have them recovered", "get them back", "collect them"],
            fieldF: [].concat(this._gcmPG.pilot),
            fieldG: [].concat(this._gcmPG.is_willing_to),
            fieldH: [].concat(this._gcmPG.travel_there),
            fieldI: ["recover them", "pick them up", "collect them", "scoop them", "gather them up"],
            fieldJ: ["return them", "bring them back"],
            fieldK: ["quite valuable", "very precious", "quite expensive", "quite costly", "valuable", "expensive", "costly"],
            fieldL: ["Due to (D50=a rather|an) unfortunate (D50=sequence|series) of events, one of our trade (D50=ships|vessels) was", "Because of (D50=a rather|an) unfortunate (D50=sequence|series) of events, one of our trade (D50=ships|vessels) was", "An unfortunate (D50=sequence|series) of events has resulted in one of our trade (D50=ships|vessels) being", "Largely due to (D50=a rather|an) unfortunate (D50=sequence|series) of events, one of our trade (D50=ships|vessels) was", "Largely because of (D50=a rather|an) unfortunate (D50=sequence|series) of events, one of our trade (D50=ships|vessels) was", "Mainly because of (D50=a rather|an) unfortunate (D50=sequence|series) of events, one of our trade (D50=ships|vessels) was"],
            fieldM: [],
            fieldN: [].concat(this._gcmPG.attacked),
            fieldO: ["was [5]", "came under fire", "was fired upon", "has come under fire"],
            fieldP: [].concat(this._gcmPG.destroyed),
            fieldQ: ["exploded", "detonated", "disintegrated", "blew up"],
            fieldR: [].concat(this._gcmPG.contract),
            sentences: [
                "A 2 of \\[target\\] cargo 1 has been 3 in an asteroid field in the \\[destination\\] system, \\[position\\]. The contents of these canisters is {2, so we are (D50=obviously|naturally) (D50=keen to|eager to) }5. We are 4 a 6 who 7 8, find the \\[target\\] lost cargo 1, 9 and {1, intact.",
                "One of our trade (D50=ships|vessels) {6 in the \\[destination\\] system, \\[position\\], resulting in the }6 having to (D50=flee|eject) and the (D50=ship|vessel) {7. However, the }6 has notified us that the cargo was 3 in an asteroid field before the (D50=ship|vessel) {8, which gives us a chance to recover it. Your 9 is to }8, find the \\[target\\] missing cargo 1, 9 and {1 here.",
                "{3 (D50=forced|compelled) to dump his 2 cargo in an asteroid field in \\[destination\\], \\[position\\]. We are (D50=keen|eager) to }5, so we are 4 a 6 who 7 8, find the \\[target\\] 3 cargo 1, 9 and {1(D50= here|).",
                "A trade (D50=ship|vessel) (D50=carrying|hauling) valuable cargo {6 in an asteroid field in the \\[destination\\] system, \\[position\\], and was (D50=forced|compelled) to dump the cargo rather than risking it to being destroyed with his ship. (D50=Due to|Because of) the value of this cargo, we are (D50=understandably|most) (D50=keen|eager) to get it back. So we're }4 a 6 who 7 8, find the \\[target\\] 1, 9 and {1(D50= here|)."
            ],
        };
    } else {
        def = {
            fieldA: [].concat(this._gcmPG.happened),
            fieldB: ["an unusual", "a strange", "a weird", "a bizarre"],
            fieldC: [].concat(this._gcmPG.pilot),
            fieldD: [].concat(this._gcmPG.due_to),
            fieldE: ["past run-ins with GalCop", "a dislike for our cuisine", "the fallout from a broken relationship", "a falling out with a business partner", "having called a local politician [6]", "having called the local police chief [6]", "having called a local gangland leader [6]", "a bad case of indigestion"],
            fieldF: ["a[7] skunk", "a[7] trumble", "a[7] rat", "a[7] snake", "a[7] rodent", "a[7] gerbil", "a[7] micro-organism", "a[7] food-processor"],
            fieldG: ["n incontinent", " lop-sided", "n odorous", " miserable", " glutinous", "n overweight", "n oversized", " low-life", " dirty", " moronic"],
            fieldH: [].concat(this._gcmPG.travel_there),
            fieldI: ["recover them", "pick them up", "collect them", "scoop them", "gather them up"],
            fieldJ: ["abandoned", "dumped", "dropped", "discarded", "ditched"],
            fieldK: [].concat(this._gcmPG.containers),
            fieldL: ["has declined", "has refused", "was unwilling"],
            fieldM: ["quite valuable", "very precious", "quite expensive", "quite costly", "(D50=rather |)valuable", "(D50=rather |)expensive", "(D50=rather |)costly"],
            fieldN: [],
            fieldO: [],
            fieldP: [],
            fieldQ: [],
            fieldR: [],
            sentences: [
                "2 set of circumstances has 1 with one of our shipments. The 3 {3 to (D50=actually |)enter our system }4 5. So instead, they have deposited the cargo in the \\[destination\\] system, in an asteroid field \\[position\\]. We (D50=need|require) a 3 to 8, find the \\[target\\] {1 cargo 2, }9 and return them here.",
                "The 3 of one of our trade (D50=ships|vessels) {3 to (D50|actually |)enter our system }4 5, so instead they have {1 their 4 cargo in an asteroid field in the \\[destination\\] system, \\[position\\]. We (D50=need|require) a }3 to 8, find the \\[target\\] {1 2, }9 and bring them back here safely."
            ],
        };
    }
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType7 = function $missionType7() {
    var def = {
        fieldA: [].concat(this._gcmPG.in_search_of),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: ["heading there", "going there", "flying there", "travelling there", "heading over there", "flying over there"],
        fieldD: [].concat(this._gcmPG.containers),
        fieldE: ["recovering them", "picking them up", "collecting them", "scooping them", "gathering them up"],
        fieldF: ["hauling", "carrying", "bringing"],
        fieldG: ["be a breeze", "be a walk in the park", "be simple enough", "be a piece of cake"],
        fieldH: ["were rewarded with", "scored big-time with", "had success with", "netted", "scored"],
        fieldI: ["Unfortunately they didn't have enough free cargo space", "But they were short on cargo space with which", "As luck would have it, they were already quite full from other raids and so were unable", "But as fate would have it, there wasn't enough cargo space on the ships in the convoy so they couldn't"],
        fieldJ: [].concat(this._gcmPG.attacked),
        fieldK: ["very juicy", "very satisfying", "most excellent", "impressive", "significant", "sizeable", "large, juicy", "very lucrative", "very sweet"],
        fieldL: ["loot", "cargo pods", "cargo canisters", "cargo", "cargo containers", "swag", "booty", "plunder", "spoils"],
        fieldM: ["scoop it all", "pick it all up", "collect it all", "recover it all"],
        fieldN: ["incursion into", "invasion into", "raid in", "attack in", "foray into", "sortie into"],
        fieldO: ["someone else finds it", "it falls into the wrong hands", "it is lost forever", "it disappears for good"],
        fieldP: ["collected", "scooped up", "recovered", "gathered up"],
        fieldQ: ["at commodity markets that aren't, shall we say, operated in complete compliance with the law", "among the many traders that pass through that system", "by extracting our cargo tax from passing ships"],
        fieldR: [", unfortunately, they didn't have enough free cargo space", " they were short on cargo space", ", as luck would have it, they were already quite full from other raids", ", as fate would have it, there wasn't enough cargo space on the ships in the convoy "],
        sentences: [
            "One of our teams {1 a convoy in the \\[destination\\], and }8 a {2 haul of 3. }9 to {4. We're }1 a 2 who can help us out by 3, finding the \\[target\\] cargo 4 in an asteroid field \\[position\\], 5 and 6 them back here. Slip in and out - should 7.",
            "During a recent {5 \\[destination\\], we }8 a {2 bounty, but some of it had to be left behind simply because9. This cargo, \\[target\\] }4 of it, located in an asteroid field \\[position\\], needs to be {7 and (D50=returned|brought back) here, before {6.",
            "Our raiding (D50=crews|teams) (D50=often|regularly) head to \\[destination\\], as there can be some {2 deals to be had {8. (D50=One trip recently|One recent trip) was (D50=particularly|especially) fruitful - so fruitful in fact that there wasn't enough cargo space to 4. So we are looking for a }2 to head over to \\[destination\\], find the \\[target\\] 4 in an asteroid field \\[position\\], and haul them all back here."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType8 = function $missionType8() {
    var def = {
        fieldA: [].concat(this._gcmPG.deteriorated),
        fieldB: [].concat(this._gcmPG.due_to),
        fieldC: [].concat(this._gcmPG.crime),
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.backbone),
        fieldG: [].concat(this._gcmPG.ability),
        fieldH: [].concat(this._gcmPG.containers),
        fieldI: ["Our relations", "Our trade relations", "Trade relations"],
        fieldJ: [].concat(this._gcmPG.steal),
        fieldK: [].concat(this._gcmPG.helpful_reason),
        fieldL: [].concat(this._gcmPG.broken_down),
        fieldM: [].concat(this._gcmPG.stealing),
        fieldN: [].concat(this._gcmPG.be_sufficient),
        fieldO: [].concat(this._gcmPG.inexperienced),
        fieldP: [].concat(this._gcmPG.we_believe),
        fieldQ: ["inequitable", "unfair", "discriminatory", "unjust"],
        fieldR: ["take matters into our own hands", "deal with this issue directly", "do something about it"],
        sentences: [
            "9 with the \\[destination\\] system have 1, 2 3, and we are now actively pursuing all possible avenues for restitution. We are 4 >5 67 to find traders in the \\[destination\\] system and {1 their cargo. We (D50=need|require) \\[target\\] }8 of any type collected.",
            "Despite (D50=lengthy negotiations|our best efforts), and the presence of {2, our trade relations with \\[destination\\] have }1 to the point where they have {3, although many believe it was }2 3. To force them back to the negotiation table we are looking put a limit on their trade by sending willing >5 into their system and {1 \\[target\\] cargo }8. Any type of cargo will {5.",
            "2 some {6 negotiators and }3, trade relations with \\[destination\\] have 1 to the point where they have {3. But we are determined to bring them back to the negotiation table, so to force their hand we are looking to limit their trade by sending willing >}5 67 to go on a mission to {1 \\[target\\] cargo }8 from anywhere within their system. Any type of cargo will {5.",
            "In an embarrassing turn of events for our government, 2 3, we have been forced into a rather {8 trade deal with \\[destination\\]. But instead of just living with the economic (D50=limitations|implications) we are choosing to 9. We are authorising attacks on traders in \\[destination\\], with the goal of 4 \\[target\\] cargo }8 of any type. {7 this action will enable us to re-negotiate the trade agreement and bring about economic growth."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType9 = function $missionType9() {
    var def = {
        fieldA: [].concat(this._gcmPG.steal),
        fieldB: [].concat(this._gcmPG.containers),
        fieldC: [].concat(this._gcmPG.in_search_of),
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: ["Easy as pie", "Like stealing candy from a baby", "Money for jam", "Piece of cake", "As easy as 123", "Nice and simple", "No complications whatsoever"],
        fieldF: ["Roll up! Roll up!", "Come on in, commander!", "Welcome to paradise, commander!", "Welcome to the new world order!", "Come and see how the other half lives!"],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We are pirates. We hunt traders for cargo. 5 - fly to \\[destination\\] and 1 \\[target\\] 2 of anything. It all sells.",
            "We want to increase the pressure we've been exerting on \\[destination\\], to make them feel the pain. One way we do that is to target their trade. We need anyone with a ship and a scoop to head over there and 1 \\[target\\] 2 of anything and everything.",
            "We are 3 >4 to join our team. The cost of entry is pretty low - just head to \\[destination\\] and 1 \\[target\\] 2 of anything. Even food has value, if you got it for nothing.",
            "6 This is your chance to live the life of a pirate. All you need is a ship and a scoop, and you're ready to roll. Head on over to \\[destination\\] and 1 \\[target\\] 2 of anything, and (D50=return|bring) it back here.",
            "This is as simple as it gets for pirates. Go to some location, in this case, \\[destination\\], and 1 \\[target\\] 2 of any commodity you can extract from traders in that system. Then (D50=return|bring) it back here. 5."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType10 = function $missionType10() {
    var def = {
        fieldA: [].concat(this._gcmPG.deteriorated),
        fieldB: [].concat(this._gcmPG.due_to),
        fieldC: [].concat(this._gcmPG.crime),
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.backbone),
        fieldG: [].concat(this._gcmPG.ability),
        fieldH: [].concat(this._gcmPG.containers),
        fieldI: ["Our relations", "Our trade relations", "Trade relations"],
        fieldJ: [].concat(this._gcmPG.steal),
        fieldK: [].concat(this._gcmPG.helpful_reason),
        fieldL: [].concat(this._gcmPG.broken_down),
        fieldM: [].concat(this._gcmPG.stealing),
        fieldN: [],
        fieldO: [].concat(this._gcmPG.inexperienced),
        fieldP: [].concat(this._gcmPG.we_believe),
        fieldQ: ["inequitable", "unfair", "discriminatory", "unjust"],
        fieldR: ["take matters into our own hands", "deal with this issue directly", "do something about it"],
        sentences: [
            "9 with the \\[destination\\] system have 1, 2 3, and we are now actively pursuing all possible avenues for restitution. We are 4 >5 67 to find traders in the \\[destination\\] system and {1 their cargo. We (D50=need|require) \\[target\\] }8 of \\[commodity\\] collected. Remember, only \\[commodity\\].",
            "Despite (D50=lengthy negotiations|our best efforts), and the presence of {2, our trade relations with \\[destination\\] have }1 to the point there they have {3, although many believe it was }2 3. To force them back to the negotiation table we are looking put a limit on their trade by sending willing >5 into their system and {4 \\[target\\] cargo }8 of \\[commodity\\] only.",
            "2 some {6 negotiators and }3, trade relations with \\[destination\\] have 1 to the point where they have {3. But we are determined to bring them back to the negotiation table, so to force their hand we are looking to limit their trade by sending willing >}5 67 to go on a mission to {1 \\[target\\] }8 of \\[commodity\\] from anywhere within their system.",
            "In an embarrassing turn of events for our government, 2 3, we have been forced into a rather {8 trade deal with \\[destination\\]. But instead of just living with the economic (D50=limitations|implications) we are choosing to 9. We are authorising attacks on traders in \\[destination\\], with the goal of 4 \\[target\\] cargo }8 of \\[commodity\\]. {7 this action will enable us to re-negotiate the trade agreement and bring about economic growth."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType11 = function $missionType11() {
    var def = {
        fieldA: [].concat(this._gcmPG.in_search_of),
        fieldB: [].concat(this._gcmPG.steal),
        fieldC: [].concat(this._gcmPG.containers),
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: [],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We are pirates, and in order to maintain our position in this system, we raid neighbouring systems and 2 their cargo. At the moment, we have a need for \\[commodity\\], \\[target\\] 3 of it actually. We are 1 any 4 who can head to \\[destination\\] and do this job for us.",
            "Demand for \\[commodity\\] has gone up locally, but our raiding parties into \\[destination\\] haven't been able to meet that demand. We're looking for anyone with ship and a scoop to head over to \\[destination\\] and 2 \\[target\\] 3 of it.",
            "After a couple of disastrous runs into \\[destination\\], we're tight pressed to meet the demand for \\[commodity\\]. We need you to head over to that system, and 2 \\[target\\] 3 of it, and bring it back here."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType12 = function $missionType12() {
    var def = {
        fieldA: [].concat(this._gcmPG.attention),
        fieldB: [].concat(this._gcmPG.large),
        fieldC: ["shipped into", "transported into", "conveyed through", "distributed through"],
        fieldD: [].concat(this._gcmPG.vile),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.backbone),
        fieldG: [].concat(this._gcmPG.ability),
        fieldH: ["dump", "drop", "abandon", "ditch", "discard"],
        fieldI: [].concat(this._gcmPG.be_sufficient),
        fieldJ: ["GalCop security transcripts", "trade manifest data", "reports from both inside and outside the system"],
        fieldK: [].concat(this._gcmPG.in_search_of),
        fieldL: ["constant", "continual", "continuous", "regular"],
        fieldM: [].concat(this._gcmPG.we_believe),
        fieldN: [].concat(this._gcmPG.achievable),
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "1 that a 2 number of slaves are being 3 \\[destination\\], (D50=most|many) of whom are not willing participants. We are attempting to (D50=clamp|crack) down on this 4 trade by finding >5 67 to demand all traders in the \\[destination\\] system to 8 their cargo of slaves. A target of \\[target\\]t will 9. The slaves will be handed over to Amnesty Intergalactic for repatriation.",
            "According to {1, nearby \\[destination\\] has become a hub through which a }2 number of trader (D50=ships|vessels) are travelling while carrying an equivalently large number of slaves. This trafficking in life is a 4 practice, and we are attempting to stamp it out where we can. We are {2 >}5 67 to (D50=enter|visit) the \\[destination\\] system and demand all traders to 8 their cargo of slaves. We are looking for a target of \\[target\\] containers, and all slaves must be handed over to Amnesty Intergalactic for repatriation.",
            "A {3 flow of slave trafficking both into and through the \\[destination\\] system has prompted us to respond. We are 2 a }5 67 to enter \\[destination\\], find any ships carrying slaves, and demanding they be handed over. All slaves will then be repatriated through Amnesty Intergalactic. {4 a target of \\[target\\] containers of slaves will be 5.",
            "We have been monitoring the activities of slave traders in and around our system and have discovered that \\[destination\\] has become something of a hub for this 4 trade. We believe in freedom from slavery, so we are {2 a }5 67 to enter the \\[destination\\] system, find any ships carrying slaves, and demand they be handed over. All slaves will then be repatriated through Amnesty Intergalactic. A target of \\[target\\] containers should be {5."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType13 = function $missionType13() {
    var def = {
        fieldA: [].concat(this._gcmPG.plenty_of),
        fieldB: [].concat(this._gcmPG.in_search_of),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.venture_into),
        fieldE: [].concat(this._gcmPG.destroyed),
        fieldF: [].concat(this._gcmPG.destroy),
        fieldG: ["constantly", "continually", "continuously", "regularly"],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Our scientists are keen to study Thargoid technology, particularly in relation to their warships. There are 1 sources for their drone ships, but almost none for wreckage from larger Thargoid (D50=ships|vessels).\\n\\nWe are 2 a 3 who can 4 interstellar space and collect \\[target\\]t of alloys and wreckage from 5 Thargoid warships.",
            "With the war against the Thargoids continuing, we are conducting research into their technology in order to improve our ships and weapons against them. And while we have a steady supply of the Thargoid Robotic drone ships, we have far fewer samples of Thargoid warships to enable our technicians and scientists to adequately cover their tests and studies.\\n\\nSo we are 2 a 3 who can 4 interstellar space, 6 several Thargoid warships and collect \\[target\\]t of any alloys or wreckage left behind.",
            "War is (D50=escalating|intensifying) with the Thargoids, and we are 7 2 ways to improve our technology to better match theirs. In order to do that, our scientists (D50=need|require) access to Thargoid equipment and technology. We already have a steady supply of their drone ships - what we (D50=need|require) is wreckage from their warships.\\n\\nSo we are 2 a 3 who can 4 interstellar space and collect \\[target\\]t of alloys or wreckage from 5 Thargoid warships.",
            "While there is always a market for Thargoid drone ships, there is less of a one for the wreckage of Thargoid warships. But in order to study Thargoid technology and get the upper hand in the ongoing war, our scientists need \\[target\\]t of alloys or wreckage from 5 Thargoid warships. Studying this wreckage will give us the opportunity to study their technology (D50=intently|closely) and perhaps finding a way to eliminate the Thargoid menace."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType14 = function $missionType14() {
    var def = {
        fieldA: ["It has come to our attention", "We have reason to believe", "We have it on good authority"],
        fieldB: [].concat(this._gcmPG.in_search_of),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.travel_to),
        fieldE: [].concat(this._gcmPG.are_willing_to),
        fieldF: ["escalating", "mounting", "intensifying", "increasing"],
        fieldG: ["learned", "discovered", "ascertained", "found out"],
        fieldH: ["more than a little concerning", "extremely disquieting", "a cause for deep concern", "a reason to be seriously concerned"],
        fieldI: [].concat(this._gcmPG.agents),
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "1 that certain elements within the society on \\[destination\\] are importing high quality firearms from high-tech worlds. Given the level of antagonism that exists between both our worlds, we feel this development is 8. So we are 2 >3 who can 4 \\[destination\\] and collect \\[target\\]t of firearms from ships entering or leaving the system, and return them here. That way, we can restrict the flow of these weapons while at the same time making it unlikely the government on \\[destination\\] will hear about it.",
            "1 that the government of \\[destination\\] is stock-piling high-tech firearms. Given the relations between our two worlds has never been cordial, we are concerned these weapons may be used against us. We (D50=need|require) >3 to 4 their space and collect \\[target\\]t of firearms from ships entering or leaving the system, by whatever means necessary, and return them here.",
            "One of our 9, working in the \\[destination\\] system, has (D50=uncovered|discovered) large stockpiles of high-tech firearms, hidden away by the government. This constitutes a serious threat to the stability and safety of all local systems. To combat this, we are 2 >3 who 5 enter \\[destination\\] and collect \\[target\\]t of firearms from any (D50=ships|vessels) entering or leaving the system. Do not purchase firearms on the open market, as these transactions can be traced back to us. Once you have them, return them here.",
            "Tensions with a neighbouring system, \\[destination\\], are 6. We have 7 through unofficial channels that they are bringing in large quantities of firearms, which local military leaders have found 8. To combat this serious threat to our stability and safety we are authorising >3 to 4 \\[destination\\] and collect \\[target\\]t of firearms from any (D50=ships|vessels) entering or leaving the system, and return them here. Firearms purchases on the open market will not be accepted."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType15 = function $missionType15() {
    var def = {
        fieldA: [].concat(this._gcmPG.pilot),
        fieldB: [].concat(this._gcmPG.travel_to),
        fieldC: [].concat(this._gcmPG.steal),
        fieldD: [].concat(this._gcmPG.achievable),
        fieldE: [],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Local drug lords have reported that their trade has been interrupted by drug runners taking a larger than usual proportion of imported drugs to \\[destination\\]. Therefore they are commissioning >1 to 2 \\[destination\\] and 3 \\[target\\]t of narcotics from any and all ships in the system.",
            "In a move designed intentionally to disrupt the local illicit drug trade, dealers in the \\[destination\\] system have been bribing traders to bypass our system completely when delivering imported drugs. So to combat this development, the local dealers have combined to commission >1 to 2 \\[destination\\] and 3 \\[target\\]t of narcotics from any and all ships in the system.",
            "GalCop and local system authority (D50=ships|vessels) have been (D50=clamp|crack)ing down on our business, which means there's a shortage of drugs here and we are struggling to meet demands. We (D50=need|require) >1 to 2 \\[destination\\], 3 \\[target\\]t of narcotics from any and all ships in the system, and return them here.",
            "A series of bad crops and over-zealous police raids has led to a shortfall in our supply of recreational drugs. But we've heard that the \\[destination\\] system has plenty of narcotics being shipped in. We feel it's only fair for us to raid their trade ships and bring back some of these narcotics to help with local demand. A total of \\[target\\]t should be 4."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType16 = function $missionType16() {
    var def = {
        fieldA: ["hard at us", "hitting us hard", "attacking us hard"],
        fieldB: ["recently", "in recent days", "over the past few weeks"],
        fieldC: ["cutting in on our operations", "limiting our operations", "limiting our field of operations", "shutting down some of our operations", "restricting what we can do", "getting into our business interests"],
        fieldD: ["making life difficult", "making our lives unpleasant", "making life miserable", "making our lives here frustrating"],
        fieldE: ["flunkies", "hell-spawn", "hangers-on", "groupies", "mindless drones", "enforcers"],
        fieldF: ["But the time has come [7]", "We want [7]", "So we want [7]", "We're planning [7]", "So we're planning [7]", "We're going [7]", "So we're going [7]", "But all that is going to change", "But they are in for an unpleasant surprise", "But they are going to regret tangling with us"],
        fieldG: ["to fight back", "to give them a message they'll never forget", "to hit them where it hurts the most", "to hit them where they feel it most", "to attack them in the most fundamental way", "to take back what is ours", "to give them some payback"],
        fieldH: ["The mission we have for you", "The task we want you to perform", "The job we have in store for you", "The mission we need you to perform"],
        fieldI: [].concat(this._gcmPG.destroy),
        fieldJ: ["We know this is(D50= going to be|)", "We understand this is(D50= going to be|)", "We fully understand this is(D50= going to be|)"],
        fieldK: ["a dangerous mission", "a very dangerous mission", "a serious undertaking", "not something to be done on a whim", "not going to be easy"],
        fieldL: ["will mean gaining a fugitive status(D50= and make life harder for you [4]|)", "will result in a fugitive status(D50= for you|)(D5= and make life harder [4]|)", "will make life harder for you [4]", "will have serious ramifications for you [4]"],
        fieldM: ["in the future", "going forward", "in the short and long term"],
        fieldN: ["We trust", "We hope", "We think", "But we trust", "But we think", "But we hope"],
        fieldO: ["fee", "payment", "agreed price", "reward"],
        fieldP: ["will compensate [8]", "will go some way towards compensating you", "will make the task a little more agreeable [8]", "will enable you to perform this task with [9]"],
        fieldQ: ["in some way", "to some degree", "to a certain extent"],
        fieldR: ["enthusiasm", "vigour", "ferocity"],
        sentences: [
            "GalCop have been 1 2, 3 and generally 4. 6.\\n\\n8 is to 9 \\[target\\] police vessels in this system. {1 2, and 3. 5 that 6 7.",
            "GalCop have been 4 2, by 3. 6. {1 2, and 3. 5 that 6 7.\\n\\n}8 is to 9 \\[target\\] police ships in this system. (D50=Do you reckon \\[gcm_youre_fix\\]|Are you) (D50=ready|up for the challenge)?",
            "2 GalCop have been 4 by 1 and 3. 6.\\n\\n8 is to 9 \\[target\\] police craft in this system. {1 2, and 3. 5 that 6 7.",
            "2 GalCop Vipers and their 5 have been 1, 4 and 3. 6.\\n\\n8 is to 9 \\[target\\] police ships in this system. {1 2, and 3. 5 that 6 7."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType17 = function $missionType17() {
    var def = {
        fieldA: ["large", "significant", "considerable", "significant"],
        fieldB: ["waste", "rubbish", "garbage", "refuse", "trash", "junk"],
        fieldC: [].concat(this._gcmPG.destroyed),
        fieldD: [].concat(this._gcmPG.containers),
        fieldE: ["intense", "extreme", "concentrated"],
        fieldF: ["an oversupply", "a glut", "a surplus", "an excess"],
        fieldG: [].concat(this._gcmPG.due_to),
        fieldH: ["we know everyone loves", "who doesn't love", "you've gotta love"],
        fieldI: [].concat(this._gcmPG.destroying),
        fieldJ: [].concat(this._gcmPG.hard),
        fieldK: [].concat(this._gcmPG.sexy),
        fieldL: [].concat(this._gcmPG.pilot),
        fieldM: ["wonderful", "marvellous", "fantastic", "superb", "brilliant"],
        fieldN: [].concat(this._gcmPG.is_willing_to),
        fieldO: [].concat(this._gcmPG.calling_for),
        fieldP: [].concat(this._gcmPG.contract),
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We have a 1 stockpile of radioactive 2 that we need 3. But we can't just blow up the waste in orbit, or in any area of local space. Instead, we need these cargo 4 dumped close to the sun, to be 3 by the 5 heat. At present, we have \\[target\\] containers of waste to dump.\\n\\nPlease note: the 4 must be destroyed by the heat of the sun, and in no other way, for this {7 to be completed successfully.",
            "Our planet-side waste storage (D50=facilities|centres) are (D50=currently|presently) experiencing 6 of radioactive 2. 7 the nature of this waste, it can't simply be 3 in orbit or in any area of local space. The only way to ensure this waste is completely 3 is to (D50=drop|dump) it in close proximity to the sun, and allow the 5 heat to destroy them. We have \\[target\\] containers waiting to be destroyed in this way.\\n\\nPlease note: the 4 must be destroyed by the heat of the sun, and in no other way, otherwise this {7 will be considered terminated.",
            ">{3! A 4 opportunity awaits to earn extra cash, and }8 extra cash. Our planet-side waste processing (D50=facilities|centres) are {6 any pilot who 5 fly \\[target\\] containers of radioactive }2 to the sun and dump them close by so that the 5 heat destroys them. So, no dropping them off at a nearby system, or 9 them in the station aegis. Yes, we know it's not {1, or even 2, but it pays. Do you have what it takes?",
            "With planet-side waste storage and processing at maximum capacity, we have been looking at other potential solutions for our radioactive 2. Our technicians have determined that this refuse can be effectively and safely 3 if it's dumped near the sun, after which the 5 heat will do the rest. But it can only be dumped hear the sun - destroying it any other way will render this {7 null and void. \\[target\\]t of waste has been transferred to the station, ready to be hauled away.",
            "Our power generation (D50=facilities|centres) are unable to process a large quantity of radioactive material on the planet, due to the high risk of radiation escaping the processing plant. So we regularly commission >{3 to take this waste and dump it near the sun, where the }5 heat can do all the work for us. We have \\[target\\] containers waiting for a willing {3 to step up and do this job for us.\\n\\nPlease note: the }4 must be destroyed by the heat of the sun, and in no other way, for this {7 to be payable."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType18 = function $missionType18() {
    var def = {
        fieldA: ["always been a little on the chilly side", "never been what you could call 'friendly'", "rarely been civil"],
        fieldB: ["we've been firing on their ships", "we've been stealing their cargo", "we've been undermining their illicit operations"],
        fieldC: ["[4]", "complete [4]", "a load of [4]", "complete and utter [4]", "utter [4]"],
        fieldD: ["rubbish", "bollox", "garbage", "nonsense", "drivel", "tosh", "hogwash", "twaddle", "bosh"],
        fieldE: ["we are now at war", "a rival gang war has begun", "we have declared war on them", "they have declared war on us", "things have escalated into all-out war"],
        fieldF: [].concat(this._gcmPG.in_search_of),
        fieldG: ["For the duration of this mission", "For the entirety of this mission", "While this mission is active", "During the timeframe of this mission"],
        fieldH: ["ensure", "make sure", "see to it that"],
        fieldI: [].concat(this._gcmPG.travel_to),
        fieldJ: [].concat(this._gcmPG.destroy),
        fieldK: [].concat(this._gcmPG.pilot),
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Our relationship with the pirates from \\[destination\\] has 1, but now they have claimed that 2, which is 3. So 5. We are 6 a {2 to head over to \\[destination\\] and 1 \\[target\\] pirate (D50=ships|vessels). }7 we will 8 our own ships stay away, so there will be no chance of accidentally shooting the wrong pirate ship.",
            "Claims and counter-claims have been levelled at both sides, but regardless of the reasons, it has devolved into open war. We are 6 a {2 to head over to \\[destination\\] and {1 \\[target\\] pirate (D50=ships|vessels). }7 we will 8 our own ships stay away, so there will be no chance of accidentally shooting the wrong pirate ship.",
            "It probably comes as no surprise that our relationship with the pirates from \\[destination\\] has 1. But recent events have taken things to a whole new level. 5, and so we are 6 a {2 to help us take the war to them. We need you to }9 \\[destination\\] and {1 \\[target\\] of the blighters. We'll make sure none of our group will be in their space }7."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType20 = function $missionType20() {
    var def = {
        fieldA: [].concat(this._gcmPG.information_about),
        fieldB: [].concat(this._gcmPG.destroyed),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.travel_to),
        fieldF: ["notification of", "information [7]", "reports [7]"],
        fieldG: [].concat(this._gcmPG.concerning),
        fieldH: [].concat(this._gcmPG.broadcasting),
        fieldI: [].concat(this._gcmPG.pirates),
        fieldJ: [].concat(this._gcmPG.received),
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Planetary authorities have {1 }1 a (D50=ship|vessel) that was 2 in \\[destination\\], \\[position\\]. And while the 3 managed to flee in an escape capsule, unfortunately the transponder was damaged and we have lost all contact with it. With a shortage of rescue (D50=ships|vessels) in the area, we are 4 a 3 to 5 \\[destination\\] and find this missing escape pod before the life support system fails.",
            "We have received 6 the destruction of one of our (D50=ships|vessels) while in transit in \\[destination\\], \\[position\\]. The 3 managed to get out in time in their escape capsule, but it was damaged in the explosion and the transponder is no longer 8 a tracking signal. Combined with a shortage of rescue (D50=ships|vessels) and a slowly failing life-support system in the escape pod, we are 4 a 3 to 5 \\[destination\\] and rescue it.",
            "An emergency situation has arisen in the \\[destination\\] system. One of our (D50=ships|vessels) was in transit when they were set upon by 9 and destroyed. The 3 managed to eject, but the capsule must have been damaged in the explosion and the transponder appears to be offline and worse, their life-support system is slowly failing. All we know is that their last known (D50=location|position) was \\[position\\]. Are you able to find this pilot for us?"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType21 = function $missionType21() {
    var def = {
        fieldA: [].concat(this._gcmPG.information_about),
        fieldB: [].concat(this._gcmPG.destroyed),
        fieldC: [].concat(this._gcmPG.we_believe),
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: [].concat(this._gcmPG.in_search_of),
        fieldF: [].concat(this._gcmPG.venture_into),
        fieldG: [].concat(this._gcmPG.is_willing_to),
        fieldH: [].concat(this._gcmPG.forced),
        fieldI: [].concat(this._gcmPG.received),
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Planetary authorities have 9 1 a (D50=ship|vessel) that was 2 in interstellar space, somewhere between \\[system\\] and \\[destination\\], and 3 the 4 managed to flee in an escape capsule. Unfortunately, the transponder for this escape capsule was damaged and we have lost all contact with it. With a shortage of rescue (D50=ships|vessels) in the area, we are 5 a 4 to 6 interstellar space between \\[system\\] and \\[destination\\] and find this missing escape pod before the life support system fails.",
            "A 4 was 8 to abandon his (D50=ship|vessel) in interstellar space, somewhere between \\[system\\] and \\[destination\\]. We are receiving intermittent transponder signals that make us suspect the escape capsule was damaged in the destruction of the (D50=ship|vessel). We are 5 a 4 who 7 6 interstellar space and rescue the escape pod before the life support systems fails or they are discovered and destroyed by Thargoids.",
            "We have received word of the destruction of one of our (D50=ships|vessels) in interstellar space, somewhere between \\[system\\] and \\[destination\\]. Some intermittent transponder signals have come through that lead us to believe the pilot managed to eject in their escape capsule, but there are strong indications the capsule was damaged in the explosion. We (D50=need|require) a 4 to head into interstellar space and rescue our pilot before their life-support fails or they are discovered and destroyed by Thargoids."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType22 = function $missionType22() {
    var def = {
        fieldA: [].concat(this._gcmPG.information_about),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: [].concat(this._gcmPG.forced),
        fieldD: [].concat(this._gcmPG.imperative),
        fieldE: [].concat(this._gcmPG.we_believe),
        fieldF: [].concat(this._gcmPG.broadcasting),
        fieldG: [].concat(this._gcmPG.in_search_of),
        fieldH: [].concat(this._gcmPG.travel_to),
        fieldI: [].concat(this._gcmPG.pirates),
        fieldJ: [].concat(this._gcmPG.received),
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We have {1 }1 a (D50=ship|vessel) that was abandoned in \\[destination\\] after the 2 was 3 to flee in an escape capsule. We (D50=need|have) to recover the black box from the (D50=ship|vessel) as information of 4 strategic importance is contained in it. 5 the (D50=ship|vessel) is 6 an intermittent signal that will show up on your space compass. We are 7 a 2 to 8 \\[destination\\], find this derelict (D50=ship|vessel) and recover the black box so we can act on the information.",
            "The 2 of a trade (D50=ship|vessel) from our fleet was 3 to abandon his (D50=ship|vessel) in \\[destination\\], and fortunately they were rescued shortly thereafter. However, 5 the (D50=ship|vessel)'s black box recorded some sensitive and time-critical information. We are 7 a 2 to 8 \\[destination\\], find the derelict (D50=ship|vessel), and recover the black box intact. 5 the derelict (D50=ship|vessel) is transmitting a signal intermittently, which will enable you to find it with your space compass.",
            "One of our >2 was allegedly attacked by 9 in the \\[destination\\] system and managed to eject. However, data from the escape pod has caused the insurers to question the story of the pilot. In order to fully process their claim, we (D50=need|require) a 2 to 8 \\[destination\\], find the derelict ship and recover the black box from within. The ship is giving off an irregular navigational signal, which should help you find it with your space compass. Retrieving this unit will enable the insurance claim to be processed quickly and get our 2 back in the black."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType23 = function $missionType23() {
    var def = {
        fieldA: [].concat(this._gcmPG.information_about),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: [].concat(this._gcmPG.forced),
        fieldD: [].concat(this._gcmPG.imperative),
        fieldE: [].concat(this._gcmPG.in_search_of),
        fieldF: [].concat(this._gcmPG.venture_into),
        fieldG: [].concat(this._gcmPG.we_believe),
        fieldH: [].concat(this._gcmPG.received),
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We have 8 1 a (D50=ship|vessel) that was abandoned in interstellar space, somewhere between \\[system\\] and \\[destination\\], after the 2 was 3 to flee in an escape capsule. We (D50=have|need) to recover the black box from the (D50=ship|vessel) as information of 4 strategic importance is contained in it. We are 5 a 2 to 6 interstellar space between \\[system\\] and \\[destination\\], find this derelict (D50=ship|vessel) and recover the black box so we can act on the information.",
            "The 2 of a trade (D50=ship|vessel) from our fleet was 3 to abandon his (D50=ship|vessel) in interstellar space, somewhere between \\[system\\] and \\[destination\\], and fortunately they were rescued shortly thereafter. However, 7 the (D50=ship|vessel)'s black box recorded some sensitive and time-critical information. We are 5 a 2 to 6 interstellar space between \\[system\\] and \\[destination\\], find the derelict (D50=ship|vessel), and recover the black box intact.",
            "One of our >2 was allegedly attacked in interstellar space, somewhere between \\[system\\] and \\[destination\\], but managed to eject. However, data from the escape pod has caused the insurers to question the story of the pilot. In order to fully process their claim, we (D50=need|require) a 2 to enter interstellar space between \\[system\\] and \\[destination\\], find the derelict ship and recover the black box from within. This will enable the insurance claim to be processed quickly and get our 2 back in the black."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType24 = function $missionType24() {
    var def = {
        fieldA: ["abandon", "eject from", "escape from", "leave"],
        fieldB: [].concat(this._gcmPG.forced),
        fieldC: [].concat(this._gcmPG.in_search_of),
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: [].concat(this._gcmPG.travel_to),
        fieldF: [].concat(this._gcmPG.destroyed),
        fieldG: [].concat(this._gcmPG.pirates),
        fieldH: [].concat(this._gcmPG.we_believe),
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "One of our couriers was 2 to 1 their (D50=ship|vessel) in the \\[destination\\] system, \\[position\\]. We have reports, however, that the ship wasn't completely 6 and that the special cargo on board may still be intact. We are 3 a 4 to 5 \\[destination\\], find the derelict (D50=ship|vessel), destroy it and return any cargo that is freed from the hulk.",
            "Our of our couriers was in transit in the \\[destination\\] system and on their way back here when they were set upon by 7 and 2 to 1 their (D50=ship|vessel) \\[position\\]. However, we don't believe the ship was destroyed, meaning there is a chance to recover the special cargo on board. We are 3 a 4 to 5 \\[destination\\], find the derelict (D50=ship|vessel), destroy it and return any cargo that is freed from the hulk.",
            "We take pride in the ability of our courier pilots to deliver on time, every time. Unfortunately, one was attacked by 7 in the \\[destination\\] system, and while the 4 is safe, having ejected from his ship, 8 the special cargo stored on his ship is still intact. This gives us the opportunity to send a willing 4 into \\[destination\\], with the job of finding the derelict (D50=ship|vessel), which is \\[position\\], destroying it, and returning any cargo that floats free."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType25 = function $missionType25() {
    var def = {
        fieldA: [].concat(this._gcmPG.information_about),
        fieldB: [].concat(this._gcmPG.destroyed),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.travel_to),
        fieldF: ["notification of", "information [7]", "reports [7]"],
        fieldG: [].concat(this._gcmPG.concerning),
        fieldH: [].concat(this._gcmPG.attention),
        fieldI: [].concat(this._gcmPG.happened),
        fieldJ: ["battle", "skirmish", "fight", "mêlée", "scuffle", "fracas", "altercation"],
        fieldK: [].concat(this._gcmPG.received),
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Planetary authorities have {2 }1 a (D50=ship|vessel) that was 2 in \\[destination\\]. And while the 3 managed to flee in an escape capsule, the ejection system malfunctioned, sending the pod away from the crash site at tremendous speed. The pod is now in danger of leaving the system completely. With a shortage of rescue (D50=ships|vessels) in the area, we are 4 a 3 to 5 \\[destination\\] and find this missing escape pod before the life support system fails. GalCop tracking satellites will provide a regular waypoint update of the pod's last known (D50=location|position).",
            "We have received 6 the destruction of one of our (D50=ships|vessels) while in transit in \\[destination\\]. The 3 managed to escape in their escape capsule, but because of an engine malfunction, the pod was ejected from the ship at tremendous speed. The pod is now in danger of leaving the system entirely. Combined with a shortage of rescue (D50=ships|vessels) and a slowly failing life-support system in the escape pod, we are 4 a 3 to 5 \\[destination\\] and catch it. GalCop tracking satellites will provide a regular waypoint update of the pod's last known (D50=location|position).",
            "8 that one of our (D50=ships|vessels) was 2 while in transit in the \\[destination\\] system. Fortunately, the pilot managed to eject in an escape capsule, but we are concerned that something must have 9 as we have odd telemetry coming from the capsule's transponder. It's possible the capsule was damaged during the {1. With a shortage of rescue (D50=ships|vessels) in the area, we are }4 a 3 to 5 the system, and track down the escape capsule. GalCop tracking satellites will provide a regular waypoint update of the pod's last known (D50=location|position)."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType30 = function $missionType26() {
    var def = {
        fieldA: [].concat(this._gcmPG.accept),
        fieldB: [].concat(this._gcmPG.contract),
        fieldC: [].concat(this._gcmPG.be_given),
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: [],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "\\[name\\] would like to meet with you in \\[destination\\], at a (D50=location|position) marked with a waypoint, to discuss another opportunity. Because these are dangerous times, when you 1 this 2 you will 3 a special passcode you will (D50=have|need) to transmit when you are in range of the ship at the destination to prove you are who you say you are. While you are free to accept or decline the additional opportunity, he is confident you will find the offer compelling.",
            "A certain 4, going by the name of \\[name\\], wants to meet up with you at a waypoint in \\[destination\\]. If you 1 this 2 you will 3 a security passcode you will (D50=have|need) to transmit to the ship at the destination. \\[name\\] believes the opportunity to present to you will be compelling, but you will be free to accept or decline as you see fit.",
            "A request has come through from a 4 by the name of \\[name\\]. They would like to meet with you in the \\[destination\\] system, at a waypoint you'll 3 if you 1 this 2. You'll also 3 a security passcode that you will (D50=have|need) to transmit to the ship at the destination in order to prove your identity. \\[name\\] has an opportunity to present to you, but you are free to accept it or decline it, based on your judgement."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType31 = function $missionType31() {
    var def = {
        fieldA: ["uncovered", "discovered", "unearthed", "found", "stumbled upon"],
        fieldB: [].concat(this._gcmPG.important),
        fieldC: [].concat(this._gcmPG.information),
        fieldD: ["urgency", "critical nature", "importance"],
        fieldE: [].concat(this._gcmPG.in_search_of),
        fieldF: [].concat(this._gcmPG.pilot),
        fieldG: [].concat(this._gcmPG.travel_to),
        fieldH: [].concat(this._gcmPG.short),
        fieldI: [].concat(this._gcmPG.imperative),
        fieldJ: [].concat(this._gcmPG.accept),
        fieldK: [].concat(this._gcmPG.contract),
        fieldL: [].concat(this._gcmPG.be_given),
        fieldM: ["pass on", "hand over", "provide"],
        fieldN: [].concat(this._gcmPG.set_up),
        fieldO: ["happening", "taking place", "transpiring", "occurring"],
        fieldP: [].concat(this._gcmPG.collect),
        fieldQ: ["\\n\\nIf the communication relay beacon signal is lost, you can turn it on for another 30 seconds or so by activating the 'Comms Relay Beacon Switch' primable equipment item (issued as part of the mission). The device can only be used 3 times before the transceiver inside it degrades, rendering it inoperable. Additionally, activating the device will incur a 5 percent penalty on the payment amount for the mission."],
        fieldR: [].concat(this._gcmPG.agents),
        sentences: [
            "One of our {9, (D50=operating|working) in the \\[destination\\] system, has }1 some 2 3, but they have been unable to transmit this data without compromising their cover. Instead, they have prepared an information drop point in the system, marked with a 'D' on your space compass. Because of the 4 of this matter, and to avoid sending official GalCop (D50=ships|vessels) into the area, we are 5 a 6 to 7 this spot, {7 the information cache and return. The relay can only (D50=broadcast|transmit) its (D50=location|position) for a }8 (D50=amount|period) of time, so it is 9 that the cache is collected and acted upon quickly.{8\\n\\nIf you {1 this 2, you will 3 security codes that will enable the data package to be delivered to your (D50=ship|vessel).",
            "One of our {9, who is (D50=operating|working) in the \\[destination\\] system, has made contact with us in order to 4 }2 3. Unfortunately, they were unable to transmit the entire data cache, but they have {5 a communications relay at drop-point in the \\[destination\\] system, marked with 'D' on your space compass. Because of the }4 of this matter, and to avoid sending official GalCop [gcm_ships] into the area, we are 5 a 6 to 7 this spot, {7 the information cache and return. The beacon on the relay can only (D50=broadcast|transmit) its (D50=location|position) for a }8 (D50=amount|period) of time, so it is 9 that the cache is collected quickly.{8\\n\\nIf you {1 this 2, you will 3 security codes that will enable the data package to be delivered to your (D50=ship|vessel).",
            "In these days of trouble, we need to be extra vigilant. We keep an eye on what is {6 in nearby systems in order to stop potential threats to the safety of our citizens. One of our 9, working in the \\[destination\\] system, has }2 3 to give us, but wasn't able to transmit it to us directly without exposing himself. So they have prepared an information drop, which, should you {1 this 2, will be marked as 'D' on your space compass. You will also be provided with a security code to unlock the data cache.\\n\\nThe relay can only (D50=broadcast|transmit) its (D50=location|position) for a }8 (D50=amount|period) of time, so it is 9 that the cache is collected quickly.{8"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType32 = function $missionType32() {
    var def = {
        fieldA: ["sensitivity", "classified nature", "top secret nature", "sensitive nature"],
        fieldB: [].concat(this._gcmPG.information),
        fieldC: [].concat(this._gcmPG.accept),
        fieldD: [].concat(this._gcmPG.contract),
        fieldE: [].concat(this._gcmPG.be_given),
        fieldF: [].concat(this._gcmPG.pilot),
        fieldG: [].concat(this._gcmPG.due_to),
        fieldH: [].concat(this._gcmPG.in_search_of),
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "An encrypted data package needs to be (D50=delivered|transported) to a (D50=ship|vessel), (D50=currently in the \\[destination\\] system|in the \\[destination\\] system at the moment). The 1 of this 2 necessitates the use of a pilot and ship, rather than over standard transmission channels. If you 3 this 4, you will 5 the waypoint co-ordinates for the (D50=ship|vessel). You'll (D50=have|need) to fly to the waypoint and give the data package to the 6 waiting there.",
            "A (D50=ship|vessel), (D50=currently in the \\[destination\\] system|in the \\[destination\\] system at the moment), is expecting delivery of an encrypted data package. 7 the 1 of this data, it cannot be transmitted over standard communication channels. If you 3 this 4 you will be required to fly to a waypoint in that system and transmit the data package to the 6 waiting there.",
            "We have an encrypted data package to deliver, but due to the 1 of the information we cannot (D50=broadcast|transmit) it over standard communication channels. Instead, we are 8 a 6 who can take the data package and delivery it to a (D50=ship|vessel) waiting in the \\[destination\\] system, at a waypoint you'll 5 upon acceptance of this 4."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType33 = function $missionType33() {
    var def = {
        fieldA: [].concat(this._gcmPG.pilot),
        fieldB: [].concat(this._gcmPG.in_search_of),
        fieldC: ["one of our [4] facilities", "a [4] facility"],
        fieldD: ["secure", "government", "secret", "high security", "military"],
        fieldE: ["The thief, however,", "But the thief"],
        fieldF: ["follow", "track", "trace"],
        fieldG: ["We're confident", "We believe", "We're fairly confident", "Our information suggests"],
        fieldH: ["a window of opportunity", "a chance to intercept", "the opportunity to intercept the agent", "the possibility of recovering the items"],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Some \\[missionstolenItemType\\] have been stolen from 3. 5 has left a trail we've been able to 6. 7 the ship is on its way to \\[destination\\] via the shortest route possible, which gives us 8. We (D50=need|require) a 1 who has a (D50=ship|vessel) capable of reaching the destination system ahead of the thief, who is flying a customised \\[missiontargetShipType\\], and recovering the items. Extreme force should only be used as a last resort.",
            "A foreign agent has infiltrated 3 and stolen some \\[missionstolenItemType\\]. However, when they departed our system we were able to track their ship, and we discovered they are headed to \\[destination\\] via the shortest route possible. This gives us 8. We are 2 a 1 who has a (D50=ship|vessel) capable of reaching the destination system ahead of the thief and recovering the items. The agent is flying a customised \\[missiontargetShipType\\]. Extreme force should only be used as a last resort.",
            "Our security specialists have detected that a foreign agent has stolen some \\[missionstolenItemType\\] from 3. They weren't able to stop the thief from escaping, but they were able to 6 their ship, a customised \\[missiontargetShipType\\]. Tracking data suggests the agent is heading for \\[destination\\], but by the shortest route, which gives us 8. We are 2 a 1 who can fly to \\[destination\\] by the quickest route, intercept the agent, and recover the \\[missionstolenItemType\\]. Extreme force should only be used as a last resort."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType40 = function $missionType40() {
    var def = {
        fieldA: [].concat(this._gcmPG.desires),
        fieldB: [].concat(this._gcmPG.be_given),
        fieldC: [],
        fieldD: [].concat(this._gcmPG.contract),
        fieldE: [],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Our client 1 a cargo of \\[target\\]t of \\[commodity\\] (D50=delivered|transported) to \\[destination\\]. However, this client does not want this transaction to appear in the standard GalCop shipping logs. Instead of (D50=delivering|transporting) the cargo to the main station, you will 2 a waypoint in \\[destination\\] to which you must travel and dump the cargo for the waiting ship.\\n\\nAlong with the payment listed on this bulletin board entry, you also will be paid with precious metals or gems once the cargo is scooped.",
            "In order to avoid the attention of GalCop, our client 1 a cargo of \\[target\\]t of \\[commodity\\] (D50=delivered|transported) to \\[destination\\], but not to the main station or any other facility. Instead, you will 2 a waypoint marker to which you must fly and then dump the cargo for the waiting (D50=ship|vessel).\\n\\nAlong with the payment listed on this bulletin board entry, you also will be paid with precious metals or gems once the cargo is scooped.",
            "GalCop record all trade (D50=ships|vessels) arriving and departing from stations, along with all cargo manifests. Our client (D50=needs|wants) to avoid this attention by having \\[target\\]t of \\[commodity\\] (D50=delivered|transported) to a waypoint in the \\[destination\\], the co-ordinates of which you will 2 upon acceptance of this 4.\\n\\nAlong with the payment listed on this bulletin board entry, you also will be paid with precious metals or gems once the cargo is scooped."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType41 = function $missionType41() {
    var def = {
        fieldA: [].concat(this._gcmPG.desires),
        fieldB: [].concat(this._gcmPG.set_up),
        fieldC: [].concat(this._gcmPG.short),
        fieldD: [].concat(this._gcmPG.be_given),
        fieldE: ["\\n\\nIf the communication relay beacon signal is lost, you can turn it on for another 30 seconds or so by activating the 'Comms Relay Beacon Switch' primable equipment item (issued as part of the mission). The device can only be used 3 times before the transceiver inside it degrades, rendering it inoperable. Additionally, activating the device will incur a 5 percent penalty on the payment amount for the mission."],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Our client 1 a certain cargo (D50=delivered|transported) to their location. However, this particular client does not want this transaction, the type of cargo or their current (D50=location|position) to appear in the standard GalCop shipping logs. A communications relay has been 2 in \\[destination\\] to which you must travel - it will be marked as 'D' on your space compass. The relay will only (D50=broadcast|transmit) its (D50=location|position) for a 3 (D50=amount|period) of time, so you will (D50=have|need) to act quickly when you arrive.5\\n\\nTransmit the passcode to this unit and you will 4 information about the type and amount of cargo you (D50=have|need) to (D50=deliver|transport), and a waypoint in another system, which should then allow you to purchase the cargo from any station. The cost of the cargo will be offset to you using the main station's prices at the final destination.\\n\\nYou must then travel to the final destination location and dump the cargo for the waiting ship.\\n\\nAlong with the payment listed on this bulletin board entry, you also will be paid with precious metals or gems once the cargo is scooped.",
            "In order to avoid the attention of GalCop, our client wants a particular cargo delivered to a specific location. But the client does not want this transaction, the type of cargo or their current (D50=location|position) to appear in the standard GalCop shipping logs. Instead, a communications relay has been 2 in the \\[destination\\] system, and it will be marked as 'D' on your space compass. The relay will only (D50=broadcast|transmit) its (D50=location|position) for a 3 (D50=amount|period) of time, so you must act quickly to reach it.5\\n\\nA security passcode, which you will 4 as part of the mission data package, must be transmitted to the relay, at which point it will communicate the type and amount of cargo, and where you need to deliver it to. You will then need to purchase the cargo from any station, the cost of which will be offset to you using the main station's prices at the destination system. When you reach the destination a waypoint will activate to enable you to find the pickup point where you must dump the cargo.\\n\\nAlong with the payment listed on this bulletin board entry, you also will be paid with precious metals or gems once the cargo is scooped.",
            "GalCop record all trade (D50=ships|vessels) arriving and departing from stations, along with all cargo manifests. Our client is keen to avoid this unwanted attention completely, so has set up a communications relay in the \\[destination\\] system which will appear, for a 3 (D50=amount|period) of time, as a 'D' on your space compass, so you need to act quickly to reach it.5\\n\\nAfter transmitting the required security code to the unit, you will be given the amount and type of cargo, and details of a waypoint in another system.\\n\\nAlong with the payment listed on this bulletin board entry, you also will be paid with precious metals or gems once the cargo is scooped."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType42 = function $missionType42() {
    var def = {
        fieldA: [].concat(this._gcmPG.accept),
        fieldB: [].concat(this._gcmPG.contract),
        fieldC: ["difficult", "hard", "challenging", "demanding", "tricky", "tough", "grim"],
        fieldD: [],
        fieldE: [],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We (D50=have|need) to (D50=deliver|transport) some specialised computer equipment to one of our (D50=ships|vessels) that is (D50=currently in the \\[destination\\] system|in the \\[destination\\] system at the moment). However, it is vital that the computers be (D50=delivered|transported) without there being any possibility of someone learning of the delivery, which is why this isn't being offered through the usual cargo contract channels. If you 1 this 2, you cannot dock at any GalCop station with the computers on board until they are delivered. Doing so will result in the computers being discovered and the 2 will be marked as failed.\\n\\nThe (D50=ship|vessel) will be waiting \\[position\\].",
            "We (D50=have|need) to (D50=deliver|transport) some highly specialised computer surveillance equipment to one of our (D50=ships|vessels), which is (D50=currently in the \\[destination\\] system|in the \\[destination\\] system at the moment). However, (D50=due to|because of) the nature of these computers and their ultimate purpose, we do not want there to be any chance that the delivery be discovered. Docking at any GalCop station while transporting these items would result in some very 3 questions being asked at a diplomatic level. If you 1 this 2 you will find our contact waiting \\[position\\].",
            "One of our (D50=ships|vessels) is (D50=currently in the \\[destination\\] system|in the \\[destination\\] system at the moment), \\[position\\], waiting for some specialised computer equipment. The mission is to (D50=deliver|transport) the cargo to the ship, but it is imperative that no one know about the transfer. If you dock at a GalCop station with these computers on board they will be discovered and the 2 will be terminated.",
            "We have a (D50=ship|vessel) (D50=currently in the \\[destination\\] system|in the \\[destination\\] system at the moment, \\[position\\]), awaiting the delivery of some specialised computer equipment. However, (D50=due to|because of) the nature of these computers, and their intended purpose, you cannot dock at any GalCop-aligned station while you have them on board. If you do, the computers will be discovered and confiscated, and the 2 will be terminated."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType43 = function $missionType43() {
    var def = {
        fieldA: [].concat(this._gcmPG.in_search_of),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: [].concat(this._gcmPG.we_believe),
        fieldD: [].concat(this._gcmPG.attention),
        fieldE: [].concat(this._gcmPG.attacked),
        fieldF: ["was [5]", "came under fire", "was fired upon", "has come under fire"],
        fieldG: ["imperative", "vital", "crucial", "essential"],
        fieldH: [].concat(this._gcmPG.battle),
        fieldI: [].concat(this._gcmPG.travel_to),
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "One of our traders, (D50=operating|working) in the \\[destination\\] system, has become stranded. Their main engines are offline, and they lack critical parts to get it (D50=operational|functional) again. To make matters worse, their life support systems are beginning to fail, and they aren't fitted with an escape pod. We are 1 a 2 to take the required equipment to the (D50=ship|vessel) and dump it nearby to enable them to scoop it and effect repairs. 3 the (D50=ship|vessel) is stranded \\[position\\].",
            "4 that one of our trade (D50=ships|vessels), (D50=operating|working) in the \\[destination\\] system, 6 and their main engine has been damaged. The crew lack certain critical parts to enable them to restart their (D50=ship|vessel), and with their life support systems beginning to fail, and no escape pod on board, it is 7 we (D50=deliver|transport) the required equipment to them. We are fairly sure the (D50=ship|vessel) is stranded \\[position\\].",
            "We have received word that one of our trade (D50=ships|vessels), (D50=operating|working) in the \\[destination\\] system, was involved in 8 that has left the ship with no power to engines, and limited power to the life support system. The 2 was also, in hindsight quite foolishly, operating without an escape pod. We understand that a critical part is needed so the 2 can repair their ship, so we are 1 a 2 who can 9 \\[destination\\], find the (D50=ship|vessel) \\[position\\], and (D50=deliver|transport) the required equipment.",
            "We have received an emergency communication from one of our trade (D50=ships|vessels). They were involved in 8 in \\[destination\\] that has rendered their main engine inoperable. With no way to repair the engine, and with limited life-support remaining, we (D50=need|require) a 2 to take some equipment, 9 \\[destination\\], locate the stranded ship \\[position\\], and dump the equipment nearby. This will allow the stranded pilot to repair their ship and continue their journey."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType44 = function $missionType44() {
    var def = {
        fieldA: [].concat(this._gcmPG.on_route_to),
        fieldB: [].concat(this._gcmPG.we_believe),
        fieldC: [].concat(this._gcmPG.in_search_of),
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: [].concat(this._gcmPG.attention),
        fieldF: [].concat(this._gcmPG.imperative),
        fieldG: [].concat(this._gcmPG.venture_into),
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "One of our traders, 1 the \\[destination\\] system, has become stranded in interstellar space. 2 their main engines are offline, and they lack critical parts to get it (D50=operational|functional) again. To make matters worse, their life support systems are beginning to fail, and they aren't fitted with an escape pod. We are 3 a 4 to take the required equipment to the (D50=ship|vessel) in interstellar space and dump it nearby to enable them to scoop it and effect repairs.",
            "5 that one of our trade (D50=ships|vessels), 1 the \\[destination\\] system, has become stranded in interstellar space when their main engine was damaged. The crew lack certain critical parts to enable them to restart their (D50=ship|vessel), and with their life support systems beginning to fail, and no escape pod on board, it is 6 we (D50=deliver|transport) the required equipment to them.",
            "We have received word that one of our trade (D50=ships|vessels), 1 the \\[destination\\] system, has become stranded in interstellar space with no power to engines, and limited power to the life support system. The 4 was also, in hindsight quite foolishly, operating without an escape pod. We understand that a critical part is needed so the 4 can repair their ship, so we are 3 a 4 who can 7 interstellar space, find the (D50=ship|vessel), and (D50=deliver|transport) the required equipment.",
            "We have received an emergency communication from one of our trade (D50=ships|vessels). A mis-jump landed them in interstellar space between here and \\[destination\\], and they came under fire from a Thargoid warship. Unfortunately, during the battle their main engine was rendered inoperable. With no way to repair the engine, and with limited life-support remaining, we (D50=need|require) a 4 to take some equipment, 7 interstellar space, locate the stranded ship, and dump the equipment nearby. This will allow the stranded pilot to repair their ship and continue their journey."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType46 = function $missionType46() {
    var def = {
        fieldA: [].concat(this._gcmPG.desires),
        fieldB: [].concat(this._gcmPG.set_up),
        fieldC: [].concat(this._gcmPG.short),
        fieldD: [].concat(this._gcmPG.be_given),
        fieldE: ["\\n\\nIn order to dump the cargo so it remains within 20km the waypoint, you will need to use a device like the Cargo Stopper, the Ejection Damper, or a Cargo Shepherd, to bring ejected cargo to a standstill."],
        fieldF: [].concat(this._gcmPG.travel_to),
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "Our client 1 a certain cargo (D50=delivered|transported) to their location. However, this particular client does not want this transaction, the type of cargo or their current (D50=location|position) to appear in the standard GalCop shipping logs. Instead, they have arranged for \\[target\\]t of \\[commodity\\] to be at a waypoint in the \\[destination\\] system. You need to 6 this (D50=location|position) and scoop all the cargo.\\n\\nYou will then need to 6 the \\[destinationA\\] system and deposit the cargo at another waypoint.5",
            "In order to avoid the attention of GalCop, our client wants a particular cargo delivered to a specific location. But the client does not want this transaction, the type of cargo or their current (D50=location|position) to appear in the standard GalCop shipping logs. Instead, a cargo drop of \\[target\\]t of \\[commodity\\] has been left at a waypoint in the \\[destination\\] system. You will need to collect this cargo, then 6 the \\[destinationA\\] system, where another waypoint has been set. Drop the cargo at this waypoint to complete the mission.5",
            "GalCop record all trade (D50=ships|vessels) arriving and departing from stations, along with all cargo manifests. Our client is keen to avoid this unwanted attention completely, so has set up a cargo drop of \\[target\\]t of \\[commodity\\] at a waypoint in the \\[destination\\] system.\\n\\nAfter collecting this cargo, you will need to 6 \\[destinationA\\], and deposit the cargo at another waypoint in that system.5"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType50 = function $missionType50() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.short),
        fieldB: [].concat(this._gcmPG.contract),
        fieldC: [].concat(this._gcmPG.be_given),
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: [bountyText],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "A recent outbreak of disease on \\[destination\\] has proved to be resistant to previous forms of antibiotic. The Galactic Centre for Disease Control have been calling for off-world medical supplies in an attempt to bring a halt to the spread of the disease. Delivering these medical supplies is time-critical - each shipment comes with a containment field that holds the antibiotics in stasis for a 1 (D50=amount|period) of time. If the shipment takes too long to arrive, the containment field will rupture, rendering the contents useless.\\n\\nUpon acceptance of this 2, you will 3 security codes that will (D50=need|have) to be transmitted to the destination station to allow your ship to dock.5",
            "In order to contain the spread of disease on \\[destination\\], the Galactic Centre for Disease Control have been calling on >4 to (D50=deliver|transport) off-world medical supplies, as the locally produced medicines are proving to be insufficient. Each delivery is time-critical, because the antibiotics are being held in stasis inside a containment field, which can only operate for a 1 time. If the containment field fails, all the medicine within is rendered useless.\\n\\nUpon acceptance of this 2, you will 3 security codes that will (D50=have|need) to be transmitted to the destination station to allow your ship to dock.5",
            "The Galactic Centre for Disease Control has issued a plea for >4 to transport antibiotics from high-tech systems to \\[destination\\] to help combat the recent outbreak of disease the planet is experiencing. Each delivery is time-critical, because the antibiotics are being held in stasis inside a containment field, which can only operate for a 1 time. If the containment field fails, all the medicine within is rendered useless.\\n\\nUpon acceptance of this 2, you will 3 security codes that will (D50=have|need) to be transmitted to the destination station to allow your ship to dock.5"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType51 = function $missionType51() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.in_search_of),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: [].concat(this._gcmPG.accept),
        fieldD: [].concat(this._gcmPG.contract),
        fieldE: [].concat(this._gcmPG.imperative),
        fieldF: ["5t"],
        fieldG: [bountyText],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "There are a number of patients who have fallen seriously ill due to the current disease outbreak, and their condition is critical. The local medical facilities are inadequate for treating patients in this condition, so we are 1 >2 who can take on the job of transferring these patients to \\[destination\\], which is the closest planet with the required techlevel, for emergency medical procedures that will hopefully save their lives.\\n\\nIf you 3 this 4, a special patient transport unit will be installed into your ship, which takes up 6 of cargo space. The unit will be removed automatically when you reach \\[destination\\].7",
            "As a result of the recent outbreak of disease planet-side, and a lack of suitable medical equipment and antibiotics, the condition of many patients has deteriorated to the point where local medical facilities are inadequate for treating their condition. It is 5 these patients are transferred to \\[destination\\], the nearest system with the required techlevel, for emergency medical treatment that will hopefully save their lives.\\n\\nIf you 3 this 4, a special patient transport unit will be installed into your ship, which takes up 6 of cargo space. The unit will be automatically removed when you reach \\[destination\\].7",
            "The ongoing fight against disease on the planet has impacted thousands of citizens. Some are in an extremely critical condition and local medical services are unable to provide sufficient care for these patients. We are 1 >2 who can take some of these critically-ill patients to waiting medical teams on \\[destination\\], the nearest system that has the required techlevel.\\n\\nIf you 3 this 4, a special patient transport unit will be installed into your ship, which takes up 6 of cargo space. The unit will be automatically removed when you reach \\[destination\\].7"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType52 = function $missionType52() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.plenty_of),
        fieldB: [].concat(this._gcmPG.due_to),
        fieldC: ["conduct$", "perform$", "carry$ out"],
        fieldD: [].concat(this._gcmPG.short),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.in_search_of),
        fieldG: [].concat(this._gcmPG.war),
        fieldH: [bountyText],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "The recent outbreak of disease has pushed local medical resources to breaking point, but in 1 cases they are inadequate 2 the mutating nature of the disease. Research into the various strains of the disease needs to be _3, and scientists on \\[destination\\] have agreed to 3 this service for us. We (D50=need|require) a batch of virus specimens (D50=delivered|transported) to the \\[destination\\] system so we can better understand and battle this outbreak.\\n\\nHowever, the containment unit around the specimens can only operate for a 4 time, and should that time expire the unit will automatically sterilise the contents to prevent any accidental contamination.8",
            "Local medical resources are limited, as is their ability to 3 sufficient testing on the different strains of the disease. However, scientists on \\[destination\\] have come forward and offered their services in this area, so we are 6 a 5 who can (D50=deliver|transport) a batch of virus specimens to this system for analysis.\\n\\nThe transportation device has a containment seal that prevents any accidental contamination, but there is a limited life in this device. If the transportation time exceeds the operational time limit, or the unit is damaged on route, the unit will automatically sterilise the contents.8",
            "The ongoing 7 against the outbreak of disease has a number of fronts, one of them being research. The more we understand the virus causing the disease, the better we are able to fight against it. Local medical facilities are swamped with the influx of patients, but scientists in the \\[destination\\] system have made their services available, if we can just get the specimens to them intact. We are 6 a 5 to (D50=deliver|transport) these virus specimens in a specialised containment device.\\n\\nBecause of the virulence of the disease, we can take no chances that the virus escapes. The containment unit has its own power supply, and should the power fail, or the unit is damaged, the contents will be automatically sterilised.8"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType60 = function $missionType60() {
    var def = {
        fieldA: ["some time", "quite some time", "a while", "quite a while"],
        fieldB: [].concat(this._gcmPG.in_search_of),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.contract),
        fieldE: [].concat(this._gcmPG.be_given),
        fieldF: [].concat(this._gcmPG.constant),
        fieldG: [].concat(this._gcmPG.pirates),
        fieldH: ["recently", "in recent months", "of late", "lately"],
        fieldI: [].concat(this._gcmPG.we_believe),
        fieldJ: ["(WBSA)"],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "For 1 we have been concerned that GalCop are not doing enough to stem the tide of pirate (D50=ships|vessels) entering our system. They insist that their analysis of transit data shows that we are no different from any other system. However, GalCop will not release the transit data to us so we can perform our own analysis.\\n\\nTherefore, we are 2 a 3 who can take a customised Witchpoint Beacon Security Access {1 device and upload a small data package to the witchpoint marker at \\[destination\\]. This package will send us a regular data stream of transit data in and out of that system, so we can better communicate with GalCop.\\n\\nPlease be aware that using the WBSA device within range of any police (D50=ships|vessels) will get you marked as a criminal and will render this contract void, with associated penalties.\\n\\nWhen the }4 is accepted you will 5 the equipment, which must be primed and then activated while the witchpoint marker is targeted.",
            "We have been under siege by a 6 flow of 7 8, although GalCop insists this is a chart-wide issue, not (D50=limited|restricted) to our system. But without the transit data from our neighbouring systems we are not able to make GalCop aware of our difficulties.\\n\\nSo, we are 2 a 3 who can take a customised Witchpoint Beacon Security Access {1 device and upload a small software package to the witchpoint marker at \\[destination\\]. This will send us a regular data stream of transit data in and out of that system, so we can be better armed in negotiations with GalCop.\\n\\nPlease be aware that using the WBSA device within range of any police (D50=ships|vessels) will mean you will be marked as a criminal, and will also render this }4 void, with the associated penalties.\\n\\nWhen the 4 is accepted you will 5 the equipment, which must be primed and then activated while the witchpoint marker is targeted.",
            "We are in a political battle with GalCop, attempting to get accurate real-time data on the identity of all ships entering our system from \\[destination\\]. 9 some undesirable groups from that system have been using our system as a staging point, but because GalCop only provide the transit data on a semi-regular basis, it's impossible to do anything about it.\\n\\nHowever, we have come into possession of a customised Witchpoint Beacon Security Access {1 device, which gives us the opportunity to bypass GalCop and access the data stream directly. We are }2 a 3 who can take the device to \\[destination\\], get within range of the witchpoint beacon, and upload a small patch to the software.\\n\\nBe aware that using the WBSA device within range of any police (D50=ships|vessels) will result in you being marked as a criminal, and will also render this 4 void.\\n\\nUpon acceptance of this mission, you will 5 the WBSA device, which must be primed and then activated while the witchpoint market is targeted."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType61 = function $missionType61() {
    var def = {
        fieldA: [].concat(this._gcmPG.on_route_to),
        fieldB: [].concat(this._gcmPG.in_search_of),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.be_given),
        fieldE: [].concat(this._gcmPG.contract),
        fieldF: [].concat(this._gcmPG.happening),
        fieldG: [].concat(this._gcmPG.we_believe),
        fieldH: ["(WBSA)"],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "It has come to the attention of GalCop security personnel that the witchpoint marker in \\[destination\\] has been tampered with, the end result being that a high number of (D50=ships|vessels) have gone missing 1 this system, or report of experiencing a mis-jump when attempting to reach the system.\\n\\nTherefore, we are 2 a 3 who can take a customised Witchpoint Beacon Security Access 8 device to the \\[destination\\] system and upload a software patch that will restore the beacon into its original state.\\n\\nWhen the 5 is accepted you will 4 the equipment, which must be primed and then activated while the witchpoint marker is targeted.",
            "GalCop security personnel have been alerted to a software fault on the witchpoint marker in \\[destination\\] that is resulting in a large number of (D50=ships|vessels) disappearing 1 the system, or arriving with reports of mis-jumps. The cause of this software fault is unknown.\\n\\nTherefore, we are 2 a 3 who can take a customised Witchpoint Beacon Security Access 8 device to \\[destination\\] and upload a software patch that will restore the beacon into its original state.\\n\\nWhen the 5 is accepted you will 4 the equipment, which must be primed and then activated while the witchpoint marker is targeted.",
            "GalCop closely monitors system traffic, 2 trends and hot spots. It has become clear that there are a high number of mis-jumps 6 for (D50=ships|vessels) attempting to enter the \\[destination\\] system. 7 some unscrupulous individuals have managed to hack the witchpoint beacon in \\[destination\\], so that it occasionally gives off an invalid transponder signal and sends ships into interstellar space.\\n\\nTherefore, we are 2 a 3 to take a Witchpoint Beacon Security Access 8 device to the \\[destination\\] system, target and get in range of the beacon, then prime and activate the WBSA device. This will remove any invalid code from its memory core."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType62 = function $missionType62() {
    var def = {
        fieldA: [].concat(this._gcmPG.destroying),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: [].concat(this._gcmPG.contract),
        fieldD: ["(that is, you are attacked by any pirate ships)"],
        fieldE: ["creating [6]", "proving to be a [7] nuisance", "wreaking havoc", "causing [6]"],
        fieldF: ["havoc", "chaos", "mayhem", "bedlam", "troubles", "difficulties", "headaches"],
        fieldG: ["considerable", "serious", "major", "significant"],
        fieldH: ["50kms"],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "GalCop has become concerned about the increase in pirate activity coming from \\[destination\\]. Of particular interest is a pirate Rock Hermit located in that system, as it appears to have become the hub for an entire pirate network operating in that area. However, rather than simply 1 the Hermit base, we want to monitor all traffic coming in and out of the Rock Hermit to gain a better understanding of the way pirates move around various systems.\\n\\nSo we need a security software package installed at the Rock Hermit. You will (D50=have|need) to approach the Hermit without being detected by any pirate vessels. If you are detected within 8 of the station 4 the 3 will not be able to be completed. Once docked, you must install a security package into the Rock Hermit's computer systems.",
            "A notorious gang of pirates from \\[destination\\] have been 5 in our spacelanes recently. Our Viper squadrons do their best to maintain safety for all >2, but there is only so much they can do in our system. So we have decided to take the war to the pirates. Of particular interest is a pirate Rock Hermit in \\[destination\\]. Evidence suggests this gang has either set up base there, or is using it as a hub for their operations. In any case, we need more information about this Hermitage and the ships that dock there.\\n\\nWe have a security package that we would like installed at the Rock Hermit. You'll (D50=have|need) to approach the Hermitage without being detected and attacked by any pirate vessels within 8 range of it. If you are then the mission is scrubbed. Once docked, you (D50=have|need) to install the security package onto the computer systems."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType63 = function $missionType63() {
    var def = {
        fieldA: ["difficult", "hard", "challenging", "demanding", "arduous", "tricky", "tough", "grim"],
        fieldB: [].concat(this._gcmPG.in_search_of),
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: [].concat(this._gcmPG.travel_there),
        fieldE: ["1"],
        fieldF: ["30"],
        fieldG: ["48"],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "GalCop and elements within the system government have been making life 1 for those of us on the fringes of their economy. We want to change that. We have gained access to a software package that, when installed at the main station, will make all their commodity prices 5cr. We are 2 a 3 who can fly to the main station in this system and install the software package. There's two parts to this: first, you must initiate a data stream to the main station, but only when you have docking clearance. Once the data stream completed, you can dock and install the software. Don't dock before it completes. There are risks, however. While the data stream is being sent, you have to stay well clear of any police. You'll only have about 6 seconds after the data stream completes to install the software package. And you'll (D50=have|need) to launch within a few of seconds of installing the software and not return to the main station in less than 7 hours, otherwise their security detection systems will be able to track the security breach to your (D50=ship|vessel).",
            "Life in this system has become increasingly 1 for those of us on the fringes of society. GalCop have been colluding with the local government, cracking down on anything that doesn't add to their bottom lines. We want to send them a message that says we won't stand for this. We've recently come into possession of a piece of software that, when installed at the main station, will force all commodities to cost just 5cr. But we (D50=need|require) a 3 to 4 and install it. There are two steps: first, you have to transmit a data stream to the main station, but only while you have docking clearance. Don't let any police (D50=ships|vessels) get close while transmitting. When the data stream is completed, and only then, you can dock and install the software package. You'll have about 6 seconds after the data stream completes to do this. Once installed you'll (D50=have|need) to clear out of the place fast and not return for at least 7 hours, otherwise GalCop will be able to trace the security breach back to your (D50=ship|vessel)."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType64 = function $missionType64() {
    var def = {
        fieldA: [].concat(this._gcmPG.pilot),
        fieldB: ["48"],
        fieldC: ["30"],
        fieldD: [],
        fieldE: [],
        fieldF: [],
        fieldG: [],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "GalCop has sided with the system government and is trying to limit our ability to maintain our own ships at the main station. We're after some payback, and as luck would have it, we have acquired a software package that can be installed at the main station which will disrupt all the ship maintenance computers and make it nigh on impossible for anyone to get anything done to their ships. We (D50=need|require) a 1 to dock at the main station and install the package. There are two parts to this: first, you have to transmit a special data stream to the main station while you have docking clearance. Keep clear of any police (D50=ships|vessels) while you're transmitting. After this, you (D50=have|need) to dock within 3 seconds and install the software package. You'll (D50=have|need) to stay sharp, though - you'll only have a few seconds from installing the package before GalCop discovers the breach and slaps you with a large fine. And best not dock there for at least 2 hours, for the same reason.",
            "It has become clear that GalCop and the local planetary government are in collusion, with the aim of restricting traffic not headed for their stations, but we're not taking this lying down. We have 'acquired' a software package that can be installed at the main station which will disrupt all ship maintenance computers and prevent pretty much every type of outfit, from simple refuelling operations to equipment installs and overhauls. All you (D50=have|need) to do is transmit a special data stream to the main station while you have docking clearance, and then dock at the main station inside 3 seconds and install the package. Simple. But don't let police ships get close while transmitting that data stream. And be ready to launch after installing the software package - you'll only have a few seconds to get clear of the station before GalCop discover the breach and fine you, big time. You should probably stay away from the main station for a good 2 hours afterwards, as well."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType65 = function $missionType65() {
    var def = {
        fieldA: ["officials", "representatives"],
        fieldB: ["difficult", "hard", "challenging", "demanding", "arduous", "tricky", "tough", "grim"],
        fieldC: [].concat(this._gcmPG.pilot),
        fieldD: ["stringent", "strict", "rigorous"],
        fieldE: [].concat(this._gcmPG.in_search_of),
        fieldF: ["50,000"],
        fieldG: ["7"],
        fieldH: ["canister", "container", "pod"],
        fieldI: [].concat(this._gcmPG.collect),
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "GalCop and local government 1 have been making life 2 for all of us outside their sanitised worldview. But we think we've found a way to make life 2 for them. We are 5 a 3 who can (D50=assist|help) us acquire some highly restricted security software from under their noses. The first step is to 9 a 'customised' escape pod from any non-GalCop station in \\[destination\\], bring it back to this system and activate it at more than 6 kms from the main station. We've learnt that security protocols aren't as 4 when dealing with escape pods, so the customised version can inject some specific commands into the computer system at the main station. After being rescued, you can launch, but the first police (D50=ship|vessel) that launches after you will self-destruct sometime in the following 7 minutes. When it does, a security software package will be ejected in a cargo 8. Scoop that cargo, and bring it back here. Easy right?",
            "One of our technical boffins has uncovered a rather unorthodox method of obtaining some highly restricted security software from GalCop. The method involves collecting a 'customised' escape capsule from any non-GalCop station in \\[destination\\], bringing it back here and activating it at more than 6 kms from the main station. Apparently security protocols aren't as 4 with escape pods, which enables this customised version to inject some specific commands into the computers back at the main station. When you launch after being rescued, the next police (D50=ship|vessel) to launch will have catastrophic failure in all systems and self-destruct, sometime in the following 7 minutes. When it does, a security software package will be ejected in a cargo 8. Scoop the cargo, and return it here. This will (D50=assist|help) us level the playing field with GalCop and the local government, who have been making life 2 for us out here on the fringes of society."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType70 = function $missionType70() {
    var def = {
        fieldA: [].concat(this._gcmPG.imperative),
        fieldB: [].concat(this._gcmPG.constant),
        fieldC: [].concat(this._gcmPG.many_different),
        fieldD: [].concat(this._gcmPG.repercussions),
        fieldE: [].concat(this._gcmPG.harmful),
        fieldF: [].concat(this._gcmPG.happen),
        fieldG: [].concat(this._gcmPG.in_search_of),
        fieldH: [].concat(this._gcmPG.pilot),
        fieldI: [].concat(this._gcmPG.approximately),
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "It is 1 that we maintain 2 watch on the solar activity of our sun, with 3 of scanners, in order to limit the 4 planet-side when 5 levels of activity 6.\\n\\nWith this in mind, we are 7 >8 with a suitably equipped ship who can take some specialised scanning equipment to within fuel-scooping range of the sun and perform a scan. Once installed, the scanner is started by priming it, and then activating it with your ships standard interface. The scan will take 9 one minute to complete, and will drain a significant amount of energy while operating.\\n\\nThe scanner is also quite fragile, so you will (D50=have|need) to be careful with it.",
            "With the planet-side 4 of solar activity a 2 concern for us, it is 1 that we maintain watch on our sun, with 3 of scanners, watching for when 5 levels of activity 6.\\n\\nWith this in mind, we have some specialised scanning equipment that needs to be activated inside fuel-scoop range of our sun. We're 7 >8 with a suitably equipped ship who can do this for us. Once installed, the scanner is started by priming it, and then activating it with your ships standard interface. It will take 9 one minute to complete, and will drain a significant amount of energy while operating.\\n\\nThe scanner is also quite fragile, so you will (D50=have|need) to be careful with it."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType71 = function $missionType71() {
    var def = {
        fieldA: [].concat(this._gcmPG.imperative),
        fieldB: [].concat(this._gcmPG.constant),
        fieldC: [].concat(this._gcmPG.many_different),
        fieldD: [].concat(this._gcmPG.harmful),
        fieldE: [].concat(this._gcmPG.happen),
        fieldF: [].concat(this._gcmPG.in_search_of),
        fieldG: [].concat(this._gcmPG.pilot),
        fieldH: [].concat(this._gcmPG.repercussions),
        fieldI: [].concat(this._gcmPG.collect),
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "It is 1 that we 2ly monitor the solar activity of our sun, with 3 of scanners, so that we can more accurately predict when a 4 event is about to 5.\\n\\nOne such scanning system involves two solar monitors in positions near the sun, one on the near side and one of the far side. Data from these monitors must be collected from within close proximity as there is too much interference from the sun for reliable long-distance data transfer. Our regular data gatherer (D50=ships|vessels) are grounded for maintenance and we are overdue on our data collection schedule. We are 6 a 7 to visit each of the solar monitors (marked with 'M' on your space compass), transmit a security code and 9 the data.",
            "With the planet-side 8 of solar activity a 2 concern for us, it is 1 that we maintain watch on our sun, with 3 of scanners, watching for when 4 levels of activity 5.\\n\\nOnce such scanning system involves two solar monitors in positions near the sub, one on the near side and one on the far side. Because of the solar interference we must manually 9 data from these monitors by flying a ship to within close range of them. But our regular data gatherer (D50=ships|vessels) have been grounded for maintenance, and we are overdue on our data collection schedule. We (D50=need|require) a 7 to visit both of the solar monitors (marked with 'M' on your space compare), transmit a security code and 9 the data."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType73 = function $missionType73() {
    var def = {
        fieldA: [].concat(this._gcmPG.imperative),
        fieldB: [].concat(this._gcmPG.constant),
        fieldC: [].concat(this._gcmPG.many_different),
        fieldD: [].concat(this._gcmPG.repercussions),
        fieldE: [].concat(this._gcmPG.harmful),
        fieldF: [].concat(this._gcmPG.happen),
        fieldG: [].concat(this._gcmPG.in_search_of),
        fieldH: [].concat(this._gcmPG.pilot),
        fieldI: [].concat(this._gcmPG.approximately),
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "It is 1 that we maintain 2 watch on the solar activity of our sun, with 3 of scanners, in order to limit the 4 planet-side when 5 levels of activity 6.\\n\\nWith this in mind, we are 7 >8 with a suitably equipped ship who can pick up some new, specialised scanning equipment from \\[destination\\], bring is back and take it to within fuel-scooping range of the sun and perform a scan. Once installed, the scanner is started by priming it, and then activating it with your ships standard interface. The scan will take 9 one minute to complete, and will drain a significant amount of energy while operating.\\n\\nThe scanner is also quite fragile, so you will (D50=have|need) to be careful with it.",
            "With the planet-side 4 of solar activity a 2 concern for us, it is 1 that we maintain watch on our sun, with 3 of scanners, watching for when 5 levels of activity 6.\\n\\nWith this in mind, we have ordered some new, specialised scanning equipment that needs to be collected from \\[destination\\], brought back here and activated inside fuel-scoop range of our sun. Once installed, the scanner is started by priming it, and then starting it with your ships standard interface. It will take 9 one minute to complete, and will drain a significant amount of energy while operating.\\n\\nThe scanner is also quite fragile, so you will (D50=have|need) to be careful with it."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType74 = function $missionType74() {
    var def = {
        fieldA: [],
        fieldB: [].concat(this._gcmPG.repercussions),
        fieldC: ["planet-side", "down on the planet", "on our citizens", "on our people"],
        fieldD: ["[9] be done elsewhere", "has been outsourced to other worlds", "[9] be sent off-world"],
        fieldE: ["is beyond our capabilities to process and ", "is just too complicated to be processed by our limited facilities and ", "can't be achieved with the technology we have available locally and ",],
        fieldF: ["analysis", "data analysis", "processing", "data processing"],
        fieldG: ["ultra-hi-tech", "big", "large-scale"],
        fieldH: [", due to the amount of information, ", ", due to the sheer scale of the dataset, ", ", because of the size of the dataset, ", " "],
        fieldI: ["needs to", "must", "has to"],
        fieldJ: ["We need a set of solar scans delivered", "We have a set of solar scans that needs to be delivered", "We require a set of solar scans be delivered", "We've got a set of solar scans that needs to be delivered"],
        fieldK: ["so this analysis can be performed", "in order to get this deep analysis performed", "to allow this data processing to take place", "to enable the processing to be completed", "so this processing can be carried out"],
        fieldL: ["We [7] [5] [6] our sun", "As part of our public safety (D50=initiatives|programmes), we [7] [5] [6] our sun", "With a sun like ours, it is [4] we [7] [5] [6] it", "By nature of the volatility of our sun, we [7] [5] [6] it"],
        fieldM: [].concat(this._gcmPG.imperative),
        fieldN: ["a lot", "masses", "huge amounts", "a substantial amount", "a considerable amount", "quite a bit"],
        fieldO: ["of data about", "of information about"],
        fieldP: ["collect", "gather", "accumulate"],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "{3, trying to minimise the }2 of solar disturbances 3. (D50=But|However)8some of the 6 54, making use of 7 data processing facilities available at hi-tech worlds. {1 to \\[destination\\] 2."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType80 = function $missionType80() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [""],
        fieldB: [].concat(this._gcmPG.calling_for),
        fieldC: [].concat(this._gcmPG.war),
        fieldD: [].concat(this._gcmPG.donation),
        fieldE: ["repatriation programs", "programs of returning victims to their home system", "rehabilitation programs", "repatriation and rehabilitation programs"],
        fieldF: [].concat(this._gcmPG.are_expensive),
        fieldG: [],
        fieldH: [bountyText],
        fieldI: ["that don't want us around", "that aren't on board with us", "that aren't on the same page", "that don't see eye to eye with us", "that stand to lose out if the slave trade is stopped", "that turn a blind eye to slave trading"],
        fieldJ: ["need", "want", "desire", "require"],
        fieldK: [].concat(this._gcmPG.assistance),
        fieldL: ["long", "hard", "difficult", "never-ending", "long running"],
        fieldM: ["win", "succeed", "triumph"],
        fieldN: ["aspect", "factor", "component"],
        fieldO: ["money", "cash", "credits"],
        fieldP: ["to run", "to operate", "to maintain", "to manage", "to organise", "to carry out"],
        fieldQ: ["extensive", "wide-ranging", "complex", "intricate"],
        fieldR: ["to continue their work", "to fund the development of additional programs", "to help resettle even more victims", "to pay for further expansions of these programs", "to pay for the legal and transfer costs of victims"],
        sentences: [
            "Amnesty Intergalactic is 2 {2 in the on-going }3 against the slave trade. Their 5 6 {7, (D50=especially|particularly) in (D50=systems|places) }9. So they {1 your 2. They are }2 a 4 of \\[target\\] cr {9.}8",
            "The 3 against the slave trade is {3, but Amnesty Intergalactic is determined to {4. One 5 of this }3 is financial. Amnesty Intergalactic run {8 }5, and these 6 {7, (D50=especially|particularly) in (D50=systems|places) }9. So they are 2 a 4 of \\[target\\] cr {9.}8",
            "Amnesty Intergalactic, in its 3 against the slave trade, run a number of 5, but they 6 {7, (D50=especially|particularly) in (D50=systems|places) }9. Amnesty Intergalactic are 2 {2, by the }4 of {6, \\[target\\] cr specifically, {9.}8"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType81 = function $missionType81() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.care_for),
        fieldB: ["don't come home", "are missing in action", "have been killed somewhere in the vastness of space", "have fallen victim to pirates or hunters", "fail to come home"],
        fieldC: ["monetary", "cash", "financial"],
        fieldD: ["pirates and interstellar disasters", "tragedy and heartbreak", "devastating mishaps"],
        fieldE: ["coming in daily", "arriving almost minute by minute", "announced daily"],
        fieldF: [].concat(this._gcmPG.need_for),
        fieldG: [bountyText],
        fieldH: [].concat(this._gcmPG.grown_exponentially),
        fieldI: [].concat(this._gcmPG.are_expensive),
        fieldJ: [].concat(this._gcmPG.calling_for),
        fieldK: [].concat(this._gcmPG.assistance),
        fieldL: ["news", "details", "information", "reports"],
        fieldM: [].concat(this._gcmPG.donation),
        fieldN: ["go a long way towards", "help us enormously in", "make a big difference in", "help us in"],
        fieldO: ["keeping our services running", "meeting our financial commitments", "meeting the needs of our clients", "keeping our services cost effective", "keeping us in the black", "keeping us solvent", "paying our creditors"],
        fieldP: [].concat(this._gcmPG.operating_mantra),
        fieldQ: [],
        fieldR: ["work the spacelanes", "make their living among the stars", "must fly there regularly"],
        sentences: [
            "The ~gcmTWOSTitle$G~ is an organisation whose {7 is to }1 those left behind when traders 2. With {3 of }4 5, the 6 our services(D50= and facilities|) 8. But those services 9, so we are {1 2 in the form of }3 >{4. \\[target\\] cr will 5 6.}7",
            "Space is a sometimes dangerous place, (D50=especially|particularly) for those who {9. And sometimes traders }2. {3 of }4 are 5. The ~gcmTWOSTitle$G~ exists solely to <1 those left behind, and the 6 our services(D50= and facilities|) 8. But those services 9, so we are {1 2. A }3 {4 of \\[target\\] cr will 5 6.}7",
            "{3 of }4 are 5, which usually means pilots 2. When }4 strike, the ~gcmTWOSTitle$G~ is there to 1 those left behind. As you can imagine, the 6 our services(D50= and facilities|) 8, and these services 9. We are {1 2. A 4 of \\[target\\] cr will 5 6.}7"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType82 = function $missionType82() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: ["aim", "seek", "attempt"],
        fieldB: [].concat(this._gcmPG.care_for),
        fieldC: ["those in the repair and services sector", "all mechanics and technicians", "those working in the repair workshops of stations across the Eight"],
        fieldD: ["get a solid education", "receive educational opportunities", "have educational experiences", "get quality educational opportunities"],
        fieldE: ["they otherwise wouldn't receive", "they might not be able to receive normally", "beyond what they would normally receive"],
        fieldF: [].concat(this._gcmPG.need_for),
        fieldG: [bountyText],
        fieldH: [].concat(this._gcmPG.grown_exponentially),
        fieldI: ["expense", "cost", "price tag"],
        fieldJ: [].concat(this._gcmPG.struggling),
        fieldK: ["to make ends meet", "to pay all our expenses", "to cover all our costs", "to meet the requirements of our clients", "to provide quality services", "to maintain the quality of our services"],
        fieldL: [].concat(this._gcmPG.calling_for),
        fieldM: [].concat(this._gcmPG.assistance),
        fieldN: [].concat(this._gcmPG.donation),
        fieldO: ["go a long way towards", "help enormously in", "make a big difference in", "help us in"],
        fieldP: ["keeping our services running", "meeting our financial commitments", "meeting the needs of our clients", "keeping our services cost effective", "keeping us in the black", "keeping us solvent", "paying our creditors"],
        fieldQ: ["into the future", "going forward", "in the coming days", "in the short term", "for the time being"],
        fieldR: ["cash", "credits", "money", "a financial sort"],
        sentences: [
            "The ~gcmMCEPTitle$G~ >1 to 2 the children of 3, enabling them to 4 5. The 6 our services(D50= and facilities|) 8, but so too has the 9. We are {1 2. To this end, we are 3 4 of a financial sort, a 5 of \\[target\\] cr, which will 6 7 8.}7",
            "The children of 3 can sometimes find themselves {1 to }4. The ~gcmMCEPTitle$G~ exists to give these children something 5. With the 6 our services(D50= and facilities|) growing, so too has the 9, leaving us {1 2. With this in mind, we are 3 >5 of 9, \\[target\\] cr, which will 6 7 8.}7",
            "The ~gcmMCEPTitle$G~ is all about the kids. We 1 to 2 the children of 3 to aim higher and enable them to 4 5. As 6 our services(D50= and facilities|) 8, so too has the 9, to the point where we are {1 2. We are now 3 a financial type of 4, a 5 of \\[target\\] cr, which will 6 7 8.}7"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType83 = function $missionType83() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.operating_mantra),
        fieldB: ["provide quality mental health services for", "look after the mental health needs of", "provide compassionate, effective mental health services to"],
        fieldC: ["of every make and model", "of every kind, from traders to bounty hunters", "of every sort, from the mostly harmless to Elite"],
        fieldD: ["regardless of their history with GalCop", "regardless of their current legal standing", "regardless of their criminal associations, past or present"],
        fieldE: ["costs are rising", "costs are climbing", "our operating costs are climbing", "the cost of operating our services is escalating"],
        fieldF: [bountyText],
        fieldG: ["law and order breaks down", "GalCop struggles to maintain safety on the spacelanes", "the dangers from pirate raids increase", "the threat of Thargoid invasion looms large"],
        fieldH: ["cost effective", "cheap", "affordable", "attainable", "maintainable"],
        fieldI: ["the full spread of available mental illness treatments", "the latest advances in mental health treatment"],
        fieldJ: ["hard", "difficult", "almost impossible", "more and more difficult"],
        fieldK: [].concat(this._gcmPG.calling_for),
        fieldL: [].concat(this._gcmPG.pilot),
        fieldM: ["support us", "partner with us", "work with us"],
        fieldN: [].concat(this._gcmPG.donation),
        fieldO: [].concat(this._gcmPG.enable_us),
        fieldP: ["expand and develop our services and facilities", "provide the best medical treatments and solutions", "ensure we can continue to look after the needs of our patients"],
        fieldQ: ["help all of our patients", "assist all of our patients", "enable all our patients to"],
        fieldR: ["have a positive, hopeful future", "feel confident about their future", "look forward to their future with hope", "face their future without fear"],
        sentences: [
            "The 1 of the ~gcmPMHSTitle$G~ is to 2 pilots 3. 5 though, (D50=especially|particularly) as 7. Providing commanders with 8 treatments, or giving them access to 9 is proving to be {1. We are 2 >3 who can 4 in the form of a 5 of \\[target\\] cr, that will 6 to 7 and 8 9.}6",
            "Contributing to ~gcmPMHSTitle$G~ isn't like making >{5 to other charities. The }1 of the fund is to 2 pilots 3, in a 8 way. We provide this service to pilots 4. But 5, (D50=especially|particularly) as 7. It is becoming {1 to give our patients access to }9. By making a {5 of \\[target\\] cr, you will 6 to 7 and 8 9.}6"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType84 = function $missionType84() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: ["wants to", "is seeking to", "is trying to", "is determined to", "is actively trying to", "is involved in a long-term mission, to"],
        fieldB: ["free", "release", "set free", "repatriate and rehabilitate"],
        fieldC: ["the road is long", "there is a long way to go", "it's going to take a long time"],
        fieldD: ["there are limited resources", "there are many hurdles to jump", "there are many obstacles to overcome", "there are many who are actively trying to stop us", "there are many complications and pitfalls"],
        fieldE: ["on this station", "on this very station", "right here", "right now"],
        fieldF: ["lives being shattered", "lives being treated as little more than cargo", "lives being torn apart", "lives being destroyed"],
        fieldG: [].concat(this._gcmPG.vile),
        fieldH: [].concat(this._gcmPG.calling_for),
        fieldI: [].concat(this._gcmPG.assistance),
        fieldJ: [].concat(this._gcmPG.war),
        fieldK: ["currently on the commodities market", "available right now on the commodities market"],
        fieldL: ["bringing them here", "bringing them to us", "giving them to us"],
        fieldM: ["for rehabilitation and repatriation", "so we can help them recover and return them to their homeland", "so we can start the process of rebuilding their lives"],
        fieldN: ["purchase", "buy", "acquire"],
        fieldO: ["there is hope", "there is something you can do", "there are ways you can help", "there is a way for you to make a difference", "there is still hope"],
        fieldP: ["sentient beings", "people", "unfortunate individuals"],
        fieldQ: ["respond", "reply", "participate", "help them out", "help them make a difference"],
        fieldR: [bountyText],
        sentences: [
            "Amnesty Intergalactic 1 2 every slave and end the slave-trade itself. But 3, and 4. Even 5 there are 6 because of this 7 trade. We are 8 your 9 in this on-going {1 by purchasing the \\[target\\] t of slaves 2, and 3 4.{9",
            "{2, }5, there are 6, as the 7 slave trade is allowed to continue. Amnesty Intergalactic 1 2 every single slave, although 4. 3 in this {1, but with your }9, we can make a difference. We are 8 for you to {5 all slaves available on this station, all \\[target\\] t, then 3 4.{9",
            "Seeking to 2 every slave should be everyone's concern, and is of prime concern to Amnesty Intergalactic. 4, so 3. Even now, 5, there are 6 as {7 are forced to participate in the }7 slave trade. But {6, if you can 5 the \\[target\\] t of slaves 2. Then 3 4. Amnesty Intergalactic are }8 your help. Will you {8?9"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType85 = function $missionType85() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.care_for),
        fieldB: ["don't come home", "are missing in action", "have been killed somewhere in the vastness of space", "have fallen victim to pirates or hunters", "fail to come home"],
        fieldC: [],
        fieldD: ["pirates and interstellar disasters", "tragedy and heartbreak", "devastating mishaps"],
        fieldE: ["coming in daily", "arriving almost minute by minute", "announced daily"],
        fieldF: [].concat(this._gcmPG.need_for),
        fieldG: [bountyText],
        fieldH: [].concat(this._gcmPG.grown_exponentially),
        fieldI: [].concat(this._gcmPG.are_expensive),
        fieldJ: [].concat(this._gcmPG.calling_for),
        fieldK: [].concat(this._gcmPG.assistance),
        fieldL: ["news", "details", "information", "reports"],
        fieldM: [].concat(this._gcmPG.donation),
        fieldN: ["go a long way towards", "help enormously in", "make a big difference in", "help us in"],
        fieldO: ["keeping our services running", "meeting our financial commitments", "meeting the needs of our clients", "keeping our services cost effective", "keeping us in the black", "keeping us solvent", "paying our creditors"],
        fieldP: [].concat(this._gcmPG.operating_mantra),
        fieldQ: ["cargo is just as useful as credits", "cargo is just as valid a [4] as credits", "cargo is a valid [4]"],
        fieldR: ["work the spacelanes", "make their living among the stars", "must fly there regularly"],
        sentences: [
            "The ~gcmTWOSTitle$G~ is an organisation whose {7 is to }1 those left behind when traders 2. With {3 of }4 5, the 6 our services(D50= and facilities|) 8. But those services 9, so we are {1 2 in the form of cargo >4. \\[target\\] \\[unit\\] of \\[commodity\\] will 5 6.}7",
            "Space is a sometimes dangerous place, (D50=especially|particularly) for those who {9. And sometimes traders }2. {3 of }4 are 5. The ~gcmTWOSTitle$G~ exists solely to <1 those left behind, and the 6 our services(D50= and facilities|) 8. But those services 9, so we are {1 2. 8, so a {4 of \\[target\\] \\[unit\\] of \\[commodity\\] will 5 6.}7",
            "{3 of }4 are 5, which usually means pilots 2. When }4 strike, the ~gcmTWOSTitle$G~ is there to 1 those left behind. As you can imagine, the 6 our services(D50= and facilities|) 8, and these services 9. We are {1 2. 8 to us, so a 4 of \\[target\\] \\[unit\\] of \\[commodity\\] will 5 6.}7"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType86 = function $missionType86() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: ["aim", "seek", "attempt"],
        fieldB: [].concat(this._gcmPG.care_for),
        fieldC: ["those in the repair and services sector", "all mechanics and technicians", "those working in the repair workshops of stations across the Eight"],
        fieldD: ["get a solid education", "receive educational opportunities", "have educational experiences", "get quality educational opportunities"],
        fieldE: ["they otherwise wouldn't receive", "they might not be able to receive normally", "beyond what they would normally receive"],
        fieldF: [].concat(this._gcmPG.need_for),
        fieldG: [bountyText],
        fieldH: [].concat(this._gcmPG.grown_exponentially),
        fieldI: ["expense", "cost", "price tag"],
        fieldJ: [].concat(this._gcmPG.struggling),
        fieldK: ["to make ends meet", "to pay all our expenses", "to cover all our costs", "to meet the requirements of our clients", "to meet the expectations of our clients", "to provide quality services", "to maintain the quality of our services"],
        fieldL: [].concat(this._gcmPG.calling_for),
        fieldM: [].concat(this._gcmPG.assistance),
        fieldN: [].concat(this._gcmPG.donation),
        fieldO: ["go a long way towards", "help enormously in", "make a big difference in", "help us in"],
        fieldP: ["keeping our services running", "meeting our financial commitments", "meeting the needs of our clients", "keeping our services cost effective", "keeping us in the black", "keeping us solvent", "paying our creditors"],
        fieldQ: ["into the future", "going forward", "in the coming days", "in the short term", "for the time being"],
        fieldR: ["cargo is just as useful as credits", "cargo is just as valid a [5] as credits", "cargo is a valid [5]"],
        sentences: [
            "The ~gcmMCEPTitle$G~ >1 to 2 the children of 3, enabling them to 4 5. The 6 our services(D50= and facilities|) 8, but so too has the 9. We are {1 2. To this end, we are 3 4 in the form of a 5. 9, so a 5 of \\[target\\] \\[unit\\] of \\[commodity\\] will 6 7 8.}7",
            "The children of 3 can sometimes find themselves {1 to }4. The ~gcmMCEPTitle$G~ exists to give these children something 5. With the 6 services(D50= and facilities|) growing, so too has the 9, leaving us {1 2. With this in mind, we are 3 >5 of cargo, \\[target\\] \\[unit\\] of \\[commodity\\] specifically, which will 6 7 8.}7",
            "The ~gcmMCEPTitle$G~ is all about the kids. We 1 to 2 the children of 3, helping them to aim higher and enabling them to 4 5. As 6 services(D50= and facilities|) 8, so too has the 9, to the point where we are {1 2. We are now 3 4 by way of a 5 of \\[target\\] \\[unit\\] of \\[commodity\\], which will 6 7 8.}7"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType87 = function $missionType87() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.operating_mantra),
        fieldB: ["provide quality mental health services for", "look after the mental health needs of", "provide compassionate, effective mental health services to"],
        fieldC: ["of every make and model", "of every kind, from traders to bounty hunters", "of every sort, from the mostly harmless to Elite"],
        fieldD: ["regardless of their history with GalCop", "regardless of their current legal standing", "regardless of their criminal associations, past or present"],
        fieldE: ["costs are rising", "costs are climbing", "our operating costs are climbing", "the cost of operating our services is escalating"],
        fieldF: [bountyText],
        fieldG: ["law and order breaks down", "GalCop struggles to maintain safety on the spacelanes", "the dangers from pirate raids increase", "the threat of Thargoid invasion looms large"],
        fieldH: ["cost effective", "cheap", "affordable", "attainable", "maintainable"],
        fieldI: ["the full spread of available mental illness treatments", "the latest advances in mental health treatment"],
        fieldJ: ["hard", "difficult", "almost impossible", "more and more difficult"],
        fieldK: [].concat(this._gcmPG.calling_for),
        fieldL: ["pilots", "commanders", "anyone", "those"],
        fieldM: ["support us", "partner with us", "work with us"],
        fieldN: [].concat(this._gcmPG.donation),
        fieldO: [].concat(this._gcmPG.enable_us),
        fieldP: ["expand and develop our services and facilities", "provide the best medical treatments and solutions", "ensure we can continue to look after the needs of our patients"],
        fieldQ: ["help all of our patients", "assist all of our patients", "enable all our patients to"],
        fieldR: ["have a positive, hopeful future", "feel confident about their future", "look forward to their future with hope", "face their future without fear"],
        sentences: [
            "The 1 of the ~gcmPMHSTitle$G~ is to 2 pilots 3. 5 though, (D50=especially|particularly) as 7. Providing commanders with 8 treatments, or giving them access to 9 is proving to be {1. We are 2 3 who can 4 in the form of a cargo 5, \\[target\\] \\[unit\\] of \\[commodity\\], that will 6 to 7 and 8 9.}6",
            "Contributing to ~gcmPMHSTitle$G~ isn't like making >{5 to other charities. The }1 of the fund is to 2 pilots 3, in a 8 way. We provide this service to pilots 4. But 5, (D50=especially|particularly) as 7. It is becoming {1 to give our patients access to }9. By making a cargo {5 of \\[target\\] \\[unit\\] of \\[commodity\\], you will 6 to 7 and 8 9.}6"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType90 = function $missionType90() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.in_search_of),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: [].concat(this._gcmPG.accept),
        fieldD: [].concat(this._gcmPG.contract),
        fieldE: [].concat(this._gcmPG.imperative),
        fieldF: ["10t"],
        fieldG: [bountyText],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "The ongoing civil war planet-side has left many homeless and destitute. Refugee camps are filled to capacity and more come every day. Many of these refugees have elected to be taken off-planet, to \\[destination\\] where a safe-haven has been established by the authorities. There are a high number of refugees who need to be transported, so we are 1 >2 who can take on the job of transferring them to \\[destination\\].\\n\\nIf you 3 this 4, a special passenger transport unit will be installed into your ship, which takes up 6 of cargo space, and accommodates fifty passengers. The unit will be removed automatically when you reach \\[destination\\].7",
            "As a result of the recent outbreak of civil war planet-side, many citizens have been displaced from their homes and are now filling refugee camps. Unfortunately, due to the limited space available in these camps, we are at risk of a breakdown of law and order due to overcrowding and insufficient supplies. To alleviate the problem, we have been in contact with authorities on \\[destination\\] and have established a safe-haven for some of these refugees. We are 1 >2 who can transfer some of these refugees to \\[destination\\] safely.\\n\\nIf you 3 this 4, a special passenger transport unit will be installed into your ship, which takes up 6 of cargo space, and can accommodate fifty refugees. The unit will be automatically removed when you reach \\[destination\\].7",
            "The ongoing hostilities on the planet have impacted thousands of citizens. Many of them have had their homes destroyed and are now filling up refugee camps in various places around the globe. But these camps are now at capacity, and with more refugees coming in daily we need a different solution.\\n\\nWe have been in contact with authorities on \\[destination\\] and have set up a safe-haven for some of these refugees, and we are now in need of pilots and ships who can take on the task of transporting the refugees there.\\n\\nIf you 3 this 4, a special passenger transport unit will be installed into your ship, which takes up 6 of cargo space. The unit will be automatically removed when you reach \\[destination\\].7"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType91 = function $missionType91() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.in_search_of),
        fieldB: [].concat(this._gcmPG.pilot),
        fieldC: [].concat(this._gcmPG.accept),
        fieldD: [].concat(this._gcmPG.contract),
        fieldE: [].concat(this._gcmPG.imperative),
        fieldF: ["5t"],
        fieldG: [bountyText],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "There are a number of patients who have been seriously injured in a recent outbreak of civil war, and their condition is critical. The local medical facilities are unable to cope with the volume of patients needing care, so we are 1 >2 who can take on the job of transferring these patients to \\[destination\\], which is the closest planet with the required techlevel, for the medical procedures that will hopefully save their lives.\\n\\nIf you 3 this 4, a special patient transport unit will be installed into your ship, which takes up 6 of cargo space. The unit will be removed automatically when you reach \\[destination\\].7",
            "As a result of the recent outbreak of civil war planet-side, and a lack of sufficient medical facilities, the condition of many injured patients has become critical. It is 5 these patients are transferred to \\[destination\\], the nearest system with the required techlevel, for emergency medical treatment that will hopefully save their lives.\\n\\nIf you 3 this 4, a special patient transport unit will be installed into your ship, which takes up 6 of cargo space. The unit will be automatically removed when you reach \\[destination\\].7",
            "The ongoing hostilities on the planet has impacted thousands of citizens. Some of the injured are in an extremely critical condition and local medical services are unable to provide sufficient care for these patients. We are 1 >2 who can take some of these critically-ill patients to waiting medical teams on \\[destination\\], the nearest system that has the required techlevel.\\n\\nIf you 3 this 4, a special patient transport unit will be installed into your ship, which takes up 6 of cargo space. The unit will be automatically removed when you reach \\[destination\\].7"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType100 = function $missionType100() {
    var def = {
        fieldA: ["operate", "run", "maintain"],
        fieldB: ["in order to", "to help us", "so we can"],
        fieldC: [].concat(this._gcmPG.many_different),
        fieldD: ["our regular scanning (D50=vessel|ship)", "the (D50=vessel|ship) that normally performs these scans"],
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.in_search_of),
        fieldG: ["fly to", "get", "approach to"],
        fieldH: ["1, 2, 3 and 4"],
        fieldI: ["approximately", "about", "around"],
        fieldJ: ["1km"],
        fieldK: ["With the threat of earthquakes a very present reality for our system, it's", "Earthquakes (D50=are|may be) a fact of life for our citizens, but that doesn't mean we're unprepared. It's", "The threat from earthquake disasters in our system is constant, which means"],
        fieldL: [].concat(this._gcmPG.constant),
        fieldM: ["watch", "monitor", "examine", "check"],
        fieldN: ["and be on the lookout for any possible disasters", "and minimise the possible loss of life these disasters bring", "and limit the damage these catastrophes cause"],
        fieldO: [].concat(this._gcmPG.imperative),
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We 1 3 of scanning equipment on our planet 2 get as much advance notice of any possible earthquakes. (D50=However,|But) 4 is offline, so we are 6 a 5 to take a seismic resonance scanner to four positions in low orbit, marked as scan points 8 on your space compass, 7 within {1 of each point, and perform the scans for us.\\n\\nThe scanner is started by priming it, and then activating it with your ships standard interface. The scanner will take }9 one minute to complete, and will drain a significant amount of energy while operating.\\n\\nThe scan is also quite fragile, so you will (D50=have|need) to be careful with it.",
            "{2 6 we 3ly 4 all seismic activity 5. One method we employ is using a seismic resonance scanner from low orbit. (D50=However,|But) }4 is offline, so we are 6 a 5 to take one of these scanners to four different positions, marked as scan points 8 on your space compass, 7 within {1 of each point, and perform the scans for us.\\n\\nThe scanner is started by priming it, and then activating it with your ships standard interface. The scanner will take }9 one minute to complete, and will drain a significant amount of energy while operating.\\n\\nThe scan is also quite fragile, so you will (D50=have|need) to be careful with it."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType101 = function $missionType101() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: ["A spate of recent", "Recent", "A recent spate of"],
        fieldB: ["devastated", "wrecked", "ravaged", "severely damaged", "wreaked havoc on"],
        fieldC: ["several", "a number of", "a few of"],
        fieldD: ["But", "Combined with this,", "In addition,"],
        fieldE: [].concat(this._gcmPG.large),
        fieldF: [].concat(this._gcmPG.calling_for),
        fieldG: ["someone", "a [8]"],
        fieldH: [].concat(this._gcmPG.pilot),
        fieldI: [].concat(this._gcmPG.accept),
        fieldJ: [bountyText],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "1 earthquakes have 2 3 our larger cities, and the death toll has been high. 4 we also have a 5 number of critically injured patients who need urgent medical care which can best be found on hi-tech worlds. We are 6 7 to 9 the job of (D50=transporting|transferring) these patients to \\[destination\\], which is the (D50=nearest|closest) planet with the (D50=necessary|requisite) technology level where their injuries can be treated.\\n\\nIf you take this job, a special patient transport unit will be installed into your ship, which takes up five tons of cargo space. The unit will be removed automatically when you reach \\[destination\\].{1"
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType102 = function $missionType102() {
    var def = {
        fieldA: ["several", "many", "a number of", "a large number of"],
        fieldB: ["imposing", "sizeable", "substantial", "significant"],
        fieldC: ["worrying", "disconcerting", "troubling"],
        fieldD: [].concat(this._gcmPG.pilot),
        fieldE: ["find", "track down", "locate", "pinpoint"],
        fieldF: [].concat(this._gcmPG.destroy),
        fieldG: ["A recent asteroid storm", "A mining transportation accident", "An asteroid storm, followed by a Miners' Guild industrial dispute,"],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "7 has left 1 2 asteroids hanging in low orbit. For any other system this (D50=may|might) not be a problem, but for us, the gravitational impact of those asteroids is having 3 effects on the stability of the tectonic plates on the planet surface. We need a 4 to circle the planet, 5 the \\[target\\] asteroids and 6 them."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType104 = function $missionType104() {
    var def = {
        fieldA: ["collect a lot of data about", "do a lot of deep-level seismic scans of", "need to be ever-vigilant in the observations of"],
        fieldB: ["trying to", "in an attempt to", "so as to", "doing our best to"],
        fieldC: ["impact of", "effects from", "disastrous consequences of", "fallout from"],
        fieldD: ["cities and population", "people", "services and infrastructure"],
        fieldE: ["the analysis", "the number-crunching", "the heavy-lifting"],
        fieldF: ["a set of seismic scans", "some particularly complex seismic scan data", "a complex set of seismic scan data"],
        fieldG: ["so this analysis can be performed", "so we can learn everything we can from the data", "so we can get the maximum possible value from the data"],
        fieldH: [],
        fieldI: [],
        fieldJ: [],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We 1 our planet, 2 minimise the 3 earthquakes on our 4. But some of 5 needs to be done elsewhere, making use of off-planet ultra-hi-tech data processing facilities. We need 6 delivered to \\[destination\\] 7. Some of the analysis is time critical, so the data must be delivered without delay.",
        ],
    }
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType110 = function $missionType110() {
    var def = {
        fieldA: ["One of our [9] has informed us that", "It has come to our attention that", "We have reason to believe", "We have it on good authority that", "Top-secret dispatches from [9] have revealed", "We believe"],
        fieldB: ["the \\[destination\\] system", "\\[destination\\]"],
        fieldC: ["However, we can't [4] as", "It would be unwise to [4] as", "While it would be tempting to simply [4], prudence dictates this option is unacceptable as"],
        fieldD: ["send one of our official ships to get the data,", "have one of our official vessels extract the data,"],
        fieldE: ["the exposure risk is too high", "the potential exposure risks are too high", "the political fallout and inflammation of security tensions should we be discovered are too great to ignore"],
        fieldF: ["We also can't", "Neither can we", "And it would be inappropriate to"],
        fieldG: [].concat(this._gcmPG.in_search_of),
        fieldH: [].concat(this._gcmPG.travel_to),
        fieldI: [].concat(this._gcmPG.agents),
        fieldJ: ["vital to our security", "of a critical nature", "of some significance", "which has a direct bearing on our security", "of strategic importance", "which threatens the peace and safety of our world", "undermines the stability of our region"],
        fieldK: [].concat(this._gcmPG.pilot),
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: ["15"],
        sentences: [
            "1 information {1 has been collected on surveillance satellites in }2. 3 5. 6 request the authorities in \\[destination\\] to simply hand over potentially sensitive and volatile security information.\\n\\nSo we are 7 an independent {2 who can }8 \\[destination\\] and extract all data captured on any surveillance satellites that can be found there. All surveillance satellites in the system will need to be accessed.\\n\\nTo access the satellite you will be given a one-time passcode you will need to transmit to the unit. This passcode will activate when you jump into \\[destination\\], and will remain active for {9 minutes. You will not be able to access the satellites after this time."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType111 = function $missionType111() {
    var def = {
        fieldA: ["Sensitive communications between some high-level security personnel", "A number of extremely volatile communications, between some highly placed military figures,", "Our of our [9] has been forced to transmit some highly sensitive data packets that"],
        fieldB: ["the \\[destination\\] system", "\\[destination\\]"],
        fieldC: ["However, we can't [4] as", "It would be unwise to [4] as", "While it would be tempting to simply [4] prudence dictates this option is unacceptable as"],
        fieldD: ["send one of our official ships to get the data,", "have one of our official vessels seen extracting the data,"],
        fieldE: ["the exposure risk is too high", "the potential exposure risks are too high", "the political fallout and inflammation of security tensions should we be discovered are too great to ignore"],
        fieldF: ["We also can't", "Neither can we", "And it would be inappropriate to"],
        fieldG: [].concat(this._gcmPG.in_search_of),
        fieldH: [].concat(this._gcmPG.travel_to),
        fieldI: [].concat(this._gcmPG.agents),
        fieldJ: [].concat(this._gcmPG.pilot),
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: ["15"],
        sentences: [
            "1 have been captured and logged on COM satellites in 2, and we believe these messages have a direct bearing on our security here. 3 5. 6 request the authorities in \\[destination\\] to simply hand over potentially sensitive and volatile communication logs.\\n\\nSo we are 7 an independent 9 who can 8 \\[destination\\] and extract communication log data recorded on any communication satellites found there. All communication satellites (both COM and COMLR) will need to be accessed.\\n\\nTo access the satellite you will be given a one-time passcode you will need to transmit to the unit. This passcode will activate when you jump into \\[destination\\], and will remain active for {9 minutes. You will not be able to access the satellites after this time."
        ],
        custom: [0x0],
        splitRules: []
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType122 = function $missionType122() {
    var def = {
        fieldA: ["attempt to", "trying to", "attempting to", "going through a complicated process to"],
        fieldB: ["delete", "excise", "remove", "wipe out"],
        fieldC: ["relevant records", "offending material", "problematic data"],
        fieldD: ["easier", "more expedient", "faster and easier", "faster", "quicker"],
        fieldE: [].concat(this._gcmPG.in_search_of),
        fieldF: [].concat(this._gcmPG.pilot),
        fieldG: [].concat(this._gcmPG.travel_to),
        fieldH: [],
        fieldI: ["incur", "receive", "get", "be slapped with"],
        fieldJ: ["A series of errors by security personnel has resulted in confidential and sensitive information being", "The actions of some of our military personnel have unfortunately been", "The lives of some of our [9] hang in the balance, because some operational material has been"],
        fieldK: [],
        fieldL: [],
        fieldM: [],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [].concat(this._gcmPG.agents),
        sentences: [
            "{1 captured by surveillance and communication satellites in \\[destination\\]. Rather than }1 2 the 3 from the satellites, it is 4 to simply destroy them.\\n\\nWe are 5 a 6 who can 7 \\[destination\\] and destroy all surveillance and communication satellites that can be found there. Satellite destruction can potentially result in a legal penalty in \\[destination\\], but we hope the payment will cover the inconvenience of carrying whatever bounty you might 9."
        ],
    };
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType150 = function $missionType150() {
    var def = {
        fieldA: [].concat(this._gcmPG.information),
        fieldB: ["suggests", "indicates", "leads us to believe"],
        fieldC: ["are short of", "have minimal", "have a limited number of", "have a shortage of"],
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.travel_to),
        fieldG: ["prevent", "stop"],
        fieldH: ["available", "operational"],
        fieldI: ["at the moment", "at the present time", "right now", "at present", "just now"],
        fieldJ: ["most likely", "in all likelihood", "with very high probability of", "and we expect they will be", "and we suspect they will be"],
        fieldK: ["pirates", "a band of pirates", "a group of pirates", "a pirate group", "a pirate gang", "mercenaries", "a group of mercenaries", "a band of mercenaries", "a mercenary group", "a mercenary gang"],
        fieldL: ["200"],
        fieldM: [].concat(this._gcmPG.received),
        fieldN: ["in transit", "on route", "on it's way"],
        fieldO: ["several", "a number of", "a few"],
        fieldP: ["[6] important government dignitaries", "[6] important government officials", "an important trade delegation", "a party of foreign diplomats"],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We have {4 }1 that 2 {2 are planning an attack in the \\[destination\\] system, 1 targeting an Anaconda transport carrying 7 that is already 5 and will be in system by the time you arrive. We }3 8 system defence (D50=ships|vessels) 9, so we are 4 >5 who can 6 \\[destination\\] and 7 the group from destroying this ship, and potentially destabilising the already volatile political environment.\\n\\nThe transport ship is called '\\[missiontargetShipName\\]'.\\n\\nA bonus of {3cr will be awarded for each attacking vessel you destroy.",
            "+{2 in the \\[destination\\] system have been increasing their activity, and we have 4 }1 that 2 they will be attempting to destroy an Anaconda transport carrying {7 that is already 5 and will be in system by the time you arrive. Unfortunately, we }3 8 system defence (D50=ships|vessels) 9, which creates a problem for us, that we are hoping to address by finding >5 who can 6 \\[destination\\] and 7 this group from achieving their goals.\\n\\nThe transport ship is called '\\[missiontargetShipName\\]'.\\n\\nA bonus of {3cr will be awarded for each attacking vessel you destroy."
        ],
    }
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType152 = function $missionType152() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.information),
        fieldB: ["suggests", "indicates", "leads us to believe"],
        fieldC: ["are short of", "have minimal", "have a limited number of", "have a shortage of"],
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.travel_to),
        fieldG: ["prevent", "stop"],
        fieldH: ["available", "operational"],
        fieldI: ["at the moment", "at the present time", "right now", "at present", "just now"],
        fieldJ: ["most likely", "in all likelihood", "with very high probability of", "and we expect they will be", "and we suspect they will be"],
        fieldK: ["pirates", "a band of pirates", "a group of pirates", "a pirate group", "a pirate gang", "mercenaries", "a group of mercenaries", "a band of mercenaries", "a mercenary group", "a mercenary gang"],
        fieldL: ["200"],
        fieldM: [].concat(this._gcmPG.received),
        fieldN: ["500"],
        fieldO: ["general disruption", "mayhem and destruction", "widespread mayhem and general disruption"],
        fieldP: [bountyText],
        fieldQ: [],
        fieldR: [],
        sentences: [
            "We have {4 }1 that 2 {2 are planning an attack on the \\[destination\\] system, 1 targeting the main station. While 6 is likely their goal, our intelligence operatives have provided evidence to suggest these pirates have come into possession of some very high-powered missiles, dubbed 'Station Killers', which they may attempt to fire at the station.\\n\\nWe }3 8 system defence (D50=ships|vessels) 9, so we are 4 >5 who can 6 \\[destination\\] and 7 this group from destroying or damaging the station.\\n\\nA bonus of {3cr will be awarded for each attacking vessel you destroy, plus 5cr for each Station Killer missile you shoot down.{7",
            "+{2 in the \\[destination\\] system have been increasing their activity, and we have 4 }1 that 2 they will be targeting the main station in that system, to cause as much {6 as possible. Our intelligence operatives have also uncovered information that indicates these brigands have taken possession of some high-energy missiles, called 'Station Killers', which they might attempt to shoot at the station.\\n\\nUnfortunately, we }3 8 system defence (D50=ships|vessels) 9, which creates a problem for us, which we are hoping to address by finding >5 who can 6 \\[destination\\] and 7 this group from achieving their goals.\\n\\nA bonus of {3cr will be awarded for each attacking vessel you destroy, plus 5cr for each Station Killer missile you shoot down.{7"
        ],
    }
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType154 = function $missionType154() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.information),
        fieldB: ["suggests", "indicates", "leads us to believe"],
        fieldC: ["are short of", "have minimal", "have a limited number of", "have a shortage of"],
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.travel_to),
        fieldG: ["prevent", "stop"],
        fieldH: ["available", "operational"],
        fieldI: ["at the moment", "at the present time", "right now", "at present", "just now"],
        fieldJ: ["most likely", "in all likelihood", "with very high probability of", "and we expect they will be", "and we suspect they will be", "and our smartest minds are saying they will be"],
        fieldK: ["500"],
        fieldL: [].concat(this._gcmPG._received),
        fieldM: [bountyText],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: ["We have received 1 that 2 the Thargoids are planning an attack on the \\[destination\\] system, {1 targeting the main station. We }3 8 system defence (D50=ships|vessels) 9, so we are 4 >5 who can 6 \\[destination\\] and 7 the Thargoids from destroying or damaging the station.\\n\\nA bonus of {2cr will be awarded for each Thargoid warship you destroy.{4"],
    }
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$missionType156 = function $missionType156() {
    var bountyText = "";
    if (player.ship.bounty > 0) bountyText = "\\nCompleting this mission will reduce any outstanding bounty you may have.";
    var def = {
        fieldA: [].concat(this._gcmPG.information),
        fieldB: ["suggests", "indicates", "leads us to believe"],
        fieldC: ["are short of", "have minimal", "have a limited number of", "have a shortage of"],
        fieldD: [].concat(this._gcmPG.in_search_of),
        fieldE: [].concat(this._gcmPG.pilot),
        fieldF: [].concat(this._gcmPG.travel_to),
        fieldG: ["prevent", "stop"],
        fieldH: ["available", "operational"],
        fieldI: ["at the moment", "at the present time", "right now", "at present", "just now"],
        fieldJ: ["most likely", "in all likelihood", "with very high probability of", "and we expect they will be", "and we suspect they will be", "and our smartest minds are saying they will be"],
        fieldK: ["500"],
        fieldL: [].concat(this._gcmPG.received),
        fieldM: [bountyText],
        fieldN: [],
        fieldO: [],
        fieldP: [],
        fieldQ: [],
        fieldR: [],
        sentences: ["We have {3 }1 that 2 the Thargoids are planning an attack on the \\[destination\\] system, {1 targeting the witchpoint beacon. We }3 8 system defence (D50=ships|vessels) 9, so we are 4 >5 who can 6 \\[destination\\] and 7 the Thargoids from destroying the beacon.\\n\\nA bonus of {2cr will be awarded for each Thargoid warship you destroy.{4"],
    }
    if (this._debug) this._setDef = def;
    var text = this._phraseGen._makePhrase(def);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$shipName = function $shipName() {
    var text = this._phraseGen._makePhrase(this._phraseGen.$pool.GNN_NamesShips);
    return this.$cleanUp(text);
}

//-------------------------------------------------------------------------------------------------------------
this.$cleanUp = function $cleanUp(text) {
    // remove any of the \ chars that prefix any [ and ] chars
    text = text.replace(/\\\[/g, "[");
    text = text.replace(/\\\]/g, "]");
    // look for any \n operatives, and replace them with an actual new line
    text = text.replace(/\\n/g, String.fromCharCode(13) + String.fromCharCode(10));
    // look for any "missionVariable" lookups and put the correct prefix in
    text = text.replace(/\[mission/g, "[mission_");
    return text;
}

//-------------------------------------------------------------------------------------------------------------
// this function takes the phraseGen definition of a particular mission type and dumps all the options and text
// to the log file to assist in data entry/debugging on the phraseGen html page
this.$outputDef = function $outputDef(missType) {
    var ws = worldScripts.GalCopBB_MissionDetails;
    var fn = "$missionType" + missType;
    ws[fn]();
    if (ws._setDef != null) {
        var list = ["A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R"];
        for (var i = 0; i <= 1; i++) {
            for (var j = 0; j <= 8; j++) {
                var idx = (i * 9) + j;
                var fld = "field" + list[idx];
                var ary = ws._setDef[fld];
                var out = fld + " (" + (j + 1) + ")";
                if (ary.length > 0) {
                    for (var k = 0; k < ary.length; k++) {
                        out += "\n" + ary[k];
                    }
                    log(this.name, out);
                } else {
                    log(this.name, out + "\n{empty}");
                }
            }
        }
        var st = ws._setDef.sentences;
        var out = "Sentences";
        for (var i = 0; i < st.length; i++) {
            out += "\n" + st[i].replace(/\\/g, "\\\\");
        }
        log(this.name, out);
    }
    ws._setDef = null;
}

this.$outputAllSamples = function () {
    for (var i = 1; i <= 156; i++) {
        var fn = "$missionType" + i.toString();
        if (this[fn]) {
            log(this.name, i.toString() + " -- " + this[fn]());
        }
    }
}
