"use strict";
this.name = "LMSS_Core";
this.author = "phkb";
this.copyright = "2015 phkb";
this.description = "Laser Mount Switching System - Core code for weapon switcher";
this.licence = "CC BY-NC-SA 4.0";

this._forwardAltKey = "EQ_WEAPON_NONE";          // .
this._aftAltKey = "EQ_WEAPON_NONE";              //  | stores the equipment keys of the secondary laser mounts
this._portAltKey = "EQ_WEAPON_NONE";             //  |
this._starboardAltKey = "EQ_WEAPON_NONE";        // .
this._forwardHoldKey = "";                       // .
this._aftHoldKey = "";                           //  | Holding variables, used when switching lasers
this._portHoldKey = "";                          //  |
this._starboardHoldKey = "";                     // .
this._disableEquipmentSale = false;              // flag to allow ship respray to know to keep any secondary lasers
this._switching = false;

this._defaultDelay = 10;                         // default amount of time it takes to switch weapons in flight
this._incompatible = ["EQ_WEAPON_LASER_COOLER"]; // array of laser keys that can't be switched

//-------------------------------------------------------------------------------------------------------------
this.startUpComplete = function () {
	var p = player.ship;
	// check whether the actuator is installed, and if so, reduce the switching delay
	if (p.equipmentStatus("EQ_LMSS_ACTUATOR") === "EQUIPMENT_OK") this._defaultDelay = 6;

	// read the secondary lasers from mission variables
	this._forwardAltKey = missionVariables.LMSS_ForwardKey;
	if (this._forwardAltKey === null) this._forwardAltKey = "EQ_WEAPON_NONE";

	this._aftAltKey = missionVariables.LMSS_AftKey;
	if (this._aftAltKey === null) this._aftAltKey = "EQ_WEAPON_NONE";

	this._portAltKey = missionVariables.LMSS_PortKey;
	if (this._portAltKey === null) this._portAltKey = "EQ_WEAPON_NONE";

	this._starboardAltKey = missionVariables.LMSS_StarboardKey;
	if (this._starboardAltKey === null) this._starboardAltKey = "EQ_WEAPON_NONE";

	// some cleanup from previous versions
	if (missionVariables.LMSS_ForwardName) delete missionVariables.LMSS_ForwardName;
	if (missionVariables.LMSS_AftName) delete missionVariables.LMSS_AftName;
	if (missionVariables.LMSS_PortName) delete missionVariables.LMSS_PortName;
	if (missionVariables.LMSS_StarboardName) delete missionVariables.LMSS_StarboardName;

	this.$resetInterface();
}

//-------------------------------------------------------------------------------------------------------------
this.playerWillSaveGame = function () {
	missionVariables.LMSS_ForwardKey = this._forwardAltKey;
	missionVariables.LMSS_ForwardName = this._forwardAltName;
	missionVariables.LMSS_AftKey = this._aftAltKey;
	missionVariables.LMSS_AftName = this._aftAltName;
	missionVariables.LMSS_PortKey = this._portAltKey;
	missionVariables.LMSS_PortName = this._portAltName;
	missionVariables.LMSS_StarboardKey = this._starboardAltKey;
	missionVariables.LMSS_StarboardName = this._starboardAltName;
}

//-------------------------------------------------------------------------------------------------------------
this.shipWillDockWithStation = function (station) {
	var p = player.ship;

	this.$stopTimers();

	// if an assembly was damaged during switching (or we were switching while docking), put the hold laser back in now
	// the assumption is that we kicked the machine and got main laser back, but the switch assembly is still damaged
	if (this._forwardHoldKey != "") p.forwardWeapon = this._forwardHoldKey;
	if (this._aftHoldKey != "") p.aftWeapon = this._aftHoldKey;
	if (this._portHoldKey != "") p.portWeapon = this._portHoldKey;
	if (this._starboardHoldKey != "") p.starboardWeapon = this._starboardHoldKey;
	this._forwardHoldKey = "";
	this._aftHoldKey = "";
	this._portHoldKey = "";
	this._starboardHoldKey = "";

	if (p.equipmentStatus("EQ_LMSS_ACTIVATOR") === "EQUIPMENT_OK") {
		this.$initInterface(station);
	}
	this.$updateManifest();
}

//-------------------------------------------------------------------------------------------------------------
this.shipDied = function (whom, why) {
	this.$stopTimers();
}

//-------------------------------------------------------------------------------------------------------------
this.equipmentDamaged = function (equipmentKey) {
	// check if the associated timer is running - if it is, stop it - we just lost the laser!
	switch (equipmentKey) {
		case "EQ_LMSS_ACTUATOR":
			// if the actuator gets damaged, switch back to the longer delay
			this._defaultDelay = 10;
			break;
		case "EQ_LMSS_FRONT":
			if (this._forwardTimer && this._forwardTimer.isRunning) this._forwardTimer.stop();
			break;
		case "EQ_LMSS_AFT":
			if (this._aftTimer && this._aftTimer.isRunning) this._aftTimer.stop();
			break;
		case "EQ_LMSS_PORT":
			if (this._portTimer && this._portTimer.isRunning) this._portTimer.stop();
			break;
		case "EQ_LMSS_STARBOARD":
			if (this._starboardTimer && this._starboardTimer.isRunning) this._starboardTimer.stop();
			break;
	}
}

//-------------------------------------------------------------------------------------------------------------
// handles any repair that happens mid-flight (via repair bots)
this.equipmentRepaired = function (equipmentKey) {
	var p = player.ship;
	if (p.status != "STATUS_IN_FLIGHT") return;

	switch (equipmentKey) {
		case "EQ_LMSS_ACTUATOR":
			// if the actuator gets repaired in flight (via repair bots), switch to the shorter delay
			this._defaultDelay = 6;
			break;
		case "EQ_LMSS_FRONT":
			if (this._forwardHoldKey != "") {
				p.forwardWeapon = this._forwardHoldKey;
				this._forwardHoldKey = "";
				this.$updateManifest();
			}
			break;
		case "EQ_LMSS_AFT":
			if (this._aftHoldKey != "") {
				p.aftWeapon = this._aftHoldKey;
				this._aftHoldKey = "";
				this.$updateManifest();
			}
			break;
		case "EQ_LMSS_PORT":
			if (this._portHoldKey != "") {
				p.portWeapon = this._portHoldKey;
				this._portHoldKey = "";
				this.$updateManifest();
			}
			break;
		case "EQ_LMSS_STARBOARD":
			if (this._starboardHoldKey != "") {
				p.starboardWeapon = this._starboardHoldKey;
				this._starboardHoldKey = "";
				this.$updateManifest();
			}
			break;
	}
}

//-------------------------------------------------------------------------------------------------------------
this.equipmentAdded = function (equipmentKey) {
	// auto add the switcher utility, so we only get one of them
	if (equipmentKey === "EQ_LMSS_FRONT" || equipmentKey === "EQ_LMSS_AFT" || equipmentKey === "EQ_LMSS_PORT" || equipmentKey === "EQ_LMSS_STARBOARD") {
		var p = player.ship;
		if (p.equipmentStatus("EQ_LMSS_ACTIVATOR") != "EQUIPMENT_OK") {
			p.awardEquipment("EQ_LMSS_ACTIVATOR");
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.equipmentRemoved = function (equipmentKey) {
	// auto-remove the switcher utility
	if (equipmentKey === "EQ_LMSS_FRONT" || equipmentKey === "EQ_LMSS_AFT" || equipmentKey === "EQ_LMSS_PORT" || equipmentKey === "EQ_LMSS_STARBOARD") {
		var p = player.ship;
		if (p.equipmentStatus("EQ_LMSS_FRONT") != "EQUIPMENT_OK" && p.equipmentStatus("EQ_LMSS_AFT") != "EQUIPMENT_OK" && p.equipmentStatus("EQ_LMSS_PORT") != "EQUIPMENT_OK" && p.equipmentStatus("EQ_LMSS_STARBOARD") != "EQUIPMENT_OK") {
			if (p.equipmentStatus("EQ_LMSS_ACTIVATOR") == "EQUIPMENT_OK") {
				p.removeEquipment("EQ_LMSS_ACTIVATOR");
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------------
this.guiScreenChanged = function (to, from) {
	// monitor which screens we go to, and reset the "disable equipment sale" flag when we move away from one one these screens
	if (guiScreen != "GUI_SCREEN_EQUIP_SHIP" && guiScreen != "GUI_SCREEN_MISSION" && guiScreen != "GUI_SCREEN_STATUS") {
		this._disableEquipmentSale === false
	}
}

//-------------------------------------------------------------------------------------------------------------
this.playerBoughtEquipment = function (equipmentKey) {
	var p = player.ship;

	// if the player purchases a respray for their ship, turn on a flag to stop the sale of equipment items during the "playerBoughtNewShip" event
	if (equipmentKey === "EQ_SHIP_RESPRAY" || equipmentKey === "EQ_SHIP_RESPRAY_180") {
		this._disableEquipmentSale = true;
		return;
	}
	// add the interface if this is our first LMSS purchase
	if (equipmentKey === "EQ_LMSS_FRONT" || equipmentKey === "EQ_LMSS_AFT" || equipmentKey === "EQ_LMSS_PORT" || equipmentKey === "EQ_LMSS_STARBOARD") {
		this.$initInterface(p.dockedStation);
	}
	// are we removing one of the positions?
	if (equipmentKey.indexOf("EQ_LMSS_") >= 0 && equipmentKey.indexOf("_REMOVAL") >= 0) {
		if (equipmentKey === "EQ_LMSS_FRONT_REMOVAL") {
			p.removeEquipment("EQ_LMSS_FRONT");
			p.removeEquipment("EQ_LMSS_FRONT_REMOVAL");
			if (this._forwardAltKey != "" && this._forwardAltKey != "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_FRONT_WEAPON");
				this.$refundItemAfterNewShip(this._forwardAltKey);
			}
			this._forwardAltKey = "EQ_WEAPON_NONE";
		}
		if (equipmentKey === "EQ_LMSS_AFT_REMOVAL") {
			p.removeEquipment("EQ_LMSS_AFT");
			p.removeEquipment("EQ_LMSS_AFT_REMOVAL");
			if (this._aftAltKey != "" && this._aftAltKey != "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_AFT_WEAPON");
				this.$refundItemAfterNewShip(this._aftAltKey);
			}
			this._aftAltKey = "EQ_WEAPON_NONE";
		}
		if (equipmentKey === "EQ_LMSS_PORT_REMOVAL") {
			p.removeEquipment("EQ_LMSS_PORT");
			p.removeEquipment("EQ_LMSS_PORT_REMOVAL");
			if (this._portAltKey != "" && this._portAltKey != "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_PORT_WEAPON");
				this.$refundItemAfterNewShip(this._portAltKey);
			}
			this._portAltKey = "EQ_WEAPON_NONE";
		}
		if (equipmentKey === "EQ_LMSS_STARBOARD_REMOVAL") {
			p.removeEquipment("EQ_LMSS_STARBOARD");
			p.removeEquipment("EQ_LMSS_STARBOARD_REMOVAL");
			if (this._starboardAltKey != "" && this._starboardAltKey != "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_STARBOARD_WEAPON");
				this.$refundItemAfterNewShip(this._starboardAltKey);
			}
			this._starboardAltKey = "EQ_WEAPON_NONE";
		}
		// if we've removed all the positions, then remove the activator as well
		if (p.equipmentStatus("EQ_LMSS_FRONT") != "EQUIPMENT_OK" && p.equipmentStatus("EQ_LMSS_AFT") != "EQUIPMENT_OK" &&
			p.equipmentStatus("EQ_LMSS_PORT") != "EQUIPMENT_OK" && p.equipmentStatus("EQ_LMSS_STARBOARD") != "EQUIPMENT_OK") {
			p.removeEquipment("EQ_LMSS_ACTIVATOR");
		}
		this.$resetInterface();
	}
	if (equipmentKey === "EQ_LMSS_ACTUATOR_REMOVAL") {
		p.removeEquipment("EQ_LMSS_ACTUATOR");
		p.removeEquipment("EQ_LMSS_ACTUATOR_REMOVAL");
		// reset the delay back to the original time
		this._defaultDelay = 10;
	}
	if (equipmentKey === "EQ_LMSS_ACTUATOR") {
		this._defaultDelay = 6;
	}
}

//-------------------------------------------------------------------------------------------------------------
// if a new ship is bought, refund any secondary lasers to the player
this.playerBoughtNewShip = function (ship, price) {
	// if the new ship is part of a respray, don't sell anything, but set a timer to reset the interfaces after the sale has completed.
	if (this._disableEquipmentSale === true) {
		this._resetInterface = new Timer(this, this.$resetInterface, 1, 0);
		return;
	}

	if (this._forwardAltKey != "" && this._forwardAltKey != "EQ_WEAPON_NONE") this.$refundItemAfterNewShip(this._forwardAltKey);
	if (this._aftAltKey != "" && this._aftAltKey != "EQ_WEAPON_NONE") this.$refundItemAfterNewShip(this._aftAltKey);
	if (this._portAltKey != "" && this._portAltKey != "EQ_WEAPON_NONE") this.$refundItemAfterNewShip(this._portAltKey);
	if (this._starboardAltKey != "" && this._starboardAltKey != "EQ_WEAPON_NONE") this.$refundItemAfterNewShip(this._starboardAltKey);
	// reset the secondary laser variables
	this._forwardAltKey = "EQ_WEAPON_NONE";
	this._aftAltKey = "EQ_WEAPON_NONE";
	this._portAltKey = "EQ_WEAPON_NONE";
	this._starboardAltKey = "EQ_WEAPON_NONE";

	this.$resetInterface();
}

//-------------------------------------------------------------------------------------------------------------
// used to reset the interface screen after equipment purchase
this.$resetInterface = function $resetInterface() {
	this.$updateManifest();
	this.$initInterface(player.ship.dockedStation);
}

//-------------------------------------------------------------------------------------------------------------
// refund price of equipment to player
this.$refundItemAfterNewShip = function $refundItemAfterNewShip(equipment) {
	var laser = EquipmentInfo.infoForKey(equipment);
	player.credits += laser.price / 10;
}

//-------------------------------------------------------------------------------------------------------------
// updates the manifest screen with details of all secondary lasers
this.$updateManifest = function $updateManifest() {
	var list = [];
	var p = player.ship;
	list.push(expandDescription("[lmss_secondary_header]") + ":");

	if (p.equipmentStatus("EQ_LMSS_FRONT_WEAPON") === "EQUIPMENT_OK") {
		if (this._forwardHoldKey != "") {
			list.push(expandDescription("[lmss_forward]") + ": " + expandDescription("[lmss_being_switched]"));
		} else {
			list.push(expandDescription("[lmss_forward]") + ": " + this.$getEquipmentName(this._forwardAltKey, "FORWARD", false));
		}
	}
	if (p.equipmentStatus("EQ_LMSS_AFT_WEAPON") === "EQUIPMENT_OK") {
		if (this._aftHoldKey != "") {
			list.push(expandDescription("[lmss_aft]") + ": " + expandDescription("[lmss_being_switched]"));
		} else {
			list.push(expandDescription("[lmss_aft]") + ": " + this.$getEquipmentName(this._aftAltKey, "AFT", false));
		}
	}
	if (p.equipmentStatus("EQ_LMSS_PORT_WEAPON") === "EQUIPMENT_OK") {
		if (this._portHoldKey != "") {
			list.push(expandDescription("[lmss_port]") + ": " + expandDescription("[lmss_being_switched]"));
		} else {
			list.push(expandDescription("[lmss_port]") + ": " + this.$getEquipmentName(this._portAltKey, "PORT", false));
		}
	}
	if (p.equipmentStatus("EQ_LMSS_STARBOARD_WEAPON") === "EQUIPMENT_OK") {
		if (this._starboardHoldKey != "") {
			list.push(expandDescription("[lmss_starboard]") + ": " + expandDescription("[lmss_being_switched]"));
		} else {
			list.push(expandDescription("[lmss_starboard]") + ": " + this.$getEquipmentName(this._starboardAltKey, "STARBOARD", false));
		}
	}

	if (list.length === 1) {
		// no secondary lasers installed
		mission.setInstructions(null);
	} else {
		mission.setInstructions(list, this.name);
	}

	this.$updateXenonHUD();

}

//-------------------------------------------------------------------------------------------------------------
// establishes the F4 interface for switching laser mounts while docked
this.$initInterface = function $initInterface(station) {
	var p = player.ship;
	if (p.equipmentStatus("EQ_LMSS_ACTIVATOR") === "EQUIPMENT_OK") {
		station.setInterface(this.name, {
			title: expandDescription("[lmss_interface_title]"),
			category: expandDescription("[interfaces-category-ship-systems]"),
			summary: expandDescription("[lmss_interface_summary]"),
			callback: this.$showLMSSList.bind(this)
		});
	} else {
		station.setInterface(this.name, null);
	}
}

//-------------------------------------------------------------------------------------------------------------
// displays the dockside switching options
this.$showLMSSList = function $showLMSSList() {
	var p = player.ship;
	var curChoices = {};
	var text = expandDescription("[lmss_select_laser]") + ":\n\n";
	var def = "99_EXIT";

	if (worldScripts.LaserArrangement && p.weaponFacings > 1) {
		curChoices["00_LA"] = {
			text: "[lmss_primary_laser]"
		};
	}

	if (p.equipmentStatus("EQ_LMSS_FRONT") === "EQUIPMENT_OK") {
		text += expandDescription("[lmss_forward]") + ":\n";
		text += "  " + expandDescription("[lmss_primary]") + ": " + (p.forwardWeapon.equipmentKey === "EQ_WEAPON_NONE" ? expandDescription("[lmss_none]") : this.$getEquipmentName(p.forwardWeapon.equipmentKey, "FORWARD", true)) + "\n";
		text += "  " + expandDescription("[lmss_secondary]") + ": " + this.$getEquipmentName(this._forwardAltKey, "FORWARD", false) + "\n";
		curChoices["01_FORWARD"] = {
			text: "[lmss_switch_forward]",
			color: "orangeColor"
		};
	}
	if (p.equipmentStatus("EQ_LMSS_AFT") === "EQUIPMENT_OK") {
		text += expandDescription("[lmss_aft]") + ":\n";
		text += "  " + expandDescription("[lmss_primary]") + ": " + (p.aftWeapon.equipmentKey === "EQ_WEAPON_NONE" ? expandDescription("[lmss_none]") : this.$getEquipmentName(p.aftWeapon.equipmentKey, "AFT", true)) + "\n";
		text += "  " + expandDescription("[lmss_secondary]") + ": " + this.$getEquipmentName(this._aftAltKey, "AFT", false) + "\n";
		curChoices["02_AFT"] = {
			text: "[lmss_switch_aft]",
			color: "orangeColor"
		};
	}
	if (p.equipmentStatus("EQ_LMSS_PORT") === "EQUIPMENT_OK") {
		text += expandDescription("[lmss_port]") + ":\n";
		text += "  " + expandDescription("[lmss_primary]") + ": " + (p.portWeapon.equipmentKey === "EQ_WEAPON_NONE" ? expandDescription("[lmss_none]") : this.$getEquipmentName(p.portWeapon.equipmentKey, "PORT", true)) + "\n";
		text += "  " + expandDescription("[lmss_secondary]") + ": " + this.$getEquipmentName(this._portAltKey, "PORT", false) + "\n";
		curChoices["03_PORT"] = {
			text: "[lmss_switch_port]",
			color: "orangeColor"
		};
	}
	if (p.equipmentStatus("EQ_LMSS_STARBOARD") === "EQUIPMENT_OK") {
		text += expandDescription("[lmss_starboard]") + ":\n";
		text += "  " + expandDescription("[lmss_primary]") + ": " + (p.starboardWeapon.equipmentKey === "EQ_WEAPON_NONE" ? expandDescription("[lmss_none]") : this.$getEquipmentName(p.starboardWeapon.equipmentKey, "STARBOARD", true)) + "\n";
		text += "  " + expandDescription("[lmss_secondary]") + ": " + this.$getEquipmentName(this._starboardAltKey, "STARBOARD", false) + "\n";
		curChoices["04_STARBOARD"] = {
			text: "[lmss_switch_starboard]",
			color: "orangeColor"
		};
	}

	curChoices["99_EXIT"] = {text: "[lmss_exit]"};

	var opts = {
		screenID: "oolite-lmss-dockside-map",
		title: expandDescription("[lmss_screen_title]"),
		allowInterrupt: true,
		exitScreen: "GUI_SCREEN_INTERFACES",
		overlay: {
			name: "lmss-arrow.png",
			height: 546
		},
		choices: curChoices,
		initialChoicesKey: def,
		message: text
	};
	mission.runScreen(opts, this.$screenHandler, this);
}

//-------------------------------------------------------------------------------------------------------------
// handles user input from the interface screen
this.$screenHandler = function $screenHandler(choice) {
	if (choice == null) return;

	switch (choice) {
		case "00_LA":
			worldScripts.LaserArrangement.$showInitial();
			return;
		case "01_FORWARD":
			this.$switchImmediate("FORWARD");
			break;
		case "02_AFT":
			this.$switchImmediate("AFT");
			break;
		case "03_PORT":
			this.$switchImmediate("PORT");
			break;
		case "04_STARBOARD":
			this.$switchImmediate("STARBOARD");
			break;
	}

	if (choice != "99_EXIT") {
		this.$showLMSSList();
	}
}

//-------------------------------------------------------------------------------------------------------------
// switches laser mounts immediately (used only when docked)
this.$switchImmediate = function $switchImmediate(position) {
	var p = player.ship;
	var key = "";

	this._switching = true;

	switch (position) {
		case "FORWARD":
			if (this._forwardAltKey === "EQ_WEAPON_NONE") p.awardEquipment("EQ_LMSS_FRONT_WEAPON");
			if (p.forwardWeapon.equipmentKey === "EQ_WEAPON_NONE") p.removeEquipment("EQ_LMSS_FRONT_WEAPON");
			key = p.forwardWeapon.equipmentKey;
			p.forwardWeapon = this._forwardAltKey;
			this._forwardAltKey = key;
			break;
		case "AFT":
			if (this._aftAltKey === "EQ_WEAPON_NONE") p.awardEquipment("EQ_LMSS_AFT_WEAPON");
			if (p.aftWeapon.equipmentKey === "EQ_WEAPON_NONE") p.removeEquipment("EQ_LMSS_AFT_WEAPON");
			key = p.aftWeapon.equipmentKey;
			p.aftWeapon = this._aftAltKey;
			this._aftAltKey = key;
			break;
		case "PORT":
			if (this._portAltKey === "EQ_WEAPON_NONE") p.awardEquipment("EQ_LMSS_PORT_WEAPON");
			if (p.portWeapon.equipmentKey === "EQ_WEAPON_NONE") p.removeEquipment("EQ_LMSS_PORT_WEAPON");
			key = p.portWeapon.equipmentKey;
			p.portWeapon = this._portAltKey;
			this._portAltKey = key;
			break;
		case "STARBOARD":
			if (this._starboardAltKey === "EQ_WEAPON_NONE") p.awardEquipment("EQ_LMSS_STARBOARD_WEAPON");
			if (p.starboardWeapon.equipmentKey === "EQ_WEAPON_NONE") p.removeEquipment("EQ_LMSS_STARBOARD_WEAPON");
			key = p.starboardWeapon.equipmentKey;
			p.starboardWeapon = this._starboardAltKey;
			this._starboardAltKey = key;
			break;
	}
	// switch breakable laser info, if present
	if (worldScripts["laserBooster_worldScript.js"]) worldScripts["laserBooster_worldScript.js"].$switchPosition(position);

	this._switching = false;

	this.$updateManifest();
}

//-------------------------------------------------------------------------------------------------------------
// stops any running timers
this.$stopTimers = function $stopTimers() {
	if (this._forwardTimer && this._forwardTimer.isRunning) this._forwardTimer.stop();
	if (this._aftTimer && this._aftTimer.isRunning) this._aftTimer.stop();
	if (this._portTimer && this._portTimer.isRunning) this._portTimer.stop();
	if (this._starboardTimer && this._starboardTimer.isRunning) this._starboardTimer.stop();
}

//-------------------------------------------------------------------------------------------------------------
// does player have the activator and a secondary laser in the forward mount?
this.$hasForwardLMSS = function $hasForwardLMSS() {
	var p = player.ship;
	if (p.equipmentStatus("EQ_LMSS_FRONT") === "EQUIPMENT_OK" &&
		p.equipmentStatus("EQ_LMSS_FRONT_WEAPON") === "EQUIPMENT_OK" &&
		this._incompatible.indexOf(p.forwardWeapon.equipmentKey) === -1) {
		return true;
	} else {
		return false;
	}
}

//-------------------------------------------------------------------------------------------------------------
// does player have the activator and a secondary laser in the aft mount?
this.$hasAftLMSS = function $hasAftLMSS() {
	var p = player.ship;
	if (p.equipmentStatus("EQ_LMSS_AFT") === "EQUIPMENT_OK" &&
		p.equipmentStatus("EQ_LMSS_AFT_WEAPON") === "EQUIPMENT_OK" &&
		this._incompatible.indexOf(p.aftWeapon.equipmentKey) === -1) {
		return true;
	} else {
		return false;
	}
}

//-------------------------------------------------------------------------------------------------------------
// does player have the activator and a secondary laser in the port mount?
this.$hasPortLMSS = function $hasPortLMSS() {
	var p = player.ship;
	if (p.equipmentStatus("EQ_LMSS_PORT") === "EQUIPMENT_OK" &&
		p.equipmentStatus("EQ_LMSS_PORT_WEAPON") === "EQUIPMENT_OK" &&
		this._incompatible.indexOf(p.portWeapon.equipmentKey) === -1) {
		return true;
	} else {
		return false;
	}
}

//-------------------------------------------------------------------------------------------------------------
// does player have the activator and a secondary laser in the starboard mount?
this.$hasStarboardLMSS = function $hasStarboardLMSS() {
	var p = player.ship;
	if (p.equipmentStatus("EQ_LMSS_STARBOARD") === "EQUIPMENT_OK" &&
		p.equipmentStatus("EQ_LMSS_STARBOARD_WEAPON") === "EQUIPMENT_OK" &&
		this._incompatible.indexOf(p.starboardWeapon.equipmentKey) === -1) {
		return true;
	} else {
		return false;
	}
}

//-------------------------------------------------------------------------------------------------------------
// start the process of switching the forward laser
this.$switchForward = function $switchForward() {
	var p = player.ship;
	// first, remove the existing laser and store a value so that we can put it back
	player.consoleMessage(expandDescription("[lmss_forward_deactivated]"));
	// put the key in the holding spot
	this._forwardHoldKey = p.forwardWeapon.equipmentKey;
	p.forwardWeapon = "EQ_WEAPON_NONE";
	this.$updateManifest();
	// start a timer to load up the secondary laser
	this._forwardTimer = new Timer(this, this.$installLaserForward, this._defaultDelay, 0);
	this.$playSound();
}

//-------------------------------------------------------------------------------------------------------------
// start the process of switching the aft laser
this.$switchAft = function $switchAft() {
	var p = player.ship;
	// first, remove the existing laser and store a value so that we can put it back
	player.consoleMessage(expandDescription("[lmss_aft_deactivated]"));
	// put the key in the holding spot
	this._aftHoldKey = p.aftWeapon.equipmentKey;
	p.aftWeapon = "EQ_WEAPON_NONE";
	this.$updateManifest();
	// start a timer to load up the secondary laser
	this._aftTimer = new Timer(this, this.$installLaserAft, this._defaultDelay, 0);
	this.$playSound();
}

//-------------------------------------------------------------------------------------------------------------
// start the process of switching the port laser
this.$switchPort = function $switchPort() {
	var p = player.ship;
	// first, remove the existing laser and store a value so that we can put it back
	player.consoleMessage(expandDescription("[lmss_port_deactivated]"));
	// put the key in the holding spot
	this._portHoldKey = p.portWeapon.equipmentKey;
	p.portWeapon = "EQ_WEAPON_NONE";
	this.$updateManifest();
	// start a timer to load up the secondary laser
	this._portTimer = new Timer(this, this.$installLaserPort, this._defaultDelay, 0);
	this.$playSound();
}

//-------------------------------------------------------------------------------------------------------------
// start the process of switching the starboard laser
this.$switchStarboard = function $switchStarboard() {
	var p = player.ship;
	// first, remove the existing laser and store a value so that we can put it back
	player.consoleMessage(expandDescription("[lmss_starboard_deactivated]"));
	// put the key in the holding spot
	this._starboardHoldKey = p.starboardWeapon.equipmentKey;
	p.starboardWeapon = "EQ_WEAPON_NONE";
	this.$updateManifest();
	// start a timer to load up the secondary laser
	this._starboardTimer = new Timer(this, this.$installLaserStarboard, this._defaultDelay, 0);
	this.$playSound();
}

//-------------------------------------------------------------------------------------------------------------
// end points for the timers so we can call one function
this.$installLaserForward = function $installLaserForward() {
	this.$installLaser("FORWARD");
}
this.$installLaserAft = function $installLaserAft() {
	this.$installLaser("AFT");
}
this.$installLaserPort = function $installLaserPort() {
	this.$installLaser("PORT");
}
this.$installLaserStarboard = function $installLaserStarboard() {
	this.$installLaser("STARBOARD");
}

//-------------------------------------------------------------------------------------------------------------
// install the lasers after the appropriate delay
this.$installLaser = function $installLaser(position) {
	var p = player.ship;

	this._switching = true;

	// get the equipment key of the original laser
	switch (position) {
		case "FORWARD":
			p.forwardWeapon = this._forwardAltKey;
			this._forwardAltKey = this._forwardHoldKey;
			this._forwardHoldKey = "";
			if (this._forwardAltKey === "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_FRONT_WEAPON");
			}
			break;
		case "AFT":
			p.aftWeapon = this._aftAltKey;
			this._aftAltKey = this._aftHoldKey;
			this._aftHoldKey = "";
			if (this._aftAltKey === "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_AFT_WEAPON");
			}
			break;
		case "PORT":
			p.portWeapon = this._portAltKey;
			this._portAltKey = this._portHoldKey;
			this._portHoldKey = "";
			if (this._portAltKey === "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_PORT_WEAPON");
			}
			break;
		case "STARBOARD":
			p.starboardWeapon = this._starboardAltKey;
			this._starboardAltKey = this._starboardHoldKey;
			this._starboardHoldKey = "";
			if (this._starboardAltKey === "EQ_WEAPON_NONE") {
				p.removeEquipment("EQ_LMSS_STARBOARD_WEAPON");
			}
			break;
	}
	// switch breakable laser info, if present
	if (worldScripts["laserBooster_worldScript.js"]) worldScripts["laserBooster_worldScript.js"].$switchPosition(position);

	this._switching = false;
	player.consoleMessage(position + " laser switched");
	this.$updateManifest();
}

//-------------------------------------------------------------------------------------------------------------
// plays the hydraulic sound of the lasers being switched
this.$playSound = function $playSound() {
	var mySound = new SoundSource;
	if (this._defaultDelay === 6) {
		mySound.sound = "actuator-6.ogg";
	} else {
		mySound.sound = "actuator-10.ogg";
	}
	mySound.loop = false;
	mySound.play();
}

//-------------------------------------------------------------------------------------------------------------
// updates the crosshairs on Xenon HUD (which uses images rather than standard crosshairs
this.$updateXenonHUD = function $updateXenonHUD() {
	var p = player.ship;
	var xHUDList = ["XenonHUD.plist", "XenonHUD_hicontrast.plist", "XenonHUD_alt.plist", "XenonHUD_narrow.plist", "XenonHUD_narrow_hicontrast.plist",
		"XenonHUD_amber.plist", "XenonHUD_hicontrast_amber.plist", "XenonHUD_alt_amber.plist", "XenonHUD_narrow_amber.plist", "XenonHUD_narrow_hicontrast_amber.plist"];

	if (p.status === "STATUS_IN_FLIGHT" && xHUDList.indexOf(p.hud) >= 0) {
		var w = worldScripts.XenonHUD;
		w.viewDirectionChanged(p.viewDirection);
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns the name of a particular equipment item
this.$getEquipmentName = function $getEquipmentName(equipment, position, isPrimary) {
	var laser = EquipmentInfo.infoForKey(equipment);
	var lb = worldScripts["laserBooster_worldScript.js"];
	var extra = "";
	if (lb && equipment == lb._damagedEq) {
		if (isPrimary) {
			laser = EquipmentInfo.infoForKey(lb._holdDamage[position].primary);
		} else {
			laser = EquipmentInfo.infoForKey(lb._holdDamage[position].secondary);
		}
		extra = " " + expandDescription("[lmss_damaged]");
	}
	if (!laser || laser.equipmentKey === "EQ_WEAPON_NONE") {
		return expandDescription("[lmss_none]");
	} else {
		return laser.name + extra;
	}
}

//-------------------------------------------------------------------------------------------------------------
// returns a JSON string of all settings (for Ship Storage Helper)
this.$retrieveLaserSettings = function $retrieveLaserSettings() {
	var ret = [];
	ret.push({
		position: "forward",
		key: this._forwardAltKey
	});
	ret.push({
		position: "aft",
		key: this._aftAltKey
	});
	ret.push({
		position: "port",
		key: this._portAltKey
	});
	ret.push({
		position: "starboard",
		key: this._starboardAltKey
	});
	return JSON.stringify(ret);
}

//-------------------------------------------------------------------------------------------------------------
// parses a JSON string and reads values from the array into all settings (for Ship Storage Helper)
this.$restoreLaserSettings = function $restoreLaserSettings(dataString) {
	var arr = JSON.parse(dataString);

	// reset the keys
	this._forwardAltKey = "EQ_WEAPON_NONE";
	this._aftAltKey = "EQ_WEAPON_NONE";
	this._portAltKey = "EQ_WEAPON_NONE";
	this._starboardAltKey = "EQ_WEAPON_NONE";

	// load values from the array
	for (var i = 0; i < arr.length; i++) {
		switch (arr[i].position) {
			case "forward":
				this._forwardAltKey = arr[i].key;
				break;
			case "aft":
				this._aftAltKey = arr[i].key;
				break;
			case "port":
				this._portAltKey = arr[i].key;
				break;
			case "starboard":
				this._starboardAltKey = arr[i].key;
				break;
		}
	}
	this.$resetInterface();
}

//-------------------------------------------------------------------------------------------------------------
// returns true if there is a secondary laser in any position, otherwise false
this.$weaponInstalled = function $weaponInstalled() {
	if (this.$hasForwardLMSS() || this.$hasAftLMSS() || this.$hasPortLMSS() || this.$hasStarboardLMSS()) return true;
	return false;
}
