
"use strict";
this.name = "in-system_taxi";
this.description = "In-system passenger contracts";

/*
	These two function calls override the properties of the player.ship so that other OXP's will get the 
	correct number of passengers and capacity
	However, if an OXP uses the player.ship.passengers array, the in-system taxi passenger will be missing
*/
//-------------------------------------------------------------------------------------------------------------
(function substitutePlayerShipPassengerCount() {
	Object.defineProperty(player.ship, "passengerCount", {
		get: function() {
			return this.passengers.length + (this.equipmentStatus("EQ_TAXI_PASSENGER_BERTH") == "EQUIPMENT_OK" ? 1 : 0);
		}
	});    	
}).call(this);

//-------------------------------------------------------------------------------------------------------------
(function substitutePlayerShipPassengerCapacity() {
	Object.defineProperty(player.ship, "passengerCapacity", {
		get: function() {
			var hold = false;
			if (worldScripts.GalCopAdminServices) {
				hold = worldScripts.GalCopAdminServices._disableContracts;
				worldScripts.GalCopAdminServices._disableContracts = true;
			}
			var count = 0;
			do {
				var result = this.addPassenger("temp passenger " + count.toString(), 0, 0, clock.seconds + 3600, 0);
				count += 1;
			} while (result == true);
			var berths = count - 1;
			count = 0;
			do {
				var result = this.removePassenger("temp passenger " + count.toString(), 0, 0, clock.seconds + 3600, 0);
				count += 1;
			} while (result == true);
			if (worldScripts.GalCopAdminServices) {
				worldScripts.GalCopAdminServices._disableContracts = hold;
			}
			return berths + (this.equipmentStatus("EQ_TAXI_PASSENGER_BERTH") == "EQUIPMENT_OK" ? 1 : 0);
		}
	});
}).call(this);

this.startUp = function () {
	// USER TOGGLEABLE VARIABLES START
	this.$availableOnLaunch = true;
	// true sets taxi to available on launch, false sets taxi to unavailable on launch
	this.$maxDistance = 1000000; //maximum distance of pickup point (in meters)

	// USER TOGGLEABLE VARIABLES END

	// this.$currentOffer = [offer_status, from_name, to_name, deadline, base_fare, passenger_name, system.ID, whole_time]
	// status = 0 no offer or offer declined
	// status = 1 offer waiting for approval
	// status = 2 offer accepted
	// status = 3 passenger on board
	this.$currentOffer = [0];
	if (missionVariables.inSystemTaxi != null && player.ship.equipmentStatus("EQ_TAXI_PASSENGER_BERTH") == "EQUIPMENT_OK") {//remove possible passenger
		this.$restorePassengerCabin();
	}
	this.$deleteMissionInstructions() // remove possible mission instructions
	this.$stationOffers = []; // array for station offers
	this.$fireMissionScreen = 0; // flag for selecting different mission screens
	this.$readyToOffer = false; // start offering only after first launch
	this.$refTime = clock.seconds; // time keeping for refreshing station contracts
	this.$addedBeacon = [false, false]; // flags for temporary beacons
	this.$premium = 0; // needed in different functions, thus global variable
	this.$offerFrequency = 90; // frequency at which new in-flight offers are generated and possibly offered. min 60
	this.$taxiAvailable = this.$availableOnLaunch; // status of availability
}

// fare, based on distance
this.$fare = function (distance) {
	return Math.floor(5 + 0.000005 * distance); // 5 cr + 0.5 cr/100 km
}

this.playerWillSaveGame = function () {
	if (this.$currentOffer[0] == 3) { // mark in-system passenger to be removed on load
		var extra = this.$getExtraInfo(null, this.$currentOffer[2]);
		missionVariables.inSystemTaxi = expandDescription("[taxi_passenger]", { name: this.$currentOffer[5], dest: this.$currentOffer[2].displayName + extra.to });
	} else
		missionVariables.inSystemTaxi_current = null;
}

// Taxi Comms comes through this function
this.$commsMessage = function (message) {
	var notifySound = new SoundSource;
	notifySound.sound = "beep.ogg";
	notifySound.play();
	player.commsMessage(expandDescription("[taxi_message]", { message: message }), 6);
}

// find suitable stations
this.$listOfStations = function () {
	function $isAllowed(entity) {
		if (entity.scanClass == "CLASS_STATION") {
			// excludes that have scanClass CLASS_STATION
			if (entity.hasRole("gates_jumpGate")) return false;
			if (entity.hasRole("fuelStation_satellite")) return false;
			if (entity.hasRole("fuelStation_station")) return false;
			if (entity.hasRole("jaguar_company_base")) return false;
			return true;
		}
		if (entity.isStation) {
			// includes that have isStation but not scanClass CLASS_STATION
			if (entity.hasRole("liners_liner")) return true;
			if (entity.hasRole("casinoship")) return true;
			if (entity.hasRole("random_hits_any_spacebar")) return true;
			if (entity.hasRole("planetFall2_surface")) return true; // for planetfall v2
		}
		if (entity.isPlanet && entity.displayName != null) {
			// planetfall (v1) && planetary compass
			if (worldScripts.PlanetFall) {
				if (!entity.isSun && !entity.hasOwnProperty("solarGasGiant") && !entity.hasOwnProperty("isGasGiant") && !entity.hasOwnProperty("PFNoLand") && !entity.hasOwnProperty("PFNoLandQuiet") && player.ship.equipmentStatus("EQ_PLANETFALL") == "EQUIPMENT_OK") return true;
			}
		}
		return false;
	}
	return system.filteredEntities(this, $isAllowed);
}

// IN-FLIGHT OFFERING FUNCTIONS START HERE

// set taxi to not available on condition red
this.alertConditionChanged = function (newCondition, oldCondition) {
	if (newCondition == 3) {
		//worldScripts["in-system_taxi"].$commsMessage("Taxi Comms switched off");
		this.$stopInFlightTimer();
	}
}

// start in-flight offer timer
this.$startOfferTimer = this.playerCancelledJumpCountdown = function () {
	// show in-flight contract offers
	if (this.$taxiAvailable && this.$currentOffer[0] < 2 && (player.ship.passengerCapacity - player.ship.passengerCount) > 0 && player.ship.equipmentStatus("EQ_IN_SYSTEM_TAXI_LICENCE") == "EQUIPMENT_OK") {
		this.$startInFlightTimer();
		return true;
	}
	return false;
}

// start in-flight offer timer on launch only if available on launch is set.
this.shipLaunchedFromStation = function (station) {
	if (this.$availableOnLaunch)
		this.$taxiAvailable = true;
	this.$startOfferTimer();
}

// stop in-flight offer timer
this.playerStartedJumpCountdown = function () {
	this.$stopInFlightTimer();
}

// remove in-system contract when exiting system
this.shipExitedWitchspace = function () {
	if (this.$currentOffer[0] == 3) { 
		var message = expandDescription("[taxi_passenger_disappointed]", { name: this.$currentOffer[5], sysname: System.infoForSystem(system.info.galaxyID, this.$currentOffer[6]).name });
		var delayedMessage = new Timer(this, function () { this.$commsMessage(message) }, 10);
	}
	if (this.$availableOnLaunch)
		this.$taxiAvailable = true;
	this.$endContract();
}

//create new in-flight offers
this.$taxiOffer = function () {
	if (system.isInterstellarSpace) return; // sanity check;
	var listOfStations = this.$listOfStations();
	var probability = 0.05 * listOfStations.length; // 0.05 * number of stations => 0.10 for 2 stations ... 1 for 20 stations
	if (Math.random() < probability && listOfStations.length > 1) {
		var to = listOfStations.splice(Math.floor(Math.random() * listOfStations.length), 1)[0];
		var from = to;
		var to_check = to;
		if (to.hasRole("planetFall2_surface")) to_check = to.linkedVE;
		var from_check = from;

		var found = false;
		while (listOfStations.length > 0 && !found) { // select suitable pickup point
			from = listOfStations.splice(Math.floor(Math.random() * listOfStations.length), 1)[0];
			from_check = from;
			from_extra = "";
			if (from.hasRole("planetFall2_surface")) from_check = from.linkedVE;
			if (player.ship.position.distanceTo(from_check) < this.$maxDistance)
				found = true;
		}
		if (!found) return; // no suitable pickup points
		if (to.isPlanet && from.isPlanet)
			var distance = from.position.distanceTo(to) - to.radius - from.radius;
		else if (to.isPlanet)
			var distance = from.position.distanceTo(to) - to.radius;
		else if (from.isPlanet)
			var distance = from.position.distanceTo(to) - from.radius;
		else
			var distance = from_check.position.distanceTo(to_check);

		var extra = this.$getExtraInfo(from, to);

		var baseTime = 0.0002 * distance + 1200; // 20 secs per 100 km + 20 mins for reach, launch and incidents.
		var time = clock.seconds + baseTime;
		var fee = this.$fare(distance);
		var name = global.randomName() + " " + global.randomName();
		this.$commsMessage(expandDescription("[taxi_new_passenger]", { name: name, from: from.displayName + extra.from, to: to.displayName + extra.to, time: Math.floor((time - clock.seconds) / 60), fare: formatCredits(fee, true, true) }));
		this.$removeOfferTimer();
		this.$offerTimer = new Timer(this, function () { this.$currentOffer = [0]; }, 60);//offer ends in 60 secs
		// time contract, so it can be removed if deadline is met before picking the passenger up.
		this.$removeContractTimer();
		this.$contractTimer = new Timer(this, this.$endContract, time - clock.seconds);
		this.$currentOffer = [1, from, to, time, fee, name, system.ID, baseTime];
	}
}

// accept offer
this.$acceptOffer = function () {
	this.$stopInFlightTimer();
	this.$removeOfferTimer();
	this.$currentOffer[0] = 2;
	this.$addMissionInstructions();
	var extra = this.$getExtraInfo(this.$currentOffer[1])
	this.$commsMessage(expandDescription("[taxi_accepted]", { location: this.$currentOffer[1].displayName + extra.from }));
	this.$addBeacon();
}

// cancel contract
this.$cancelContract = function () {
	this.$endContract();
	this.$commsMessage(expandDescription("[taxi_cancelled]"));
}

// cancel contract before picking passenger up.
this.$endContract = function () {
	if (this.$currentOffer[0] < 3) { // passenger not on board yet.
		this.$removeBeacon();
		this.$deleteMissionInstructions();
		this.$currentOffer = [0];
		this.$removeContractTimer();
		this.$startOfferTimer();
	}
}

// add mission instructions when accepting contract
this.$addMissionInstructions = function () {
	var deadline = new Date(0, 0, 0, 0, 0, this.$currentOffer[3]);
	var extra = this.$getExtraInfo(this.$currentOffer[1], this.$currentOffer[2]);
	var instructions = [expandDescription("[taxi_instructions_header]")];
	if (system.ID != this.$currentOffer[6]) {
		var passngr = this.$splitLines(expandDescription("[taxi_passenger]", {name: this.$currentOffer[5], dest: System.systemNameForID(this.$currentOffer[6])}));
	} else {
		var passngr = this.$splitLines(expandDescription("[taxi_instructions]", { name: this.$currentOffer[5], from: this.$currentOffer[1].displayName + extra.from, dest: this.$currentOffer[2].displayName + extra.to, deadline: deadline.toLocaleTimeString() }));
	}
	instructions = instructions.concat(passngr);
	mission.setInstructions(instructions, this.name);
}

// splits a single line into an array of lines that fit the screen bounds.
this.$splitLines = function(text) {
	var words = text.split(" ");
	var result = [];
	var line = "";
	for (var i = 0; i < words.length; i++) {
		if (defaultFont.measureString(line + " " + words[i]) > 30.8) {
			result.push(line);
			line = "  ";
		}
		line += (line == "" ? "" : " ") + words[i];
	}
	result.push(line);
	return result;
}

this.$getExtraInfo = function (from, to) {
	var result = { from: "", to: "" };
	if (from && from.hasRole("planetFall2_surface")) result.from = expandDescription("[taxi_on_planet]", { planet: this.$getPlanetNameForEntity(from) });
	if (to && to.hasRole("planetFall2_surface")) result.to = expandDescription("[taxi_on_planet]", { planet: this.$getPlanetNameForEntity(to) });
	return result;
}

// delete mission instructions when cancelling contract
this.$deleteMissionInstructions = function () {
	mission.setInstructions(null, this.name);
}

// IN-FLIGHT OFFERING FUNCTIONS END HERE

// STATION OFFERING FUNCTIONS START HERE

// reate new station offers
this.$createStationOffers = function () {
	if (system.isInterstellarSpace) return; // sanity check;
	var i;
	var listOfStations = this.$listOfStations();
	for (i = 0; i < listOfStations.length; i++) {
		if (listOfStations[i] != player.ship.dockedStation && Math.random() < 0.1) { // contract probability of 0.1 per station in system.
			var from = player.ship.dockedStation;
			var from_check = from;
			if (from.hasRole("planetFall2_surface")) from_check = from.linkedVE;
			var to = listOfStations[i];
			var to_check = to;
			if (to.hasRole("planetFall2_surface")) to_check = to.linkedVE;
			if (to.isplanet)
				var distance = from_check.position.distanceTo(to_check) - to.radius;
			else
				var distance = from_check.position.distanceTo(to_check);

			var baseTime = 0.0002 * distance + 900; // 20 secs per 100 km + 15 mins for launch and incidents.			
			var time = clock.seconds + baseTime;
			var fee = this.$fare(distance);
			var name = global.randomName() + " " + global.randomName();
			this.$stationOffers.push([1, from, to, time, fee, name, system.ID, baseTime]);
			if (this.$stationOffers.length == 8) break; // cap possible offers to 8
		}
	}
}

// create station offers when docking
this.shipDockedWithStation = function (station) {
	this.$stationOffers = [];
	this.$createStationOffers();
	this.$stationOffersInterface();
	this.$refTime = clock.seconds;
}

// handle f4-screen
this.$stationOffersInterface = function () {
	// show interface if no contract accepted, no room or no offers
	if (this.$readyToOffer && this.$stationOffers.length > 0 && this.$currentOffer[0] < 2 && player.ship.passengerCapacity - player.ship.passengerCount > 0) {
		player.ship.dockedStation.setInterface("in-system_taxi", {
			title: expandDescription("[taxi_interface_title]", { avail: this.$stationOffers.length }),
			category: expandDescription("[interfaces-category-deliveries]"),
			summary: expandDescription("[taxi_interface_summary]"),
			callback: this.$showStationOffers.bind(this)
		});
	}
	else {
		player.ship.dockedStation.setInterface("in-system_taxi", null); // remove interface if contract accepted, no room or no offers
	}
}

// clean old offers and add new offers time to time
this.guiScreenChanged = function (to, from) {
	var p = player.ship;
	if (p.docked && !system.isInterstellarSpace && clock.seconds - this.$refTime > 600) { // refresh every 10 mins of game time
		this.$createStationOffers();
		this.$clearOldOffers();
		this.$refTime = clock.seconds;
	}
}

this.$clearOldOffers = function () {
	var i;
	var validOffers = [];
	for (i = 0; i < this.$stationOffers.length; i++) {
		// leave only ones that have at least 5 mins of fly time left, 10 goes to launching
		if ((Math.floor((this.$stationOffers[i][3] - clock.seconds))) > 900) {
			validOffers.push(this.$stationOffers[i]);
		}
	}
	this.$stationOffers = validOffers;
	this.$stationOffersInterface();
}

// mission screen for showing offers
this.$showStationOffers = function () {
	var options = {};
	var emptyMessage = "";
	this.$clearOldOffers();
	if (this.$stationOffers.length == 0) {
		emptyMessage = expandDescription("[taxi_too_late]");
	}
	else {
		var i;
		for (i = 0; i < this.$stationOffers.length; i++) {
			var extra = this.$getExtraInfo(null, this.$stationOffers[i][2]);
			options[i + "_TAXI"] = this.$stationOffers[i][5] + " to " + this.$stationOffers[i][2].displayName + extra.to + " in " + Math.floor((this.$stationOffers[i][3] - clock.seconds) / 60) + " minutes. Fare: " + formatCredits(this.$stationOffers[i][4], true, true);
		}
	}
	options["99_NO_THANKS"] = expandDescription("[taxi_menu_exit]");
	mission.runScreen({
		title: expandDescription("[taxi_screen_title]"),
		message: expandDescription("[taxi_screen_text]") + emptyMessage,
		choices: options,
		exitScreen: "GUI_SCREEN_INTERFACES",
		overlay: { name: "tg_is_overlay.png", height: 546 }
	},
		function (choice) {
			if (!choice || choice == "99_NO_THANKS") return;
			else {
				var ws = worldScripts["in-system_taxi"];
				ws.$removeOfferTimer(); // in case there is an offerTimer running from in-flight offers.
				ws.$currentOffer = ws.$stationOffers[choice.charAt(0)];
				ws.$currentOffer[0] = 2;
				player.ship.dockedStation.setInterface("in-system_taxi", null); // remove station interface after accepting contract.
			}
		});
}

// STATION OFFERING FUNCTIONS END HERE

// CONTRACT HANDLING

// add temporary beacons
this.$addBeacon = function () {
	// add temporary from beacon
	if (this.$currentOffer[0] == 2 && !this.$currentOffer[1].beaconCode) {
		this.$currentOffer[1].beaconCode = expandDescription("[taxi_beacon]");
		this.$addedBeacon[0] = true;
	}
	// add temporary to beacon
	if (this.$currentOffer[0] == 3 && !this.$currentOffer[2].beaconCode) {
		this.$currentOffer[2].beaconCode = expandDescription("[taxi_beacon]");
		this.$addedBeacon[1] = true;
	}
}

// remove temporary beacons
this.$removeBeacon = function () {
	// remove temporary from beacon
	if (this.$addedBeacon[0]) {
		this.$currentOffer[1].beaconCode = null;
		this.$addedBeacon[0] = false;
	}
	// remove temporary to beacon
	if (this.$addedBeacon[1]) {
		this.$currentOffer[2].beaconCode = null;
		this.$addedBeacon[1] = false;
	}
}

this.shipWillLaunchFromStation = function (station) {
	this.$readyToOffer = true; // after frist launch, allow offering
	this.$addBeacon(); // add temporary beacons
}

this.$restorePassengerCabin = function() {
	var p = player.ship;
	this.$deleteMissionInstructions();
	p.removeEquipment("EQ_TAXI_PASSENGER_BERTH");
	p.awardEquipment("EQ_PASSENGER_BERTH");
}

this.$replacePassengerCabin = function() {
	var p = player.ship;
	this.$addMissionInstructions();
	p.removeEquipment("EQ_PASSENGER_BERTH");
	p.awardEquipment("EQ_TAXI_PASSENGER_BERTH");
}

// handle passenger when docking
this.shipWillDockWithStation = function (station) {
	this.$removeBeacon(); // remove temporary beacons
	this.$stopInFlightTimer(); // stop in-flight offering
	// is there a passenger on board?
	if (this.$currentOffer[0] == 3) {
		// over 15 minutes (15*60 = 900 secs) late, cancel contract if possible.
		if (this.$currentOffer[3] - clock.seconds < -900) {
			if (station != this.$currentOffer[2]) {
				this.$fireMissionScreen = 6;
				this.$restorePassengerCabin();
				this.$currentOffer[0] = 0;
			}
			// if we're already at home, no switching is required, just fire mission screen
			else {
				this.$fireMissionScreen = 5;
				this.$restorePassengerCabin();
				this.$currentOffer[0] = 0;
			}
			return;
		}
		// destination check
		var rightDestination = false;
		if (this.$currentOffer[2].isPlanet) {
			if (worldScripts.PlanetFall.lastPlanet == this.$currentOffer[2] && (station.hasRole("planetFall_surface")))
				rightDestination = true;
		}
		else if (station == this.$currentOffer[2])
			rightDestination = true;
		// right destination, 0 - 15 mins late, remove passenger, pay fare
		if (this.$currentOffer[3] - clock.seconds < 0 && rightDestination) {
			player.credits = player.credits + this.$currentOffer[4];
			this.$fireMissionScreen = 4;
			this.$restorePassengerCabin();
			this.$currentOffer[0] = 0;
			return;
		}
		// right destination in time, remove passenger, pay fare and possibly premium
		if (rightDestination) {
			this.$premium = 30 * ((this.$currentOffer[3] - clock.seconds) / this.$currentOffer[7]); // tip 30*proportion
			player.credits = player.credits + this.$currentOffer[4] + this.$premium;
			this.$fireMissionScreen = 2;
			this.$restorePassengerCabin();
			this.$currentOffer[0] = 0;
			return;
		}
	}
}


// handle unhiding the passenger and passenger picking. show mission screens, when there's something to show.
this.missionScreenOpportunity = function () {
	var extra = this.$getExtraInfo(this.$currentOffer[1], this.$currentOffer[2]);
	// If we are at the right starting point, let's pick the passenger on board.
	if (player.ship.docked && this.$currentOffer[0] == 2) {
		var rightStartingPoint = false;
		if (this.$currentOffer[1].isPlanet) { // is it a planetary starting point?
			if (worldScripts.PlanetFall.lastPlanet == this.$currentOffer[1] && (player.ship.dockedStation.hasRole("planetFall_surface")))
				rightStartingPoint = true;
		}
		else if (player.ship.dockedStation == this.$currentOffer[1])
			rightStartingPoint = true;
		if (rightStartingPoint) { // right starting point
			this.$deleteMissionInstructions();
			// is there room for the passenger?
			if (player.ship.passengerCapacity - player.ship.passengerCount == 0) {
				this.$fireMissionScreen = 3;
			}
			// all is probably well, let's add new contract
			else {
				this.$replacePassengerCabin();
				this.$currentOffer[0] = 3;
				this.$currentOffer[7] = this.$currentOffer[3] - clock.seconds; // premium is based on basetime. basetime is defined when picking up passenger.
				this.$fireMissionScreen = 1;
			}
		}
	}
	switch (this.$fireMissionScreen) {
		case 1:
			mission.runScreen({
				title: expandDescription("[taxi_passenger_boarding]"),
				message: expandDescription("[taxi_passenger_boarding_detail]", { name: this.$currentOffer[5], dest: this.$currentOffer[2].displayName + extra.to, time: Math.floor((this.$currentOffer[3] - clock.seconds) / 60), fare: formatCredits(this.$currentOffer[4], true, true) }),
				overlay: { name: "tg_is_overlay.png", height: 546 }
			});
			this.$fireMissionScreen = 0;
			break;
		case 2:
			var tipTalk = "";
			if (this.$premium > 0.05)
				tipTalk = expandDescription("[taxi_delivery_tip]", { premium: formatCredits(this.$premium, true, true) });
			mission.runScreen({
				title: expandDescription("[taxi_passenger_success]"),
				message: expandDescription("[taxi_passenger_success_detail]", { name: this.$currentOffer[5], fare: formatCredits(this.$currentOffer[4], true, true) }) + tipTalk,
				overlay: { name: "tg_is_overlay.png", height: 546 }
			});
			player.increasePassengerReputation();
			this.$currentOffer = [0];
			this.$fireMissionScreen = 0;
			break;
		case 3:
			mission.runScreen({
				title: expandDescription("[taxi_no_room]"),
				message: expandDescription("[taxi_no_room_cancelled]", { name: this.$currentOffer[5] }),
				overlay: { name: "tg_is_overlay.png", height: 546 }
			});
			this.$currentOffer = [0];
			this.$fireMissionScreen = 0;
			break;
		case 4:
			mission.runScreen({
				title: expandDescription("[taxi_missed_deadline]"),
				message: expandDescription("[taxi_missed_deadline_detail]", { name: this.$currentOffer[5], fare: formatCredits(this.$currentOffer[4], true, true) }),
				overlay: { name: "tg_is_overlay.png", height: 546 }
			});
			player.decreasePassengerReputation();
			this.$currentOffer = [0];
			this.$fireMissionScreen = 0;
			break;
		case 5:
			mission.runScreen({
				title: expandDescription("[taxi_unhappy]"),
				message: expandDescription("[taxi_unhappy_detail]", { name: this.$currentOffer[5] }),
				overlay: { name: "tg_is_overlay.png", height: 546 }
			});
			player.decreasePassengerReputation();
			this.$currentOffer = [0];
			this.$fireMissionScreen = 0;
			break;
		case 6:
			mission.runScreen({
				title: expandDescription("[taxi_demanding]"),
				message: expandDescription("[taxi_demanding_detail]", { name: this.$currentOffer[5], sysname: System.infoForSystem(system.info.galaxyID, this.$currentOffer[6]).name }),
				overlay: { name: "tg_is_overlay.png", height: 546 }
			});
			this.$fireMissionScreen = 0;
			break;
	}
}

// Timer functions
this.$startInFlightTimer = function () {
	if (this.$showOfferTimer == null)
		this.$showOfferTimer = new Timer(this, this.$taxiOffer, 60, this.$offerFrequency);
	else this.$showOfferTimer.start();
};

this.$stopInFlightTimer = function () {
	if (this.$showOfferTimer != null)
		this.$showOfferTimer.stop();
};

this.$removeInFlightTimer = function () {
	if (this.$showOfferTimer != null) {
		this.$showOfferTimer.stop();
		delete this.$showOfferTimer;
	}
};

this.$removeOfferTimer = function () {
	if (this.$offerTimer != null) {
		this.$offerTimer.stop();
		delete this.$offerTimer;
	}
};

this.$removeContractTimer = function () {
	if (this.$contractTimer != null) {
		this.$contractTimer.stop();
		delete this.$contractTimer;
	}
};

this.playerBoughtEquipment = function (equipment) {
	if (equipment === "EQ_IN_SYSTEM_TAXI_LICENCE_REMOVAL") {
		player.ship.removeEquipment("EQ_IN_SYSTEM_TAXI_LICENCE");
		player.ship.removeEquipment("EQ_IN_SYSTEM_TAXI_LICENCE_REMOVAL");
	}
}

this.$getPlanetNameForEntity = function (station) {
	if (!station.linkedPlanetPos) return "";
	if (!worldScripts.PlanetFall2) return "";
	var name = worldScripts.PlanetFall2.$getPlanetFromCoords(station.linkedPlanetPos).displayName;
	if (name.indexOf("(") >= 0) {
		name = name.substring(0, name.indexOf("(")).trim();
	}
	return name;
}