// worldScripts["AI Trading Fallback Datalogger"]
"use strict";

this.name    = "AI Trading Fallback Datalogger";
this.version = "3.0";

/* ====================================================================================
            DYNAMIC COMMODITY DATA STRUCTURE
======================================================================================= */
this.$commodityData = {}; // Object to store all commodity data dynamically

/* ====================================================================================
            ENSURE COMMODITY DATA EXISTS
======================================================================================= */
this._ensureCommodityData = function(cargo) {
    if (!this.$commodityData[cargo]) {
        // Initialize data structure for new commodity
        this.$commodityData[cargo] = {
            ledger: 0,
            totalBought: 0,
            totalSold: 0,
            longRunBalance: 0,
            longRunBought: 0,
            longRunSold: 0
        };
        log(this.name, "New commodity detected: " + cargo);
    }
};

/* ====================================================================================
            CLOSE COMMODITY BOOKS
======================================================================================= */
this._closeCommodityBooks = function(cargo) {
    if (this.$commodityData[cargo]) {
        // Transfer to long-run accounts
        this.$commodityData[cargo].longRunBalance -= this.$commodityData[cargo].ledger;
        this.$commodityData[cargo].longRunBought += this.$commodityData[cargo].totalBought;
        this.$commodityData[cargo].longRunSold += this.$commodityData[cargo].totalSold;
        
        // Reset current ledger
        this.$commodityData[cargo].ledger = 0;
        this.$commodityData[cargo].totalBought = 0;
        this.$commodityData[cargo].totalSold = 0;
    }
};

/* ====================================================================================
            STARTUP INITIALIZATION
======================================================================================= */
this.startUp = function() {
    delete this.startUp;
    
    // Don't run if AI Trading Assistant exists
    if (worldScripts["AI Trading Assistant"]) {
        delete this.$commodityData;
        delete this.playerWillSaveGame;
        delete this.playerBoughtCargo;
        delete this.playerSoldCargo;
        return;
    }
    
    // Load saved data
    if (missionVariables.tradingAI_database) {
        try {
            this.$commodityData = JSON.parse(missionVariables.tradingAI_database);
            log(this.name, "AI Trading Fallback Datalogger loaded with " + 
                Object.keys(this.$commodityData).length + " commodities.");
        } catch (e) {
            log(this.name, "Error loading saved data: " + e.message);
            this.$commodityData = {};
        }
    } else {
        log(this.name, "AI Trading Fallback Datalogger initialized.");
    }
    
    player.consoleMessage("AI Trading Fallback Datalogger v" + this.version + " is working.", 10);
};

/* ====================================================================================
            SAVE GAME HANDLER
======================================================================================= */
this.playerWillSaveGame = function () {
    missionVariables.tradingAI_tradingData = JSON.stringify(this.$commodityData);
};

/* ====================================================================================
            PLAYER BUYING CARGO
======================================================================================= */
this.playerBoughtCargo = function(cargo, quantity, unitPrice) {
    this._ensureCommodityData(cargo);
    this.$commodityData[cargo].ledger += (unitPrice * quantity);
    this.$commodityData[cargo].totalBought += quantity;
};

/* ====================================================================================
            PLAYER SELLING CARGO
======================================================================================= */
this.playerSoldCargo = function(cargo, quantity, unitPrice) {
    this._ensureCommodityData(cargo);
    this.$commodityData[cargo].ledger -= (unitPrice * quantity);
    this.$commodityData[cargo].totalSold += quantity;
    
    // Check if we should close the books on this commodity
    if (!manifest[cargo] || manifest[cargo] === 0) {
        this._closeCommodityBooks(cargo);
    }
};