/*Javascript version of Oolite Galaxy code 
Copyright (C) 2023 Paul Cooper
Version 0.8.0

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
MA 02110-1301, USA.*/

var galArray = [];
var galOverlaps = [];

var lookupArrays = {
	Species: ["Rodents", "Frogs", "Lizards", "Lobsters", "Birds", "Humanoids", "Feline", "Insects", "Human Colonists"],
	Economies: ["Rich Industrial", "Average Industrial", "Poor Industrial", "Mainly Industrial", "Mainly Agricultural", "Rich Agricultural", "Average Agricultural", "Poor Agricultural"],
	Governments: ["Anarchy", "Feudal", "Multi-Government", "Dictatorship", "Communist", "Confederacy", "Democracy", "Corporate State"],
	"Tech Levels": [...Array(15).keys()].map(e => e + 1),
	Productivity: [1, 5000, 10000, 15000, 20000, 30000, 40000, 50000],
	Population: [0.1, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0],
	"Planet Radius": [2000, 3000, 4000, 5000, 6000, 7000]
}

 
function createGalaxyCoordsArray(galaxyNumber, galArray, galSeed)
{
	// This function orignal purpose is to create the list of coordinates for systems in the current galaxy.
	// Now extended to add other planet details to the array.

	var w0;
	var w1;
	var w2;

	if (galSeed === null || galSeed === undefined)
	{
		w0 = 23114;
		w1 = 584;
		w2 = 46931;
	}
	else
	{
		w0 = galSeed[0] + 256 * galSeed[1];
		w1 = galSeed[2] + 256 * galSeed[3];
		w2 = galSeed[4] + 256 * galSeed[5];
	}

	var new_w;
	var n1;
	var n2;
	var lo;
	var hi;

	if (galaxyNumber > 0)  //Rotate the seed to get seed for appropriate galaxy number
	{
		var p2 = Math.pow(2, 8 - galaxyNumber);
		var p3 = Math.pow(2, galaxyNumber);

		n1 = w0 % 256;
		n2 = Math.floor(w0/256);
		lo = n1 % p2;
		hi = Math.floor(n1 / p2);
		n1 = lo * p3 + hi;
		lo = n2 % p2;
		hi = Math.floor(n2 / p2);
		n2 = lo * p3 + hi;
		w0 = n2 * 256 + n1;

		n1 = w1 % 256;
		n2 = Math.floor(w1/256);
		lo = n1 % p2;
		hi = Math.floor(n1 / p2);
		n1 = lo * p3 + hi;
		lo = n2 % p2;
		hi = Math.floor(n2 / p2);
		n2 = lo * p3 + hi;
		w1 = n2 * 256 + n1;

		n1 = w2 % 256;
		n2 = Math.floor(w2/256);
		lo = n1 % p2;
		hi = Math.floor(n1 / p2);
		n1 = lo * p3 + hi;
		lo = n2 % p2;
		hi = Math.floor(n2 / p2);
		n2 = lo * p3 + hi;
		w2 = n2 * 256 + n1;

	}

	for (var z = 0; z <= 1020; z++)
	{
		if ((z % 4) == 0)
		{
			galArray.push([Math.floor(w1/256), Math.floor(w0/256)/2, ((DRAW3D) ? Math.floor(w2/256/4)/2 : 16) , null, 
				{
					name:getSystemName(w0, w1, w2),
					species: getSpecies(w0, w1, w2),
					government: getGovernment(w0, w1, w2),
					economy: getEconomy(w0, w1, w2),
					techlevel: getTechLevel(w0, w1, w2),
					population: getPop(w0, w1, w2),
					productivity: getProductivity(w0, w1, w2),
					radius: getRadius(w0, w1, w2),
					description: getDescription(w0, w1, w2)
				}]);
			//alert("End");
		}
		new_w = (w0 + w1 + w2) % 65536;
		w0 = w1;
		w1 = w2;
		w2 = new_w;		
	}
	
	systemLinks = findAllLinkedSystems(galArray);

	//reposition z axis values
	
	if (DRAW3D)
	{
		var count=0, old_count = -1;
		//showAlerts=false;
		while ((count < 256) || (count != old_count))
		{
			old_count = count;
			count = interateSystemsRepositioning(galArray, systemLinks);
		}
	}


	//find overlaps

	for (var idx = 0; idx < galArray.length; idx++)
	{
		var overlapList = checkForOverlapSystems(galArray, idx, galArray[idx][0], galArray[idx][1], galArray[idx][2]);
		if (overlapList.length > 0) galArray[idx][3] = overlapList;
	}

	//if (!DRAW3D)
	//{
		galOverlaps = dedupedOverlapsList(galArray);
		calcOverlapSepDirections();
	//}

	return;
}

function ooliteDist(x1, y1, z1, x2, y2, z2)
{
	return Math.floor(Math.sqrt((x2-x1) * (x2-x1) + Math.floor(Math.abs(y2-y1)) * Math.floor(Math.abs(y2-y1)) + Math.floor(Math.abs(z2-z1)) * Math.floor(Math.abs(z2-z1)))) * 0.4;
}

function distanceBetween(systemNumber1, systemNumber2, galArray)
{
	return ooliteDist(galArray[systemNumber1][0], galArray[systemNumber1][1], galArray[systemNumber1][2], galArray[systemNumber2][0], galArray[systemNumber2][1], galArray[systemNumber2][2]);
}

function findLinkedSystems(systemNumber, galArray)
{
	var list = [];
	for(var idx = 0; idx < 256; idx++)
	{
		if (idx != systemNumber)
		{
			if (ooliteDist(galArray[systemNumber][0], galArray[systemNumber][1], 0/*galArray[systemNumber][2]*/, galArray[idx][0], galArray[idx][1], 0/*galArray[idx][2]*/) <= 7) 
			{
				range = getRange(galArray[systemNumber][0], galArray[systemNumber][1], galArray[idx][0], galArray[idx][1]);
				list.push([idx, range[0], range[1]]);
			}
		}
	}

	return list;
}

function findAllLinkedSystems(galArray)
{
	var systemLinks = [];
	for(var idx = 0; idx < 256; idx++)
	{
		systemLinks.push(findLinkedSystems(idx, galArray));
	}

	return systemLinks;
}

function getRange(x1, y1, x2, y2)
{
	var min_z = 0;
	var max_z = 0;

	for (var z = -8; z <= 8; z+=0.5)
	{
		if (ooliteDist(x1, y1, 0, x2, y2, z) <= 7)
		{
			if (z < min_z) min_z = z;
			if (z > max_z) max_z = z;
		}
	}
	return [min_z, max_z];
}

function interateSystemsRepositioning(galArray, systemLinks) //Z repositioning.
{
	var count = 0;
	for (var idx = 0; idx < 256; idx++)
	{
		var min_z;
		var max_z;
		for(var s = 0; s < systemLinks[idx].length; s++)
		{
			var min_r = systemLinks[idx][s][1] + galArray[systemLinks[idx][s][0]][2]; //min range + z
			var max_r = systemLinks[idx][s][2] + galArray[systemLinks[idx][s][0]][2]; //max range + z
			if (s == 0)
			{
				min_z = min_r;
				max_z = max_r;
			}
			else
			{
				if (min_r > min_z) min_z = min_r;
				if (max_r < max_z) max_z = max_r;
			}
			
		}

		var flag = false;
		var old_z = galArray[idx][2];
		if (min_z <= max_z) 
		{
			if ((min_z <= galArray[idx][2]) && (galArray[idx][2] <= max_z)) {count++; flag=true;}
			else if (galArray[idx][2] < min_z) galArray[idx][2] = min_z;
			else galArray[idx][2] = max_z;
		}
		else 
		{
			if (galArray[idx][2] < max_z) galArray[idx][2] = max_z;
			else if (min_z < galArray[idx][2]) galArray[idx][2] = min_z;
			else if (galArray[idx][2] = max_z) galArray[idx][2] = min_z;
			else if (min_z = galArray[idx][2]) galArray[idx][2] = max_z;
		}
		if (galArray[idx][3] !== null) // check if this is a system that shares an x and y with another system and check if the z values are the same and if so tweek position.
		{
			for(var o = 0; o < galArray[idx][3].length; o++)
			{
				if (galArray[idx][2] == galArray[galArray[idx][3][o]][2])
				{
					if (old_z < (galArray[idx][2]-0.5)) galArray[idx][2] -= 0.5;
					else if (old_z == (galArray[idx][2]-0.5)) galArray[idx][2] += 0.5;
					else if (old_z > (galArray[idx][2]+0.5)) galArray[idx][2] += 0.5;
					else galArray[idx][2] -= 0.5;
				}
			}
		}
		if ((!flag) && showAlerts) alert("idx=" + idx + " min=" + min_z + " max=" + max_z + " old=" + old_z + " new=" + galArray[idx][2]);
	}
	return count;
}

function getSystemName(w0, w1, w2)
{
	//setup static variable
	if (typeof getSystemName.digrams == 'undefined') getSystemName.digrams = "@@LEXEGEZACEBISOUSESARMAINDIREA'ERATENBERALAVETIEDORQUANTEISRION";

	var w3 = (w0 + w1 + w2) % 65536;
	var w4 = (w1 + w2 + w3) % 65536;
	var w5 = (w2 + w3 + w4) % 65536;
	
	var str = getSystemName.digrams.substr(((w2>>>8)&31)*2, 2);
	str += getSystemName.digrams.substr(((w3>>>8)&31) * 2, 2);
	str += getSystemName.digrams.substr(((w4>>>8)&31) * 2, 2);
	str += getSystemName.digrams.substr(((w5>>>8)&31) * 2, 2);
	str = str.substr(0, ((w0&64)>>>5)+6);
	return stringReplaceAll(stringReplaceAll(str, "@", ""), "'", "");
}

function getSpecies(w0, w1, w2)
{
	var species = "";

	if ((w2&128) == 0) species = "Human Colonists";
	else
	{
		switch ((w2>>>10)&7)
		{
			case 0: 
				species += "Large";
				break;
			case 1:
				species += "Fierce";
				break;
	 		case 2: 
				species += "Small";
				break;
		}
		if (species.length > 0) species += " ";
		switch ((w2>>>13)&7)
		{
			case 0: 
				species += "Green";
				break;
			case 1: 
				species += "Red";
				break;
			case 2: 
				species += "Yellow";
				break;
			case 3: 
				species += "Blue";
				break;
			case 4: 
				species += "Black";
				break;
			case 5: 
				species += "Harmless";
				break;
		}
		if ((species.length > 0) && (species.substr(species.length -1 , 1) != " ")) species += " ";
		switch (((w0>>>8)&7) ^ ((w1>>>8)&7))
		{
			case 0: 
				species += "Slimy";
				break;
			case 1: 
				species += "Bug-Eyed";
				break;
			case 2: 
				species += "Horned";
				break;
			case 3: 
				species += "Bony";
				break;
			case 4: 
				species += "Fat";
				break;
			case 5: 
				species += "Furry";
				break;
		}
		if ((species.length > 0) && (species.substr(species.length -1 , 1) != " ")) species += " ";
		switch (((((w0>>>8)&7) ^ ((w1>>>8)&7)) + ((w2>>>8)&3))&7)
		{
			case 0: 
				species += "Rodents";
				break;
			case 1: 
				species += "Frogs";
				break;
			case 2: 
				species += "Lizards";
				break;
			case 3: 
				species += "Lobsters";
				break;
			case 4: 
				species += "Birds";
				break;
			case 5: 
				species += "Humanoids";
				break;
			case 6: 
				species += "Feline";
				break;
			case 7: 
				species += "Insects";
				break;
		}
	}
	return species;
}

function getGovernment(w0, w1, w2)
{
	switch ((w1>>>3)&7)
	{
		case 0: 
			return "Anarchy";
		case 1: 
			return "Feudal";
		case 2: 
			return "Multi-Government";
		case 3: 
			return "Dictatorship";
		case 4: 
			return "Communist";
		case 5: 
			return "Confederacy";
		case 6: 
			return "Democracy";
		case 7: 
			return "Corporate State";
	}
	return "";
}

function getEconomy(w0, w1, w2)
{
	var e = (w0>>>8)&7;
	if (((w1>>>3)&7) < 2) e = e|2;
	switch (e)
	{
		case 0: 
			return "Rich Industrial";
		case 1: 
			return "Average Industrial";
		case 2: 
			return "Poor Industrial";
		case 3: 
			return "Mainly Industrial";
		case 4: 
			return "Mainly Agricultural";
		case 5: 
			return "Rich Agricultural";
		case 6: 
			return "Average Agricultural";
		case 7: 
			return "Poor Agricultural";
	}
	return "";
}

function getTechLevel(w0, w1, w2)
{
	var e = (w0>>>8)&7;
	var g = (w1>>>3)&7;
	if (g < 2) e = e|2;
	return (~e&7) + ((w1>>>8)&3) + (g>>>1) + (g&1) + 1;
}

function getPop(w0, w1, w2)
{
	var e = (w0>>>8)&7;
	var g = (w1>>>3)&7;
	if (g < 2) e = e|2;
	var tl = (~e&7) + ((w1>>>8)&3) + (g>>>1) + (g&1);
	return (tl*4 + g + e + 1)/10;
}

function getProductivity(w0, w1, w2)
{
	var e = (w0>>>8)&7;
	var g = (w1>>>3)&7;
	if (g < 2) e = e|2;
	var tl = (~e&7) + ((w1>>>8)&3) + (g>>>1) + (g&1);
	var p = (tl*4 + g + e + 1)/10;
	return Math.round(((~e&7) + 3) * (g + 4) * p * 80, 0);
}

function getRadius(w0, w1, w2)
{
	return ((w2&3840)+2816) + (w1>>>8);
}

function getDescription(w0, w1, w2)
{
	//setup static variables
	if (typeof getDescription.lookups == 'undefined') getDescription.lookups = [["fabled", "notable", "well known", "famous", "noted"], ["very", "mildly", "most", "reasonably", ""], ["ancient", "[20]", "great", "vast", "pink"], ["[29] [28] plantations", "mountains", "[27]", "[19] forests", "oceans"], ["shyness", "silliness", "mating traditions", "loathing of [5]", "love for [5]"], ["food blenders", "tourists", "poetry", "discos", "[13]"], ["talking tree", "crab", "bat", "lobst", "%R"], ["beset", "plagued", "ravaged", "cursed", "scourged"], ["[21] civil war", "[26] [23] [24]s", "[26] disease", "[21] earthquakes", "[21] solar activity"], ["its [2] [3]", "the %I [23] [24]", "its inhabitants' [25] [4]", "[32]", "its [12] [13]"], ["juice", "brandy", "water", "brew", "gargle blasters"], ["%R", "%I [24]", "%I %R", "%I [26]", "[26] %R"], ["fabulous", "exotic", "hoopy", "unusual", "exciting"], ["cuisine", "night life", "casinos", "sit coms", "[32]"], ["%H", "The planet %H", "The world %H", "This planet", "This world"], ["n unremarkable", " boring", " dull", " tedious", " revolting"], ["planet", "world", "place", "little planet", "dump"], ["wasp", "moth", "grub", "ant", "%R"], ["poet", "arts graduate", "yak", "snail", "slug"], ["tropical", "dense", "rain", "impenetrable", "exuberant"], ["funny", "weird", "unusual", "strange", "peculiar"], ["frequent", "occasional", "unpredictable", "dreadful", "deadly"], ["[1] [0] for [9]", "[1] [0] for [9] and [9]", "[7] by [8]", "[1] [0] for [9] but [7] by [8]", "a[15] [16]"], ["[26]", "mountain", "edible", "tree", "spotted"], ["[30]", "[31]", "[6]oid", "[18]", "[17]"], ["ancient", "exceptional", "eccentric", "ingrained", "[20]"], ["killer", "deadly", "evil", "lethal", "vicious"], ["parking meters", "dust clouds", "ice bergs", "rock formations", "volcanoes"], ["plant", "tulip", "banana", "corn", "%Rweed"], ["%R", "%I %R", "%I [26]", "inhabitant", "%I %R"], ["shrew", "beast", "bison", "snake", "wolf"], ["leopard", "cat", "monkey", "goat", "fish"], ["[11] [10]", "%I [30] [33]", "its [12] [31] [33]", "[34] [35]", "[11] [10]"], ["meat", "cutlet", "steak", "burgers", "soup"], ["ice", "mud", "Zero-G", "vacuum", "%I ultra"], ["hockey", "cricket", "karate", "polo", "tennis"]];
	if (typeof getDescription.digrams == 'undefined') getDescription.digrams = "ABOUSEITILETSTONLONUTHNOALLEXEGEZACEBISOUSESARMAINDIREA'ERATENBERALAVETIEDORQUANTEISRION"
	
	var str = "[14] is [22].";
	var a = w1, b = w2, c = 0, new_a, new_b, p, p2, token, rep;
	var count = 0;

	function descriptionTwist()
	{
		new_a = (b + (a&65280) + (((a&255)<<1)&255) + ((a>>>7)&1))&65535;
		new_b = (a&65280) + (((a&255)<<1)&255);
		a = new_a;
		b = new_b;
	}

	function descriptionRandomName()
	{
		var str2 = "";
		descriptionTwist();
		var n = ((a>>>8)&3) + 1;
		for (var z = 0; z < n; z++)
		{
			descriptionTwist();
			str2 += getDescription.digrams.substr(((a>>>9)&31)*2, 2);
		}
		return str2;
	}

	while ((str.indexOf("[") >= 0) && (count < 40))
	{
		descriptionTwist();
		c = Math.min(Math.floor(a/13056),4);
		p = str.indexOf("[");
		p2 = str.indexOf("]", p);
		token = str.substring(p, p2+1);
		rep = getDescription.lookups[parseInt(str.substring(p+1, p2))][c];
		str = str.replace(token, rep); //single replace
		count++;
	}

	if (str.indexOf("%R") >=0)
	{
		rep = descriptionRandomName();
		str = stringReplaceAll(str, "%R", rep.substr(0,1) + rep.substr(1).toLowerCase());
	}

	if (str.indexOf("%H") >=0)
	{
		rep = getSystemName(w0, w1, w2);
		str = stringReplaceAll(str, "%H", rep.substr(0,1) + rep.substr(1).toLowerCase());
	}

	if (str.indexOf("%I") >=0)
	{
		rep = getSystemName(w0, w1, w2);
		switch (rep.substr(rep.length - 1, 1))
		{
			case "A":
				rep += "N";
				break;
			case "E":
				rep += "SE";
				break;
			case "I":
				rep += "AN";
				break;
			case "O":
				rep += "ESE";
				break;
			case "U":
				rep += "AN";
				break;
			default:
				rep += "IAN";
				break;
		}
		str = stringReplaceAll(str, "%I", rep.substr(0,1) + rep.substr(1).toLowerCase());
	}

	str = stringReplaceAll(str, "  ", " ");

	return str;
}

function stringReplaceAll(target_string, replace_text, with_text)
{
	var str = target_string.concat("");
	var out_str = "";

	while (str.indexOf(replace_text) >= 0)
	{
		out_str += str.substring(0, str.indexOf(replace_text)) + with_text;
		str = str.substr(str.indexOf(replace_text) + replace_text.length);
	}
	out_str += str;
	return out_str;
}

function getMaxZ(galArray)
{
	var max_z;

	for (var idx = 0; idx < galArray.length; idx++)
	{
		if ((max_z === undefined) || (max_z < galArray[idx][2])) max_z = galArray[idx][2];
	}

	return max_z;
}

function getMinZ(galArray)
{
	var min_z;

	for (var idx = 0; idx < galArray.length; idx++)
	{
		if ((min_z === undefined) || (min_z > galArray[idx][2])) min_z = galArray[idx][2];
	}

	return min_z;
}

function checkForOverlapSystems(galArray, systemNum, x, y, z)
{
	var new_list = [];
	for (var idx = 0; idx < galArray.length; idx++)
	{
		if ((galArray[idx][0] == x) && (galArray[idx][1] == y) && (Math.abs(galArray[idx][2] - z) <= 4) && (idx != systemNum))
		{
			//Add to list
			new_list.push(idx);
		}
	}
	return new_list;
}

function dedupedOverlapsList(galArray)
{
	var list = [];
	var min1, newList, list2;
	for(var idx = 0; idx < galArray.length; idx++)
	{
		if (galArray[idx][3] !== null) 
		{
			var list1 = galArray[idx][3].slice(0);
			list1.push(idx); // add self to list as this won't be in list
			min1 = Math.min.apply(null, list1);
			newList = true;
			for(var idx2 = 0; idx2 < list.length; idx2++)
			{
				list2 = list[idx2][0];
				if (Math.min.apply(null, list2) == min1)
				{
					newList = false;
					break;
				}
			}
			if (newList) list.push([list1.sort(), null, null, null]);
		}
	}
	return list;
}

function RouteElement(inLocation, inParent, inCost, inDistance, inTime)
{
	this.location = inLocation;
	this.parent = inParent;
	this.cost = inCost;
	this.distance = inDistance;
	this.time = inTime;
}

function routeFromSystem(start, goal, optimizeBy, galArray)
{
	var i, j;
	
	if (start > 255 || goal > 255) return null;
	
	var neighbours = findAllLinkedSystems(galArray);
	var cheapest = new Array ( 256 );
	for (i = 0; i < 256; i++) cheapest[i] = null;
	
	var maxCost = (optimizeBy == "time") ? 256 * (7 * 7) : 256 * (7 * 256 + 7);

	var curr = [];
	cheapest[start] = new RouteElement(start, -1, 0, 0, 0);
	curr.push(cheapest[start]);

	next = [];
	while (curr.length != 0)
	{
		for (i = 0; i < curr.length; i++) 
		{
			var ns = neighbours[curr[i].location];
			for (j = 0; j < ns.length; j++)
			{
				var ce = cheapest[curr[i].location];
				var n = ns[j][0];
				var c = ce.location;

				var lastDistance = distanceBetween(c, n, galArray);
				var lastTime = lastDistance * lastDistance;

				var distance = ce.distance + lastDistance;
				var time = ce.time + lastTime;
				var cost = ce.cost + ((optimizeBy == "time") ? lastTime : 7 * 256 + lastDistance);

				if (cost < maxCost && (cheapest[n] === null || cheapest[n].cost > cost)) 
				{
					var e = new RouteElement(n, c, cost, distance, time);
					cheapest[n] = e;
					next.push(e);

					if (n == goal && cost < maxCost) 
					{
						maxCost = cost;
					}
				}
			}
		}
		curr = next.slice(0); //deepcopy
		next.length = 0; //remove all entries
	}
	
	if (!cheapest[goal]) return null;
	
	var route = [];
	var e = cheapest[goal];
	for (;;)
	{
		route.splice(0, 0, e.location);
		if (e.parent == -1) break;
		e = cheapest[e.parent];
	}
	
	return [route, Math.round(cheapest[goal].distance*100)/100, Math.round(cheapest[goal].time*100)/100];

}

function searchSystemsInfo(parameter, searchValue)
{
	var list = [];
	for(var idx = 0; idx < galArray.length; idx++)
	{
		if ((galArray[idx][4][parameter]+" ").substr(0, searchValue.length) == searchValue.toUpperCase()) list.push(idx);
	}
	return list;
}

function calcOverlapSepDirections()
{
	for (var idx = 0; idx < galOverlaps.length; idx++)
	{
		var system1 = galOverlaps[idx][0][0];
		var angles = [];
		var links = findLinkedSystems(system1, galArray);
		var links1 = [];
		for (var s = 0; s < links.length; s++)
		{
			links1.push(links[s][0]);
		}
		var links2 = links1.slice(0);
		for (var s = 0; s < links1.length; s++)
		{
			var inList = galOverlaps[idx][0].indexOf(links1[s]);
			if (inList > -1) 
			{
				pos = links2.indexOf(links1[s]);
				links2.splice(pos, 1);
			}
		}
		for (var s = 0; s < links2.length; s++)
		{
			var dy = galArray[links2[s]][1]-galArray[system1][1];
			var dx = galArray[links2[s]][0]-galArray[system1][0];
			var angle;
			if (dx == 0) angle = Math.PI/2;
			else angle = Math.atan(dy/dx); // actually want 3rd and 4th quadrant angles to map to 180 degrees round.
			angles.push(angle);
		}
		angles.sort();
		var maxgap = 0;
		var maxgappos = null;
		for (var a = 0; a < angles.length; a++)
		{
			var gap = 0;
			if (a == (angles.length - 1)) gap = angles[0] + Math.PI - angles[a];
			else angles[a + 1] - angles[a];
			if (gap > maxgap) 
			{
				maxgap = gap;
				maxgappos = a;
			}	
		}
		galOverlaps[idx][1] = [galArray[system1][0], galArray[system1][1]];
		galOverlaps[idx][2] = angles[maxgappos] + (maxgap / 2);
		if (galOverlaps[idx][2] > Math.PI/2) galOverlaps[idx][2] -= Math.PI;

		//find system with highest z
		var sys_mh = null;
		var z_val = null;
		for (var s = 0; s < galOverlaps[idx][0].length; s++)
		{
			var sys = galOverlaps[idx][0][s];
			if (z_val === null)
			{
				sys_mh = sys;
				z_val = galArray[sys][2];
			}
			else
			{
				if (galArray[sys][2] > z_val)
				{
					sys_mh = sys;
					z_val = galArray[sys][2];
				}
			}
		}
		galOverlaps[idx][3] = sys_mh;
	
	}
}

