/*

personalities.js

Script to make special personalities appear in the Ooniverse.


Oolite
Copyright © 2004-2009 Giles C Williams and contributors

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
MA 02110-1301, USA.

*/
"use strict";


this.name = "personalities";
this.author = "Commander McLane, modified by Alnivel";
this.copyright = "� 2009 Commander McLane";
this.description = "Script for spawning of special personalities";
this.version = "0.9.2";

/* Params */
this.$chanceOnExitWitchspace = 65;
this.$chanceOnLaunch = 65;
/* Params end */

/**
 * Add role that can appear on conditions (in centrain systems, after some event etc)
 * @param {String} shipUniqueRole 
 * @param {Function} weightFunction () -> Number
 * @param {Function}  [positionFunction] ("EXIT_WITCHSPACE"|"LAUNCH") -> Vector3D?
 */
this._addConditionalAppearance = function (shipUniqueRole, weightFunction, positionFunction) {
	if (typeof shipUniqueRole !== "string")
		throw new Error("Argument 'shipUniqueRole' must be a string");
	if (weightFunction instanceof Function === false)
		throw new Error("Argument 'weightFunction' must be a function");
	if (positionFunction !== undefined && positionFunction instanceof Function === false)
		throw new Error("Optional argument 'positionFunction' must be a function");
	this.$conditionalAppearances.push({
		role: shipUniqueRole,
		weightFunction: weightFunction,
		positionFunction: positionFunction
	});
};

/**
 * Add callback when escape pod scooped by player 
 * @param {String} personalityName 
 * @param {Function} callback (personalityName) -> void
 */
this._addActionAtPodScooped = function (personalityName, callback) {
	if (typeof personalityName !== "string")
		throw new Error("Argument 'personalityName' must be a string");
	if (callback instanceof Function === false)
		throw new Error("Argument 'callback' must be a function");
	this.$actionsAtPodScooped[personalityName] = callback;
};

/**
 * Add callback when escape pod bringed to station by player
 * @param {*} personalityName 
 * @param {*} callback 
 */
this._addActionAtPodUnloaded = function (personalityName, callback) {
	if (typeof personalityName !== "string")
		throw new Error("Argument 'personalityName' must be a string");
	if (callback instanceof Function === false)
		throw new Error("Argument 'callback' must be a function");
	this.$actionsAtPodUnloaded[personalityName] = callback;
};

/**
 * To balance the distribution of the roles of ships that can appear in any galaxy and which are only in this galaxy, 
 * only the number of ships in the corresponding role is used, without taking into account the weight of the role.
 * 
 * Such an error in one or two ships will not do anything particularly bad, but let there be a way to correct this, 
 * in case someone suddenly decides to use 100 ship variants with a weight of (0.01) each.
 * @param {"any"|number} galaxy 
 * @param {number} adjustment 
 */
this._adjustGalaxyRoleWeight = function (galaxy, adjustment) {
	if (galaxy !== "any" || isNaN(galaxy) || galaxy < 0 || galaxy > 7)
		throw new Error("Argument 'galaxy' must be \"any\" or number in range [0; 7]");
	if (isFinite(adjustment) === false)
		throw new Error("Argument 'adjustment' must be a finite number");

	this.$galaxyRoleAdjustments[galaxy] += adjustment;
	this.$revalAnyGalaxyPersonalityChance("adjustment");
};




this.$logging = true;

this.startUp = this.reset = function () {
	missionVariables.personalities_systemname = System.systemNameForID(Math.floor(Math.random() * 256));

	this._addConditionalAppearance("personalities_daddyhoggy",
		function weight() {
			return galaxyNumber == 0 && system.ID == 147 ? 100 : 0;
		}/* EXAMPLE
		,		
		function position(type) {
			if(type === "LAUNCH")
				return undefined; // Use standart rules: spawn near station in space or via launching
			if(type === "EXIT_WITCHSPACE")
				return Math.random() < 0.5? new Vector3D() : undefined; // in half cases spawn near witchpoint
		}*/);

	// The object to hold a reference to the timer and its function
	// All these guts could be stored directly in the world script
	// But I use separate object to not mix by accident with the OXP logic
	this.$personalityObject_hesperus = {
		releaseTrumble: function () {
			this.displayTimer = null;
			player.consoleMessage(expandDescription("[personalities-hesperus-trumbleEscaped]"), 6);
			player.ship.awardEquipment("EQ_TRUMBLE");
		},
		releaseTrumbleTimer: null
	};

	this._addActionAtPodScooped("hesperus", function () {
		if (Math.random() < 0.7) {
			this.hesperusObject.releaseTrumbleTimer = new Timer(this, this.hesperusObject.releaseTrumble, 7, 0);
		}
	}.bind(this));

	this._addActionAtPodUnloaded("hesperus", function () {
		// player.addMessageToArrivalReport(expandDescription("[personalities-hesperus-captured]")); <-- already handled
		player.credits += 132;
		// Should I prevent default unload actions??
	});
};


this.$Config = {
	Name: this.name,
	Display: "Personalities",
	Alive: "$Config",
	SInt: {
		S0: {
			Name: "$chanceOnExitWitchspace",
			Min: 0,
			Max: 100,
			Def: 65,
			Desc: "After witchspace"
		},
		S1: {
			Name: "$chanceOnLaunch",
			Min: 0,
			Max: 100,
			Def: 65,
			Desc: "After launch"
		},
		Info: "Adjusting the chances of personalities to appear."
	}
};

this.startUpComplete = function () {
	let storedChanceOnExitWitchspace = missionVariables.personalities_chanceOnExitWitchspace;
	let storedChanceOnLaunch = missionVariables.personalities_chanceOnLaunch;

	this.$chanceOnExitWitchspace = storedChanceOnExitWitchspace || storedChanceOnExitWitchspace === 0 ? storedChanceOnExitWitchspace : 65;
	this.$chanceOnLaunch = storedChanceOnLaunch || storedChanceOnLaunch === 0 ? storedChanceOnLaunch : 65;

	if (worldScripts.Lib_Config)
		worldScripts.Lib_Config._registerSet(this.$Config);

	let storedScoopedPersonalities = missionVariables.personalities_scoopedPersonalities;
	this.$scoopedPersonalities = storedScoopedPersonalities ? JSON.parse(storedScoopedPersonalities) : [];

	this.$revalAnyGalaxyPersonalityChance("full");
};

this.playerWillSaveGame = function (message) {
	missionVariables.personalities_chanceOnExitWitchspace = this.$chanceOnExitWitchspace;
	missionVariables.personalities_chanceOnLaunch = this.$chanceOnLaunch;

	missionVariables.personalities_scoopedPersonalities = JSON.stringify(this.$scoopedPersonalities);
};

this.playerEnteredNewGalaxy = function(galaxyNumber) {
	this.$revalAnyGalaxyPersonalityChance("newGalaxy");
}

this.$galaxyRoleAdjustments = { "any": 0, "0": 0, "1": 0, "2": 0, "3": 0, "4": 0, "5": 0, "6": 0, "7": 0 };
this.$anyGalaxyRoleBaseWeight = 0;
this.$currentGalaxyRoleBaseWeight = 0;

this.$chanceAnyGalaxyPersonality = 100;


/**
 * 
 * @param {"full"|"newGalaxy"|"adjustment"} recalculationType 
 */
this.$revalAnyGalaxyPersonalityChance = function(recalculationType) {
	switch(recalculationType) {
		default:
		case "full":
			let shipsAnyGalaxy = Ship.keysForRole("personalities_galaxy-any");
			this.$anyGalaxyRoleBaseWeight = shipsAnyGalaxy? shipsAnyGalaxy.length : 0;
		case "newGalaxy":
			let shipsCurrentGalaxy = Ship.keysForRole("personalities_galaxy-" + galaxyNumber);
			this.$currentGalaxyRoleBaseWeight = shipsCurrentGalaxy? shipsCurrentGalaxy.length : 0;
			this.$currentGalaxyPersonalityRole =  "personalities_galaxy-" + galaxyNumber;
			this.$currentGalaxyPersonalityRoleLauncher = "personalities_launcher_galaxy-" + galaxyNumber;
		case "adjustment":			
	}

	const anyGalaxyRoleWeight 	  = this.$anyGalaxyRoleBaseWeight + this.$galaxyRoleAdjustments["any"];
	const currentGalaxyRoleWeight = this.$currentGalaxyRoleBaseWeight + this.$galaxyRoleAdjustments[galaxyNumber];

	// Just on a whim, double the chance for a galaxy-specific personality to appear 
	this.$chanceAnyGalaxyPersonality = (100*anyGalaxyRoleWeight) / (anyGalaxyRoleWeight + 2*currentGalaxyRoleWeight); 
};

//#region Personalities spawn
this.$anyGalaxyPersonalityRole =  "personalities_galaxy-any";
this.$anyGalaxyPersonalityRoleLauncher = "personalities_launcher_galaxy-any";
this.$currentGalaxyPersonalityRole; // set in this.$revalAnyGalaxyPersonalityChance
this.$currentGalaxyPersonalityRoleLauncher;

this.$delayedSpawnCallback = function (role) { // Used with timer to delay spawn
	let ship = system.addShips(role, 1);
	this.$waitUntilSpawnTimer = null;

	this.$proccessAddedShips([ship]);
};

this.shipWillExitWitchspace = function () {
	//system.addShips("personalities_disembodied", 5);
	if (Math.random() * 100 >= this.$chanceOnExitWitchspace) return;
	if (system.isInterstellarSpace) return;
	missionVariables.personalities_systemname = System.systemNameForID(Math.floor(Math.random() * 256));
	if (system.countShipsWithRole("personalities") > 0) return;

	let addedShips = null;

	let personalitiesAppearance = Math.random();
	if (personalitiesAppearance < 0.5) { // In 50% of cases try to find suitable "special" appearence
		let appearenceParams = this.$getRandomConditionalAppearance();

		if (appearenceParams) {
			let position = appearenceParams.positionFunction && appearenceParams.positionFunction("EXIT_WITCHSPACE");

			if (position)
				addedShips = system.addShips(appearenceParams.role, 1, position);
			else if (Math.random() < 5 / 8)
				addedShips = system.addShips(appearenceParams.role, 1);
			else
				addedShips = system.addShipsToRoute(appearenceParams.role, 1);

			this.$proccessAddedShips(addedShips);
			return;
		}
	}

	if (personalitiesAppearance > 0.5) {
		personalitiesAppearance -= 0.5;
	}

	let personalityRole;
	if(Math.random() * 100 <= this.$chanceAnyGalaxyPersonality) {
		personalityRole = this.$anyGalaxyPersonalityRole;
	}		
	else {
		personalityRole = this.$currentGalaxyPersonalityRole;
	}
	
	if (personalitiesAppearance < 0.25) { // 
		if (Math.random() < 5 / 8) // Spawn with delay
			this.$waitUntilSpawnTimer = new Timer(this, 
				this.$delayedSpawnCallback.bind(this, personalityRole), 
				((Math.random() * 7) + 5) // Delay
			);
		else // Spawn immediately
			addedShips = system.addShips(personalityRole, 1);
	}
	else /* 25% */ {
		addedShips = system.addShipsToRoute(personalityRole, 1);
	}

	this.$proccessAddedShips(addedShips);
};

this.shipWillLaunchFromStation = function (station) {
	//station.launchShipWithRole("personalities_disembodied")
	if (Math.random() * 100 >= this.$chanceOnLaunch) return;
	if (!station.isMainStation || system.countShipsWithRole("personalities") > 0) return;

	let addedShips = null;

	let personalitiesAppearance = Math.random();
	if (personalitiesAppearance < 0.5) { // In 50% of cases try to find suitable "special" appearence
		let appearenceParams = this.$getRandomConditionalAppearance();

		if (appearenceParams) {
			let position = appearenceParams.positionFunction && appearenceParams.positionFunction("LAUNCH");

			if (position)
				addedShips = system.addShips(appearenceParams.role, 1, position);
			else if (Math.random() < 0.5)
				addedShips = [station.launchShipWithRole(appearenceParams.role)];
			else {
				let offsetPlane = station.orientation.rotateZ(Math.random() * 2).vectorRight().multiply(Math.random() * 5000 + 2500);
				let offsetFromDock = station.vectorForward.multiply(Math.random() * 2800 + 6000);

				addedShips = system.addShips(appearenceParams.role, 1, station.position.add(offsetFromDock).add(offsetPlane));
			}				

			this.$proccessAddedShips(addedShips);
			return;
		}
	}

	if (personalitiesAppearance > 0.5) {
		personalitiesAppearance -= 0.5;
	}

	let personalityRole, personalityRoleLauncher;
	if(Math.random() * 100 <= this.$chanceAnyGalaxyPersonality) {
		personalityRole = this.$anyGalaxyPersonalityRole;
		personalityRoleLauncher = this.$anyGalaxyPersonalityRoleLauncher;
	}		
	else {
		personalityRole = this.$currentGalaxyPersonalityRole;
		personalityRoleLauncher = this.$currentGalaxyPersonalityRoleLauncher;
	}

	if (personalitiesAppearance < 0.25) {
		if (Math.random() < 0.5)
			addedShips = [station.launchShipWithRole(personalityRoleLauncher)];
		else {

			let offsetPlane = station.orientation.rotateZ(Math.random() * 2).vectorRight().multiply(Math.random() * 5000 + 2500);
			let offsetFromDock = station.vectorForward.multiply(Math.random() * 2800 + 6000);
			
			addedShips = system.addShips(personalityRole, 1, station.position.add(offsetFromDock).add(offsetPlane));
		}
			
	}
	else /* 25% */ {
		addedShips = system.addShips(personalityRole, 1);
	}
	this.$proccessAddedShips(addedShips);
};

this.$rememeberedAppearedPersonalities = new Array(5);
this.$rememeberedAppearedPersonalitiesIndex = 0;
this.$rememberAppearedPersonality = function (name) {
	if (!name)
		return;
	this.$rememeberedAppearedPersonalities[this.$rememeberedAppearedPersonalitiesIndex++] = name;
	this.$rememeberedAppearedPersonalitiesIndex %= this.$rememeberedAppearedPersonalities.length;
};
this.$isRecentlyAppearedPersonality = function (name) {
	return this.$rememeberedAppearedPersonalities.indexOf(name) !== -1;
};

this.$proccessAddedShips = function (ships) {
	if(!ships) 
		return;

	const length = ships.length;
	for (let i = 0; i < length; i++) {
		const ship = ships[i];
		const shipScriptInfo = ship && ship.scriptInfo;
		const personalityName = shipScriptInfo && shipScriptInfo.name;

		if (this.$isRecentlyAppearedPersonality(personalityName))
			ship.remove(true);
		else
			this.$rememberAppearedPersonality(personalityName);
	}
};

this.$conditionalAppearances = [];
this.$getRandomConditionalAppearance = function () {
	const appearenceCount = this.$conditionalAppearances.length;
	let cummulativeWeights = new Array(appearenceCount);
	let cummulativeWeight = 0;

	let appearenceIndex = appearenceCount;
	while (appearenceIndex--) {
		let appearanceParams = this.$conditionalAppearances[appearenceIndex];
		let roleWeight = +appearanceParams.weightFunction();

		cummulativeWeight += roleWeight > 0 ? roleWeight : 0; // NaN > 0 === false, so ok
		cummulativeWeights[appearenceIndex] = cummulativeWeight;
	}

	const randomValue = Math.random() * cummulativeWeight;

	appearenceIndex = appearenceCount;
	while (appearenceIndex--) {
		if (randomValue < cummulativeWeights[appearenceIndex])
			return this.$conditionalAppearances[appearenceIndex];
	}

	return null;
};

//#endregion

//#region Personalities escape pod capturing
this.$actionsAtPodScooped = {};
this.$actionsAtPodUnloaded = {};

this.$scoopedPersonalities = [];
this.shipScoopedOther = function (whom) {
	// if(whom.crew && cargo.crew.length > 0) {
	// 	const character = whom.crew[0];

	// }

	let name = whom.$$personalities_name;
	let capturedMessage = whom.$$personalities_capturedMessage;

	if (name) {
		let entry = { name: name, capturedMessage: capturedMessage };
		this.$scoopedPersonalities.push(entry);

		let shipWasDumpedHandler = whom.script.shipWasDumped;
		whom.script.shipWasDumped = this.$personalityPodWasDumped.bind(whom, this.$scoopedPersonalities, entry, shipWasDumpedHandler);

		let callback = this.$actionsAtPodScooped[name];
		if (callback)
			callback(name);
	}
};

this.$personalityPodWasDumped = function (scoopedPersonalities, personalityEntry, shipWasDumpedHandler, dumper) {
	let index = scoopedPersonalities.indexOf(personalityEntry);
	if (index !== -1) {
		scoopedPersonalities.splice(index, 1);
	}
	else {
		log("Personalities", "Dumped pod for " + personalityEntry.name + " was not in $scoopedPersonalities.");
	}

	if (shipWasDumpedHandler)
		shipWasDumpedHandler.call(this, dumper);
};

this.shipWillDockWithStation = function () { // Should replace this with a character script?
	for (let i = 0; i < this.$scoopedPersonalities.length; i++) {
		let personalitity = this.$scoopedPersonalities[i];
		if (personalitity.capturedMessage)
			player.addMessageToArrivalReport(personalitity.capturedMessage);

		let callback = this.$actionsAtPodUnloaded[personalitity.name];
		if (callback)
			callback(personalitity.name);

	}
	// if (this.$hesperusCaptured) {
	// 	player.addMessageToArrivalReport(expandDescription("[personalities-hesperus-captured]"));
	// 	player.credits += 132;
	// 	this.$hesperusCaptured = false;
	// }
	this.$scoopedPersonalities.length = 0;
};
//#endregion

//#region Plist ship script
this.shipBeingAttacked = function (whom) {
	if (!whom) return;
	if (whom.hasRole("personalities")) {
		whom.script.hitcounter--;
		if (whom.script.hitcounter < 1) {
			whom.script.$broadcastMessage("hitting");
			whom.script.hitcounter = Math.ceil(Math.random() * 10) + 10;
		}
	}
};

this.shipDestroyedTarget = function (target) {
	if (target.hasRole("personalities") && !missionVariables.personalities_killed_name) {
		if (target.hasRole("personalities_privateer") && Math.random() < 0.5) return;
		missionVariables.personalities_killed_name = target.scriptInfo.displayName;
	}
};

this.alertConditionChanged = function (newCondition, oldCondition) {
	if (newCondition == 3 && oldCondition != 3) {
		system.shipsWithRole("personalities", player.ship, 25600).forEach(function (ship) { ship.script.alertConditionChanged(newCondition, oldCondition); });
	}
};
//#endregion

//#region PriorityAI script
this.$messageTypeAlternativeNames = { //!!!!!!!!! NOT FINISHED !!!!!!!!!
	"attack": "oolite_beginningAttack", // beginningFight?
	"flee": "oolite_startFleeing",
	"jump": "oolite_engageWitchspaceDriveFlee",
	"kill": "oolite_killedTarget",
	"witchspace": "oolite_engageWitchspaceDrive",

	// personalitiesOXP prefix
	"greet": "personalitiesOXP_greet", // new
	"chatter": "personalitiesOXP_chatter",
	"launch": "personalitiesOXP_launch",
	"attacked": "personalitiesOXP_attacked", //?
	"dead": "personalitiesOXP_dead", 
	"captured": "personalitiesOXP_captured", // Well, it's not comms message, but I still should somehow proccess it
};

this.$getNormalizedCommunicationsKey = function (messageType) {
	const unaliasedName = this.$messageTypeAlternativeNames[messageType];
	if (unaliasedName)
		return unaliasedName;
	if (messageType.indexOf("oolite_") === 0 || messageType.indexOf("personalitiesOXP_") === 0)
		return message;
	else if (messageType.charAt(0) === "_")
		return messageType.substring(1);
	else 
		return "oolite_" + messageType;
};

this.$registredCommsPersonalities = [];
this._registerCommsOfPersonalitiesWithPriorityAI = function (personalities) {
	const libPriorityAI = worldScripts["oolite-libPriorityAI"];

	for (let i = 0; i < personalities.length; i++) {
		const personalityName = personalities[i];

		const alreadyRegistred = this.$registredCommsPersonalities.indexOf(personalityName) !== -1;
		if(alreadyRegistred){
			log("Personalities_CommsDebug", "Info for " + personalityName + ": comms already registred");
			continue;
		} 
		else {
			this.$registredCommsPersonalities.push(personalityName);
		}
		
		const personalityNameNoFallback = "_" + personalityName; // prevent fall back to generic
		const typesAsString = expandDescription("[personalities-" + personalityName + "-messageTypes]");
		if (typesAsString.indexOf("-messageTypes") !== -1) {
			log("Personalities_CommsDebug", "Error for " + personalityName + ": Failed to find comm message types from description.plist");
			continue;
		}

		const types = typesAsString.split(" ");
		for (let typeIndex = 0; typeIndex < types.length; typeIndex++) {
			const messageType = types[typeIndex];
			const communicationsKey = this.$getNormalizedCommunicationsKey(messageType);

			log("Personalities_CommsDebug", "Info for " + personalityNameNoFallback + ": Set [personalities-" + personalityName + "-"+ messageType + "] for key " + communicationsKey);
			libPriorityAI._setCommunication("_personalitiesOXP", personalityNameNoFallback,
				communicationsKey,
				"[personalities-" + personalityName + "-" + messageType + "]"
			);

		}

	}
};
//#endregion